# phredator/cli.py
import argparse
import json
from phredator.parser.fastqc_parser import FastQCParser


def _run_parse(input_path: str, output: str, verbose: bool) -> int:
    if verbose:
        print(f"[INFO] Parsing FastQC data from: {input_path}")

    parser_obj = FastQCParser(input_path)
    report = parser_obj.parse()

    # Convert dataclass to dict for JSON
    report_dict = {
        "sample_name": report.sample_name,
        "per_base_quality": report.per_base_quality,
        "gc_content": report.gc_content,
        "duplication_levels": report.duplication_levels,
        "adapter_content": report.adapter_content,
        "overrepresented_sequences": report.overrepresented_sequences,
    }

    with open(output, "w") as f:
        json.dump(report_dict, f, indent=4)

    if verbose:
        print(f"[INFO] Report saved to: {output}")

    return 0


def main() -> int:
    parser = argparse.ArgumentParser(
        description="Phredator: Rule-based FastQC parser and QC analyzer"
    )

    subparsers = parser.add_subparsers(dest="command", required=True)

    parse_parser = subparsers.add_parser("parse", help="Parse FastQC data")
    parse_parser.add_argument("input_path", type=str, help="Path to FastQC zip or folder")
    parse_parser.add_argument("--output", type=str, default="phredator_report.json",
                              help="Output file (JSON)")
    parse_parser.add_argument("--verbose", action="store_true", help="Show progress")

    args = parser.parse_args()

    if args.command == "parse":
        return _run_parse(args.input_path, args.output, args.verbose)

    parser.print_help()
    return 1


if __name__ == "__main__":
    raise SystemExit(main())