"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResourceExtractor = exports.ResourceExtractorShareMethod = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ssm_1 = require("aws-cdk-lib/aws-ssm");
const cfnStore_1 = require("./cfnStore");
const flattener_1 = require("./flattener");
const resourceTransformer_1 = require("./resourceTransformer");
/**
 * The available value sharing methods to pass values from the extracted stack
 * onto the original stack(s).
 */
var ResourceExtractorShareMethod;
(function (ResourceExtractorShareMethod) {
    ResourceExtractorShareMethod["CFN_OUTPUT"] = "CFN_OUTPUT";
    ResourceExtractorShareMethod["SSM_PARAMETER"] = "SSM_PARAMETER";
    ResourceExtractorShareMethod["API_LOOKUP"] = "API_LOOKUP";
})(ResourceExtractorShareMethod = exports.ResourceExtractorShareMethod || (exports.ResourceExtractorShareMethod = {}));
/**
 * This Aspect takes a CDK application, all synthesized CloudFormationStackArtifact,
 * a value share method, and a list of Cloudformation resources that should be
 * pulled out of the main CDK application, which should be synthesized to a
 * cloudformation template that an external team (e.g. security team) to deploy,
 * and adjusting the CDK application to reference pre-created resources already pulled out
 *
 * @example
    const app = App()
    const stack = new Stack(app, 'MyStack');
    extractedStack = new Stack(app, 'ExtractedStack');
    const synthedApp = app.synth();

    Aspects.of(app).add(new ResourceExtractor({
      extractDestinationStack: extractedStack,
      stackArtifacts: synthedApp.stacks,
      valueShareMethod: ResourceExtractorShareMethod.CFN_OUTPUT,
      resourceTypesToExtract: [
        'AWS::IAM::Role',
        'AWS::IAM::Policy',
        'AWS::IAM::ManagedPolicy',
        'AWS::IAM::InstanceProfile',
      ],
    });
    app.synth({ force: true });
 */
class ResourceExtractor {
    constructor(props) {
        this.valueShareMethod = ResourceExtractorShareMethod.CFN_OUTPUT;
        /** Save props */
        this.extractDestinationStack = props.extractDestinationStack;
        this.resourceTypesToExtract = props.resourceTypesToExtract;
        if (props.valueShareMethod) {
            this.valueShareMethod = props.valueShareMethod;
        }
        aws_cdk_lib_1.Annotations.of(this.extractDestinationStack).addWarning('❗ ResourceExtractor is in experimental mode. \
      Please be sure to validate synthesized assets prior to deploy. \
      Please open any issues found at https://github.com/cdklabs/cdk-enterprise-iac/issues');
        /** Initialize CfnStore to save templates and mappings */
        this.cfn = new cfnStore_1.CfnStore({
            extractedStackName: props.extractDestinationStack.stackName,
            region: props.extractDestinationStack.region,
            stackArtifacts: props.stackArtifacts,
            valueShareMethod: this.valueShareMethod,
        });
        /** Initialize resource transformer */
        this.resourceTransformer = new resourceTransformer_1.ResourceTransformer({
            cfnStore: this.cfn,
            additionalTransforms: props.additionalTransforms,
        });
        /** Save resources that are not getting extracted */
        this.logicalIdsNotGettingExtracted =
            this.getLogicalIdsNotGettingExtracted();
    }
    /** Entrypoint */
    visit(node) {
        // Ignore the extracted stack
        const anyNode = node;
        if ('stack' in anyNode &&
            anyNode.stack.toString() === this.extractDestinationStack.stackName) {
            return;
        }
        // Process all resources
        if (node instanceof aws_cdk_lib_1.CfnResource) {
            if (this.resourceTypesToExtract.includes(node.cfnResourceType)) {
                this.processCfnResource(node);
            }
        }
    }
    /**
     * Takes in a CfnResource object and processes it by recreating it in
     * the Security Stack and deleting it from the stack in which it came
     * from.
     *
     * @param node the CfnResource to process
     */
    processCfnResource(node) {
        const stackName = node.stack.stackName;
        const logicalId = node.stack.resolve(node.logicalId);
        const cfnResourceJson = this.cfn.templates[stackName].Resources[logicalId];
        const props = cfnResourceJson.Properties;
        // Create a new object that contains other allowed top level properties
        // other than `Type` and `Properties` so that we can ensure that they
        // are correctly set on the extracted object
        const allowedTopLevelKeys = ['UpdateReplacePolicy', 'DeletionPolicy'];
        const topLevelProps = Object.keys(cfnResourceJson)
            .filter((k) => allowedTopLevelKeys.includes(k))
            .reduce((obj, key) => {
            obj[key] = cfnResourceJson[key];
            return obj;
        }, {});
        // Check if props include references to resources that _aren't_ being
        // extracted
        const flattenedProps = flattener_1.Flattener.flattenObject(props);
        const propsToAdjust = Object.keys(flattenedProps).filter((key) => Object.keys(this.logicalIdsNotGettingExtracted).includes(flattenedProps[key]));
        // If properties being extracted to another stack contain
        // references to resources in the original stack, we need to adjust
        let modifiedProps = {};
        if (propsToAdjust) {
            modifiedProps = this.modifyExtractedResourceProperties({
                props,
                propsToAdjust,
            });
        }
        // Add to extracted stack
        if (!this.extractDestinationStack.node.tryFindChild(node.stack.resolve(logicalId))) {
            const r = new aws_cdk_lib_1.CfnResource(this.extractDestinationStack, logicalId, {
                type: node.cfnResourceType,
                properties: modifiedProps || props,
            });
            Object.keys(topLevelProps).forEach((k) => {
                r.addOverride(k, topLevelProps[k]);
            });
        }
        // Look in the node's stack for any refs to the resource being extracted
        const foundRefsAllStacks = Object.keys(this.cfn.flatTemplates).filter((key) => this.cfn.flatTemplates[key] === logicalId);
        const foundRefs = foundRefsAllStacks.filter((k) => k.split('.')[0] === stackName);
        // loop through resources in stack with Cfn intrinsic functions
        // referencing extracted resources
        for (const foundRef of foundRefs) {
            // See if the found ref is also getting extracted. Ignore if so.
            if (this.isRefAlsoGettingExtracted(foundRef))
                continue;
            // Get the CfnResource that is referencing this extracted resource
            const foundRefNode = this.getResourceFromFoundRef(node, foundRef);
            // Figure out the pattern of how extracted resource is being referenced
            // e.g. using `Fn::GetAtt` or `Ref`
            const exportValue = this.cfn.determineExportValue(node, foundRef);
            if (exportValue) {
                // Generate export in ExportedStack, and return the `Fn::ImportValue`
                // method to use when referencing in the App stack
                const importValue = this.exportValue(node.stack, this.extractDestinationStack.resolve(logicalId), exportValue);
                // Override any ref to extracted resource
                this.overrideFoundRefWithImportValue({
                    foundRefNode,
                    importValue,
                    flattenedKey: foundRef,
                });
            }
            // Remove any DependsOn references
            if (foundRefNode instanceof aws_cdk_lib_1.CfnResource) {
                this.deletionOverrideDependsOn({
                    foundRefNode,
                    flattenedKey: foundRef,
                });
            }
        }
        /** Delete from original stack */
        const removed = node.stack.node.tryRemoveChild(node.node.id);
        if (!removed) {
            const parent = node.node.scope?.node;
            parent?.tryRemoveChild(node.node.id);
        }
    }
    /**
     * This will adjust properties in extracted resources that have
     * CloudFormation intrinsic functions referencing resources in the original
     * stack.
     *
     * Most commonly found in IAM policies that scope IAM actions to specific
     * resources in the Stack the policy is being extracted from. In this case
     * it will take a look at the resource type and transform into a partial ARN.
     *
     * @param props modifyExtractedResourcePropertiesProps
     * @returns Json of modified properties, including partial wildcard matchers
     */
    modifyExtractedResourceProperties(props) {
        let modifiedProps = props.props;
        for (const key in props.propsToAdjust) {
            if (Object.prototype.hasOwnProperty.call(props.propsToAdjust, key)) {
                const resourceLogicalIdToReplace = flattener_1.Flattener.getValueByPath(props.props, props.propsToAdjust[key]);
                const partial = this.resourceTransformer.toPartial(resourceLogicalIdToReplace);
                const splitKey = props.propsToAdjust[key].split('.');
                if (splitKey.slice(-1)[0] == 'Ref') {
                    flattener_1.Flattener.setToValue(modifiedProps, splitKey.slice(0, -1).join('.'), partial);
                }
                else if (splitKey.slice(-2)[0] == 'Fn::GetAtt') {
                    flattener_1.Flattener.setToValue(modifiedProps, splitKey.slice(0, -2).join('.'), partial);
                }
            }
        }
        return modifiedProps;
    }
    /**
     * This takes the flattened key of the resource that's referencing an
     * extracted resource via Cfn intrinsic function, the CfnResource itself
     * (e.g. CfnFunction) as well as the import value (from
     * determineExportValue), and adds a Property override.
     *
     * @param props overrideFoundRefWithImportValueProps
     */
    overrideFoundRefWithImportValue(props) {
        // find property to override
        const splitKey = props.flattenedKey.split('.');
        let propertyOverridePath;
        if (splitKey.slice(-1)[0] == 'Ref') {
            propertyOverridePath = splitKey.slice(4, -1).join('.');
        }
        else if (splitKey.slice(-2)[0] == 'Fn::GetAtt') {
            propertyOverridePath = splitKey.slice(4, -2).join('.');
        }
        else {
            propertyOverridePath = 'notFound';
        }
        if (this.valueShareMethod == ResourceExtractorShareMethod.CFN_OUTPUT) {
            const newValue = props.foundRefNode.stack.resolve(props.importValue);
            if (props.foundRefNode instanceof aws_cdk_lib_1.CfnOutput) {
                props.foundRefNode.value = newValue;
            }
            else if (props.flattenedKey.includes('Fn::Join')) {
                // Get Fn::Join path and save new value to fnJoins table
                const fnJoinFlatJsonPath = props.flattenedKey.split('.Fn::Join')[0] + '.Fn::Join';
                const fnJoinOverridePath = propertyOverridePath.split('.Fn::Join')[0];
                this.cfn.fnJoins[props.flattenedKey] = newValue;
                // Rebuild Fn::Join object
                const joined = this.cfn.rebuildFnJoin(fnJoinFlatJsonPath);
                props.foundRefNode.addPropertyOverride(fnJoinOverridePath, joined);
            }
            else {
                props.foundRefNode.addPropertyOverride(propertyOverridePath, newValue);
            }
        }
        else if (this.valueShareMethod == ResourceExtractorShareMethod.SSM_PARAMETER) {
            const newValue = aws_ssm_1.StringParameter.valueFromLookup(props.foundRefNode.stack, props.importValue);
            if (props.foundRefNode instanceof aws_cdk_lib_1.CfnOutput) {
                props.foundRefNode.value = newValue;
            }
            else if (props.flattenedKey.includes('Fn::Join')) {
                // Get Fn::Join path and save new value to fnJoins table
                const fnJoinFlatJsonPath = props.flattenedKey.split('.Fn::Join')[0] + '.Fn::Join';
                const fnJoinOverridePath = propertyOverridePath.split('.Fn::Join')[0];
                this.cfn.fnJoins[props.flattenedKey] = newValue;
                // Rebuild Fn::Join object
                const joined = this.cfn.rebuildFnJoin(fnJoinFlatJsonPath);
                props.foundRefNode.addPropertyOverride(fnJoinOverridePath, joined);
            }
            else {
                props.foundRefNode.addPropertyOverride(propertyOverridePath, newValue);
            }
        }
        else if (this.valueShareMethod == ResourceExtractorShareMethod.API_LOOKUP) {
            const importValue = props.foundRefNode.stack.resolve(props.importValue)['Fn::ImportValue'];
            const newValue = this.cfn.extractedStackExports[importValue] ||
                `dummy-value-for-${importValue}`;
            if (props.foundRefNode instanceof aws_cdk_lib_1.CfnOutput) {
                props.foundRefNode.value = newValue;
            }
            else if (props.flattenedKey.includes('Fn::Join')) {
                // Get Fn::Join path and save new value to fnJoins table
                const fnJoinFlatJsonPath = props.flattenedKey.split('.Fn::Join')[0] + '.Fn::Join';
                const fnJoinOverridePath = propertyOverridePath.split('.Fn::Join')[0];
                this.cfn.fnJoins[props.flattenedKey] = newValue;
                // Rebuild Fn::Join object
                const joined = this.cfn.rebuildFnJoin(fnJoinFlatJsonPath);
                props.foundRefNode.addPropertyOverride(fnJoinOverridePath, joined);
            }
            else {
                props.foundRefNode.addPropertyOverride(propertyOverridePath, newValue);
            }
        }
    }
    /**
     * Remove a `DependsOn` reference when not needed
     *
     * @param props
     */
    deletionOverrideDependsOn(props) {
        const splitKey = props.flattenedKey.split('.');
        if (splitKey.slice(-2)[0] == 'DependsOn') {
            props.foundRefNode.addDeletionOverride(`DependsOn.${splitKey.slice(-1)[0]}`);
        }
    }
    /**
     * Finds the CDK resource for a given flattened key by looking it up in the
     * context of the provided node's stack.
     *
     * @param node
     * @param flattendKey
     * @returns CfnResource
     */
    getResourceFromFoundRef(node, flattendKey) {
        const foundRefLogicalId = flattendKey.split('.')[2];
        return node.stack.node.findAll().find((x) => {
            if ('stack' in x && 'logicalId' in x) {
                return x.stack.resolve(x.logicalId) == foundRefLogicalId;
            }
            else {
                return undefined;
            }
        });
    }
    /**
     * Given a flattened key, determine if the resouce is also getting
     * extracted so we know whether or not to adjust CloudFormation Intrinsic
     * functions.
     *
     * @param flattendKey
     * @returns boolean
     */
    isRefAlsoGettingExtracted(flattendKey) {
        const splitKey = flattendKey.split('.');
        const pathToType = flattener_1.Flattener.getValueByPath(this.cfn.templates, splitKey.slice(0, 3).concat(['Type']).join('.'));
        return this.resourceTypesToExtract.includes(pathToType);
    }
    /**
     * Get a list of logical ids of resources _not_ getting exported.
     *
     * @returns flat json in the form of `{ LogicalId: 'ResourceType' }`
     */
    getLogicalIdsNotGettingExtracted() {
        const logicalIds = {};
        for (const key in this.cfn.flatTemplates) {
            if (Object.prototype.hasOwnProperty.call(this.cfn.flatTemplates, key)) {
                const splitKey = key.split('.');
                if (splitKey.slice(-1)[0] == 'Type' &&
                    splitKey.slice(1, 2)[0] == 'Resources') {
                    if (!this.resourceTypesToExtract.includes(this.cfn.flatTemplates[key])) {
                        logicalIds[splitKey.slice(-2)[0]] = this.cfn.flatTemplates[key];
                    }
                }
            }
        }
        return logicalIds;
    }
    /**
     * Exports a value using a consistent standard for the different value
     * share methods.
     *
     * @param stack - the CDK `Stack` object for the resource to import
     * @param name - the name of the export, normally the LogicalId of the
     * resource being exported
     * @param value - the value to export
     */
    exportValue(stack, name, value) {
        const stackName = stack.stackName;
        let shareName = `${stackName}:${name}`;
        // Intrinsic Function will resolve the value upon deployment
        const intrinsic = stack.resolve(value);
        // Support for multiple references to the same object
        if ('Fn::GetAtt' in intrinsic) {
            const attribute = intrinsic['Fn::GetAtt'][1];
            shareName = shareName + `:${attribute}`;
        }
        if (this.valueShareMethod == ResourceExtractorShareMethod.SSM_PARAMETER) {
            shareName = shareName.replace(/:/g, '/');
            const paramName = `/${shareName}`;
            const paramLogicalId = `p${shareName}`;
            if (!this.extractDestinationStack.node.tryFindChild(paramLogicalId)) {
                new aws_cdk_lib_1.CfnResource(this.extractDestinationStack, paramLogicalId, {
                    type: 'AWS::SSM::Parameter',
                    properties: {
                        Name: paramName,
                        Type: 'String',
                        Value: intrinsic,
                    },
                });
            }
            return paramName;
        }
        else {
            // CFN_OUTPUT and API_LOOKUP share the same method of exporting values
            if (!this.extractDestinationStack.node.tryFindChild(`Export${shareName}`)) {
                return this.extractDestinationStack.exportValue(intrinsic, {
                    name: shareName,
                });
            }
            else {
                return aws_cdk_lib_1.Fn.importValue(shareName);
            }
        }
    }
}
exports.ResourceExtractor = ResourceExtractor;
_a = JSII_RTTI_SYMBOL_1;
ResourceExtractor[_a] = { fqn: "@cdklabs/cdk-enterprise-iac.ResourceExtractor", version: "0.0.263" };
//# sourceMappingURL=data:application/json;base64,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