"""@package docstring
File IO to load/save tsv/csv using the built-in csv module

This unit is generated by AI chatbot using prompts made by Qianqian Fang
This unit is under the public domain
"""

import csv
import gzip
from typing import Dict, List, Any, Optional, Union
import numpy as np

__all__ = [
    "load_csv_tsv",
    "loadcsv",
    "loadtsv",
    "save_csv_tsv",
]

##====================================================================================
## dependent libraries
##====================================================================================


def load_csv_tsv(
    filename: str,
    delimiter: Optional[str] = None,
    return_dict: bool = True,
    convert_numeric: bool = True,
    header: bool = True,
    encoding: str = "utf-8",
    **kwargs,
) -> Union[Dict[str, List], List[List[str]]]:
    """
    Read CSV, TSV, and their gzipped variants using Python's csv module

    Args:
        filename: Path to the file (.csv, .tsv, .csv.gz, .tsv.gz)
        delimiter: Column delimiter. If None, auto-detect from file extension
        return_dict: If True, return dict with column names as keys. If False, return list of rows
        convert_numeric: If True, attempt to convert numeric strings to numbers
        header: If True, treat first row as header
        encoding: File encoding (default: 'utf-8')
        **kwargs: Additional arguments passed to csv.reader/DictReader

    Returns:
        Dict with column names as keys and lists as values, or list of rows

    Examples:
        # Read CSV file as dictionary
        data = read_delimited_file('data.csv')

        # Read TSV file as list of rows
        rows = read_delimited_file('data.tsv', return_dict=False)

        # Read compressed file with custom delimiter
        data = read_delimited_file('data.csv.gz', delimiter=';')
    """

    # Auto-detect delimiter from filename if not provided
    if delimiter is None:
        filename_lower = filename.lower()
        if ".csv" in filename_lower:
            delimiter = ","
        elif ".tsv" in filename_lower:
            delimiter = "\t"
        else:
            delimiter = ","  # Default to comma

    # Determine if file is compressed
    is_compressed = filename.lower().endswith(".gz")

    # Open file (compressed or regular)
    if is_compressed:
        file_handle = gzip.open(filename, "rt", encoding=encoding)
    else:
        file_handle = open(filename, "r", encoding=encoding)

    try:
        # Set up CSV reader parameters
        csv_params = {
            "delimiter": delimiter,
            "quotechar": '"',
            "quoting": csv.QUOTE_MINIMAL,
            **kwargs,
        }

        if return_dict and header:
            # Use DictReader for dictionary output with headers
            reader = csv.DictReader(file_handle, **csv_params)

            # Read all rows
            rows = list(reader)

            if not rows:
                return {}

            # Convert to column-based dictionary
            data = {}
            fieldnames = reader.fieldnames or []

            for field in fieldnames:
                column_data = [row.get(field, "") for row in rows]

                if convert_numeric:
                    column_data = group_column(column_data)

                data[field] = column_data

            return data

        else:
            # Use regular reader for list output
            reader = csv.reader(file_handle, **csv_params)

            # Read all rows
            rows = list(reader)

            if not rows:
                return []

            if convert_numeric:
                # Convert numeric values in each row
                converted_rows = []
                for row in rows:
                    converted_row = [group_column(cell) for cell in row]
                    converted_rows.append(converted_row)
                rows = converted_rows

            if return_dict and not header:
                # Create dictionary with generic column names
                if rows:
                    num_cols = len(rows[0])
                    fieldnames = [f"col_{i}" for i in range(num_cols)]

                    data = {}
                    for i, field in enumerate(fieldnames):
                        column_data = [row[i] if i < len(row) else "" for row in rows]
                        data[field] = column_data

                    return data
                else:
                    return {}

            return rows

    finally:
        file_handle.close()


def tonumbers(value: str) -> Union[int, float, str]:
    """
    Convert a string value to numeric if possible

    Args:
        value: String value to convert

    Returns:
        Converted numeric value or original string
    """

    if not isinstance(value, str):
        return value

    # Handle empty strings
    if not value.strip():
        return value

    # Handle common NA values
    if value.lower() in ["na", "n/a", "nan", "null", "none", ""]:
        return np.nan

    # Try integer conversion first
    try:
        # Check if it looks like an integer (no decimal point)
        if "." not in value and "e" not in value.lower():
            return int(value)
    except (ValueError, TypeError):
        pass

    # Try float conversion
    try:
        return float(value)
    except (ValueError, TypeError):
        pass

    # Return as string if conversion fails
    return value


def group_column(column: List[str]) -> List[Any]:
    """
    Convert a column of string values to numeric where possible

    Args:
        column: List of string values

    Returns:
        List with numeric values converted
    """

    converted = [tonumbers(value) for value in column]

    # Check if all non-NaN values are numeric
    numeric_count = sum(
        1 for x in converted if isinstance(x, (int, float)) and not np.isnan(x)
    )
    nan_count = sum(1 for x in converted if isinstance(x, float) and np.isnan(x))
    total_numeric = numeric_count + nan_count

    # If most values are numeric, convert the whole column to numpy array
    if total_numeric > len(converted) * 0.8:  # 80% threshold
        try:
            # Convert to numpy array, handling mixed types
            numeric_array = []
            for value in converted:
                if isinstance(value, (int, float)):
                    numeric_array.append(value)
                elif isinstance(value, str):
                    try:
                        numeric_array.append(float(value))
                    except (ValueError, TypeError):
                        numeric_array.append(np.nan)
                else:
                    numeric_array.append(np.nan)

            return np.array(numeric_array, dtype=float)
        except Exception:
            pass

    return converted


def loadcsv(filename: str, **kwargs) -> Dict[str, List]:
    """
    Convenience function to read CSV files

    Args:
        filename: CSV file path
        **kwargs: Additional arguments for read_delimited_file

    Returns:
        Dictionary with column data
    """
    return load_csv_tsv(filename, delimiter=",", **kwargs)


def loadtsv(filename: str, **kwargs) -> Dict[str, List]:
    """
    Convenience function to read TSV files

    Args:
        filename: TSV file path
        **kwargs: Additional arguments for read_delimited_file

    Returns:
        Dictionary with column data
    """
    return load_csv_tsv(filename, delimiter="\t", **kwargs)


def save_csv_tsv(
    data: Union[Dict[str, List], List[List]],
    filename: str,
    delimiter: Optional[str] = None,
    compress: bool = False,
    encoding: str = "utf-8",
    **kwargs,
):
    """
    Write data to CSV/TSV files (with optional compression)

    Args:
        data: Dictionary with column data or list of rows
        filename: Output file path
        delimiter: Column delimiter. If None, auto-detect from filename
        compress: If True, compress output with gzip
        encoding: File encoding
        **kwargs: Additional arguments for csv.writer/DictWriter
    """

    # Auto-detect delimiter
    if delimiter is None:
        if ".tsv" in filename.lower():
            delimiter = "\t"
        else:
            delimiter = ","

    # Add .gz extension if compressing
    if compress and not filename.lower().endswith(".gz"):
        filename += ".gz"

    # Open file
    if compress or filename.lower().endswith(".gz"):
        file_handle = gzip.open(filename, "wt", encoding=encoding)
    else:
        file_handle = open(filename, "w", encoding=encoding, newline="")

    try:
        csv_params = {
            "delimiter": delimiter,
            "quotechar": '"',
            "quoting": csv.QUOTE_MINIMAL,
            **kwargs,
        }

        if isinstance(data, dict):
            # Write dictionary data
            fieldnames = list(data.keys())
            writer = csv.DictWriter(file_handle, fieldnames=fieldnames, **csv_params)

            # Write header
            writer.writeheader()

            # Write rows
            if fieldnames:
                num_rows = len(data[fieldnames[0]])
                for i in range(num_rows):
                    row = {
                        field: data[field][i] if i < len(data[field]) else ""
                        for field in fieldnames
                    }
                    writer.writerow(row)

        else:
            # Write list data
            writer = csv.writer(file_handle, **csv_params)
            writer.writerows(data)

    finally:
        file_handle.close()
