"""Chain configuration for DeFi Llama integration.

This module contains the valid chains and their aliases for use with the DeFi Llama API.
The VALID_CHAINS dictionary maps primary chain identifiers to their known aliases.
"""

# Chain configuration with aliases
VALID_CHAINS: dict[str, list[str]] = {
    "ethereum": ["eth", "eth1", "eth2"],
    "solana": ["sol"],
    "bitcoin": ["btc"],
    "bsc": ["bnb", "bsc"],
    "tron": ["trx"],
    "base": ["base"],
    "berachain": ["bera"],
    "arbitrum": ["arb"],
    "sui": ["sui"],
    "avalanche": ["avax", "ava"],
    "aptos": ["apt"],
    "polygon": ["matic", "polygon"],
    "hyperliquid": ["hyper"],
    "op_mainnet": ["op"],
    "sonic": ["sonic"],
    "core": ["core"],
    "zircuit": ["zircuit"],
    "cronos": ["cro"],
    "bitlayer": ["bit"],
    "cardano": ["ada"],
    "bsquared": ["b2"],
    "mantle": ["mntl"],
    "pulsechain": ["pulse"],
    "gnosis": ["gnosis"],
    "dydx": ["dydx"],
    "taiko": ["tk"],
    "bob": ["bob"],
    "zksync_era": ["zk", "zkSync"],
    "linea": ["linea"],
    "blast": ["blast"],
    "rootstock": ["rs"],
    "thorchain": ["thor"],
    "ailayer": ["ai"],
    "sei": ["sei"],
    "eos": ["eos"],
    "ton": ["ton"],
    "near": ["near"],
    "merlin": ["merlin"],
    "kava": ["kava"],
    "algorand": ["algo"],
    "starknet": ["stark"],
    "hedera": ["hbar"],
    "mixin": ["mixin"],
    "scroll": ["scroll"],
    "kaia": ["kaia"],
    "stacks": ["stx"],
    "ronin": ["ronin"],
    "osmosis": ["osmo"],
    "verus": ["verus"],
    "multiversx": ["x"],
    "celo": ["celo"],
    "xrpl": ["xrpl"],
    "fraxtal": ["frax"],
    "stellar": ["xlm"],
    "bouncebit": ["bounce"],
    "wemix3_0": ["wemix"],
    "filecoin": ["fil"],
    "hydration": ["hydra"],
    "fantom": ["ftm"],
    "iota_evm": ["iot"],
    "manta": ["manta"],
    "eclipse": ["eclp"],
    "flow": ["flow"],
    "injective": ["inj"],
    "tezos": ["xtz"],
    "soneium": ["son"],
    "neutron": ["neut"],
    "icp": ["icp"],
    "iotex": ["iotex"],
    "metis": ["metis"],
    "opbnb": ["opbnb"],
    "bifrost_network": ["bifrost"],
    "flare": ["flare"],
    "xdc": ["xdc"],
    "morph": ["morph"],
    "waves": ["waves"],
    "conflux": ["conflux"],
    "corn": ["corn"],
    "reya_network": ["reya"],
    "mode": ["mode"],
    "cronos_zkevm": ["cronoszk"],
    "telos": ["telos"],
    "rollux": ["rollux"],
    "zetachain": ["zeta"],
    "chainflip": ["flip"],
    "fuel_ignition": ["fuel"],
    "aurora": ["aurora"],
    "map_protocol": ["map"],
    "kujira": ["kujira"],
    "astar": ["astar"],
    "moonbeam": ["moonbeam"],
    "story": ["story"],
    "abstract": ["abstract"],
    "radix": ["radix"],
    "zklink_nova": ["zklink"],
    "duckchain": ["duck"],
    "swellchain": ["swell"],
    "apechain": ["ape"],
    "icon": ["icx"],
    "immutable_zkevm": ["immutable"],
    "eos_evm": ["eosevm"],
    "bifrost": ["bifrost"],
    "k2": ["k2"],
    "aelf": ["aelf"],
    "fsc": ["fsc"],
    "proton": ["proton"],
    "secret": ["secret"],
    "unichain": ["unichain"],
    "neo": ["neo"],
    "mayachain": ["maya"],
    "canto": ["canto"],
    "chiliz": ["chz"],
    "x_layer": ["xlayer"],
    "polynomial": ["poly"],
    "ontology": ["ont"],
    "onus": ["onus"],
    "bitcoincash": ["bch"],
    "terra2": ["terra2"],
    "polygon_zkevm": ["polyzk"],
    "ink": ["ink"],
    "sophon": ["sophon"],
    "venom": ["venom"],
    "dexalot": ["dexalot"],
    "bahamut": ["bahamut"],
    "vite": ["vite"],
    "dfs_network": ["dfs"],
    "ergo": ["ergo"],
    "wanchain": ["wan"],
    "mantra": ["mantra"],
    "doge": ["doge"],
    "lisk": ["lisk"],
    "alephium": ["alephium"],
    "vision": ["vision"],
    "dogechain": ["dogechain"],
    "nuls": ["nuls"],
    "agoric": ["agoric"],
    "defichain": ["defi"],
    "dymension": ["dym"],
    "thundercore": ["tc"],
    "godwokenv1": ["godwoken"],
    "bevm": ["bevm"],
    "litecoin": ["ltc"],
    "ux": ["ux"],
    "functionx": ["fx"],
    "oraichain": ["oraichain"],
    "dfk": ["dfk"],
    "carbon": ["carbon"],
    "beam": ["beam"],
    "gravity": ["gravity"],
    "horizen_eon": ["horizen"],
    "moonriver": ["movr"],
    "real": ["real"],
    "oasys": ["oasys"],
    "hydra": ["hydra"],
    "oktchain": ["okt"],
    "shibarium": ["shib"],
    "world_chain": ["world"],
    "interlay": ["interlay"],
    "acala": ["acala"],
    "elys": ["elys"],
    "boba": ["boba"],
    "vana": ["vana"],
    "harmony": ["harmony"],
    "lachain_network": ["lachain"],
    "theta": ["theta"],
    "ab": ["ab"],
    "defiverse": ["defiverse"],
    "kcc": ["kcc"],
    "oasis_sapphire": ["oasis"],
    "etherlink": ["etherlink"],
    "wax": ["wax"],
    "archway": ["archway"],
    "redbelly": ["redbelly"],
    "velas": ["velas"],
    "equilibrium": ["equilibrium"],
    "unit0": ["unit0"],
    "ql1": ["ql1"],
    "songbird": ["songbird"],
    "zilliqa": ["zil"],
    "rangers": ["rangers"],
    "odyssey": ["odyssey"],
    "terra_classic": ["terra"],
    "kadena": ["kadena"],
    "zero_network": ["zero"],
    "elastos": ["elastos"],
    "fluence": ["fluence"],
    "idex": ["idex"],
    "xpla": ["xpla"],
    "milkomeda_c1": ["milkomeda"],
    "taraxa": ["taraxa"],
    "bitrock": ["bitrock"],
    "persistence_one": ["persistence"],
    "meter": ["meter"],
    "arbitrum_nova": ["arbitrumnova"],
    "everscale": ["everscale"],
    "ultron": ["ultron"],
    "fuse": ["fuse"],
    "vechain": ["vet"],
    "renec": ["renec"],
    "shimmerevm": ["shimmer"],
    "obyte": ["obyte"],
    "nolus": ["nolus"],
    "airdao": ["airdao"],
    "elysium": ["elysium"],
    "xai": ["xai"],
    "starcoin": ["starcoin"],
    "oasis_emerald": ["oasisem"],
    "haqq": ["haqq"],
    "nos": ["nos"],
    "neon": ["neon"],
    "bittorrent": ["btt"],
    "csc": ["csc"],
    "satoshivm": ["satv"],
    "naka": ["naka"],
    "edu_chain": ["edu"],
    "kintsugi": ["kintsugi"],
    "energi": ["energi"],
    "rss3": ["rss3"],
    "sx_rollup": ["sx"],
    "cosmoshub": ["cosmos"],
    "saakuru": ["saakuru"],
    "boba_bnb": ["boba_bnb"],
    "ethereumclassic": ["etc"],
    "skale_europa": ["skale"],
    "degen": ["degen"],
    "mint": ["mint"],
    "juno": ["juno"],
    "viction": ["viction"],
    "evmos": ["evmos"],
    "enuls": ["enuls"],
    "lightlink": ["lightlink"],
    "sanko": ["sanko"],
    "karura": ["karura"],
    "kardia": ["kardia"],
    "superposition": ["super"],
    "crab": ["crab"],
    "genesys": ["genesys"],
    "matchain": ["matchain"],
    "chihuahua": ["chihuahua"],
    "massa": ["massa"],
    "kroma": ["kroma"],
    "tombchain": ["tomb"],
    "smartbch": ["bchsmart"],
    "ancient8": ["ancient8"],
    "penumbra": ["penumbra"],
    "ethpow": ["ethpow"],
    "omax": ["omax"],
    "migaloo": ["migaloo"],
    "bostrom": ["bostrom"],
    "energyweb": ["energyweb"],
    "libre": ["libre"],
    "defichain_evm": ["defievm"],
    "artela": ["artela"],
    "dash": ["dash"],
    "sora": ["sora"],
    "step": ["step"],
    "nibiru": ["nibiru"],
    "zkfair": ["zkfair"],
    "hela": ["hela"],
    "godwoken": ["godwoken"],
    "shape": ["shape"],
    "stargaze": ["stargaze"],
    "crossfi": ["crossfi"],
    "bitkub_chain": ["bitkub"],
    "q_protocol": ["qprotocol"],
    "loop": ["loop"],
    "parex": ["parex"],
    "alv": ["alv"],
    "nahmii": ["nahmii"],
    "shido": ["shido"],
    "electroneum": ["etn"],
    "zora": ["zora"],
    "astar_zkevm": ["astark"],
    "comdex": ["comdex"],
    "stratis": ["stratis"],
    "polkadex": ["polkadex"],
    "meer": ["meer"],
    "neo_x_mainnet": ["neo_x"],
    "aura_network": ["aura"],
    "findora": ["findora"],
    "shiden": ["shiden"],
    "swan": ["swan"],
    "crescent": ["crescent"],
    "rari": ["rari"],
    "cyber": ["cyber"],
    "redstone": ["redstone"],
    "silicon_zkevm": ["silicon"],
    "endurance": ["endurance"],
    "inevm": ["inevm"],
    "grove": ["grove"],
    "areon_network": ["areon"],
    "jbc": ["jbc"],
    "planq": ["planq"],
    "lachain": ["lachain"],
    "rei": ["rei"],
    "multivac": ["multivac"],
    "cube": ["cube"],
    "syscoin": ["syscoin"],
    "vinuchain": ["vinuchain"],
    "callisto": ["callisto"],
    "hpb": ["hpb"],
    "ham": ["ham"],
    "ethf": ["ethf"],
    "gochain": ["gochain"],
    "darwinia": ["darwinia"],
    "sx_network": ["sx"],
    "manta_atlantic": ["atlantic"],
    "ontologyevm": ["ontEvm"],
    "mvc": ["mvc"],
    "sifchain": ["sifchain"],
    "plume": ["plume"],
    "bitgert": ["bitgert"],
    "reichain": ["reichain"],
    "bitnet": ["bitnet"],
    "tenet": ["tenet"],
    "milkomeda_a1": ["milkomedaA1"],
    "aeternity": ["aeternity"],
    "palm": ["palm"],
    "concordium": ["concordium"],
    "kopi": ["kopi"],
    "asset_chain": ["asset"],
    "pego": ["pego"],
    "waterfall": ["waterfall"],
    "heco": ["heco"],
    "exsat": ["exsat"],
    "goerli": ["goerli"],
    "celestia": ["celestia"],
    "bandchain": ["band"],
    "sommelier": ["sommelier"],
    "stride": ["stride"],
    "polkadot": ["dot"],
    "kusama": ["kusama"],
    "dexit": ["dexit"],
    "fusion": ["fusion"],
    "boba_avax": ["boba"],
    "stafi": ["stafi"],
    "empire": ["empire"],
    "oxfun": ["oxfun"],
    "pryzm": ["pryzm"],
    "hoo": ["hoo"],
    "echelon": ["echelon"],
    "quicksilver": ["quick"],
    "clv": ["clv"],
    "pokt": ["pokt"],
    "dsc": ["dsc"],
    "zksync_lite": ["zkLite"],
    "nova_network": ["nova"],
    "cmp": ["cmp"],
    "genshiro": ["genshiro"],
    "lamden": ["lamden"],
    "polis": ["polis"],
    "zyx": ["zyx"],
    "ubiq": ["ubiq"],
    "heiko": ["heiko"],
    "parallel": ["parallel"],
    "coti": ["coti"],
    "kekchain": ["kek"],
    "muuchain": ["muuchain"],
    "tlchain": ["tlchain"],
    "zeniql": ["zeniql"],
    "bitindi": ["bitindi"],
    "lung": ["lung"],
    "bone": ["bone"],
    "lukso": ["lukso"],
    "joltify": ["joltify"],
}


def get_chain_from_alias(alias: str) -> str | None:
    """Get the main chain identifier from an alias.

    Args:
        alias: The chain alias to look up

    Returns:
        The main chain identifier if found, None otherwise
    """
    normalized_alias = alias.lower().strip()

    # Check if it's a main chain name
    if normalized_alias in VALID_CHAINS:
        return normalized_alias

    # Check aliases
    for chain, aliases in VALID_CHAINS.items():
        if normalized_alias in [a.lower() for a in aliases]:
            return chain

    return None


def is_valid_chain(chain: str) -> bool:
    """Check if a chain identifier is valid.

    Args:
        chain: The chain identifier to validate

    Returns:
        True if the chain is valid, False otherwise
    """
    return get_chain_from_alias(chain) is not None


def get_all_chains() -> list[str]:
    """Get a list of all valid main chain identifiers.

    Returns:
        List of all main chain identifiers
    """
    return list(VALID_CHAINS.keys())


def get_chain_aliases(chain: str) -> list[str]:
    """Get all aliases for a given chain.

    Args:
        chain: The main chain identifier

    Returns:
        List of aliases for the chain, empty list if chain not found
    """
    normalized_chain = chain.lower().strip()
    return VALID_CHAINS.get(normalized_chain, [])
