"""Core bootstrap functionality.

This module provides the fundamental bootstrap resampling functions and
core statistical utilities for bootstrap analysis.
"""

import warnings
from typing import Callable, Iterator, Optional, Union

import numpy as np
import numpy.typing as npt
from joblib import Parallel, delayed
from scipy.stats import norm

from .constants import (
    DEFAULT_BOOTSTRAP_SAMPLES,
    DEFAULT_CONFIDENCE_LEVEL,
    DEFAULT_N_JOBS,
    DEFAULT_SEED,
    EPSILON,
    ERROR_MESSAGES,
    MIN_SAMPLE_SIZE,
)
from .exceptions import InsufficientDataError, NumericalError, ValidationError


def _validate_bootstrap_params(
    number_of_bootstrap_samples: int,
    bootstrap_conf_level: float,
    sample_size: Optional[int] = None,
) -> None:
    """Validate bootstrap parameters.

    Parameters
    ----------
    number_of_bootstrap_samples : int
        Number of bootstrap samples.
    bootstrap_conf_level : float
        Confidence level for bootstrap.
    sample_size : int, optional
        Sample size for bootstrap.

    Raises
    ------
    ValidationError
        If parameters are invalid.
    """
    if number_of_bootstrap_samples <= 0:
        raise ValidationError(
            ERROR_MESSAGES["invalid_bootstrap_samples"],
            parameter="number_of_bootstrap_samples",
            value=number_of_bootstrap_samples,
        )

    if not (0 < bootstrap_conf_level < 1):
        raise ValidationError(
            ERROR_MESSAGES["invalid_confidence_level"],
            parameter="bootstrap_conf_level",
            value=bootstrap_conf_level,
        )

    if sample_size is not None and sample_size < MIN_SAMPLE_SIZE:
        raise ValidationError(
            ERROR_MESSAGES["invalid_sample_size"],
            parameter="sample_size",
            value=sample_size,
        )


def _validate_sample_array(array: npt.NDArray[np.floating], name: str) -> None:
    """Validate a sample array.

    Parameters
    ----------
    array : ndarray
        Array to validate.
    name : str
        Name of the array for error messages.

    Raises
    ------
    ValidationError
        If array is invalid.
    InsufficientDataError
        If array is too small.
    """
    if array.size == 0:
        raise ValidationError(ERROR_MESSAGES["empty_array"], parameter=name)

    if array.size < MIN_SAMPLE_SIZE:
        raise InsufficientDataError(
            ERROR_MESSAGES["insufficient_data"],
            sample_size=array.size,
            min_required=MIN_SAMPLE_SIZE,
        )


def estimate_confidence_interval(
    distribution: npt.NDArray[np.floating],
    bootstrap_conf_level: float = DEFAULT_CONFIDENCE_LEVEL,
) -> npt.NDArray[np.floating]:
    """Estimate confidence interval from a bootstrap distribution.

    Parameters
    ----------
    distribution : ndarray
        1D array containing the bootstrap distribution.
    bootstrap_conf_level : float, optional
        Confidence level between 0 and 1. Default is 0.95.

    Returns
    -------
    ndarray
        Array [lower_bound, upper_bound] representing the confidence interval.

    Raises
    ------
    ValidationError
        If distribution is empty or confidence level is invalid.
    NumericalError
        If quantile computation fails.

    Examples
    --------
    >>> distribution = np.array([1, 2, 3, 4, 5])
    >>> ci = estimate_confidence_interval(distribution, 0.95)
    >>> len(ci)
    2

    Notes
    -----
    Time complexity: O(n log n) where n is the distribution length.
    Space complexity: O(1).
    """
    _validate_sample_array(distribution, "distribution")
    _validate_bootstrap_params(1, bootstrap_conf_level)

    try:
        left_quant = (1 - bootstrap_conf_level) / 2
        right_quant = 1 - left_quant
        return np.quantile(distribution, [left_quant, right_quant])
    except (ValueError, TypeError) as e:
        raise NumericalError(
            "Failed to compute confidence interval",
            operation="quantile_computation",
            values={"confidence_level": bootstrap_conf_level},
        ) from e


def estimate_p_value(
    bootstrap_difference_distribution: npt.NDArray[np.floating],
    number_of_bootstrap_samples: int,
) -> float:
    """Estimate two-sided p-value from bootstrap difference distribution.

    Parameters
    ----------
    bootstrap_difference_distribution : ndarray
        1D array containing the bootstrap difference distribution.
    number_of_bootstrap_samples : int
        Number of bootstrap samples used to generate the distribution.

    Returns
    -------
    float
        The estimated two-sided p-value.

    Raises
    ------
    ValidationError
        If distribution is empty or number of samples is invalid.
    NumericalError
        If p-value computation fails.

    Examples
    --------
    >>> distribution = np.array([-1, 0, 1, 2, 3])
    >>> p_value = estimate_p_value(distribution, 5)
    >>> 0 <= p_value <= 1
    True

    Notes
    -----
    Time complexity: O(n) where n is the distribution length.
    Space complexity: O(1).
    """
    _validate_sample_array(
        bootstrap_difference_distribution, "bootstrap_difference_distribution"
    )
    _validate_bootstrap_params(number_of_bootstrap_samples, 0.95)

    try:
        positions = np.sum(bootstrap_difference_distribution < 0, axis=0)
        return float(
            2
            * np.minimum(positions, number_of_bootstrap_samples - positions)
            / number_of_bootstrap_samples
        )
    except (ValueError, TypeError) as e:
        raise NumericalError(
            "Failed to compute p-value",
            operation="p_value_computation",
            values={"sample_count": number_of_bootstrap_samples},
        ) from e


def estimate_bin_params(sample: npt.NDArray[np.floating]) -> tuple[float, int]:
    """Estimate optimal histogram bin parameters using Freedman-Diaconis rule.

    Parameters
    ----------
    sample : ndarray
        1D array containing observations.

    Returns
    -------
    tuple[float, int]
        A tuple (bin_width, bin_count) where:
        - bin_width : float
            Width of each bin using Freedman-Diaconis rule.
        - bin_count : int
            Number of bins for the histogram.

    Raises
    ------
    ValidationError
        If sample is empty.
    NumericalError
        If bin computation fails.

    Examples
    --------
    >>> sample = np.array([1, 2, 3, 4, 5, 6, 7, 8, 9, 10])
    >>> bin_width, bin_count = estimate_bin_params(sample)
    >>> bin_width > 0 and bin_count > 0
    True

    Notes
    -----
    Time complexity: O(n log n) where n is the sample length.
    Space complexity: O(1).
    """
    _validate_sample_array(sample, "sample")

    try:
        q1 = np.quantile(sample, 0.25)
        q3 = np.quantile(sample, 0.75)
        iqr = q3 - q1

        # Freedman-Diaconis rule
        bin_width = (2 * iqr) / (sample.shape[0] ** (1 / 3))

        # Ensure bin_width is not zero
        if bin_width < EPSILON:
            bin_width = (sample.max() - sample.min()) / 10

        bin_count = max(1, int(np.ceil((sample.max() - sample.min()) / bin_width)))

        return float(bin_width), bin_count
    except (ValueError, TypeError) as e:
        raise NumericalError(
            "Failed to compute bin parameters",
            operation="bin_computation",
            values={"sample_size": sample.size},
        ) from e


def jackknife_indices(
    sample: npt.NDArray[np.floating],
) -> Iterator[npt.NDArray[np.intp]]:
    """Generate jackknife indices for leave-one-out resampling.

    Parameters
    ----------
    sample : ndarray
        1D array containing the sample data.

    Yields
    ------
    ndarray
        Arrays of indices with one element removed for jackknife resampling.

    Raises
    ------
    ValidationError
        If sample is empty.

    Examples
    --------
    >>> sample = np.array([1, 2, 3, 4, 5])
    >>> indices_list = list(jackknife_indices(sample))
    >>> len(indices_list) == len(sample)
    True

    Notes
    -----
    Time complexity: O(n) per yielded array where n is the sample length.
    Space complexity: O(n) per yielded array.
    """
    _validate_sample_array(sample, "sample")

    base_indices = np.arange(len(sample))
    for i in range(len(sample)):
        yield np.delete(base_indices, i)


def bca_confidence_interval(
    sample: npt.NDArray[np.floating],
    bootstrap_distribution: npt.NDArray[np.floating],
    statistic: Callable[[npt.NDArray[np.floating]], float] = np.mean,
    bootstrap_conf_level: float = DEFAULT_CONFIDENCE_LEVEL,
) -> npt.NDArray[np.floating]:
    """Compute BCa (bias-corrected and accelerated) confidence interval.

    Parameters
    ----------
    sample : ndarray
        1D array containing the original sample data.
    bootstrap_distribution : ndarray
        1D array containing the bootstrap distribution of the statistic.
    statistic : callable, optional
        Function that computes the statistic of interest.
        Default is np.mean.
    bootstrap_conf_level : float, optional
        Confidence level between 0 and 1. Default is 0.95.

    Returns
    -------
    ndarray
        Array [lower_bound, upper_bound] representing the BCa confidence interval.

    Raises
    ------
    ValidationError
        If inputs are invalid.
    NumericalError
        If BCa computation fails.

    Examples
    --------
    >>> sample = np.array([1, 2, 3, 4, 5])
    >>> bootstrap_dist = np.array([2.5, 3.0, 2.8, 3.2, 2.9])
    >>> ci = bca_confidence_interval(sample, bootstrap_dist)
    >>> len(ci)
    2

    Notes
    -----
    Time complexity: O(n² + b log b) where n is sample size, b is bootstrap samples.
    Space complexity: O(n + b).
    """
    _validate_sample_array(sample, "sample")
    _validate_sample_array(bootstrap_distribution, "bootstrap_distribution")
    _validate_bootstrap_params(len(bootstrap_distribution), bootstrap_conf_level)

    try:
        number_of_bootstrap_samples = bootstrap_distribution.shape[0]
        sample_stat = statistic(sample)

        # Confidence interval alphas
        alphas = np.array(
            [(1 - bootstrap_conf_level) / 2, 1 - (1 - bootstrap_conf_level) / 2]
        )

        # Bias correction value
        z0 = norm.ppf(
            np.sum(bootstrap_distribution < sample_stat) / number_of_bootstrap_samples
        )

        # Compute jackknife statistics for acceleration
        jackknife_stats = [
            statistic(sample[indices]) for indices in jackknife_indices(sample)
        ]
        jack_mean = np.mean(jackknife_stats)

        # Acceleration value
        numerator = np.sum((jack_mean - jackknife_stats) ** 3)
        denominator = 6.0 * (np.sum((jack_mean - jackknife_stats) ** 2) ** 1.5)

        if abs(denominator) < EPSILON:
            acceleration = 0.0
            warnings.warn(
                "Acceleration value undefined due to zero denominator. "
                "Using percentile method instead.",
                UserWarning,
                stacklevel=2,
            )
        else:
            acceleration = numerator / denominator

        # Compute BCa endpoints
        z_alphas = z0 + norm.ppf(alphas)
        adjusted_alphas = norm.cdf(z0 + z_alphas / (1 - acceleration * z_alphas))

        indices = np.round((number_of_bootstrap_samples - 1) * adjusted_alphas).astype(
            int
        )
        indices = np.clip(indices, 0, number_of_bootstrap_samples - 1)

        sorted_distribution = np.sort(bootstrap_distribution)
        return sorted_distribution[indices]

    except (ValueError, TypeError, ZeroDivisionError) as e:
        raise NumericalError(
            "Failed to compute BCa confidence interval",
            operation="bca_computation",
            values={
                "sample_size": sample.size,
                "bootstrap_samples": len(bootstrap_distribution),
            },
        ) from e


def bootstrap_resampling(
    sample_function: Callable[
        [np.random.Generator], Union[float, npt.NDArray[np.floating]]
    ],
    number_of_bootstrap_samples: int = DEFAULT_BOOTSTRAP_SAMPLES,
    seed: Optional[int] = DEFAULT_SEED,
    n_jobs: int = DEFAULT_N_JOBS,
) -> npt.NDArray[np.floating]:
    """Perform bootstrap resampling with parallel processing.

    Parameters
    ----------
    sample_function : callable
        Function that takes a NumPy Generator and returns a bootstrap statistic.
        Signature: sample_function(generator: np.random.Generator) -> float or ndarray
    number_of_bootstrap_samples : int, optional
        Number of bootstrap samples to generate. Default is 10000.
    seed : int, optional
        Seed for reproducibility. Default is 42.
    n_jobs : int, optional
        Number of parallel jobs. -1 uses all available cores. Default is -1.

    Returns
    -------
    ndarray
        Array of bootstrap statistics with shape (number_of_bootstrap_samples,).

    Raises
    ------
    ValidationError
        If parameters are invalid.
    NumericalError
        If bootstrap computation fails.

    Examples
    --------
    >>> def sample_mean(rng):
    ...     return rng.normal(0, 1, 100).mean()
    >>> results = bootstrap_resampling(sample_mean, 1000)
    >>> len(results)
    1000

    Notes
    -----
    Time complexity: O(n * f) where n is bootstrap samples, f is sample function cost.
    Space complexity: O(n).
    """
    _validate_bootstrap_params(number_of_bootstrap_samples, 0.95)

    try:
        # Create independent random number generators for reproducibility
        base_seed = np.random.SeedSequence(seed)
        seeds = base_seed.spawn(number_of_bootstrap_samples)
        rngs = [np.random.Generator(np.random.PCG64(s)) for s in seeds]

        # Parallel bootstrap computation
        results = Parallel(n_jobs=n_jobs)(delayed(sample_function)(rng) for rng in rngs)

        return np.array(results)

    except (ValueError, TypeError) as e:
        raise NumericalError(
            "Failed to perform bootstrap resampling",
            operation="bootstrap_resampling",
            values={"n_samples": number_of_bootstrap_samples, "n_jobs": n_jobs},
        ) from e
