A Python client for interacting with the **Embedded Intelligence Platform (EIP)** by sumtyme.ai.

The EIP is powered by **Abstract Generalised Networks (AGNs)**, a proprietary AI architecture designed to model how complex systems evolve. Unlike traditional machine learning, AGNs autonomously rewrite their underlying algorithms as they observe new data. This allows them to holistically understand how directional shifts influence a system in domains like financial markets and weather without the need for training, fine-tuning, or retraining.

This client provides direct access to the EIP's capabilities for understanding directional changes and system evolution, rather than simply predicting the next step.

## Getting Started


### Installation

You can install this package via pip.

```bash
pip install sumtyme
```

### Client Initialisation and API Key
To use the client, you need an API key and must initialise the `EIPClient` with a subdomain.


#### Option 1: Use an Existing API Key
If you already have an API key, we recommend setting it as an environment variable named apikey. The client will automatically load it upon initialisation.

**Create a `.env` file:**
```dotenv
apikey="your-api-key-here"
```

Then, initialise the client in your code:


```python
from sumtyme import EIPClient

# Replace 'your-subdomain' with your assigned EIP subdomain.
client = EIPClient(subdomain='your-subdomain')
```


#### Option 2: Sign Up for a New API Key

If you do not yet have an API key, you can register as a new user and obtain one directly through the client's `user_signup`. This method will send a sign-up request to the EIP, return your API key and automatically save it to a local .txt file. After signing up, email team@sumtyme.ai for profile activation.


```python
# Initialise the client with your subdomain
client = EIPClient(subdomain='your-subdomain')

# Register a new user and get your API key
signup_result = client.user_signup(payload={
    "email": "your_email@example.com",
    "password": "min_password_length_8",
    "confirm_password":"min_password_length_8"
})
```

### Core Functionality

AGNs are not autoregressive and analyse the entire data window simultaneously. Therefore, any outputs observed before the last data period of a window should be used for context and not as a prediction. The recommended window size is 5001 data periods.

#### Modelling the Environment

The `model_timeseries_environment` method shows the AGN's modelling process of the entire environment based on the observed data. This method accepts a single dataset of 5,001 to 10,000 data periods.

```python
# See the entire model generated by the AGN for the submitted data

entire_model = client.model_timeseries_environment(
    data_input='folder/ohlc_data.csv', # or can be a pandas dataframe 
    interval=1,
    interval_unit='days',
    reasoning_mode='reactive',
    output_file='saved_output'
)
```

#### Identifying the Latest Directional Change

Use 'identify_timeseries_directional_change' to get a quick assessment of whether a directional change has been identified for the very last point of your dataset. This method requires at least 5,001 data periods.

```python 
# Identify if a directional change exists at the last data period

latest_change = client.identify_timeseries_directional_change(
    data_input='folder/ohlc_data.csv', # or can be a pandas dataframe 
    interval=1,
    interval_unit='days',
    reasoning_mode='reactive'
)

# The result is a list: [datetime_string, trend_value]
# Trend values: 1 (positive change), -1 (negative change), 0 (no clear change)
```

#### Rolling Window Analysis

The `get_timeseries_rolling_directional_changes` method performs a rolling analysis on your data, providing a trend identification for the last point of each window. This is ideal for backtesting or observing how trends evolve over time.

```python
# Performs a rolling window analysis on the observed data

rolling_trends = client.get_timeseries_rolling_directional_changes(
    data_input='folder/ohlc_data.csv', # or can be a pandas dataframe 
    interval=1,
    interval_unit='days',
    reasoning_mode='proactive',
    window_size=5001,
    output_file='saved_output'
)
```