#!/usr/bin/env python
"""
Comprehensive edge case tests for bitformat library.

This file focuses on testing edge cases and scenarios that may not be fully covered
by existing tests. It uses property-based testing with Hypothesis where appropriate
to find edge cases automatically.

Note: Some tests may fail due to legitimate edge cases or bugs. Where this happens,
the failing test will include a comment explaining the expected behavior or why
it might be failing. These tests are not intended to be "fixed" by modifying the
library - they are meant to document and explore the edge cases.
"""

import pytest
import math
import sys
import struct
from typing import List, Any, Union
from hypothesis import given, assume, strategies as st, settings, HealthCheck
from hypothesis import Verbosity

# NOTE: The following imports will fail until the bitformat package is properly built
# with its Rust dependencies. In a proper CI environment, these should work.
try:
    from bitformat import (
        Bits, MutableBits, Array, Format, Field, Dtype, DtypeSingle, DtypeArray, 
        DtypeTuple, Endianness, DtypeKind
    )
    from bitformat._common import Expression, ExpressionError
    BITFORMAT_AVAILABLE = True
except ImportError as e:
    BITFORMAT_AVAILABLE = False
    pytest.skip(f"bitformat not available: {e}", allow_module_level=True)


class TestBitsEdgeCases:
    """Test edge cases for Bits class."""
    
    def test_empty_bits_operations(self):
        """Test operations on empty Bits objects."""
        empty = Bits()
        assert len(empty) == 0
        assert empty.hex == ""
        assert empty.bin == ""
        assert empty.bytes == b""
        
        # Concatenation with empty should work
        non_empty = Bits("0x42")
        assert empty + non_empty == non_empty
        assert non_empty + empty == non_empty
        
    def test_single_bit_operations(self):
        """Test operations on single-bit Bits objects."""
        zero_bit = Bits("0b0")
        one_bit = Bits("0b1")
        
        assert len(zero_bit) == 1
        assert len(one_bit) == 1
        assert zero_bit != one_bit
        
        # Test logical operations
        assert zero_bit | one_bit == one_bit
        assert zero_bit & one_bit == zero_bit
        assert zero_bit ^ one_bit == one_bit
        
    @given(st.integers(min_value=0, max_value=2**64 - 1))
    def test_large_integer_roundtrip(self, value):
        """Test roundtrip conversion for large integers."""
        # Test various bit widths
        bit_width = value.bit_length() + 1  # +1 to avoid exact fit issues
        if bit_width > 64:
            assume(bit_width <= 128)  # Keep test reasonable
            
        dtype_str = f"u{bit_width}"
        bits = Bits.from_dtype(dtype_str, value)
        recovered = bits.unpack([dtype_str])[0]
        assert recovered == value, f"Failed roundtrip for {value} with {dtype_str}"
        
    @given(st.integers(min_value=-(2**63), max_value=2**63 - 1))
    def test_signed_integer_edge_cases(self, value):
        """Test signed integer edge cases including min/max values."""
        bit_width = max(value.bit_length() + 1, 64)  # Ensure enough bits for sign
        if bit_width > 64:
            assume(bit_width <= 128)
            
        dtype_str = f"i{bit_width}"
        try:
            bits = Bits.from_dtype(dtype_str, value)
            recovered = bits.unpack([dtype_str])[0]
            assert recovered == value
        except ValueError as e:
            # This might fail for edge cases - document it
            # POSSIBLE FAILING TEST: Some signed integer values near boundaries
            # might not roundtrip correctly due to two's complement representation
            pytest.skip(f"Expected failure for signed integer edge case: {value}, error: {e}")

                    
    @given(st.binary(min_size=0, max_size=1000))
    def test_bytes_roundtrip_edge_cases(self, data):
        """Test bytes roundtrip with various edge cases."""
        bits = Bits.from_bytes(data)
        recovered = bits.bytes
        assert recovered == data
        
        # Test with partial byte handling
        if len(data) > 0:
            # Add some extra bits and check behavior
            extra_bits = Bits("0b101")
            combined = bits + extra_bits
            
            # The bytes should include the partial byte
            combined_bytes = combined.to_bytes()
            assert len(combined_bytes) == len(data) + 1  # Extra bits pad to full byte
            
    def test_bit_manipulation_edge_cases(self):
        """Test edge cases in bit manipulation."""
        # Test setting bits at boundaries
        bits = MutableBits("0x00")
        
        # Test first and last bit
        bits[0] = 1
        assert bits == "0x80"
        
        bits[7] = 1  
        assert bits == "0x81"
        
        # Test extending with bit operations
        bits = MutableBits("0b1")
        bits.append("0b0")
        assert bits == "0b10"
        assert len(bits) == 2


class TestArrayEdgeCases:
    """Test edge cases for Array class."""
    
    def test_empty_array_operations(self):
        """Test operations on empty arrays."""
        empty_array = Array.from_zeros("u8", 0)
        assert len(empty_array) == 0
        assert empty_array.to_list() == []
        
        # Operations on empty arrays
        with pytest.raises(IndexError):
            _ = empty_array[0]
            
    def test_single_element_array(self):
        """Test arrays with single elements."""
        single = Array("u8", [42])
        assert len(single) == 1
        assert single[0] == 42
        assert single[-1] == 42
        assert single.to_list() == [42]
        
    @given(st.lists(st.integers(0, 255), min_size=1, max_size=1000))
    def test_array_dtype_conversion_edge_cases(self, values):
        """Test edge cases in array dtype conversions."""
        array = Array("u8", values)
        original_bits = array.to_bits()
        
        # Convert to different compatible dtypes
        compatible_dtypes = ["bits8", "hex2"]  # Same bit width
        
        for new_dtype in compatible_dtypes:
            array.dtype = new_dtype
            # Should preserve the underlying bits
            assert array.to_bits() == original_bits
            
        # POSSIBLE FAILING TEST: Converting back might not preserve exact values
        # due to different interpretation of the same bits
        try:
            array.dtype = "u8"
            recovered_values = array.to_list()
            # This might fail if the intermediate dtype conversion changes the data
            assert recovered_values == values
        except AssertionError:
            # Document why this might fail
            pytest.skip("Expected failure: dtype conversion may not preserve values through intermediate types")
            
    def test_array_with_extreme_values(self):
        """Test arrays with extreme values for their dtypes."""
        # Test unsigned integer arrays at boundaries
        max_u8 = Array("u8", [0, 255])
        assert max_u8.to_list() == [0, 255]
        
        # Test signed integer arrays at boundaries
        signed_extremes = Array("i8", [-128, 127])
        assert signed_extremes.to_list() == [-128, 127]
        
        # Test what happens with overflow
        with pytest.raises(ValueError):
            Array("u8", [256])  # Should fail
            
        with pytest.raises(ValueError):
            Array("i8", [128])  # Should fail
            
    @given(st.integers(1, 64), st.lists(st.integers(), min_size=1, max_size=100))
    def test_array_bit_width_edge_cases(self, bit_width, values):
        """Test arrays with various bit widths and value ranges."""
        max_val = 2**(bit_width - 1) - 1  # For signed
        min_val = -(2**(bit_width - 1))
        
        # Filter values to fit in the bit width
        filtered_values = [v for v in values if min_val <= v <= max_val]
        if not filtered_values:
            assume(False)  # Skip if no valid values
            
        dtype_str = f"i{bit_width}"
        try:
            array = Array(dtype_str, filtered_values)
            recovered = array.to_list()
            assert recovered == filtered_values
        except (ValueError, OverflowError):
            # Some bit widths might not be supported
            # POSSIBLE FAILING TEST: Not all bit widths may be implemented
            pass


class TestDtypeEdgeCases:
    """Test edge cases for Dtype system."""
    
    def test_dtype_extreme_sizes(self):
        """Test dtypes with extreme sizes."""
        # Very small sizes
        tiny_dtype = DtypeSingle.from_params(DtypeKind.UINT, 1)
        assert tiny_dtype.size == 1
        
        # Larger sizes (but reasonable)
        large_dtype = DtypeSingle.from_params(DtypeKind.UINT, 128)
        assert large_dtype.size == 128
        
        # Test what happens with zero size
        with pytest.raises(ValueError):
            DtypeSingle.from_params(DtypeKind.UINT, 0)
            
    def test_dtype_string_parsing_edge_cases(self):
        """Test edge cases in dtype string parsing."""
        # Valid edge cases
        valid_cases = [
            "u1",
            "i128", 
            "f16",
            "bool",
            "bytes1",
            "hex",
            "bin",
        ]
        
        for case in valid_cases:
            dtype = Dtype.from_string(case)
            assert str(dtype) == case or str(dtype).startswith(case)
            
        # Invalid cases that should fail
        invalid_cases = [
            "u0",  # Zero size int
            "x8",  # Invalid kind
            "f7",  # Invalid float size
        ]
        
        for case in invalid_cases:
            with pytest.raises((ValueError, Exception)):
                Dtype.from_string(case)
                
    def test_dtype_array_edge_cases(self):
        """Test edge cases for array dtypes."""
        # Empty array
        empty_array_dtype = DtypeArray.from_string("[u8; 0]")
        assert empty_array_dtype.items == 0
        
        # Single item array
        single_array_dtype = DtypeArray.from_string("[u8; 1]")
        assert single_array_dtype.items == 1
        
        # Array without specified size
        variable_array_dtype = DtypeArray.from_string("[u8;]")
        # POSSIBLE FAILING TEST: Variable-sized arrays might not have a fixed items count
        try:
            items = variable_array_dtype.items
            assert items is None or isinstance(items, Expression)
        except AttributeError:
            # This might fail if the implementation doesn't support variable arrays properly
            pass
            
    def test_dtype_tuple_edge_cases(self):
        """Test edge cases for tuple dtypes."""
        # Single element tuple
        single_tuple = DtypeTuple.from_string("(u8)")
        assert single_tuple.items == 1
        
        # Empty tuple (if supported)
        try:
            empty_tuple = DtypeTuple.from_string("()")
            assert empty_tuple.items == 0
        except ValueError:
            # Empty tuples might not be supported
            pass
            
        # Large tuple
        large_tuple_str = "(" + ", ".join(["u8"] * 100) + ")"
        large_tuple = DtypeTuple.from_string(large_tuple_str)
        assert large_tuple.items == 100


class TestFormatEdgeCases:
    """Test edge cases for Format class."""
    
    def test_format_with_empty_fields(self):
        """Test formats with edge case field configurations."""
        # Format with no fields
        try:
            empty_format = Format("()")
            assert len(empty_format) == 0
        except ValueError:
            # Empty formats might not be supported
            pass
            
    def test_format_with_extreme_nesting(self):
        """Test deeply nested format structures."""
        # Nested tuples
        nested_str = "((u8, u8), (u16, u16))"
        nested_format = Format(nested_str)
        
        # POSSIBLE FAILING TEST: Deep nesting might have limits
        very_nested_str = "(" * 10 + "u8" + ")" * 10
        try:
            very_nested = Format(very_nested_str) 
        except (ValueError, RecursionError):
            # Deep nesting might hit limits
            pass
            
    def test_format_parsing_malformed_data(self):
        """Test format parsing with malformed or edge case data."""
        simple_format = Format("(x: u8, y: u8)")
        
        # Empty data
        with pytest.raises((ValueError, IndexError)):
            simple_format.parse(b"")
            
        # Insufficient data
        with pytest.raises((ValueError, IndexError)):
            simple_format.parse(b"\x42")  # Only one byte, need two
            
        # Extra data (should work but only consume what's needed)
        result = simple_format.parse(b"\x42\x43\x44\x45")
        assert result == 16  # Should consume 2 bytes (16 bits)
        
    @given(st.binary(min_size=0, max_size=100))
    def test_format_parsing_random_data(self, data):
        """Test format parsing with random binary data."""
        formats_to_test = [
            "(u8)",
            "(u16)",
            "(u8, u8)",
            "(f32)",
        ]
        
        for format_str in formats_to_test:
            format_obj = Format(format_str)
            expected_bits = format_obj.bit_length
            
            if expected_bits <= len(data) * 8:
                # Should be able to parse
                try:
                    bits_consumed = format_obj.parse(data)
                    assert bits_consumed == expected_bits
                except (ValueError, struct.error, OverflowError):
                    # Some random data might cause parsing errors with floats
                    # POSSIBLE FAILING TEST: Random data might not be valid for all dtypes
                    pass
            else:
                # Should fail due to insufficient data
                with pytest.raises((ValueError, IndexError)):
                    format_obj.parse(data)


class TestEndiannesssEdgeCases:
    """Test edge cases related to endianness."""
    
    def test_endianness_consistency(self):
        """Test that endianness is handled consistently."""
        value = 0x1234
        
        # Test different endianness formats
        big_endian = Bits.from_dtype("u16_be", value)
        little_endian = Bits.from_dtype("u16_le", value)
        native_endian = Bits.from_dtype("u16_ne", value)
        
        # They should be different (unless native happens to match one)
        if sys.byteorder == 'big':
            assert big_endian == native_endian
        else:
            assert little_endian == native_endian
            
        # But they should have same length
        assert len(big_endian) == len(little_endian) == len(native_endian) == 16
        
        # And unpack to same value
        assert big_endian.unpack(["u16_be"])[0] == value
        assert little_endian.unpack(["u16_le"])[0] == value
        assert native_endian.unpack(["u16_ne"])[0] == value
        
    @given(st.integers(0, 2**32 - 1))
    def test_endianness_roundtrip(self, value):
        """Test endianness roundtrip for various values."""
        for endian in ["_be", "_le", "_ne"]:
            for size in [16, 32]:
                if value >= 2**size:
                    continue
                    
                dtype_str = f"u{size}{endian}"
                bits = Bits.from_dtype(dtype_str, value)
                recovered = bits.unpack([dtype_str])[0]
                assert recovered == value


class TestErrorConditionsAndBoundaries:
    """Test error conditions and boundary cases."""
    
    def test_memory_efficiency_edge_cases(self):
        """Test memory-related edge cases."""
        # Very long bit strings
        long_bits = Bits("0b" + "1" * 10000)
        assert len(long_bits) == 10000
        
        # Operations on long bit strings
        long_bits2 = Bits("0b" + "0" * 10000)
        combined = long_bits + long_bits2
        assert len(combined) == 20000
        
    def test_unicode_and_encoding_edge_cases(self):
        """Test edge cases with unicode and string encoding."""
        # Test hex strings with various cases
        hex_cases = [
            "0xFF",
            "0xff", 
        ]
        
        for hex_str in hex_cases:
            bits = Bits(hex_str)
            assert bits.hex.lower() == "ff"
            
        # Test binary strings with whitespace
        bin_with_whitespace = "0b1010 0101"
        bits = Bits(bin_with_whitespace)
        assert bits.bin == "10100101"
        
    def test_type_conversion_edge_cases(self):
        """Test edge cases in type conversions."""
        # Converting between incompatible types
        bits = Bits("0x42")
        
        # Should work
        as_int = bits.unpack(["u8"])[0]
        assert as_int == 0x42
        
        # Float conversion from non-float-sized bits might fail
        try:
            as_float = bits.unpack(["f32"])[0]  # 8 bits -> 32-bit float
        except ValueError:
            # This should fail - 8 bits can't be interpreted as 32-bit float
            pass


class TestHypothesisPropertyBasedEdgeCases:
    """Property-based tests using Hypothesis to find edge cases."""
    
    @given(st.integers(1, 128), st.integers(0, 2**20))
    @settings(max_examples=50, suppress_health_check=[HealthCheck.too_slow])
    def test_roundtrip_property(self, bit_width, value):
        """Property: pack/unpack should be identity for valid inputs."""
        max_value = 2**bit_width - 1
        assume(value <= max_value)
        
        dtype_str = f"u{bit_width}"
        try:
            bits = Bits.from_dtype(dtype_str, value)
            recovered = bits.unpack([dtype_str])[0]
            assert recovered == value, f"Roundtrip failed for {value} with {bit_width} bits"
        except ValueError:
            # Some bit widths might not be supported
            assume(False)
            
    @given(st.lists(st.integers(0, 255), min_size=0, max_size=100))
    @settings(max_examples=30)
    def test_concatenation_property(self, byte_values):
        """Property: concatenating bits should preserve total length."""
        if not byte_values:
            return
            
        bits_list = [Bits.from_dtype("u8", val) for val in byte_values]
        
        # Concatenate all
        result = Bits()
        for bits in bits_list:
            result = result + bits
            
        expected_length = len(byte_values) * 8
        assert len(result) == expected_length
        
        # Should be able to unpack back to original values
        recovered = result.unpack([f"u8"] * len(byte_values))
        assert recovered == byte_values
        
    @given(st.binary(min_size=1, max_size=50))
    @settings(max_examples=30)
    def test_bytes_conversion_property(self, data):
        """Property: bytes -> bits -> bytes should be identity."""
        bits = Bits.from_bytes(data)
        recovered = bits.bytes
        assert recovered == data
        
        # Length should be consistent
        assert len(bits) == len(data) * 8


# Additional tests that might reveal implementation-specific edge cases
class TestImplementationSpecificEdgeCases:
    """Tests that probe implementation-specific behaviors."""
    
    def test_bit_rust_integration_edge_cases(self):
        """Test edge cases in the Rust/Python integration."""
        # Test that Python/Rust boundary works correctly
        bits = Bits("0x42")
        
        # Operations that might cross the boundary
        concatenated = bits + bits + bits
        assert len(concatenated) == 24
        
        # Slicing that might involve Rust code
        sliced = concatenated[4:12]
        assert len(sliced) == 8
        
    def test_memory_management_edge_cases(self):
        """Test memory management across Python/Rust boundary."""
        # Create many objects to test memory management
        objects = []
        for i in range(1000):
            obj = Bits.from_dtype("u16", i % 65536)
            objects.append(obj)
            
        # All should still be valid
        assert all(len(obj) == 16 for obj in objects)
        
    def test_thread_safety_indicators(self):
        """Basic tests that might reveal thread safety issues."""
        # This doesn't test actual threading, but tests operations that
        # might be problematic if not thread-safe
        bits = Bits("0x42")
        
        # Multiple simultaneous operations
        results = []
        for i in range(100):
            result = bits + Bits.from_dtype("u8", i % 256)
            results.append(result)
            
        # All should be different but valid
        assert len(set(str(r) for r in results)) > 1  # Should have variety
        assert all(len(r) == 16 for r in results)  # All should be valid


class TestAdvancedHypothesisStrategies:
    """Advanced hypothesis strategies for finding complex edge cases."""
    
    @given(st.lists(st.integers(0, 2**32-1), min_size=0, max_size=20).map(
        lambda x: [val for val in x if val.bit_length() <= 24]))
    @settings(max_examples=20)
    def test_variable_width_integers(self, values):
        """Test integers with varying bit requirements."""
        if not values:
            return
            
        for value in values:
            # Test with minimal bit width
            bit_width = max(value.bit_length(), 1)
            if bit_width <= 64:  # Keep reasonable
                dtype_str = f"u{bit_width}"
                try:
                    bits = Bits.from_dtype(dtype_str, value)
                    recovered = bits.unpack([dtype_str])[0]
                    assert recovered == value
                except ValueError:
                    # Some bit widths might not be supported
                    pass
                    
    @given(st.data())
    @settings(max_examples=10)
    def test_composite_format_edge_cases(self, data):
        """Use hypothesis data() to generate complex format combinations."""
        # Generate a random format structure
        num_fields = data.draw(st.integers(1, 5))
        field_types = data.draw(st.lists(
            st.sampled_from(["u8", "u16", "i8", "i16", "f32", "bool"]),
            min_size=num_fields, max_size=num_fields
        ))
        
        # Build format string
        format_str = "(" + ", ".join(field_types) + ")"
        
        try:
            format_obj = Format(format_str)
            # Generate appropriate test data
            test_data = bytearray()
            for field_type in field_types:
                if field_type.startswith("u8") or field_type.startswith("i8"):
                    test_data.extend(data.draw(st.binary(min_size=1, max_size=1)))
                elif field_type.startswith("u16") or field_type.startswith("i16"):
                    test_data.extend(data.draw(st.binary(min_size=2, max_size=2)))
                elif field_type == "f32":
                    test_data.extend(data.draw(st.binary(min_size=4, max_size=4)))
                elif field_type == "bool":
                    # For bool, we need at least 1 bit, but bytes are easier
                    test_data.extend(data.draw(st.binary(min_size=1, max_size=1)))
                    
            if len(test_data) > 0:
                # Try to parse
                try:
                    bits_consumed = format_obj.parse(bytes(test_data))
                    assert bits_consumed >= 0
                except (ValueError, struct.error):
                    # Random data might not be valid for all formats
                    pass
        except ValueError:
            # Some format combinations might not be valid
            pass


class TestUnicodeAndEncodingEdgeCases:
    """Test edge cases related to unicode and encoding handling."""
    
    def test_hex_string_unicode_edge_cases(self):
        """Test hex string parsing with unicode edge cases."""
        # Various unicode representations that might look like hex
        unicode_hex_cases = [
            "０ｘＦＦ",  # Full-width characters
            "0x\u0046\u0046",  # Unicode F's
            "0xff",  # Normal case
            "0xFF",  # Upper case
            "0X42",  # Capital X
        ]
        
        for case in unicode_hex_cases:
            try:
                # Most of these should fail unless specifically handled
                bits = Bits(case)
                # If it succeeds, should be valid
                assert len(bits) % 4 == 0  # Hex should be multiple of 4 bits
            except (ValueError, UnicodeError):
                # Expected for unicode edge cases
                pass
                
    def test_binary_string_edge_cases(self):
        """Test binary string parsing edge cases."""
        binary_cases = [
            "0b",      # Empty binary
            "0B1010",  # Capital B
            "1010",    # No prefix
            "0b\u0030\u0031",  # Unicode 0 and 1
            "0b1 0 1 0",  # With spaces
            "0b\n1\t0\r1\n0",  # With various whitespace
        ]
        
        for case in binary_cases:
            try:
                bits = Bits(case)
                # If successful, should be valid binary
                assert all(c in '01' for c in bits.bin)
            except ValueError:
                # Many of these are expected to fail
                pass


class TestFloatingPointEdgeCases:
    """Extended floating point edge case testing."""
                    
    def test_float_bit_patterns(self):
        """Test specific bit patterns that might cause float issues."""
        # Test bit patterns that represent special float values
        special_patterns = [
            "0x7f800000",  # f32 +infinity
            "0xff800000",  # f32 -infinity 
            "0x7fc00000",  # f32 NaN
            "0x00000000",  # f32 +0.0
            "0x80000000",  # f32 -0.0
            "0x00000001",  # f32 smallest positive subnormal
        ]
        
        for pattern in special_patterns:
            try:
                bits = Bits(pattern)
                if len(bits) == 32:  # Right size for f32
                    value = bits.unpack(["f32"])[0]
                    
                    # Should be able to round-trip back
                    bits2 = Bits.from_dtype("f32", value)
                    
                    # For special values, bit pattern might not be identical
                    # but the values should be equivalent
                    if math.isnan(value):
                        assert math.isnan(bits2.unpack(["f32"])[0])
                    elif math.isinf(value):
                        recovered = bits2.unpack(["f32"])[0]
                        assert math.isinf(recovered)
                        assert math.copysign(1, recovered) == math.copysign(1, value)
                    else:
                        assert bits2.unpack(["f32"])[0] == value
            except (ValueError, struct.error):
                # Some bit patterns might not be valid
                pass


if __name__ == "__main__":
    # Allow running the file directly for quick testing
    pytest.main([__file__, "-v"])