"""
OSDU Performance Tests - Locust Configuration
Generated by OSDU Performance Testing Framework

# Simple API-based performance testing for s service
"""

import os
import json
from locust import HttpUser
from locust import events, tag, task
from string import Template
from osdu_perf import PerformanceUser


# STEP 1: Register custom CLI args with Locust
# Please dont remove this code as it is required for OSDU parameters.
@events.init_command_line_parser.add_listener
def add_custom_args(parser):
    """Add OSDU-specific command line arguments"""
    parser.add_argument("--partition", type=str, default=os.getenv("PARTITION"), help="OSDU Data Partition ID")
    # Note: --host is provided by Locust built-in, no need to add it here
    # Note: --token is not exposed as CLI arg for security, only via environment variable
    parser.add_argument("--appid", type=str, default=os.getenv("APPID"), help="Azure AD Application ID")


class OSDUUser(HttpUser):
    """
    OSDU Performance Test User
    
    This class automatically:
    - Handles Azure authentication and token management
    - Manages HTTP headers and request setup
    - Provides simple API methods (get, post, put, delete)
    - Manages Locust user simulation and load testing
    
    Usage:
        locust -f locustfile.py --host https://your-api.com --partition your-partition --appid your-app-id
    """
    
    # Optional: Customize user behavior
    # Default `wait_time` is provided by `PerformanceUser` (between(1, 3)).
    # To override in the generated file, uncomment and import `between` from locust:
    # from locust import between
    # wait_time = between(1, 3)  # realistic pacing (recommended)
    # wait_time = between(0, 0)  # no wait (maximum load)
    
    def on_start(self):
        """Called when a user starts - performs setup"""
        # super().on_start()

        # Access OSDU parameters from Locust parsed options or environment variables.
        # The available APIs are detailed in the README and library documentation.
        # This code generation accelerates development by removing the need to manually refer to the library docs.
        # locust client is available as self.client. Also have self.get, self.put, self.post wrapper API to use
        # One time setup can be performed here.
        # each other load function must have task decorator.
        # token is supplied via environment variable which is not passed to azure load test. 

        self.config_obj = PerformanceUser(self.environment)
        self.partition = self.config_obj.get_partition()
        self.host = self.config_obj.get_host()
        self.headers = self.config_obj.get_headers()
        self.appid = self.config_obj.get_appid()
        self.logger = self.config_obj.get_logger()
        self.record_counter = 0
        self.legal_tag_name = "performance-test-legal-tag"

        #token = "eyJ0eXAiOiJKV1QiLCJhbGciOiJSUzI1NiIsIng1dCI6InJ0c0ZULWItN0x1WTdEVlllU05LY0lKN1ZuYyIsImtpZCI6InJ0c0ZULWItN0x1WTdEVlllU05LY0lKN1ZuYyJ9.eyJhdWQiOiJodHRwczovL21hbmFnZW1lbnQuY29yZS53aW5kb3dzLm5ldC8iLCJpc3MiOiJodHRwczovL3N0cy53aW5kb3dzLm5ldC83MmY5ODhiZi04NmYxLTQxYWYtOTFhYi0yZDdjZDAxMWRiNDcvIiwiaWF0IjoxNzYyODQ0MDkyLCJuYmYiOjE3NjI4NDQwOTIsImV4cCI6MTc2Mjg0OTc3MiwiX2NsYWltX25hbWVzIjp7Imdyb3VwcyI6InNyYzEifSwiX2NsYWltX3NvdXJjZXMiOnsic3JjMSI6eyJlbmRwb2ludCI6Imh0dHBzOi8vZ3JhcGgud2luZG93cy5uZXQvNzJmOTg4YmYtODZmMS00MWFmLTkxYWItMmQ3Y2QwMTFkYjQ3L3VzZXJzLzA0ZWZmMDQ2LWQzNWYtNDUyZS1hMjk3LTY1YjViZGExZDI3NC9nZXRNZW1iZXJPYmplY3RzIn19LCJhY3IiOiIxIiwiYWNycyI6WyJwMSIsImMxMCJdLCJhaW8iOiJBY1FBTy84YUFBQUFhK0ZhaHJwOWhnaXNvZGVmUFRkM2gyQ0hxMXJhTitXMU5zYk9hTXpXcGhQaVk2ZE5icy9VeDJTdnczbUxzeEJXNTBnaTlwQStJbTlrc1hRT1ZoZ2pRSHZWRU1neEZPR2l3bVhSemp5OWNKQ2x6K1EzMTdNTEVQZXkzUVEyVXh3czFGWENMZjJNYUVDTm5zamZCTVJ3c0ZKUW1TQXJ5dk9nbERlbDJKbkJsbnRtODZ2c0RJek45TFBWR3AzQkl4TzFBRnhkTzlIbFc0THNocGN0VkdXOXRVZWlwMEZ1ZTNkU2t2RVdEOTlEZ2ZGajFJRHZFSFFYZEh4QkhzU2pQK1U5IiwiYW1yIjpbInJzYSIsIm1mYSJdLCJhcHBpZCI6IjA0YjA3Nzk1LThkZGItNDYxYS1iYmVlLTAyZjllMWJmN2I0NiIsImFwcGlkYWNyIjoiMCIsImRldmljZWlkIjoiZTZlMjk4ODMtYjRhNi00YzM2LTg3MGItOGM1Njg0ZGI0ZDllIiwiZmFtaWx5X25hbWUiOiJDSiIsImdpdmVuX25hbWUiOiJKYW5yYWoiLCJpZHR5cCI6InVzZXIiLCJpcGFkZHIiOiI0MC42OS4yMi4xMjgiLCJuYW1lIjoiSmFucmFqIENKIiwib2lkIjoiMDRlZmYwNDYtZDM1Zi00NTJlLWEyOTctNjViNWJkYTFkMjc0Iiwib25wcmVtX3NpZCI6IlMtMS01LTIxLTIxMjc1MjExODQtMTYwNDAxMjkyMC0xODg3OTI3NTI3LTU3Mzc4MDc2IiwicHVpZCI6IjEwMDMyMDAxRjc5RDA5ODEiLCJwd2RfdXJsIjoiaHR0cHM6Ly9nby5taWNyb3NvZnQuY29tL2Z3bGluay8_bGlua2lkPTIyMjQxOTgiLCJyaCI6IjEuQVJvQXY0ajVjdkdHcjBHUnF5MTgwQkhiUjBaSWYza0F1dGRQdWtQYXdmajJNQk1hQU9nYUFBLiIsInNjcCI6InVzZXJfaW1wZXJzb25hdGlvbiIsInNpZCI6IjAwOGIwOWQ5LTE2ZTAtMGFkNC1jYzlkLWVlZDJkMjdkY2JhOCIsInN1YiI6InUySlUwY3V3eHptWHRnY05TZ3BJNFJnQ2ZWZzVPeHM0ckp3S1hsTUIzd2siLCJ0aWQiOiI3MmY5ODhiZi04NmYxLTQxYWYtOTFhYi0yZDdjZDAxMWRiNDciLCJ1bmlxdWVfbmFtZSI6ImphbnJhamNqQG1pY3Jvc29mdC5jb20iLCJ1cG4iOiJqYW5yYWpjakBtaWNyb3NvZnQuY29tIiwidXRpIjoiMDFiUU0tNVVJVVNqc0E1bExrRTdBQSIsInZlciI6IjEuMCIsIndpZHMiOlsiYjc5ZmJmNGQtM2VmOS00Njg5LTgxNDMtNzZiMTk0ZTg1NTA5Il0sInhtc19hY3RfZmN0IjoiMyA1IiwieG1zX2Z0ZCI6InFQRUhvYXdOLVRSR3dna1BEZ0lhN0RISGtva05pVU9jdkQ2cFE5bEZaNjRCYTI5eVpXRnpiM1YwYUMxa2MyMXoiLCJ4bXNfaWRyZWwiOiIxIDEwIiwieG1zX3N1Yl9mY3QiOiI4IDMiLCJ4bXNfdGNkdCI6MTI4OTI0MTU0N30.LRZwOZ1hdu7xv2gybcmptYYfOr3i0qpJ13vl285UjdAsCHI1aMjIN7kfQhtKV8VFU1Ve7M7F1VQdTKFeNM1IAxI_TI1v71hD0y900KDtVynlX3qwC5FxiQl3gj66UrI3zXI9oIFA9Rnx7o57H-mGhu8xunndjeowMACIaCvgzins895qJu1Zq7MJZAdlu7l8CQFUvbm2iWIz0goAWDfhtX2bkS1G-TOyQwevuDxnpkS2uWi5eZ3yWTZgLvze4uwTGqKJcGY9ZY8dfDX63BTcrekxfooOBA_sPvNty3hOnEADgHafgSLZhsNSs__2tVdRKbNjMSmNNP06AkPrZcg4Hw"
        #self.headers["Authorization"] = "Bearer "+token
    
        self.create_legal_tag()

        self.logger.info(f"🚀 Started performance testing user")
        self.logger.info(f"   📍 Partition: {self.partition}")
        self.logger.info(f"   🌐 Host: {self.host}")
        self.logger.info(f"   🆔 App ID: {self.appid or 'Not provided'}")

    def on_stop(self):
        """Called when a user stops - performs cleanup"""
        self.logger.info("🛑 Stopped performance testing user")

    @tag("storage", "health_check")
    @task(1)
    def check_service_health(self):
        # *** Simple and clean consumer call ***
        self.get("/api/storage/v2/liveness_check")

    @tag("storage", "record_size_1KB")
    @task(2)
    def test_create_storage_record_file_generic(self):
        self.logger.info("[Test] [create a storage record of kind File.Generic] API tested: /api/storage/v2/records")
    
        # Generate unique record ID using incrementing counter
        self.record_counter += 1
        partition = self.partition
        # Construct the storage record payload following OSDU File.Generic schema
        payload = [{
            "id": f"{partition}:dataset--File.Generic:{self.record_counter}",
            "kind": f"{partition}:wks:dataset--File.Generic:1.0.0",
            "acl": {
                "owners": [f"data.default.owners@{partition}.dataservices.energy"],
                "viewers": [f"data.default.viewers@{partition}.dataservices.energy"]
            },
            "legal": {
                "legaltags": [f"{partition}-{self.legal_tag_name}"],
                "otherRelevantDataCountries": ["US"],
                "status": "compliant"
            },
            "data": {
                "Name": "Example File",
                "ResourceSecurityClassification": "Public",
                "FileSourceInfo": {
                    "FileSource": "s3://bucket/sample.csv",
                    "PreloadFilePath": "sample.csv"
                }
            }
        }]
        
        url = self.host + "/api/storage/v2/records"
        # Create the storage record using PUT method
        # Note: Storage API uses PUT for record creation/update operations
        response = self.client.put(url, data=json.dumps(payload), headers=self.headers)
        self.logger.info(f"Create record status: {response.status_code}")
        
        if response.status_code == 201:
            self.logger.info(f"✅ Successfully created record: {payload[0]['id']}")
        else:
            self.logger.error(f"❌ Failed to create record: {payload[0]['id']} Response message= {response.text}")


    @tag("storage")
    @task(3)
    def test_get_storage_record(self):
        """
        Test retrieving a specific storage record by its ID.
        
        This test retrieves a storage record using its unique identifier.
        It attempts to fetch the record that was created in the previous test.
        
        API Endpoint: GET /api/storage/v2/records/{id}
        
        Args:
            headers (dict): HTTP headers including authentication
            partition (str): Data partition ID for multi-tenant isolation
            base_url (str): Base URL for the storage service
        """
        self.logger.info("API tested: /api/storage/v2/records/{id}")
        
        # Use the most recently created record ID, or fallback to ID with counter 1

        record_id = f"{self.partition}:dataset--File.Generic:1"

        self.logger.info(f"Retrieving record with ID: {record_id}")
        endpoint = f"/api/storage/v2/records/{record_id}"
        

        # Retrieve the storage record
        response = self.client.get(endpoint, headers=self.headers)
        self.logger.info(f"Get record status: {response.status_code}")

        if response.status_code == 200:
            self.logger.info(f"Successfully retrieved record: {record_id}")
            # Optionally, validate the retrieved record structure
            try:
                record_data = response.json()
                self.logger.info(f"   Record kind: {record_data.get('kind', 'Unknown')}")
            except Exception as e:
                self.logger.error(f"   Could not parse record data: {e}")
        else:
            self.logger.error(f"Failed to retrieve record: {record_id}")


    @tag("storage")
    @task(4)
    def test_get_all_records_with_kind(self):
        """
        Test querying all storage records with a specific kind using GET method.
        
        This test retrieves all storage records that match a specific kind pattern
        using query parameters. It uses the GET method with URL parameters to
        filter records by kind, limit results, and sort order.
        
        API Endpoint: GET /api/storage/v2/query/records
        
        Args:
            headers (dict): HTTP headers including authentication
            partition (str): Data partition ID for multi-tenant isolation
            base_url (str): Base URL for the storage service
        """
        self.logger.info("[Test] [get all storage records with kind] API tested: /api/storage/v2/query/records (GET)")
        
        
        # Define query parameters for filtering and pagination
        params = {
            'limit': 1000,  # Maximum number of records to return
            'kind': f"{self.partition}:wks:dataset--File.Generic:1.0.0",  # Filter by record kind
            'deleted': 'false',  # Only retrieve active (non-deleted) records
            'sortOrder': 'DESC'  # Sort by creation time, newest first
        }
        
        
        self.logger.info(f"Retrieving records with parameters: {params}")
        
        # Execute the query request
        response = self.get("/api/storage/v2/query/records", params=params)
        self.logger.info(f"Get all records status: {response.status_code}")

        if response.status_code == 200:
            try:
                response_data = response.json()
                record_count = len(response_data.get('records', []))
                total_count = response_data.get('totalCount', 0)
                self.logger.info(f"Successfully retrieved {record_count} records (Total: {total_count})")
            except Exception as e:
                self.logger.error(f"Response received but failed to parse JSON: {e}")
        else:
            self.logger.error(f"Failed to retrieve records. Status: {response.status_code}")
            try:
                self.logger.error(f"Error response: {response.text}")
            except:
                self.logger.error("Could not read response text")

    @tag("storage", "record_size_100KB")
    @task(6)
    def test_create_storage_record_100_KB(self):
        self.logger.info("[Test] [create a 100KB storage record] API tested: /api/storage/v2/records")
    
        # Generate unique record ID using incrementing counter
        self.record_counter += 1
        
        try:
            # Load the 100KB JSON template
            import os
            script_dir = os.path.dirname(os.path.abspath(__file__))
            template_file = os.path.join(script_dir, "perf_storage_record_100KB.json")
            
            with open(template_file, 'r') as f:
                template_content = Template(f.read())
            
            # Replace template variables with actual values
            filled_content = template_content.substitute(
                partition=self.partition,
                record_counter=self.record_counter,
                legal_tag_name=self.legal_tag_name
            )
            
            # Parse the filled template
            large_payload = json.loads(filled_content)
            
            # Wrap in array format as expected by OSDU Storage API
            payload = [large_payload]
            
            url = self.host + "/api/storage/v2/records"
            # Create the storage record using PUT method
            response = self.client.put(url, data=json.dumps(payload), headers=self.headers)
            self.logger.info(f"Create 100KB record status: {response.status_code}")
            
            if response.status_code == 201:
                self.logger.info(f"✅ Successfully created 100KB record: {large_payload['id']}")
            else:
                self.logger.error(f"❌ Failed to create 100KB record: {large_payload['id']} Response: {response.text}")
                
        except Exception as e:
            self.logger.error(f"❌ Error loading 100KB template: {e}")

    @tag("storage", "record_size_1MB")
    @task(7)
    def test_create_storage_record_1_MB(self):
        self.logger.info("[Test] [create a 1MB storage record] API tested: /api/storage/v2/records")
    
        # Generate unique record ID using incrementing counter
        self.record_counter += 1
        
        try:
            # Load the 1MB JSON template
            import os
            script_dir = os.path.dirname(os.path.abspath(__file__))
            template_file = os.path.join(script_dir, "perf_storage_record_1MB.json")
            
            with open(template_file, 'r') as f:
                template_content = Template(f.read())
            
            # Replace template variables with actual values
            filled_content = template_content.substitute(
                partition=self.partition,
                record_counter=self.record_counter,
                legal_tag_name=self.legal_tag_name
            )
            
            # Parse the filled template
            large_payload = json.loads(filled_content)
            
            # Wrap in array format as expected by OSDU Storage API
            payload = [large_payload]
            
            url = self.host + "/api/storage/v2/records"
            # Create the storage record using PUT method
            response = self.client.put(url, data=json.dumps(payload), headers=self.headers)
            self.logger.info(f"Create 1MB record status: {response.status_code}")
            
            if response.status_code == 201:
                self.logger.info(f"✅ Successfully created 1MB record: {large_payload['id']}")
            else:
                self.logger.error(f"❌ Failed to create 1MB record: {large_payload['id']} Response: {response.text}")
                
        except Exception as e:
            self.logger.error(f"❌ Error loading 1MB template: {e}")

    @tag("storage")
    @task(5)
    def test_query_records_by_attributes(self):
        self.logger.info("[Test] [query storage records by attributes] API tested: /api/storage/v2/query/records,  POST")
    
        # Create the JSON payload based on the curl command
        # Using the created record ID to query for specific records
        query_payload = {
            "records": [
                f"{self.partition}:dataset--File.Generic:{self.record_counter}"  # Query for the created record
            ],
            "attributes": [
                "id",
                "kind", 
                "acl",
                "legal",
                "data"
            ]
        }


        self.logger.info(f"Querying records with payload: {query_payload}")

        response = self.post("/api/storage/v2/query/records", data=query_payload)
        self.logger.info(f"Query records status: {response.status_code}")

        if response.status_code == 200:
            try:
                response_data = response.json()
                records = response_data.get('records', [])
                record_count = len(records)
                self.logger.info(f"✅ Successfully queried {record_count} records")
                    
            except Exception as e:
                self.logger.error(f"⚠️ Response received but failed to parse JSON: {e}")
                self.logger.error(f"Raw response: {response.text}")
        else:
            self.logger.error(f"❌ Failed to query records. Status: {response.status_code}")
            try:
                self.logger.error(f"Response: {response.text}")
            except:
                self.logger.error("Could not read response text")


    def create_legal_tag(self):
        try:
            # Create legal tag required for storage records
            url = self.host + "/api/legal/v1/legaltags"

            #self.logger.info(f"self.token = {self.token}")
      
            # Set required headers
            headers = self.headers
            #headers["Content-Type"] = "application/json"
            #headers["data-partition-id"] = self.partition
            #headers['accept'] = 'application/json'
            #headers['Authorization'] = f"Bearer {self.token}"

            payload = {
                "name": self.legal_tag_name,
                "description": "Legal Tag added for Storage Performance Tests",
                "properties": {
                    "countryOfOrigin": ["US"],
                    "contractId": "No Contract Related",
                    "expirationDate": "2099-01-01",
                    "dataType": "Public Domain Data",
                    "originator": "OSDU",
                    "securityClassification": "Public",
                    "exportClassification": "EAR99",
                    "personalData": "No Personal Data"
                }
            }
            
            import requests
            response = requests.post(url, json=payload, headers=headers)
            self.logger.info(f"Create Legal Tag '{self.legal_tag_name}' url = {url} and status: {response.status_code}")
            
            if response.status_code in [200, 201, 409]:  # 409 means already exists
                self.logger.info(f" ✅ Legal tag '{self.legal_tag_name}' is ready for use")
            else:
                self.logger.error(f"❌ Failed to create legal tag: {response.text}")
                
        except Exception as e:
            self.logger.error(f"❌ Error in prehook setup: {e}")
            # Don't mark as executed if there was an error
            return
