"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const aws_wafv2_1 = require("aws-cdk-lib/aws-wafv2");
const nag_rules_1 = require("../../nag-rules");
/**
 * Rest API stages are associated with AWS WAFv2 web ACLs
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_apigateway_1.CfnStage) {
        const stageLogicalId = nag_rules_1.NagRules.resolveResourceFromIntrinsic(node, node.ref);
        const stageName = nag_rules_1.NagRules.resolveResourceFromIntrinsic(node, node.stageName);
        const restApiId = nag_rules_1.NagRules.resolveResourceFromIntrinsic(node, node.restApiId);
        let found = false;
        for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_wafv2_1.CfnWebACLAssociation) {
                if (isMatchingWebACLAssociation(child, stageLogicalId, stageName, restApiId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: (0, path_1.parse)(__filename).name });
/**
 * Helper function to check whether a given Web ACL Association is associated with the given Rest API
 * @param node the CfnWebACLAssociation to check
 * @param stageLogicalId the Cfn Logical ID of the REST API Stage
 * @param stageName the name of the REST API Stage
 * @param restApiId the ID or Cfn Resource ID of the REST API associated with the Stage
 * returns whether the CfnWebACLAssociation is associates with the given Rest API
 */
function isMatchingWebACLAssociation(node, stageLogicalId, stageName, restApiId) {
    const resourceLogicalId = JSON.stringify(aws_cdk_lib_1.Stack.of(node).resolve(node.resourceArn));
    const regexes = Array();
    regexes.push(`${restApiId}.+${stageLogicalId}(?![\\w])`);
    if (stageName !== undefined) {
        regexes.push(`${restApiId}.+${stageName}(?![\\w])`);
    }
    const regex = new RegExp(regexes.join('|'), 'gm');
    if (regex.test(resourceLogicalId)) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiQVBJR1dBc3NvY2lhdGVkV2l0aFdBRi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9ydWxlcy9hcGlndy9BUElHV0Fzc29jaWF0ZWRXaXRoV0FGLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUE7OztFQUdFO0FBQ0YsK0JBQTZCO0FBQzdCLDZDQUFpRDtBQUNqRCwrREFBc0Q7QUFDdEQscURBQTZEO0FBQzdELCtDQUE4RDtBQUU5RDs7O0dBR0c7QUFDSCxrQkFBZSxNQUFNLENBQUMsY0FBYyxDQUNsQyxDQUFDLElBQWlCLEVBQXFCLEVBQUU7SUFDdkMsSUFBSSxJQUFJLFlBQVkseUJBQVEsRUFBRSxDQUFDO1FBQzdCLE1BQU0sY0FBYyxHQUFHLG9CQUFRLENBQUMsNEJBQTRCLENBQzFELElBQUksRUFDSixJQUFJLENBQUMsR0FBRyxDQUNULENBQUM7UUFDRixNQUFNLFNBQVMsR0FBRyxvQkFBUSxDQUFDLDRCQUE0QixDQUNyRCxJQUFJLEVBQ0osSUFBSSxDQUFDLFNBQVMsQ0FDZixDQUFDO1FBQ0YsTUFBTSxTQUFTLEdBQUcsb0JBQVEsQ0FBQyw0QkFBNEIsQ0FDckQsSUFBSSxFQUNKLElBQUksQ0FBQyxTQUFTLENBQ2YsQ0FBQztRQUNGLElBQUksS0FBSyxHQUFHLEtBQUssQ0FBQztRQUNsQixLQUFLLE1BQU0sS0FBSyxJQUFJLG1CQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDO1lBQ2xELElBQUksS0FBSyxZQUFZLGdDQUFvQixFQUFFLENBQUM7Z0JBQzFDLElBQ0UsMkJBQTJCLENBQ3pCLEtBQUssRUFDTCxjQUFjLEVBQ2QsU0FBUyxFQUNULFNBQVMsQ0FDVixFQUNELENBQUM7b0JBQ0QsS0FBSyxHQUFHLElBQUksQ0FBQztvQkFDYixNQUFNO2dCQUNSLENBQUM7WUFDSCxDQUFDO1FBQ0gsQ0FBQztRQUNELElBQUksQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUNYLE9BQU8sNkJBQWlCLENBQUMsYUFBYSxDQUFDO1FBQ3pDLENBQUM7UUFDRCxPQUFPLDZCQUFpQixDQUFDLFNBQVMsQ0FBQztJQUNyQyxDQUFDO1NBQU0sQ0FBQztRQUNOLE9BQU8sNkJBQWlCLENBQUMsY0FBYyxDQUFDO0lBQzFDLENBQUM7QUFDSCxDQUFDLEVBQ0QsTUFBTSxFQUNOLEVBQUUsS0FBSyxFQUFFLElBQUEsWUFBSyxFQUFDLFVBQVUsQ0FBQyxDQUFDLElBQUksRUFBRSxDQUNsQyxDQUFDO0FBRUY7Ozs7Ozs7R0FPRztBQUNILFNBQVMsMkJBQTJCLENBQ2xDLElBQTBCLEVBQzFCLGNBQXNCLEVBQ3RCLFNBQTZCLEVBQzdCLFNBQWlCO0lBRWpCLE1BQU0saUJBQWlCLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FDdEMsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FDekMsQ0FBQztJQUNGLE1BQU0sT0FBTyxHQUFHLEtBQUssRUFBVSxDQUFDO0lBQ2hDLE9BQU8sQ0FBQyxJQUFJLENBQUMsR0FBRyxTQUFTLEtBQUssY0FBYyxXQUFXLENBQUMsQ0FBQztJQUN6RCxJQUFJLFNBQVMsS0FBSyxTQUFTLEVBQUUsQ0FBQztRQUM1QixPQUFPLENBQUMsSUFBSSxDQUFDLEdBQUcsU0FBUyxLQUFLLFNBQVMsV0FBVyxDQUFDLENBQUM7SUFDdEQsQ0FBQztJQUNELE1BQU0sS0FBSyxHQUFHLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDbEQsSUFBSSxLQUFLLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEVBQUUsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFDRCxPQUFPLEtBQUssQ0FBQztBQUNmLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG5TUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuKi9cbmltcG9ydCB7IHBhcnNlIH0gZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBDZm5SZXNvdXJjZSwgU3RhY2sgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBDZm5TdGFnZSB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1hcGlnYXRld2F5JztcbmltcG9ydCB7IENmbldlYkFDTEFzc29jaWF0aW9uIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLXdhZnYyJztcbmltcG9ydCB7IE5hZ1J1bGVDb21wbGlhbmNlLCBOYWdSdWxlcyB9IGZyb20gJy4uLy4uL25hZy1ydWxlcyc7XG5cbi8qKlxuICogUmVzdCBBUEkgc3RhZ2VzIGFyZSBhc3NvY2lhdGVkIHdpdGggQVdTIFdBRnYyIHdlYiBBQ0xzXG4gKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAqL1xuZXhwb3J0IGRlZmF1bHQgT2JqZWN0LmRlZmluZVByb3BlcnR5KFxuICAobm9kZTogQ2ZuUmVzb3VyY2UpOiBOYWdSdWxlQ29tcGxpYW5jZSA9PiB7XG4gICAgaWYgKG5vZGUgaW5zdGFuY2VvZiBDZm5TdGFnZSkge1xuICAgICAgY29uc3Qgc3RhZ2VMb2dpY2FsSWQgPSBOYWdSdWxlcy5yZXNvbHZlUmVzb3VyY2VGcm9tSW50cmluc2ljKFxuICAgICAgICBub2RlLFxuICAgICAgICBub2RlLnJlZlxuICAgICAgKTtcbiAgICAgIGNvbnN0IHN0YWdlTmFtZSA9IE5hZ1J1bGVzLnJlc29sdmVSZXNvdXJjZUZyb21JbnRyaW5zaWMoXG4gICAgICAgIG5vZGUsXG4gICAgICAgIG5vZGUuc3RhZ2VOYW1lXG4gICAgICApO1xuICAgICAgY29uc3QgcmVzdEFwaUlkID0gTmFnUnVsZXMucmVzb2x2ZVJlc291cmNlRnJvbUludHJpbnNpYyhcbiAgICAgICAgbm9kZSxcbiAgICAgICAgbm9kZS5yZXN0QXBpSWRcbiAgICAgICk7XG4gICAgICBsZXQgZm91bmQgPSBmYWxzZTtcbiAgICAgIGZvciAoY29uc3QgY2hpbGQgb2YgU3RhY2sub2Yobm9kZSkubm9kZS5maW5kQWxsKCkpIHtcbiAgICAgICAgaWYgKGNoaWxkIGluc3RhbmNlb2YgQ2ZuV2ViQUNMQXNzb2NpYXRpb24pIHtcbiAgICAgICAgICBpZiAoXG4gICAgICAgICAgICBpc01hdGNoaW5nV2ViQUNMQXNzb2NpYXRpb24oXG4gICAgICAgICAgICAgIGNoaWxkLFxuICAgICAgICAgICAgICBzdGFnZUxvZ2ljYWxJZCxcbiAgICAgICAgICAgICAgc3RhZ2VOYW1lLFxuICAgICAgICAgICAgICByZXN0QXBpSWRcbiAgICAgICAgICAgIClcbiAgICAgICAgICApIHtcbiAgICAgICAgICAgIGZvdW5kID0gdHJ1ZTtcbiAgICAgICAgICAgIGJyZWFrO1xuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfVxuICAgICAgaWYgKCFmb3VuZCkge1xuICAgICAgICByZXR1cm4gTmFnUnVsZUNvbXBsaWFuY2UuTk9OX0NPTVBMSUFOVDtcbiAgICAgIH1cbiAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5DT01QTElBTlQ7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5OT1RfQVBQTElDQUJMRTtcbiAgICB9XG4gIH0sXG4gICduYW1lJyxcbiAgeyB2YWx1ZTogcGFyc2UoX19maWxlbmFtZSkubmFtZSB9XG4pO1xuXG4vKipcbiAqIEhlbHBlciBmdW5jdGlvbiB0byBjaGVjayB3aGV0aGVyIGEgZ2l2ZW4gV2ViIEFDTCBBc3NvY2lhdGlvbiBpcyBhc3NvY2lhdGVkIHdpdGggdGhlIGdpdmVuIFJlc3QgQVBJXG4gKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuV2ViQUNMQXNzb2NpYXRpb24gdG8gY2hlY2tcbiAqIEBwYXJhbSBzdGFnZUxvZ2ljYWxJZCB0aGUgQ2ZuIExvZ2ljYWwgSUQgb2YgdGhlIFJFU1QgQVBJIFN0YWdlXG4gKiBAcGFyYW0gc3RhZ2VOYW1lIHRoZSBuYW1lIG9mIHRoZSBSRVNUIEFQSSBTdGFnZVxuICogQHBhcmFtIHJlc3RBcGlJZCB0aGUgSUQgb3IgQ2ZuIFJlc291cmNlIElEIG9mIHRoZSBSRVNUIEFQSSBhc3NvY2lhdGVkIHdpdGggdGhlIFN0YWdlXG4gKiByZXR1cm5zIHdoZXRoZXIgdGhlIENmbldlYkFDTEFzc29jaWF0aW9uIGlzIGFzc29jaWF0ZXMgd2l0aCB0aGUgZ2l2ZW4gUmVzdCBBUElcbiAqL1xuZnVuY3Rpb24gaXNNYXRjaGluZ1dlYkFDTEFzc29jaWF0aW9uKFxuICBub2RlOiBDZm5XZWJBQ0xBc3NvY2lhdGlvbixcbiAgc3RhZ2VMb2dpY2FsSWQ6IHN0cmluZyxcbiAgc3RhZ2VOYW1lOiBzdHJpbmcgfCB1bmRlZmluZWQsXG4gIHJlc3RBcGlJZDogc3RyaW5nXG4pOiBib29sZWFuIHtcbiAgY29uc3QgcmVzb3VyY2VMb2dpY2FsSWQgPSBKU09OLnN0cmluZ2lmeShcbiAgICBTdGFjay5vZihub2RlKS5yZXNvbHZlKG5vZGUucmVzb3VyY2VBcm4pXG4gICk7XG4gIGNvbnN0IHJlZ2V4ZXMgPSBBcnJheTxzdHJpbmc+KCk7XG4gIHJlZ2V4ZXMucHVzaChgJHtyZXN0QXBpSWR9Liske3N0YWdlTG9naWNhbElkfSg/IVtcXFxcd10pYCk7XG4gIGlmIChzdGFnZU5hbWUgIT09IHVuZGVmaW5lZCkge1xuICAgIHJlZ2V4ZXMucHVzaChgJHtyZXN0QXBpSWR9Liske3N0YWdlTmFtZX0oPyFbXFxcXHddKWApO1xuICB9XG4gIGNvbnN0IHJlZ2V4ID0gbmV3IFJlZ0V4cChyZWdleGVzLmpvaW4oJ3wnKSwgJ2dtJyk7XG4gIGlmIChyZWdleC50ZXN0KHJlc291cmNlTG9naWNhbElkKSkge1xuICAgIHJldHVybiB0cnVlO1xuICB9XG4gIHJldHVybiBmYWxzZTtcbn1cbiJdfQ==