"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const aws_apigatewayv2_1 = require("aws-cdk-lib/aws-apigatewayv2");
const nag_rules_1 = require("../../nag-rules");
function checkCORSMethodResponses(node) {
    const methodResponses = aws_cdk_lib_1.Stack.of(node).resolve(node.methodResponses);
    return methodResponses?.every((response) => {
        const hasCORSResponseParameter = Object.entries(response.responseParameters || {}).some(([key, value]) => key.startsWith('method.response.header.Access-Control-Allow-') &&
            value === true);
        return response.statusCode === '204' && hasCORSResponseParameter;
    });
}
/**
 * APIs implement authorization
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_apigateway_1.CfnMethod || node instanceof aws_apigatewayv2_1.CfnRoute) {
        if (node instanceof aws_apigateway_1.CfnMethod) {
            const httpMethod = nag_rules_1.NagRules.resolveIfPrimitive(node, node.httpMethod);
            if (httpMethod === 'OPTIONS' && checkCORSMethodResponses(node)) {
                return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
            }
        }
        const authorizationType = nag_rules_1.NagRules.resolveIfPrimitive(node, node.authorizationType);
        if (authorizationType == undefined ||
            authorizationType == aws_apigateway_1.AuthorizationType.NONE) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: (0, path_1.parse)(__filename).name });
//# sourceMappingURL=data:application/json;base64,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