use crate::functions::results::{compute_member_results_from_displacement, extract_displacements};
use crate::models::members::memberhinge::{classify_from_hinges, AxisMode, AxisModes};
use nalgebra::DMatrix;
use serde::{Deserialize, Serialize};
use std::collections::{BTreeMap, HashMap, HashSet};
use utoipa::ToSchema;
// use csv::Writer;
// use std::error::Error;
use crate::functions::reactions::{
    compose_support_reaction_vector_equilibrium, extract_reaction_nodes,
};
use crate::models::imperfections::imperfectioncase::ImperfectionCase;
use crate::models::loads::loadcase::LoadCase;
use crate::models::loads::loadcombination::LoadCombination;
use crate::models::members::enums::MemberType;
use crate::models::members::memberset::MemberSet;
use crate::models::members::{
    material::Material, memberhinge::MemberHinge, section::Section, shapepath::ShapePath,
};
use crate::models::results::resultbundle::ResultsBundle;
use crate::models::results::results::{ResultType, Results};
use crate::models::results::resultssummary::ResultsSummary;
use crate::models::settings::settings::Settings;
use crate::models::supports::nodalsupport::NodalSupport;
use crate::models::supports::supportconditiontype::SupportConditionType;

use crate::functions::hinge_and_release_operations::{
    apply_end_releases_to_local_beam_k, build_local_truss_translational_spring_k,
    modes_from_single_ends,
};
use crate::functions::load_assembler::{
    assemble_distributed_loads, assemble_nodal_loads, assemble_nodal_moments,
};

#[derive(Serialize, Deserialize, ToSchema, Debug)]
pub struct FERS {
    pub member_sets: Vec<MemberSet>,
    pub load_cases: Vec<LoadCase>,
    pub load_combinations: Vec<LoadCombination>,
    pub imperfection_cases: Vec<ImperfectionCase>,
    pub settings: Settings,
    pub results: Option<ResultsBundle>,
    pub memberhinges: Option<Vec<MemberHinge>>,
    pub materials: Vec<Material>,
    pub sections: Vec<Section>,
    pub nodal_supports: Vec<NodalSupport>,
    pub shape_paths: Option<Vec<ShapePath>>,
}

const AXIAL_SLACK_TOLERANCE_DEFAULT: f64 = 1.0e-6;

struct RigidElimination {
    s: DMatrix<f64>,
    full_to_red: HashMap<usize, usize>,
}

pub struct AssemblyContext<'a> {
    pub material_by_id: HashMap<u32, &'a Material>,
    pub section_by_id: HashMap<u32, &'a Section>,
    pub hinge_by_id: HashMap<u32, &'a MemberHinge>,
    pub support_by_id: HashMap<u32, &'a NodalSupport>,
}

impl<'a> AssemblyContext<'a> {
    pub fn new(model: &'a FERS) -> Self {
        let (material_by_id, section_by_id, hinge_by_id, support_by_id) = model.build_lookup_maps();
        Self {
            material_by_id,
            section_by_id,
            hinge_by_id,
            support_by_id,
        }
    }
}

const TRANSLATION_AXES: [(&str, usize); 3] = [("X", 0), ("Y", 1), ("Z", 2)];
const ROTATION_AXES: [(&str, usize); 3] = [("X", 3), ("Y", 4), ("Z", 5)];

macro_rules! get_case_insensitive {
    ($map:expr, $key:expr) => {
        $map.get($key)
            .or_else(|| $map.get(&$key.to_ascii_lowercase()))
    };
}

impl FERS {
    fn visit_unique_supported_nodes<F>(&self, mut visitor: F) -> Result<(), String>
    where
        F: FnMut(u32, usize, &NodalSupport) -> Result<(), String>,
    {
        use std::collections::{HashMap, HashSet};

        let support_by_id: HashMap<u32, &NodalSupport> =
            self.nodal_supports.iter().map(|s| (s.id, s)).collect();

        let mut visited_node_ids: HashSet<u32> = HashSet::new();

        for member_set in &self.member_sets {
            for member in &member_set.members {
                for node in [&member.start_node, &member.end_node] {
                    if !visited_node_ids.insert(node.id) {
                        continue;
                    }
                    if let Some(support_id) = node.nodal_support {
                        if let Some(nodal_support) = support_by_id.get(&support_id) {
                            let base_index = (node.id as usize - 1) * 6;
                            visitor(node.id, base_index, *nodal_support)?; // propagate
                        }
                    }
                }
            }
        }
        Ok(())
    }

    fn spring_stiffness_or_error(
        owner_id: u32,
        axis_label: &str,
        kind_label: &str,
        stiffness: Option<f64>,
    ) -> Result<f64, String> {
        let k = stiffness.ok_or_else(|| {
            format!(
                "Support {} {} {} is Spring but stiffness is missing.",
                owner_id, kind_label, axis_label
            )
        })?;
        if k <= 0.0 {
            return Err(format!(
                "Support {} {} {} Spring stiffness must be positive.",
                owner_id, kind_label, axis_label
            ));
        }
        Ok(k)
    }

    pub fn build_lookup_maps(
        &self,
    ) -> (
        HashMap<u32, &Material>,
        HashMap<u32, &Section>,
        HashMap<u32, &MemberHinge>,
        HashMap<u32, &NodalSupport>,
    ) {
        let material_map: HashMap<u32, &Material> =
            self.materials.iter().map(|m| (m.id, m)).collect();
        let section_map: HashMap<u32, &Section> = self.sections.iter().map(|s| (s.id, s)).collect();
        let memberhinge_map: HashMap<u32, &MemberHinge> = self
            .memberhinges
            .iter()
            .flatten()
            .map(|mh| (mh.id, mh))
            .collect();
        let support_map: HashMap<u32, &NodalSupport> =
            self.nodal_supports.iter().map(|s| (s.id, s)).collect();

        (material_map, section_map, memberhinge_map, support_map)
    }

    fn add_support_springs_to_operator(
        &self,
        global_stiffness_matrix: &mut nalgebra::DMatrix<f64>,
    ) -> Result<(), String> {
        self.visit_unique_supported_nodes(|_node_id, base_index, support| {
            // Translational springs
            for (axis_label, local_dof) in TRANSLATION_AXES {
                if let Some(condition) =
                    get_case_insensitive!(support.displacement_conditions, axis_label)
                {
                    if let SupportConditionType::Spring = condition.condition_type {
                        let stiffness = Self::spring_stiffness_or_error(
                            support.id,
                            axis_label,
                            "displacement",
                            condition.stiffness,
                        )?;
                        global_stiffness_matrix
                            [(base_index + local_dof, base_index + local_dof)] += stiffness;
                    }
                }
            }

            // Rotational springs
            for (axis_label, local_dof) in ROTATION_AXES {
                if let Some(condition) =
                    get_case_insensitive!(support.rotation_conditions, axis_label)
                {
                    if let SupportConditionType::Spring = condition.condition_type {
                        let stiffness = Self::spring_stiffness_or_error(
                            support.id,
                            axis_label,
                            "rotation",
                            condition.stiffness,
                        )?;
                        global_stiffness_matrix
                            [(base_index + local_dof, base_index + local_dof)] += stiffness;
                    }
                }
            }
            Ok(())
        })
    }

    fn build_operator_with_supports(
        &self,
        active_map: &std::collections::HashMap<u32, bool>,
        displacement: Option<&nalgebra::DMatrix<f64>>,
    ) -> Result<nalgebra::DMatrix<f64>, String> {
        let mut k = self.assemble_global_stiffness_matrix(active_map)?;
        if let Some(u) = displacement {
            let k_geo = self.assemble_geometric_stiffness_matrix_with_active(u, active_map)?;
            k += k_geo;
        }
        self.add_support_springs_to_operator(&mut k)?;
        Ok(k)
    }

    fn build_rigid_elimination_partial_using_hinges(&self) -> Result<RigidElimination, String> {
        use crate::models::members::enums::MemberType;
        use nalgebra::DMatrix;
        use std::collections::{HashMap, HashSet};

        let assembly_context = AssemblyContext::new(self);

        // parent: slave -> (master, r = x_b - x_a)
        let mut parent: HashMap<u32, (u32, (f64, f64, f64))> = HashMap::new();

        #[derive(Clone, Copy)]
        struct RigidInfo {
            a: u32, // master node id
            b: u32, // slave node id
            r: (f64, f64, f64),
            modes: AxisModes,
        }
        let mut rigid_elems: Vec<RigidInfo> = Vec::new();

        for set in &self.member_sets {
            for m in &set.members {
                if !matches!(m.member_type, MemberType::Rigid) {
                    continue;
                }

                let a = m.start_node.id; // master
                let b = m.end_node.id; // slave

                let r = (
                    m.end_node.X - m.start_node.X,
                    m.end_node.Y - m.start_node.Y,
                    m.end_node.Z - m.start_node.Z,
                );
                let l2 = r.0 * r.0 + r.1 * r.1 + r.2 * r.2;
                if l2 < 1.0e-24 {
                    return Err(format!("Rigid member {} has zero length.", m.id));
                }

                // unique master per slave
                if parent.contains_key(&b) {
                    return Err(format!("Node {} is slave in multiple rigid links.", b));
                }

                // cycle check: does 'a' depend on 'b' upstream?
                let mut p = a;
                let mut guard = 0usize;
                while let Some(&(pp, _)) = parent.get(&p) {
                    if pp == b {
                        return Err(format!("Rigid cycle detected involving node {}.", b));
                    }
                    p = pp;
                    guard += 1;
                    if guard > 100000 {
                        return Err("Rigid chain too long (suspected loop).".to_string());
                    }
                }

                parent.insert(b, (a, r));

                // hinge modes, then FORCE translations to rigid for a rigid link
                let a_h = m
                    .start_hinge
                    .and_then(|id| assembly_context.hinge_by_id.get(&id).copied());
                let b_h = m
                    .end_hinge
                    .and_then(|id| assembly_context.hinge_by_id.get(&id).copied());
                let mut modes = classify_from_hinges(a_h, b_h);

                // IMPORTANT: translations MUST be rigid for a rigid link
                modes.trans = [AxisMode::Rigid, AxisMode::Rigid, AxisMode::Rigid];

                rigid_elems.push(RigidInfo { a, b, r, modes });
            }
        }

        // number of full DOFs
        let n_full = self.compute_num_dofs();

        // which full dofs are eliminated
        let mut eliminated: HashSet<usize> = HashSet::new();
        for info in &rigid_elems {
            for ax in 0..3 {
                if matches!(info.modes.trans[ax], AxisMode::Rigid) {
                    eliminated.insert(FERS::dof_index(info.b, ax));
                }
                if matches!(info.modes.rot[ax], AxisMode::Rigid) {
                    eliminated.insert(FERS::dof_index(info.b, 3 + ax));
                }
            }
        }

        // map retained full -> reduced
        let mut full_to_red: HashMap<usize, usize> = HashMap::new();
        let mut red_to_full: Vec<usize> = Vec::new();
        let mut seen: HashSet<usize> = HashSet::new();
        for set in &self.member_sets {
            for m in &set.members {
                for node in [&m.start_node, &m.end_node] {
                    for d in 0..6 {
                        let fi = FERS::dof_index(node.id, d);
                        if eliminated.contains(&fi) {
                            continue;
                        }
                        if seen.insert(fi) {
                            full_to_red.insert(fi, red_to_full.len());
                            red_to_full.push(fi);
                        }
                    }
                }
            }
        }

        let n_red = red_to_full.len();
        let mut s = DMatrix::<f64>::zeros(n_full, n_red);

        // 1) identity for retained dofs
        for (fi, &col) in &full_to_red {
            s[(*fi, col)] = 1.0;
        }

        // depth helper so that masters are processed before their slaves in a chain
        fn depth_of(node: u32, parent: &HashMap<u32, (u32, (f64, f64, f64))>) -> usize {
            let mut d = 0usize;
            let mut p = node;
            while let Some(&(pp, _)) = parent.get(&p) {
                d += 1;
                p = pp;
            }
            d
        }

        // 2) write slave rows by composing master rows via C(r)
        let mut edges = rigid_elems.clone();
        edges.sort_by_key(|e| depth_of(e.a, &parent)); // masters before slaves

        for info in &edges {
            let c = FERS::rigid_map_c(info.r.0, info.r.1, info.r.2);

            for i in 0..6 {
                let is_trans = i < 3;
                let ax = if is_trans { i } else { i - 3 };
                let rigid_here = if is_trans {
                    matches!(info.modes.trans[ax], AxisMode::Rigid)
                } else {
                    matches!(info.modes.rot[ax], AxisMode::Rigid)
                };
                if !rigid_here {
                    continue;
                }

                let row_b = FERS::dof_index(info.b, i);

                // Zero the row_b to avoid accidental accumulation if multiple edges ever touched it

                // S[row_b, :] = Σ_j C[i,j] * S[row_a_j, :]
                for j in 0..6 {
                    let row_a_j = FERS::dof_index(info.a, j);
                    let coeff = c[(i, j)];
                    if coeff == 0.0 {
                        continue;
                    }
                    for col in 0..n_red {
                        s[(row_b, col)] += coeff * s[(row_a_j, col)];
                    }
                }
            }
        }

        Ok(RigidElimination { s, full_to_red })
    }

    pub fn get_member_count(&self) -> usize {
        self.member_sets.iter().map(|ms| ms.members.len()).sum()
    }

    fn assemble_element_into_global_12(
        global: &mut nalgebra::DMatrix<f64>,
        i0: usize,
        j0: usize,
        ke: &nalgebra::DMatrix<f64>,
    ) {
        debug_assert_eq!(ke.nrows(), 12);
        debug_assert_eq!(ke.ncols(), 12);
        for i in 0..6 {
            for j in 0..6 {
                global[(i0 + i, i0 + j)] += ke[(i, j)];
                global[(i0 + i, j0 + j)] += ke[(i, j + 6)];
                global[(j0 + i, i0 + j)] += ke[(i + 6, j)];
                global[(j0 + i, j0 + j)] += ke[(i + 6, j + 6)];
            }
        }
    }

    pub fn assemble_global_stiffness_matrix(
        &self,
        active_map: &std::collections::HashMap<u32, bool>,
    ) -> Result<nalgebra::DMatrix<f64>, String> {
        use crate::models::members::enums::MemberType;

        self.validate_node_ids()?;
        let assembly_context = AssemblyContext::new(self);

        let number_of_dofs: usize = self.compute_num_dofs();

        let mut global_stiffness_matrix =
            nalgebra::DMatrix::<f64>::zeros(number_of_dofs, number_of_dofs);

        for member_set in &self.member_sets {
            for member in &member_set.members {
                // Build a 12x12 GLOBAL element matrix according to the member behavior
                let element_global_opt: Option<nalgebra::DMatrix<f64>> = match member.member_type {
                    MemberType::Normal => {
                        let Some(_) = member.section else {
                            return Err(format!(
                                "Member {} (Normal) is missing a section id.",
                                member.id
                            ));
                        };

                        // 1) Local base K
                        let k_local_base = member
                            .calculate_stiffness_matrix_3d(
                                &assembly_context.material_by_id,
                                &assembly_context.section_by_id,
                            )
                            .ok_or_else(|| {
                                format!("Member {} failed to build local stiffness.", member.id)
                            })?;

                        // 2) Per-end modes from hinges (LOCAL axes)
                        let a_h = member
                            .start_hinge
                            .and_then(|id| assembly_context.hinge_by_id.get(&id).copied());
                        let b_h = member
                            .end_hinge
                            .and_then(|id| assembly_context.hinge_by_id.get(&id).copied());
                        let (a_trans, a_rot, b_trans, b_rot) = modes_from_single_ends(a_h, b_h);

                        // 3) Apply releases & semi-rigid springs (condensation in LOCAL)
                        let k_local_mod = apply_end_releases_to_local_beam_k(
                            &k_local_base,
                            a_trans,
                            a_rot,
                            b_trans,
                            b_rot,
                        )?;

                        // 4) Transform to GLOBAL
                        let t = member.calculate_transformation_matrix_3d();
                        let k_global = t.transpose() * k_local_mod * t;
                        Some(k_global)
                    }

                    MemberType::Truss => {
                        let mut k_global: nalgebra::Matrix<
                            f64,
                            nalgebra::Dyn,
                            nalgebra::Dyn,
                            nalgebra::VecStorage<f64, nalgebra::Dyn, nalgebra::Dyn>,
                        > = member
                            .calculate_truss_stiffness_matrix_3d(
                                &assembly_context.material_by_id,
                                &assembly_context.section_by_id,
                            )
                            .ok_or_else(|| {
                                format!(
                                    "Member {} (Truss) failed to build truss stiffness.",
                                    member.id
                                )
                            })?;

                        // Optional: translational node-to-ground springs from hinges (LOCAL → GLOBAL)
                        let a_h = member
                            .start_hinge
                            .and_then(|id| assembly_context.hinge_by_id.get(&id).copied());
                        let b_h = member
                            .end_hinge
                            .and_then(|id| assembly_context.hinge_by_id.get(&id).copied());
                        let (a_trans, _a_rot, b_trans, _b_rot) = modes_from_single_ends(a_h, b_h);

                        let k_s_local = build_local_truss_translational_spring_k(a_trans, b_trans);
                        if k_s_local.iter().any(|v| *v != 0.0) {
                            let t = member.calculate_transformation_matrix_3d();
                            let k_s_global = t.transpose() * k_s_local * t;
                            k_global += k_s_global;
                        }

                        Some(k_global)
                    }

                    MemberType::Tension | MemberType::Compression => {
                        let is_active: bool = *active_map.get(&member.id).unwrap_or(&true);
                        if is_active {
                            member.calculate_truss_stiffness_matrix_3d(
                                &assembly_context.material_by_id,
                                &assembly_context.section_by_id,
                            )
                        } else {
                            None
                        }
                    }
                    MemberType::Rigid => None,
                };

                if let Some(element_global) = element_global_opt {
                    let start_index = (member.start_node.id as usize - 1) * 6;
                    let end_index = (member.end_node.id as usize - 1) * 6;

                    Self::assemble_element_into_global_12(
                        &mut global_stiffness_matrix,
                        start_index,
                        end_index,
                        &element_global,
                    );
                }
            }
        }

        Ok(global_stiffness_matrix)
    }

    fn assemble_geometric_stiffness_matrix_with_active(
        &self,
        displacement: &nalgebra::DMatrix<f64>,
        active_map: &std::collections::HashMap<u32, bool>,
    ) -> Result<nalgebra::DMatrix<f64>, String> {
        use crate::models::members::enums::MemberType;
        let assembly_context: AssemblyContext<'_> = AssemblyContext::new(self);
        let n = self.compute_num_dofs();
        let mut k_geo = nalgebra::DMatrix::<f64>::zeros(n, n);

        for member_set in &self.member_sets {
            for member in &member_set.members {
                // Skip rigid: enforced by MPC; contributes no element geometry
                if matches!(member.member_type, MemberType::Rigid) {
                    continue;
                }
                // Skip deactivated tension/compression
                if matches!(
                    member.member_type,
                    MemberType::Tension | MemberType::Compression
                ) && !*active_map.get(&member.id).unwrap_or(&true)
                {
                    continue;
                }

                let n_axial = member.calculate_axial_force_3d(
                    displacement,
                    &assembly_context.material_by_id,
                    &assembly_context.section_by_id,
                );
                let k_g_local_base = member.calculate_geometric_stiffness_matrix_3d(n_axial);

                // Apply same end releases/semi-rigid springs as for material stiffness
                let a_h = member
                    .start_hinge
                    .and_then(|id| assembly_context.hinge_by_id.get(&id).copied());
                let b_h = member
                    .end_hinge
                    .and_then(|id| assembly_context.hinge_by_id.get(&id).copied());
                let (a_trans, a_rot, b_trans, b_rot) = modes_from_single_ends(a_h, b_h);

                let k_g_local_mod = apply_end_releases_to_local_beam_k(
                    &k_g_local_base,
                    a_trans,
                    a_rot,
                    b_trans,
                    b_rot,
                )?;

                // Transform to GLOBAL
                let t = member.calculate_transformation_matrix_3d();
                let k_g_global = t.transpose() * k_g_local_mod * t;

                let i0 = (member.start_node.id as usize - 1) * 6;
                let j0 = (member.end_node.id as usize - 1) * 6;
                Self::assemble_element_into_global_12(&mut k_geo, i0, j0, &k_g_global);
            }
        }
        Ok(k_geo)
    }

    pub fn validate_node_ids(&self) -> Result<(), String> {
        // Collect all node IDs in a HashSet for quick lookup
        let mut node_ids: HashSet<u32> = HashSet::new();

        // Populate node IDs from all members
        for member_set in &self.member_sets {
            for member in &member_set.members {
                node_ids.insert(member.start_node.id);
                node_ids.insert(member.end_node.id);
            }
        }

        // Ensure IDs start at 1 and are consecutive
        let max_id = *node_ids.iter().max().unwrap_or(&0);
        for id in 1..=max_id {
            if !node_ids.contains(&id) {
                return Err(format!(
                    "Node ID {} is missing. Node IDs must be consecutive starting from 1.",
                    id
                ));
            }
        }

        Ok(())
    }

    fn update_active_set(
        &self,
        displacement: &nalgebra::DMatrix<f64>,
        active_map: &mut std::collections::HashMap<u32, bool>,
        axial_slack_tolerance: f64,
        material_map: &std::collections::HashMap<u32, &Material>,
        section_map: &std::collections::HashMap<u32, &Section>,
    ) -> bool {
        use crate::models::members::enums::MemberType;

        let mut changed = false;
        for member_set in &self.member_sets {
            for member in &member_set.members {
                match member.member_type {
                    MemberType::Tension => {
                        let n = member.calculate_axial_force_3d(
                            displacement,
                            material_map,
                            section_map,
                        );
                        let should_be_active = n >= -axial_slack_tolerance;
                        if active_map.get(&member.id).copied().unwrap_or(true) != should_be_active {
                            active_map.insert(member.id, should_be_active);
                            changed = true;
                        }
                    }
                    MemberType::Compression => {
                        let n = member.calculate_axial_force_3d(
                            displacement,
                            material_map,
                            section_map,
                        );
                        let should_be_active = n <= axial_slack_tolerance;
                        if active_map.get(&member.id).copied().unwrap_or(true) != should_be_active {
                            active_map.insert(member.id, should_be_active);
                            changed = true;
                        }
                    }
                    _ => {}
                }
            }
        }
        changed
    }

    fn compute_num_dofs(&self) -> usize {
        let max_node = self
            .member_sets
            .iter()
            .flat_map(|ms| ms.members.iter())
            .flat_map(|m| vec![m.start_node.id, m.end_node.id])
            .max()
            .unwrap_or(0) as usize;
        max_node * 6
    }

    pub fn assemble_load_vector_for_combination(
        &self,
        combination_id: u32,
    ) -> Result<DMatrix<f64>, String> {
        let num_dofs = self.compute_num_dofs();
        let mut f_comb = DMatrix::<f64>::zeros(num_dofs, 1);

        // Find the combination by its load_combination_id field
        let combo = self
            .load_combinations
            .iter()
            .find(|lc| lc.load_combination_id == combination_id)
            .ok_or_else(|| format!("LoadCombination {} not found.", combination_id))?;

        // Now iterate the HashMap<u32, f64>
        for (&case_id, &factor) in &combo.load_cases_factors {
            let f_case = self.assemble_load_vector_for_case(case_id);
            f_comb += f_case * factor;
        }

        Ok(f_comb)
    }

    fn dof_index(node_id: u32, local_dof: usize) -> usize {
        (node_id as usize - 1) * 6 + local_dof
    }

    fn rigid_map_c(r_x: f64, r_y: f64, r_z: f64) -> nalgebra::SMatrix<f64, 6, 6> {
        use nalgebra::{Matrix3, SMatrix};

        let i3 = Matrix3::<f64>::identity();
        let skew = Matrix3::<f64>::new(0.0, -r_z, r_y, r_z, 0.0, -r_x, -r_y, r_x, 0.0);

        // [u_b; θ_b] = [I  -[r]_x; 0  I] [u_a; θ_a]
        let mut c = SMatrix::<f64, 6, 6>::zeros();
        c.fixed_view_mut::<3, 3>(0, 0).copy_from(&i3);
        c.fixed_view_mut::<3, 3>(0, 3).copy_from(&(-skew));
        c.fixed_view_mut::<3, 3>(3, 3).copy_from(&i3);
        c
    }

    fn reduce_system(
        k_full: &DMatrix<f64>,
        f_full: &DMatrix<f64>,
        elim: &RigidElimination,
    ) -> (DMatrix<f64>, DMatrix<f64>) {
        let k_red = elim.s.transpose() * k_full * &elim.s;
        let f_red = elim.s.transpose() * f_full;
        (k_red, f_red)
    }

    fn expand_solution(elim: &RigidElimination, u_red: &DMatrix<f64>) -> DMatrix<f64> {
        &elim.s * u_red
    }

    fn constrain_single_dof(
        &self,
        k_global: &mut DMatrix<f64>,
        rhs: &mut DMatrix<f64>,
        dof_index: usize,
        prescribed: f64,
    ) {
        for j in 0..k_global.ncols() {
            k_global[(dof_index, j)] = 0.0;
        }
        for i in 0..k_global.nrows() {
            k_global[(i, dof_index)] = 0.0;
        }
        k_global[(dof_index, dof_index)] = 1.0;
        rhs[(dof_index, 0)] = prescribed;
    }

    fn constrain_linear_constraint_penalty(
        &self,
        k_red: &mut nalgebra::DMatrix<f64>,
        rhs_red: &mut nalgebra::DMatrix<f64>,
        a_column: &nalgebra::DMatrix<f64>, // shape (n_red, 1)
        prescribed: f64,
        penalty_factor: f64,
    ) {
        let n_red_rows: usize = k_red.nrows();
        let n_red_cols: usize = k_red.ncols();
        debug_assert_eq!(n_red_rows, n_red_cols, "Reduced stiffness must be square.");
        debug_assert_eq!(
            a_column.nrows(),
            n_red_rows,
            "Constraint vector length must match reduced system size."
        );
        debug_assert_eq!(a_column.ncols(), 1, "Constraint vector must be a column.");

        // Scale penalty from the matrix magnitude to mitigate conditioning issues.
        let mut max_diag: f64 = 0.0;
        for i in 0..n_red_rows {
            let value: f64 = k_red[(i, i)].abs();
            if value > max_diag {
                max_diag = value;
            }
        }
        if max_diag <= 0.0 {
            max_diag = 1.0;
        }
        let alpha: f64 = penalty_factor * max_diag;

        // K += alpha * a * a^T
        for i in 0..n_red_rows {
            let ai: f64 = a_column[(i, 0)];
            if ai == 0.0 {
                continue;
            }
            for j in 0..n_red_rows {
                let aj: f64 = a_column[(j, 0)];
                if aj == 0.0 {
                    continue;
                }
                k_red[(i, j)] += alpha * ai * aj;
            }
        }

        // rhs += alpha * a * prescribed
        if prescribed != 0.0 {
            for i in 0..n_red_rows {
                rhs_red[(i, 0)] += alpha * a_column[(i, 0)] * prescribed;
            }
        }
    }

    /// Try to detect whether a constraint vector is essentially "one-hot" on a single reduced DOF.
    /// Returns Some(pivot_index) if exactly or approximately one-hot, otherwise None.
    fn detect_one_hot_constraint(
        &self,
        a_column: &nalgebra::DMatrix<f64>,
        tolerance_ratio: f64,
    ) -> Option<usize> {
        debug_assert_eq!(a_column.ncols(), 1, "Constraint vector must be a column.");
        let n: usize = a_column.nrows();

        // Find the entry with the largest absolute value
        let mut max_val: f64 = 0.0;
        let mut max_idx: usize = 0;
        for i in 0..n {
            let v: f64 = a_column[(i, 0)].abs();
            if v > max_val {
                max_val = v;
                max_idx = i;
            }
        }
        if max_val == 0.0 {
            return None;
        }

        // Sum of squares of all other coefficients
        let mut sum_sq_other: f64 = 0.0;
        for i in 0..n {
            if i == max_idx {
                continue;
            }
            let v: f64 = a_column[(i, 0)];
            sum_sq_other += v * v;
        }

        // If the energy of others is small relative to the pivot, treat it as one-hot
        if sum_sq_other <= (tolerance_ratio * tolerance_ratio) * (max_val * max_val) {
            Some(max_idx)
        } else {
            None
        }
    }

    fn apply_boundary_conditions_reduced(
        &self,
        elim: &RigidElimination,
        k_red: &mut nalgebra::DMatrix<f64>,
        rhs_red: &mut nalgebra::DMatrix<f64>,
    ) -> Result<(), String> {
        use crate::models::supports::supportconditiontype::SupportConditionType;
        use std::collections::HashMap;

        let support_map: HashMap<u32, &NodalSupport> =
            self.nodal_supports.iter().map(|s| (s.id, s)).collect();

        // Tunables: penalty factor and one-hot tolerance
        const BC_PENALTY_FACTOR: f64 = 1.0e8; // increase if constraints look too soft; decrease if conditioning suffers
        const ONE_HOT_TOL_RATIO: f64 = 1.0e-6; // if all other coefficients are <= 1e-6 of the pivot, treat as one-hot

        let n_red: usize = k_red.nrows();
        debug_assert_eq!(n_red, k_red.ncols(), "Reduced stiffness must be square.");
        debug_assert_eq!(
            rhs_red.nrows(),
            n_red,
            "RHS size must match reduced system size."
        );
        debug_assert_eq!(rhs_red.ncols(), 1, "RHS must be a column vector.");

        for ms in &self.member_sets {
            for m in &ms.members {
                for node in [&m.start_node, &m.end_node] {
                    let Some(support_id) = node.nodal_support else {
                        continue;
                    };
                    let Some(support) = support_map.get(&support_id) else {
                        continue;
                    };

                    let base_full: usize = (node.id as usize - 1) * 6;

                    // Handle translational constraints
                    for (axis_label, local_dof) in TRANSLATION_AXES {
                        let cond_opt =
                            support.displacement_conditions.get(axis_label).or_else(|| {
                                support
                                    .displacement_conditions
                                    .get(&axis_label.to_ascii_lowercase())
                            });
                        let is_fixed: bool = cond_opt
                            .map(|c| matches!(c.condition_type, SupportConditionType::Fixed))
                            .unwrap_or(true);
                        if !is_fixed {
                            continue;
                        }

                        let fi: usize = base_full + local_dof;
                        if let Some(ri) = elim.full_to_red.get(&fi).copied() {
                            // Retained DOF: constrain exactly
                            self.constrain_single_dof(k_red, rhs_red, ri, 0.0);
                        } else {
                            // Slave DOF: enforce (S_row * u_red) = 0
                            let mut a_column = nalgebra::DMatrix::<f64>::zeros(n_red, 1);
                            for j in 0..n_red {
                                a_column[(j, 0)] = elim.s[(fi, j)];
                            }

                            // If the constraint is essentially one-hot, constrain that single DOF exactly.
                            if let Some(pivot_j) =
                                self.detect_one_hot_constraint(&a_column, ONE_HOT_TOL_RATIO)
                            {
                                self.constrain_single_dof(k_red, rhs_red, pivot_j, 0.0);
                            } else {
                                self.constrain_linear_constraint_penalty(
                                    k_red,
                                    rhs_red,
                                    &a_column,
                                    0.0, // prescribed displacement is zero
                                    BC_PENALTY_FACTOR,
                                );
                            }
                        }
                    }

                    // Handle rotational constraints
                    for (axis_label, local_dof) in ROTATION_AXES {
                        let cond_opt = support.rotation_conditions.get(axis_label).or_else(|| {
                            support
                                .rotation_conditions
                                .get(&axis_label.to_ascii_lowercase())
                        });
                        let is_fixed: bool = cond_opt
                            .map(|c| matches!(c.condition_type, SupportConditionType::Fixed))
                            .unwrap_or(true);
                        if !is_fixed {
                            continue;
                        }

                        let fi: usize = base_full + local_dof;
                        if let Some(ri) = elim.full_to_red.get(&fi).copied() {
                            // Retained rotational DOF: constrain exactly
                            self.constrain_single_dof(k_red, rhs_red, ri, 0.0);
                        } else {
                            // Slave rotational DOF: enforce (S_row * u_red) = 0
                            let mut a_column = nalgebra::DMatrix::<f64>::zeros(n_red, 1);
                            for j in 0..n_red {
                                a_column[(j, 0)] = elim.s[(fi, j)];
                            }

                            if let Some(pivot_j) =
                                self.detect_one_hot_constraint(&a_column, ONE_HOT_TOL_RATIO)
                            {
                                self.constrain_single_dof(k_red, rhs_red, pivot_j, 0.0);
                            } else {
                                self.constrain_linear_constraint_penalty(
                                    k_red,
                                    rhs_red,
                                    &a_column,
                                    0.0,
                                    BC_PENALTY_FACTOR,
                                );
                            }
                        }
                    }
                }
            }
        }

        Ok(())
    }

    pub fn assemble_load_vector_for_case(&self, load_case_id: u32) -> DMatrix<f64> {
        let num_dofs = self.compute_num_dofs();
        let mut f = DMatrix::<f64>::zeros(num_dofs, 1);

        if let Some(load_case) = self.load_cases.iter().find(|lc| lc.id == load_case_id) {
            assemble_nodal_loads(load_case, &mut f);
            assemble_nodal_moments(load_case, &mut f);
            assemble_distributed_loads(load_case, &self.member_sets, &mut f, load_case_id);
        }
        f
    }

    fn init_active_map_tie_comp(&self) -> HashMap<u32, bool> {
        let mut map = HashMap::new();
        for ms in &self.member_sets {
            for member in &ms.members {
                if matches!(
                    member.member_type,
                    MemberType::Tension | MemberType::Compression
                ) {
                    map.insert(member.id, true);
                }
            }
        }
        map
    }

    // Do not abbreviate code

    fn solve_first_order_common(
        &mut self,
        load_vector_full: nalgebra::DMatrix<f64>,
        name: String,
        result_type: ResultType,
    ) -> Result<Results, String> {
        let tolerance: f64 = self.settings.analysis_option.tolerance;
        let max_it: usize = self.settings.analysis_option.max_iterations.unwrap_or(20) as usize;
        let axial_slack_tolerance: f64 = AXIAL_SLACK_TOLERANCE_DEFAULT;

        let mut active_map = self.init_active_map_tie_comp();
        let mut u_full = nalgebra::DMatrix::<f64>::zeros(self.compute_num_dofs(), 1);

        let assembly_context: AssemblyContext<'_> = AssemblyContext::new(self);

        // Build rigid elimination (S) once; it depends only on topology/hinges
        let elim = self.build_rigid_elimination_partial_using_hinges()?;

        let mut converged = false;
        for _iter in 0..max_it {
            // Linear operator (no geometric stiffness) for first-order analysis
            let k_full = self.build_operator_with_supports(&active_map, None)?;

            // Reduce system: K_r = Sᵀ K S,  f_r = Sᵀ f
            let (mut k_red, mut f_red) = Self::reduce_system(&k_full, &load_vector_full, &elim);

            // Apply boundary conditions in REDUCED space
            self.apply_boundary_conditions_reduced(&elim, &mut k_red, &mut f_red)?;

            // Solve reduced system
            let u_red = k_red.lu().solve(&f_red).ok_or_else(|| {
                "Reduced stiffness matrix is singular or near-singular".to_string()
            })?;

            // Expand to FULL space
            let u_full_new = Self::expand_solution(&elim, &u_red);

            // Active-set update (Tension/Compression) uses FULL displacement
            let delta = &u_full_new - &u_full;
            u_full = u_full_new;

            let changed = self.update_active_set(
                &u_full,
                &mut active_map,
                axial_slack_tolerance,
                &assembly_context.material_by_id,
                &assembly_context.section_by_id,
            );

            if delta.norm() < tolerance && !changed {
                converged = true;
                break;
            }
        }

        if !converged {
            return Err(format!(
                "Active-set iteration did not converge within {} iterations",
                max_it
            ));
        }

        // ---------------------------
        // Build reactions including MPCs
        // ---------------------------
        // Recompute the FINAL linear operator with the final active_map,
        // then form the reduced residual r_r = K_r u_r - f_r with the UNMODIFIED K_r, f_r
        // Map to FULL space: r_full = S r_red
        // Keep only true support reactions
        let r_support = compose_support_reaction_vector_equilibrium(
            self,
            &result_type,      // <- pass the ResultType you're solving
            &u_full,           // <- final full displacement vector
            Some(&active_map), // <- so ties/struts respect active set
        )?;

        let mut sum_rx = 0.0;
        let mut sum_ry = 0.0;
        let mut sum_rz = 0.0;
        for (i, val) in r_support.iter().enumerate() {
            match i % 6 {
                0 => sum_rx += val,
                1 => sum_ry += val,
                2 => sum_rz += val,
                _ => {}
            }
        }

        // Sum of EXTERNAL loads per axis (global vector you assembled)
        let mut sum_fx = 0.0;
        let mut sum_fy = 0.0;
        let mut sum_fz = 0.0;
        for (i, val) in load_vector_full.iter().enumerate() {
            match i % 6 {
                0 => sum_fx += val,
                1 => sum_fy += val,
                2 => sum_fz += val,
                _ => {}
            }
        }

        log::debug!(
            "Equil check: sum reactions [Fx,Fy,Fz] = [{:.6}, {:.6}, {:.6}]",
            sum_rx,
            sum_ry,
            sum_rz
        );
        log::debug!(
            "Equil check: sum external [Fx,Fy,Fz]  = [{:.6}, {:.6}, {:.6}]",
            sum_fx,
            sum_fy,
            sum_fz
        );

        // Store masked reactions
        let results = self
            .build_and_store_results(
                name.clone(),
                result_type.clone(),
                &u_full,
                &r_support,
                Some(&active_map),
            )?
            .clone();

        Ok(results)
    }

    fn solve_second_order_common(
        &mut self,
        load_vector_full: nalgebra::DMatrix<f64>,
        name: String,
        result_type: ResultType,
        max_iterations: usize,
        tolerance: f64,
    ) -> Result<Results, String> {
        let axial_slack_tolerance: f64 = AXIAL_SLACK_TOLERANCE_DEFAULT;

        let mut active_map = self.init_active_map_tie_comp();
        let n_full = self.compute_num_dofs();
        let mut u_full = nalgebra::DMatrix::<f64>::zeros(n_full, 1);

        let assembly_context: AssemblyContext<'_> = AssemblyContext::new(self);

        // Build rigid elimination (S) once; it depends only on topology/hinges
        let elim = self.build_rigid_elimination_partial_using_hinges()?;

        let mut converged = false;
        for _iter in 0..max_iterations {
            // Tangent operator (includes geometric stiffness) for Newton–Raphson
            let k_tangent_full = self.build_operator_with_supports(&active_map, Some(&u_full))?;

            // Reduce: K_tr = Sᵀ K_tangent S, f_r = Sᵀ f
            let (k_red_tangent, f_red) =
                Self::reduce_system(&k_tangent_full, &load_vector_full, &elim);

            // Current reduced displacement u_r = Sᵀ u
            let mut u_red = elim.s.transpose() * &u_full;

            // Residual in reduced space (before BC application): r_r = K_tr u_r − f_r
            let mut r_red = &k_red_tangent * &u_red - &f_red;

            // Apply BCs in REDUCED space, then solve for Δu_r from K_tr,BC Δu_r = −r_r,BC
            let mut k_treated = k_red_tangent.clone();
            self.apply_boundary_conditions_reduced(&elim, &mut k_treated, &mut r_red)?;
            let delta_red = k_treated
                .lu()
                .solve(&(-&r_red))
                .ok_or_else(|| "Tangent stiffness singular.".to_string())?;

            // Update reduced, expand to FULL
            u_red += &delta_red;
            let u_full_new = Self::expand_solution(&elim, &u_red);

            // Check convergence and update active set
            let delta_full = &u_full_new - &u_full;
            u_full = u_full_new;

            let changed = self.update_active_set(
                &u_full,
                &mut active_map,
                axial_slack_tolerance,
                &assembly_context.material_by_id,
                &assembly_context.section_by_id,
            );

            if delta_full.norm() < tolerance && !changed {
                converged = true;
                break;
            }
        }

        if !converged {
            return Err(format!(
                "Newton–Raphson with active set did not converge in {} iterations",
                max_iterations
            ));
        }

        // ---------------------------
        // Build reactions including MPCs
        // ---------------------------
        // Use the LINEAR operator (no geometric stiffness) to compute final reactions,
        // matching classic practice and your previous intent/comments.
        // Map to FULL space reactions
        let r_support = compose_support_reaction_vector_equilibrium(
            self,
            &result_type,      // <- pass the ResultType you're solving
            &u_full,           // <- final full displacement vector
            Some(&active_map), // <- so ties/struts respect active set
        )?;

        let results = self
            .build_and_store_results(
                name.clone(),
                result_type.clone(),
                &u_full,
                &r_support,
                Some(&active_map),
            )?
            .clone();
        Ok(results)
    }

    pub fn solve_for_load_case(&mut self, load_case_id: u32) -> Result<Results, String> {
        let load_vector = self.assemble_load_vector_for_case(load_case_id);
        let load_case = self
            .load_cases
            .iter()
            .find(|lc| lc.id == load_case_id)
            .ok_or_else(|| format!("LoadCase {} not found.", load_case_id))?;
        self.solve_first_order_common(
            load_vector,
            load_case.name.clone(),
            ResultType::Loadcase(load_case_id),
        )
    }
    pub fn solve_for_load_case_second_order(
        &mut self,
        load_case_id: u32,
        max_iterations: usize,
        tolerance: f64,
    ) -> Result<Results, String> {
        let load_vector = self.assemble_load_vector_for_case(load_case_id);
        let load_case = self
            .load_cases
            .iter()
            .find(|lc| lc.id == load_case_id)
            .ok_or_else(|| format!("LoadCase {} not found.", load_case_id))?;
        self.solve_second_order_common(
            load_vector,
            load_case.name.clone(),
            ResultType::Loadcase(load_case_id),
            max_iterations,
            tolerance,
        )
    }
    pub fn solve_for_load_combination(&mut self, combination_id: u32) -> Result<Results, String> {
        let load_vector = self.assemble_load_vector_for_combination(combination_id)?;
        let combo = self
            .load_combinations
            .iter()
            .find(|lc| lc.load_combination_id == combination_id)
            .ok_or_else(|| format!("LoadCombination {} not found.", combination_id))?;
        self.solve_first_order_common(
            load_vector,
            combo.name.clone(),
            ResultType::Loadcombination(combination_id),
        )
    }

    pub fn solve_for_load_combination_second_order(
        &mut self,
        combination_id: u32,
        max_iterations: usize,
        tolerance: f64,
    ) -> Result<Results, String> {
        let load_vector = self.assemble_load_vector_for_combination(combination_id)?;
        let combo = self
            .load_combinations
            .iter()
            .find(|lc| lc.load_combination_id == combination_id)
            .ok_or_else(|| format!("LoadCombination {} not found.", combination_id))?;
        self.solve_second_order_common(
            load_vector,
            combo.name.clone(),
            ResultType::Loadcombination(combination_id),
            max_iterations,
            tolerance,
        )
    }

    pub fn build_and_store_results(
        &mut self,
        name: String,
        result_type: ResultType,
        displacement_vector: &DMatrix<f64>,
        global_reaction_vector: &DMatrix<f64>,
        active_map: Option<&std::collections::HashMap<u32, bool>>,
    ) -> Result<&Results, String> {
        // 1) Element/member results
        let member_results = compute_member_results_from_displacement(
            self,
            &result_type,
            displacement_vector,
            active_map,
        );

        // 2) Node displacements
        let displacement_nodes = extract_displacements(self, displacement_vector);

        // 3) Reactions: already computed via compose_support_reaction_vector_equilibrium(...)
        let reaction_nodes: BTreeMap<u32, crate::models::results::reaction::ReactionNodeResult> =
            extract_reaction_nodes(self, global_reaction_vector);

        // 4) Pack results & store
        let total_members: usize = self.member_sets.iter().map(|set| set.members.len()).sum();
        let total_supports: usize = self.nodal_supports.len();

        let results = Results {
            name: name.clone(),
            result_type: result_type.clone(),
            displacement_nodes,
            reaction_nodes,
            member_results,
            summary: ResultsSummary {
                total_displacements: total_members,
                total_reaction_forces: total_supports,
                total_member_forces: total_members,
            },
            unity_checks: None,
        };

        let bundle = self.results.get_or_insert_with(|| ResultsBundle {
            loadcases: BTreeMap::new(),
            loadcombinations: BTreeMap::new(),
            unity_checks_overview: None,
        });

        match result_type {
            ResultType::Loadcase(_) => {
                if bundle.loadcases.insert(name.clone(), results).is_some() {
                    return Err(format!("Duplicate load case name `{}`", name));
                }
                Ok(bundle.loadcases.get(&name).unwrap())
            }
            ResultType::Loadcombination(_) => {
                if bundle
                    .loadcombinations
                    .insert(name.clone(), results)
                    .is_some()
                {
                    return Err(format!("Duplicate load combination name `{}`", name));
                }
                Ok(bundle.loadcombinations.get(&name).unwrap())
            }
        }
    }

    pub fn save_results_to_json(fers_data: &FERS, file_path: &str) -> Result<(), std::io::Error> {
        let json = serde_json::to_string_pretty(fers_data)?;
        std::fs::write(file_path, json)
    }
}
