"""Tests targeting specific missing coverage in server.py.

This module focuses on covering the exact lines missing from server.py
based on coverage report analysis.
"""

import os
from datetime import datetime
from unittest.mock import AsyncMock, Mock, patch

import pytest
from fastcore.xml import FT
from starlette.background import BackgroundTask
from starlette.exceptions import HTTPException
from starlette.requests import Request
from starlette.responses import FileResponse, HTMLResponse

from starhtml.server import (
    APIRouter,
    Fragment,
    JSONResponse,
    ResponseRenderer,
    RouteFuncs,
    _mk_locfunc,
    cookie,
    serve,
    signal_shutdown,
)
from starhtml.utils import HttpHeader, empty


class TestAPIRouterMissingCoverage:
    """Test APIRouter functionality missing from coverage."""

    def test_api_router_with_body_wrap(self):
        """Test APIRouter with custom body_wrap (lines 72-77)."""

        def custom_body_wrap(content):
            return f"<wrapper>{content}</wrapper>"

        router = APIRouter(prefix="/api", body_wrap=custom_body_wrap)
        assert router.prefix == "/api"
        assert router.body_wrap == custom_body_wrap
        assert router.routes == []
        assert router.wss == []
        assert isinstance(router.rt_funcs, RouteFuncs)

    def test_api_router_wrap_func(self):
        """Test APIRouter._wrap_func method (lines 79-86)."""
        router = APIRouter()

        def test_handler():
            return "test"

        wrapped = router._wrap_func(test_handler, "/test")

        # Check wrapped function properties
        assert hasattr(wrapped, "__routename__")
        assert wrapped.__routename__ == "test_handler"

        # Function should be added to rt_funcs since name not in all_meths
        assert hasattr(router.rt_funcs, "test_handler")
        assert router.rt_funcs.test_handler == wrapped

    def test_api_router_wrap_func_http_method_name(self):
        """Test _wrap_func with HTTP method name (line 84-85)."""
        router = APIRouter()

        def get():  # This is an HTTP method name
            return "get response"

        wrapped = router._wrap_func(get, "/")

        # Should NOT be added to rt_funcs since 'get' is in all_meths
        assert not hasattr(router.rt_funcs, "get")
        assert wrapped.__routename__ == "get"

    def test_api_router_callable_path(self):
        """Test APIRouter with callable path (lines 91-99)."""
        router = APIRouter()

        def index():
            return "index page"

        # When router is called with a callable, it treats the callable as the function to route
        # This triggers line 99: return f(path) if callable(path) else f
        router(index)

        # Check route was created correctly
        assert len(router.routes) == 1
        func, path, methods, name, include_in_schema, body_wrap = router.routes[0]
        assert func == index
        assert path == "/"  # Should be "/" since function name is "index"

    def test_api_router_callable_path_non_index(self):
        """Test APIRouter with callable path that's not named 'index'."""
        router = APIRouter()

        def other_func():
            return "other"

        # When using callable as the argument, the callable is treated as the handler
        router(other_func)

        # Should use function name in path since __name__ != "index"
        func, path, methods, name, include_in_schema, body_wrap = router.routes[0]
        assert func == other_func
        assert path == "/other_func"  # Should be "/" + function name since name != "index"

    def test_api_router_getattr(self):
        """Test APIRouter.__getattr__ method (lines 102-105)."""
        router = APIRouter()

        # Add a function to rt_funcs
        def test_func():
            return "test"

        router.rt_funcs.test_func = test_func

        # Should be able to access via router.test_func
        assert router.test_func == test_func

        # Test AttributeError for missing attribute
        with pytest.raises(AttributeError):
            _ = router.nonexistent_func

    def test_api_router_to_app(self):
        """Test APIRouter.to_app method (lines 107-112)."""
        router = APIRouter()
        mock_app = Mock()

        # Add some routes and websockets
        @router("/test")
        def test_route():
            return "test"

        @router.ws("/ws")
        def ws_route():
            pass

        router.to_app(mock_app)

        # Should call _add_route and _add_ws on app
        mock_app._add_route.assert_called_once()
        mock_app._add_ws.assert_called_once()

    def test_api_router_ws(self):
        """Test APIRouter.ws method (lines 114-120)."""
        router = APIRouter(prefix="/api")

        def ws_handler():
            return "websocket"

        # Store original function reference before applying decorator
        original_ws_handler = ws_handler

        # Apply websocket decorator
        result = router.ws("/websocket")(ws_handler)

        # Check websocket was added
        assert len(router.wss) == 1
        func, path, conn, disconn, name, middleware = router.wss[0]
        assert path == "/api/websocket"  # Should include prefix
        assert func == original_ws_handler  # Original function should be stored
        # The decorator returns None (result of append)
        assert result is None


class TestRouteFuncsMissingCoverage:
    """Test RouteFuncs missing coverage."""

    def test_route_funcs_setattr(self):
        """Test RouteFuncs.__setattr__ method (line 133)."""
        rt_funcs = RouteFuncs()

        def test_func():
            return "test"

        rt_funcs.test_func = test_func
        assert rt_funcs._funcs["test_func"] == test_func

    def test_route_funcs_getattr_http_method(self):
        """Test RouteFuncs.__getattr__ with HTTP method names (lines 136-141)."""
        rt_funcs = RouteFuncs()

        # Should raise AttributeError for HTTP method names
        for method in ["get", "post", "put", "delete", "patch", "head", "options"]:
            with pytest.raises(AttributeError, match="Route functions with HTTP Names are not accessible"):
                getattr(rt_funcs, method)

    def test_route_funcs_getattr_missing(self):
        """Test RouteFuncs.__getattr__ with missing function."""
        rt_funcs = RouteFuncs()

        with pytest.raises(AttributeError, match="No route named missing found"):
            _ = rt_funcs.missing

    def test_route_funcs_dir(self):
        """Test RouteFuncs.__dir__ method (line 144)."""
        rt_funcs = RouteFuncs()

        rt_funcs.func1 = lambda: "func1"
        rt_funcs.func2 = lambda: "func2"

        dir_result = dir(rt_funcs)
        assert "func1" in dir_result
        assert "func2" in dir_result
        assert isinstance(dir_result, list)


class TestServeFunctionMissingCoverage:
    """Test serve function missing coverage."""

    def test_serve_function_exists_and_callable(self):
        """Test that serve function exists and is callable (basic coverage)."""
        # Just test that the function exists and can be imported
        assert callable(serve)

        # Test that it handles appname parameter
        try:
            # This will try to import uvicorn and might fail, but we're testing the logic before that
            serve(appname=None)  # Should not crash on None appname
        except ImportError:
            # uvicorn not available for import is expected in test environment
            pass
        except Exception as e:
            # Other exceptions might be expected too (like module not found)
            assert "uvicorn" in str(e).lower() or "module" in str(e).lower()

    @patch.dict(os.environ, {"PORT": "8080"})
    def test_serve_port_env_var_handling(self):
        """Test that serve uses PORT environment variable."""
        # We can't easily test the full serve function, but we can test the PORT handling logic
        import os

        expected_port = int(os.getenv("PORT", "5001"))
        assert expected_port == 8080  # Should use env var


class TestCookieMissingCoverage:
    """Test cookie function missing coverage."""

    def test_cookie_with_datetime_expires(self):
        """Test cookie with datetime expires (lines 198-202)."""
        expires_date = datetime(2024, 12, 31, 23, 59, 59)
        result = cookie("test", "value", expires=expires_date)

        # Should format datetime as HTTP date string
        assert "expires=" in result.v
        assert "31 Dec 2024 23:59:59 GMT" in result.v

    def test_cookie_with_string_expires(self):
        """Test cookie with string expires (else branch of lines 200-202)."""
        expires_str = "Wed, 21 Oct 2025 07:28:00 GMT"
        result = cookie("test", "value", expires=expires_str)

        assert expires_str in result.v


class TestSignalShutdownMissingCoverage:
    """Test signal_shutdown missing coverage."""

    def test_signal_shutdown_returns_event(self):
        """Test signal_shutdown returns an event (lines 223-233)."""
        event = signal_shutdown()

        # Should return an asyncio Event
        assert hasattr(event, "set")
        assert hasattr(event, "is_set")
        assert not event.is_set()

        # Test that we can set the event
        event.set()
        assert event.is_set()


class TestResponseRendererMissingCoverage:
    """Test ResponseRenderer missing coverage."""

    def test_response_renderer_file_response_not_exists(self):
        """Test ResponseRenderer with FileResponse that doesn't exist (lines 289, 316-318)."""
        mock_request = Mock()
        renderer = ResponseRenderer(mock_request)

        # Create FileResponse with non-existent file
        file_response = FileResponse("nonexistent_file.txt")

        with pytest.raises(HTTPException) as exc_info:
            renderer.process(file_response)

        assert exc_info.value.status_code == 404

    def test_response_renderer_empty_resp(self):
        """Test ResponseRenderer with falsy response (line 310)."""
        mock_request = Mock()
        mock_request.injects = []
        renderer = ResponseRenderer(mock_request)

        # Test with None, empty string, empty list, etc.
        for empty_resp in [None, "", [], False, 0]:
            result = renderer.process(empty_resp)
            assert isinstance(result, HTMLResponse)

    def test_response_renderer_http_headers_background_tasks(self):
        """Test ResponseRenderer with HttpHeaders and BackgroundTasks (lines 337, 339-342)."""
        mock_request = Mock()
        mock_request.injects = []
        renderer = ResponseRenderer(mock_request)

        # Create HttpHeader and BackgroundTask
        header = HttpHeader("X-Custom", "value")

        def task_func():
            pass

        task = BackgroundTask(task_func)

        result = renderer.process(("content", header, task))

        # Should have header and background task
        assert "X-Custom" in result.headers
        assert result.headers["X-Custom"] == "value"
        assert result.background is not None

    def test_response_renderer_dict_response(self):
        """Test ResponseRenderer with dict (lines 353-354)."""
        mock_request = Mock()
        mock_request.injects = []
        renderer = ResponseRenderer(mock_request)

        result = renderer.process({"key": "value"})

        assert isinstance(result, JSONResponse)

    def test_response_renderer_string_response(self):
        """Test ResponseRenderer with string (lines 355-356)."""
        mock_request = Mock()
        mock_request.injects = []
        renderer = ResponseRenderer(mock_request)

        result = renderer.process("hello world")

        assert isinstance(result, HTMLResponse)
        assert "hello world" in result.body.decode()

    def test_response_renderer_other_response(self):
        """Test ResponseRenderer with other types (line 357)."""
        mock_request = Mock()
        mock_request.injects = []
        renderer = ResponseRenderer(mock_request)

        result = renderer.process(42)  # Number gets converted to string

        assert isinstance(result, HTMLResponse)
        assert "42" in result.body.decode()

    def test_response_renderer_cls_handling(self):
        """Test ResponseRenderer cls parameter handling (lines 361-364)."""
        mock_request = Mock()
        mock_request.injects = []
        renderer = ResponseRenderer(mock_request)

        # Test with special cls values that should be ignored
        from typing import Any

        for special_cls in [Any, FT, empty]:
            result = renderer.process("content", cls=special_cls)
            assert isinstance(result, HTMLResponse)  # Should default to HTML

    def test_response_renderer_ft_processing(self):
        """Test ResponseRenderer FT object processing (lines 374, 377-379)."""
        mock_request = Mock()
        mock_request.app = Mock(title="Test", canonical=False)
        mock_request.hdrs = []
        mock_request.ftrs = []
        mock_request.bodykw = {}
        mock_request.htmlkw = {}
        mock_request.body_wrap = lambda x: x
        mock_request.injects = []

        renderer = ResponseRenderer(mock_request)

        # Test with tuple of FT objects that have proper attrs
        ft1 = FT("div", ("content1",), {"class": "test1"})
        ft2 = FT("div", ("content2",), {"class": "test2"})

        result = renderer.process((ft1, ft2))

        assert isinstance(result, HTMLResponse)

    def test_response_renderer_target_attributes(self):
        """Test ResponseRenderer target attribute processing (lines 389-393)."""
        mock_request = Mock()
        mock_request.app = Mock(title="Test", canonical=False)
        mock_request.hdrs = []
        mock_request.ftrs = []
        mock_request.bodykw = {}
        mock_request.htmlkw = {}
        mock_request.body_wrap = lambda x: x
        mock_request.injects = []

        renderer = ResponseRenderer(mock_request)

        # Create FT with target attributes
        ft_elem = FT("div", ("content",))
        ft_elem.attrs = {"get": "/api/data", "link": "/page"}

        # Mock _url_for function
        with patch("starhtml.server._url_for") as mock_url_for:
            mock_url_for.return_value = "/mocked/url"

            result = renderer.process(ft_elem)

            assert isinstance(result, HTMLResponse)

    def test_response_renderer_full_page_check(self):
        """Test ResponseRenderer full page detection (line 403, 429)."""
        mock_request = Mock()
        mock_request.app = Mock(title="Test", canonical=False)
        mock_request.hdrs = []
        mock_request.ftrs = []
        mock_request.bodykw = {}
        mock_request.htmlkw = {}
        mock_request.body_wrap = lambda x: x
        mock_request.injects = []

        renderer = ResponseRenderer(mock_request)

        # Test with empty response
        assert not renderer._is_full_page(())
        assert not renderer._is_full_page([])

        # Test with HTML tag (should be full page)
        html_elem = FT("html", ("content",))
        assert renderer._is_full_page((html_elem,))


class TestUtilityFunctionsMissingCoverage:
    """Test utility functions missing coverage."""

    def test_to_string_patch_exists(self):
        """Test that to_string patch is applied to StringConvertor."""
        from starlette.convertors import StringConvertor

        converter = StringConvertor()

        # Test that the patch was applied and method exists
        assert hasattr(converter, "to_string")

        # Test normal conversion
        result = converter.to_string("normal_value")
        assert result == "normal_value"

        # Test slash rejection
        with pytest.raises(AssertionError, match="May not contain path separators"):
            converter.to_string("path/with/slash")

    def test_url_path_for_patch_exists(self):
        """Test that url_path_for patch exists on HTTPConnection."""
        from starlette.requests import HTTPConnection

        # Test that the patch was applied
        assert hasattr(HTTPConnection, "url_path_for")

        # The actual functionality is tested in integration tests


class TestRequestResponsePipelineMissingCoverage:
    """Test request-response pipeline missing coverage."""

    @pytest.mark.asyncio
    async def test_find_p_special_annotations(self):
        """Test _find_p with special annotations (lines 450, 452, 454)."""
        from inspect import Parameter

        from starlette.applications import Starlette

        from starhtml.server import _find_p

        mock_request = Mock(spec=Request)
        mock_request.scope = {"app": Mock(spec=Starlette), "session": {"user": "test"}}

        # Test Starlette annotation
        param = Parameter("app", Parameter.POSITIONAL_OR_KEYWORD, annotation=Starlette)
        result = await _find_p(mock_request, "app", param)
        assert result == mock_request.scope["app"]

        # Test session with body annotation (when arg starts with "session")
        body_annotation = dict  # This would be a body type
        param = Parameter("sess", Parameter.POSITIONAL_OR_KEYWORD, annotation=body_annotation)
        result = await _find_p(mock_request, "sess", param)
        assert result == {"user": "test"}

    @pytest.mark.asyncio
    async def test_find_p_special_names_no_annotation(self):
        """Test _find_p with special names and no annotation (lines 461-473)."""
        from inspect import Parameter

        from starhtml.server import _find_p

        mock_request = Mock(spec=Request)
        mock_request.scope = {"session": {"user": "test"}, "auth": {"token": "abc"}, "app": Mock()}
        mock_request.body = AsyncMock(return_value=b"body content")

        # Test various special names with empty annotation
        param = Parameter("name", Parameter.POSITIONAL_OR_KEYWORD, annotation=empty)

        # Test session
        result = await _find_p(mock_request, "sess", param)  # "session".startswith("sess")
        assert result == {"user": "test"}

        # Test scope
        result = await _find_p(mock_request, "scope", param)
        assert hasattr(result, "session")  # Should be dict2obj'd

        # Test auth
        result = await _find_p(mock_request, "auth", param)
        assert result == {"token": "abc"}

        # Test app
        result = await _find_p(mock_request, "app", param)
        assert result == mock_request.scope["app"]

        # Test body
        result = await _find_p(mock_request, "body", param)
        assert result == "body content"

        # Test special request attributes
        mock_request.hdrs = ["header1"]
        result = await _find_p(mock_request, "hdrs", param)
        assert result == ["header1"]

    @pytest.mark.asyncio
    async def test_find_p_parameter_resolution_missing_required(self):
        """Test _find_p parameter resolution with missing required field (lines 488-489)."""
        from inspect import Parameter

        from starhtml.server import _find_p

        mock_request = Mock(spec=Request)
        mock_request.path_params = {}
        mock_request.cookies = {}
        mock_request.headers = {}
        mock_request.query_params = Mock(getlist=Mock(return_value=[]))

        # Mock form parsing to return empty
        with (
            patch("starhtml.server.parse_form") as mock_parse_form,
            patch("starhtml.server.form2dict") as mock_form2dict,
        ):
            mock_parse_form.return_value = {}
            mock_form2dict.return_value = {}

            # Parameter with no default and not found anywhere
            param = Parameter("missing_param", Parameter.POSITIONAL_OR_KEYWORD, annotation=str, default=empty)

            with pytest.raises(HTTPException) as exc_info:
                await _find_p(mock_request, "missing_param", param)

            assert exc_info.value.status_code == 400
            assert "Missing required field: missing_param" in str(exc_info.value.detail)

    @pytest.mark.asyncio
    async def test_find_p_annotation_cast_error(self):
        """Test _find_p annotation casting error (line 495)."""
        from inspect import Parameter

        from starhtml.server import _find_p

        mock_request = Mock(spec=Request)
        mock_request.path_params = {"param": "not_a_number"}
        mock_request.cookies = {}
        mock_request.headers = {}
        mock_request.query_params = Mock(getlist=Mock(return_value=[]))

        # Parameter that should cast to int but can't
        param = Parameter("param", Parameter.POSITIONAL_OR_KEYWORD, annotation=int)

        with pytest.raises(HTTPException) as exc_info:
            await _find_p(mock_request, "param", param)

        assert exc_info.value.status_code == 404


class TestMkLocfuncMissingCoverage:
    """Test _mk_locfunc missing coverage."""

    def test_mk_locfunc_creation(self):
        """Test _mk_locfunc creates proper location function."""

        def test_func():
            return "test"

        loc_func = _mk_locfunc(test_func, "/test/path")

        # Should wrap the original function
        assert callable(loc_func)
        assert str(loc_func) == "/test/path"

        # Should have 'to' method for URL generation
        assert hasattr(loc_func, "to")

        # Test calling the function
        result = loc_func()
        assert result == "test"

        # Test 'to' method exists and works
        try:
            # This may fail if qp is not available, but that's OK for coverage
            url = loc_func.to(key="value")
            # If it works, it should return a string
            assert isinstance(url, str)
        except Exception:
            # qp function may not be available in test context, which is fine
            pass


class TestFragmentResponse:
    """Test Fragment response behavior for Datastar HTML responses."""

    def test_fragment_basic(self):
        """Fragment with explicit selector returns correct Datastar headers."""
        from starlette.testclient import TestClient

        from starhtml import Div, star_app

        app, rt = star_app()

        @rt("/update")
        def update():
            return Fragment(Div("Updated content", id="foo"), selector="#foo")

        client = TestClient(app)
        response = client.get("/update")

        assert response.status_code == 200
        assert response.headers["content-type"] == "text/html; charset=utf-8"
        assert response.headers["datastar-selector"] == "#foo"
        assert "datastar-mode" not in response.headers
        assert "Updated content" in response.text

    def test_fragment_auto_selector(self):
        """Fragment auto-detects selector from element id attribute."""
        from starlette.testclient import TestClient

        from starhtml import Div, star_app

        app, rt = star_app()

        @rt("/update")
        def update():
            return Fragment(Div("Auto content", id="auto-id"))

        client = TestClient(app)
        response = client.get("/update")

        assert response.headers["datastar-selector"] == "#auto-id"
        assert "Auto content" in response.text

    def test_fragment_no_selector(self):
        """Fragment without selector or id has no datastar-selector header."""
        from starlette.testclient import TestClient

        from starhtml import Div, star_app

        app, rt = star_app()

        @rt("/update")
        def update():
            return Fragment(Div("Content without selector"))

        client = TestClient(app)
        response = client.get("/update")

        assert "datastar-selector" not in response.headers
        assert "Content without selector" in response.text

    def test_fragment_mode_outer(self):
        """Fragment with outer mode (default) has no datastar-mode header."""
        from starlette.testclient import TestClient

        from starhtml import Div, star_app

        app, rt = star_app()

        @rt("/update")
        def update():
            return Fragment(Div("Outer mode"), selector="#target", mode="outer")

        client = TestClient(app)
        response = client.get("/update")
        assert "datastar-mode" not in response.headers

    def test_fragment_mode_inner(self):
        """Fragment with inner mode sets correct header."""
        from starlette.testclient import TestClient

        from starhtml import Div, star_app

        app, rt = star_app()

        @rt("/update")
        def update():
            return Fragment(Div("Inner mode"), selector="#target", mode="inner")

        client = TestClient(app)
        response = client.get("/update")
        assert response.headers["datastar-mode"] == "inner"

    def test_fragment_mode_other(self):
        """Fragment respects append, prepend, before, after, replace, remove modes."""
        from starlette.testclient import TestClient

        from starhtml import Div, star_app

        def make_handler(mode_value):
            def handler():
                return Fragment(Div(f"{mode_value} mode"), selector="#target", mode=mode_value)

            return handler

        for mode in ["append", "prepend", "before", "after", "replace", "remove"]:
            app, rt = star_app()
            rt(f"/{mode}")(make_handler(mode))

            client = TestClient(app)
            response = client.get(f"/{mode}")
            assert response.headers["datastar-mode"] == mode

    def test_fragment_view_transition(self):
        """Fragment with view transition sets correct header."""
        from starlette.testclient import TestClient

        from starhtml import Div, star_app

        app, rt = star_app()

        @rt("/update")
        def update():
            return Fragment(Div("Transition content"), selector="#result", use_view_transition=True)

        client = TestClient(app)
        response = client.get("/update")

        assert response.headers["datastar-use-view-transition"] == "true"
        assert response.headers["datastar-selector"] == "#result"

    def test_fragment_custom_headers(self):
        """Fragment supports additional custom headers."""
        from starlette.testclient import TestClient

        from starhtml import Div, star_app

        app, rt = star_app()

        @rt("/update")
        def update():
            return Fragment(Div("Custom header content"), selector="#result", **{"X-Custom": "test-value"})

        client = TestClient(app)
        response = client.get("/update")

        assert response.headers["X-Custom"] == "test-value"
        assert response.headers["datastar-selector"] == "#result"
