# @Coding: UTF-8
# @Time: 2024/9/22 17:15
# @Author: xieyang_ls
# @Filename: spirit_application.py

import os

import json

import inspect

import importlib.util

from threading import Lock

from typing import Callable

from cgi import FieldStorage

from types import FunctionType, MethodType

from pyutils_spirit.util.set import Set, HashSet

from pyutils_spirit.util.json_util import deep_dumps

from logging import info, INFO, basicConfig, exception

from pyutils_spirit.style.resources import draw_spirit_banner

from pyutils_spirit.util.assemble import Assemble, HashAssemble

from pyutils_spirit.spirit_container.request_result import Result

from http.server import BaseHTTPRequestHandler, ThreadingHTTPServer

from pyutils_spirit.spirit_container.multipart_file import MultipartFile

from pyutils_spirit.exception.exception import BusinessException, SpiritContainerServiceException

from pyutils_spirit.spirit_container.spirit_application_container import SpiritApplicationContainer

basicConfig(level=INFO)

SIGNATURE_SET: set[str] = {"Component", "Mapper", "Service", "Controller",
                           "WebSocketServerEndPoint", "RequestInterceptor", "ExceptionAdvice"}


class SpiritApplication:
    __SPIRIT_APPLICATION: object = None

    REQUEST_BODY_KEY: str = None

    UNIFIED_RESPONSE_TYPE: bool = None

    __CONTAINER: SpiritApplicationContainer = None

    __LOCK: Lock = Lock()

    __CONTROLLER_PATH_SET: set[str] = None

    EXCEPTION_ADVICE_ASSEMBLE: Assemble[type, MethodType] = None

    __INTERCEPTOR_PATH_SET: set[str] = None

    INTERCEPTOR_BEFORE: Callable[[BaseHTTPRequestHandler], tuple[int, bool]] = None

    INTERCEPTOR_AFTER: Callable[[BaseHTTPRequestHandler], None] = None

    def __init__(self, host: str, port: int,
                 unified_response_type: bool = True,
                 request_body_key: str = "body") -> None:
        SpiritApplication.UNIFIED_RESPONSE_TYPE = unified_response_type
        SpiritApplication.REQUEST_BODY_KEY = request_body_key
        self.__host: str = host
        self.__port: int = port
        self.__auto_wired_set = set()

    def __new__(cls, host: str, port: int,
                unified_response_type: bool = True,
                request_body_key: str = "body") -> object:
        cls.__LOCK.acquire()
        try:
            if cls.__SPIRIT_APPLICATION is None:
                cls.__CONTAINER = SpiritApplicationContainer()
                cls.__CONTROLLER_PATH_SET = set()
                cls.EXCEPTION_ADVICE_ASSEMBLE = HashAssemble()
                cls.__INTERCEPTOR_PATH_SET = set()
                cls.__SPIRIT_APPLICATION = object.__new__(cls)
            else:
                raise SpiritContainerServiceException
            return cls.__SPIRIT_APPLICATION
        finally:
            cls.__LOCK.release()

    def __call__(self, cls) -> type:
        module = inspect.getmodule(cls)
        self.__current_file__ = module.__file__
        draw_spirit_banner()
        self.__scan_modules(os.getcwd())
        self.__start_service(host=self.__host, port=self.__port)
        return cls

    def __scan_modules(self, work_directory: str):
        for dirpath, dir_names, filenames in os.walk(work_directory):
            for file_name in filenames:
                if file_name == "__init__.py":
                    continue
                if file_name.endswith('.py'):
                    file_path = os.path.join(dirpath, file_name)
                    self.__load_module(file_path)
        self.__auto_injected()

    def __load_module(self, file_path):
        if file_path == self.__current_file__:
            return None
        module_path = file_path[:-3]
        module_name = os.path.basename(module_path)
        spec = importlib.util.spec_from_file_location(module_name, file_path)
        module = importlib.util.module_from_spec(spec)
        spec.loader.exec_module(module)
        self.__analyze_module(module)

    def __analyze_module(self, module) -> None:
        unique_modules: Set[object] = HashSet()
        for name, obj in inspect.getmembers(module):
            if obj is None:
                continue
            if obj not in unique_modules:
                unique_modules.add(obj)
                if isinstance(obj, FunctionType | type):
                    decorator = getattr(obj, "__decorator__", None)
                    if decorator in SIGNATURE_SET:
                        instance = obj()
                        self.__auto_wired_set.add(instance)
                        if decorator == "Controller":
                            path = getattr(obj, "__decorator_path__")
                            SpiritApplication.__CONTROLLER_PATH_SET.add(path)
                        elif decorator == "ExceptionAdvice":
                            for method_name, method in inspect.getmembers(instance):
                                if isinstance(method, MethodType):
                                    decorator = getattr(method, "__decorator__", None)
                                    if decorator == "ThrowsException":
                                        params = getattr(method, "__decorator_params__")
                                        SpiritApplication.EXCEPTION_ADVICE_ASSEMBLE.put(params, method)
                        elif decorator == "RequestInterceptor":
                            SpiritApplication.__INTERCEPTOR_PATH_SET = getattr(obj, "__decorator_params__")
                            for method_name, method in inspect.getmembers(instance):
                                if isinstance(method, MethodType):
                                    decorator = getattr(method, "__decorator__", None)
                                    if decorator == "InterceptorBefore":
                                        SpiritApplication.INTERCEPTOR_BEFORE = method
                                    elif decorator == "InterceptorAfter":
                                        SpiritApplication.INTERCEPTOR_AFTER = method

    def __auto_injected(self) -> None:
        for instance in self.__auto_wired_set:
            for method_name, method in inspect.getmembers(instance):
                if isinstance(method, MethodType):
                    decorator = getattr(method, "__decorator__", None)
                    if decorator == "Resource":
                        if hasattr(method, "__self__") and isinstance(method.__self__, type):
                            method(cls=type(instance), resources={})
                        else:
                            method(self=instance, resources={})

    def __start_service(self, host: str, port: int):
        server_address: tuple[str, int] = (host, port)
        service = ThreadingHTTPServer(server_address, self.RequestHandler)
        info("Spirit Container Service Startup successfully")
        info(f"Listening on Server: {host}:{port}")
        service.serve_forever()

    @classmethod
    def do_interceptor_function(cls, request_path: str) -> bool:
        if len(cls.__INTERCEPTOR_PATH_SET) == 0:
            return False
        for path in cls.__INTERCEPTOR_PATH_SET:
            path_list = request_path.split(path, maxsplit=1)
            if len(path_list) == 1:
                return False
            if path_list[0] == "" and path_list[1][0] == "?":
                return True
        return False

    @classmethod
    def get_func_kwargs(cls, path: str, method_type: str) -> [object, callable, dict]:
        cls.__LOCK.acquire()
        try:
            for controller_path in cls.__CONTROLLER_PATH_SET:
                path_list = path.split(controller_path, maxsplit=1)
                if len(path_list) == 1:
                    continue
                if path_list[0] != "":
                    continue
                controller_func_path = path_list[1]
                if len(path_list) == 2 and controller_func_path[0] == "/":
                    controller = cls.__CONTAINER.get_resource(signature=controller_path)
                    for method_name, method in inspect.getmembers(controller):
                        decorator = getattr(method, "__decorator__", None)
                        if decorator is method_type:
                            func_path = getattr(method, "__decorator_path__")
                            func_path_list = controller_func_path.split(func_path, maxsplit=1)
                            if len(func_path_list) == 1:
                                continue
                            func_args = func_path_list[1]
                            if len(func_path_list) > 1:
                                if func_args == "":
                                    return method, None
                                elif func_args[0] == "?":
                                    kwargs = dict()
                                    for param in func_args.split("?")[1].split("&"):
                                        if "=" in param:
                                            key = param.split("=")[0].strip()
                                            value = param.split("=")[1].strip()
                                            kwargs[key] = value
                                    return method, kwargs
        finally:
            cls.__LOCK.release()
        raise ValueError(f"please check the path {path} of {method_type} Method")

    class RequestHandler(BaseHTTPRequestHandler):

        __data = None

        __response = None

        __is_interceptor: bool = False

        __is_pass: bool = False

        __response_code: int = 200

        def __init__(self, *args, **kwargs):
            super().__init__(*args, **kwargs)

        def __request_exception_advice(self, e: Exception):
            exception(e)
            self.__response_code = 500
            self.__response = str(e)
            method = SpiritApplication.EXCEPTION_ADVICE_ASSEMBLE.get(key=type(e))
            if method is not None:
                self.__response = method(e)
            elif SpiritApplication.UNIFIED_RESPONSE_TYPE is True:
                if isinstance(e, BusinessException):
                    self.__response = Result.WARN(self.__response)
                else:
                    self.__response = Result.ERROR(self.__response)

        def __is_interceptor_func(self):
            self.__is_interceptor = SpiritApplication.do_interceptor_function(self.path)
            if self.__is_interceptor:
                if SpiritApplication.INTERCEPTOR_BEFORE is not None:
                    self.__response_code, self.__is_pass = SpiritApplication.INTERCEPTOR_BEFORE(self)
                    if self.__is_pass is False:
                        self.__response = Result.WARN(f"the request '{self.path}' is be intercepted")
                        return True
            return False

        def __request_end_func(self):
            self.send_response(self.__response_code)
            intercepted = self.headers["X-Intercepted"]
            if intercepted == "True":
                self.send_header(keyword="X-Intercepted", value="True")
            else:
                self.send_header(keyword="X-Intercepted", value="False")
            self.send_header(keyword='Access-Control-Allow-Origin', value='*')
            self.send_header(keyword='Content-type', value='application/json')
            self.end_headers()
            self.wfile.write(deep_dumps(self.__response).encode('utf-8'))
            if self.__is_interceptor and self.__is_pass:
                if SpiritApplication.INTERCEPTOR_AFTER is not None:
                    SpiritApplication.INTERCEPTOR_AFTER(self)

        def do_OPTIONS(self):
            self.send_response(200)
            self.send_header('Access-Control-Allow-Origin', '*')
            self.send_header('Access-Control-Allow-Methods', 'GET, POST, PUT, DELETE, OPTIONS')
            self.send_header('Access-Control-Allow-Headers', '*')
            self.end_headers()

        def do_GET(self):
            try:
                if self.__is_interceptor_func():
                    return None
                func, kwargs = SpiritApplication.get_func_kwargs(self.path, "GET")
                if kwargs is not None:
                    self.__response = func(**kwargs)
                else:
                    self.__response = func()
                if SpiritApplication.UNIFIED_RESPONSE_TYPE is True:
                    self.__response = Result.SUCCESS(self.__response)
            except Exception as e:
                self.__request_exception_advice(e)
            finally:
                self.__request_end_func()

        def __get_post_file(self, post_body_length) -> (dict, bool):
            if self.headers['Content-Type'].startswith('multipart/form-data'):
                # 解析请求体
                field_storage = FieldStorage(fp=self.rfile,
                                             headers=self.headers,
                                             environ={'REQUEST_METHOD': 'POST'})
                form_data: dict = {}
                for form in field_storage.list:
                    if form.name in form_data:
                        data = form_data[form.name]
                        if not isinstance(data, list):
                            form_data[form.name] = [data]
                        if form.filename is not None and isinstance(form.value, bytes):
                            form_data[form.name].append(MultipartFile(filename=form.filename,
                                                                      file_bytes=form.value))
                        else:
                            form_data[form.name].append(form.value)
                    else:
                        if form.filename is not None and isinstance(form.value, bytes):
                            form_data[form.name] = MultipartFile(filename=form.filename,
                                                                 file_bytes=form.value)
                        else:
                            form_data[form.name] = form.value
                return form_data
            else:
                data = self.rfile.read(post_body_length)
                return {
                    SpiritApplication.REQUEST_BODY_KEY: json.loads(data)
                }

        def do_POST(self):
            post_body_length = int(self.headers['Content-Length'])
            try:
                if self.__is_interceptor_func():
                    return None
                func, kwargs = SpiritApplication.get_func_kwargs(self.path, "POST")
                if post_body_length > 0:
                    self.__data = self.__get_post_file(post_body_length)
                else:
                    self.__response_code = 400
                    raise ValueError("Post Method: Request Body must be not empty.")
                if kwargs is not None:
                    self.__response = func(**kwargs, **self.__data)
                else:
                    self.__response = func(**self.__data)
                if SpiritApplication.UNIFIED_RESPONSE_TYPE is True:
                    self.__response = Result.SUCCESS(self.__response)
            except Exception as e:
                self.__request_exception_advice(e)
            finally:
                self.__request_end_func()

        def __set_request_arguments(self, kwargs, func):
            if kwargs is not None:
                if self.__data is None:
                    self.__response = func(**kwargs)
                else:
                    self.__response = func(**kwargs, **self.__data)
            else:
                if self.__data is None:
                    self.__response = func()
                else:
                    self.__response = func(**self.__data)

        def do_PUT(self):
            put_body_length = int(self.headers['Content-Length'])
            if put_body_length > 0:
                data = self.rfile.read(put_body_length)
                self.__data = {
                    SpiritApplication.REQUEST_BODY_KEY: json.loads(data)
                }
            else:
                self.__data = None
            try:
                if self.__is_interceptor_func():
                    return None
                func, kwargs = SpiritApplication.get_func_kwargs(self.path, "PUT")
                self.__set_request_arguments(kwargs=kwargs, func=func)
                if SpiritApplication.UNIFIED_RESPONSE_TYPE is True:
                    self.__response = Result.SUCCESS(self.__response)
            except Exception as e:
                self.__request_exception_advice(e)
            finally:
                self.__request_end_func()

        def do_DELETE(self):
            content = self.headers['Content-Length']
            if content is not None:
                delete_body_length = int(content)
                if delete_body_length > 0:
                    data = self.rfile.read(delete_body_length)
                    self.__data = {
                        SpiritApplication.REQUEST_BODY_KEY: json.loads(data)
                    }
                else:
                    self.__data = None
            else:
                self.__data = None
            try:
                if self.__is_interceptor_func():
                    return None
                func, kwargs = SpiritApplication.get_func_kwargs(self.path, "DELETE")
                self.__set_request_arguments(kwargs=kwargs, func=func)
                if SpiritApplication.UNIFIED_RESPONSE_TYPE is True:
                    self.__response = Result.SUCCESS(self.__response)
            except Exception as e:
                self.__request_exception_advice(e)
            finally:
                self.__request_end_func()
