# @Coding: UTF-8
# @Time: 2024/9/20 12:29
# @Author: xieyang_ls
# @Filename: websocket.py

import time

import socket

import base64

import hashlib

import inspect

from abc import ABC

from pyutils_spirit.style.resources import draw_websocket_banner

from logging import info, basicConfig, INFO, exception

from pyutils_spirit.concurrent_thread.thread_executor import ThreadExecutor

from pyutils_spirit.spirit_container.spirit_application_container import SpiritApplicationContainer

basicConfig(level=INFO)


class WebSocket(ABC):
    __SOCKET_SERVER: socket.socket = None

    __LISTENER_COUNT = None

    __BUFFER_SIZE: int = None

    WEBSOCKET_END_POINTS: set[type] = None

    __EXECUTOR: ThreadExecutor = None

    CONTAINER: SpiritApplicationContainer = None

    __CLOSE_SIGNATURE = None

    @classmethod
    def start_server(cls, host, port):
        cls.__SOCKET_SERVER = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        cls.__LISTENER_COUNT = 100000
        cls.__BUFFER_SIZE = 4096
        cls.__EXECUTOR = ThreadExecutor(executor_count=cls.__LISTENER_COUNT + 1)
        cls.__SOCKET_SERVER.bind((host, port))
        cls.__SOCKET_SERVER.listen(cls.__LISTENER_COUNT)
        cls.__EXECUTOR.execute(cls.__listener_connection)
        cls.__CLOSE_SIGNATURE = ""
        info(f"WebSocket Server listening on ws://{host}:{port}")

    class Session:

        def __init__(self,
                     socket_client: socket.socket,
                     address: tuple[str, int],
                     headers: dict[str, str],
                     params: dict[str, str | int]):
            self.__socket: socket.socket = socket_client
            self.__address: tuple[str, int] = address
            self.__headers: dict[str, str] = headers
            self.__params: dict[str, str | int] = params
            self.__is_connected: bool = True
            self.__start_connected_timer: float = time.time()
            self.__close_connected_timer: float | None = None

        def get_socket_client(self) -> socket.socket:
            return self.__socket

        def get_is_connected(self) -> bool:
            return self.__is_connected

        def get_address(self):
            if self.__address is None:
                self.__address = self.__socket.getpeername()
            return self.__address

        def get_headers(self) -> dict:
            if self.__headers is not None:
                return self.__headers  # 已缓存的 headers 则直接返回

        def get_params(self) -> dict:
            return self.__params

        def send_message(self, message: str, encoding: str = "utf-8", opcode: int = 0x1) -> None:
            try:
                # 验证操作码
                if opcode not in (0x1, 0x2):
                    raise ValueError("Opcode must be 0x1 (text) or 0x2 (binary)")
                # 编码消息
                message_bytes = message.encode(encoding)
                fin = 0x80  # FIN=1（不分片）
                mask_bit = 0x00  # 服务器必须不掩码
                # 构造帧头
                header = bytearray()
                payload_len = len(message_bytes)
                # 处理长度字段
                if payload_len <= 125:
                    header += bytes([fin | opcode, payload_len | mask_bit])
                elif payload_len <= 65535:
                    header += bytes([fin | opcode, 126 | mask_bit])
                    header += payload_len.to_bytes(length=2, byteorder='big', signed=False)
                else:
                    header += bytes([fin | opcode, 127 | mask_bit])
                    header += payload_len.to_bytes(length=8, byteorder='big', signed=False)
                # 发送完整帧
                self.__socket.send(header + message_bytes)
            except (UnicodeEncodeError, ValueError) as e:
                exception(f"Encoding/Validation error: {str(e)}")
            except Exception as e:
                exception(f"Send error: {str(e)}")

        def __send_close_frame(self, code: int, reason: str) -> None:
            payload = code.to_bytes(length=2, byteorder='big', signed=False)
            if reason:
                payload += reason.encode('utf-8')
            # 构造帧头
            fin = 0x80
            opcode = 0x08  # 关闭帧
            length = len(payload)
            if length <= 125:
                frame = bytes([fin | opcode, length]) + payload
            elif length <= 65535:
                frame = bytes([fin | opcode, 126]) + length.to_bytes(length=2, byteorder='big') + payload
            else:
                frame = bytes([fin | opcode, 127]) + length.to_bytes(length=8, byteorder='big') + payload
            self.__socket.send(frame)

        def close(self, close_code: int = 1000, reason: str = "See you again~~") -> None:
            if self.__is_connected is True:
                self.__is_connected = False
                self.__close_connected_timer = time.time()
                info(f"Disconnecting Session: {self}")
                self.__send_close_frame(code=close_code, reason=reason)
                self.__socket.close()
                __endpoint = WebSocket.CONTAINER.remove_resource(signature=str(self.__address))
                __onclose: callable = __endpoint.ws_funcs["OnClose"]
                if __onclose is not None:
                    __onclose(__endpoint, session=self)
                session_id = id(self.__socket)
                if session_id in __endpoint.message_buffer:
                    del __endpoint.message_buffer[session_id]
                if self.__socket in __endpoint.partial_data:
                    del __endpoint.partial_data[self.__socket]

        def __str__(self) -> str:
            if self.__is_connected:
                connected_timer: str = f"current_connected_timer={time.time() - self.__start_connected_timer}"
            else:
                connected_timer = f"total_connected_timer={self.__close_connected_timer - self.__start_connected_timer}"
            return ("{" + f"socket={self.__socket}, " +
                    f"address={self.__address}, " +
                    f"{connected_timer}" + "}")

    @classmethod
    def __handshaking(cls, socket_client: socket.socket, ws_headers: dict) -> bool:
        # WebSocket 握手
        try:
            # 检查必要字段是否存在
            ws_key = ws_headers.get("Sec-WebSocket-Key", "")
            if not ws_key:
                ws_key = ws_headers.get("sec-websocket-key", "")
                if not ws_key:
                    raise ValueError("Missing Sec-WebSocket-Key")

            # 验证 WebSocket 版本
            ws_version = ws_headers.get("Sec-WebSocket-Version", "13")
            if ws_version != "13":
                response = (
                    "HTTP/1.1 426 Upgrade Required\r\n"
                    "Sec-WebSocket-Version: 13\r\n\r\n"
                )
                socket_client.send(response.encode())
                return False

            # 生成 Accept Key
            magic_guid = "258EAFA5-E914-47DA-95CA-C5AB0DC85B11"
            accept_key = base64.b64encode(
                hashlib.sha1((ws_key + magic_guid).encode()).digest()
            ).decode()

            # 构建响应头（增加版本确认）
            response_headers = [
                "HTTP/1.1 101 Switching Protocols",
                "Upgrade: websocket",
                "Connection: Upgrade",
                f"Sec-WebSocket-Accept: {accept_key}",
                "Sec-WebSocket-Version: 13",  # 明确返回支持的版本
            ]

            # 发送握手响应
            response = "\r\n".join(response_headers) + "\r\n\r\n"
            socket_client.send(response.encode())
            return True

        except (KeyError, ValueError) as e:
            # 处理无效请求
            exception(f"Key / Value Error: {e}")
            error_response = "HTTP/1.1 400 Bad Request\r\n\r\n"
            socket_client.send(error_response.encode())
            return False

        except Exception as e:
            # 处理其他异常（如连接中断）
            exception(f"Handshake error: {str(e)}")
            return False

    @classmethod
    def __listener_connection(cls) -> None:
        while True:
            try:
                socket_client, addr = cls.__SOCKET_SERVER.accept()
                # 缓冲区分块接收，直到遇到 \r\n\r\n
                header_data: bytes | None = None

                buffer = bytearray()
                while True:
                    chunk = socket_client.recv(cls.__BUFFER_SIZE)  # 每次接收 4KB
                    if not chunk:  # 连接关闭
                        break
                    buffer.extend(chunk)
                    if b"\r\n\r\n" in buffer:
                        # 找到头部结束位置并截断
                        header_end = buffer.index(b"\r\n\r\n")
                        header_data = buffer[:header_end]
                        break
                try:
                    # 解码并解析头部
                    ws_endpoint_cls: type | None = None
                    ws_headers: dict[str, str] = {}
                    params: dict[str, str | int] = {}
                    header_str = header_data.decode("utf-8")
                    lines = header_str.split("\r\n")
                    for end_point_cls in cls.WEBSOCKET_END_POINTS:
                        signature = getattr(end_point_cls, "__decorator_signature__")
                        paths = lines[0].split(signature)
                        if len(paths) == 2 and paths[0][-1] == " " and paths[1][0] in (" ", "?"):
                            ws_endpoint_cls = end_point_cls
                            break
                    if ws_endpoint_cls is None:
                        ex_msg = f"Error: WebSocket URL({lines[0].split()[1]}) cannot match endpoint signature"
                        ex_str = (
                            "HTTP/1.1 400 Bad Request\r\n"
                            "Content-Type: text/plain\r\n"
                            "Connection: close({})\r\n\r\n"
                        ).format(ex_msg)
                        socket_client.send(ex_str.encode("utf-8"))
                        socket_client.close()
                        raise (ValueError(f"{ex_msg}"))
                    arguments = lines[0].split()[1].split("?")
                    if len(arguments) > 1:
                        for param in arguments[1].split("&"):
                            key, value = param.split(sep="=", maxsplit=1)
                            params[key] = value
                    for line in lines[1:]:  # 跳过请求行（如 GET / HTTP/1.1）
                        if ": " in line:
                            key, value = line.split(sep=": ", maxsplit=1)
                            ws_headers[key] = value
                except UnicodeDecodeError as e:
                    # 处理编码错误
                    exception(f"Header decoding error: {e}")
                    continue
                except ValueError as e:
                    # 处理键值对解析错误
                    exception(f"Invalid header format: {e}")
                    continue
                if cls.__handshaking(socket_client=socket_client, ws_headers=ws_headers):
                    __endpoint = ws_endpoint_cls()
                    cls.CONTAINER.set_resource(signature=str(addr), resource=__endpoint)
                    session = cls.Session(socket_client=socket_client,
                                          address=addr,
                                          headers=ws_headers,
                                          params=params)
                    info(f"Session: {session} connection successful")
                    cls.__EXECUTOR.execute(task=cls.__listener_message, end_point=__endpoint, session=session)
                    __onopen: callable = __endpoint.ws_funcs["OnOpen"]
                    if __onopen is not None:
                        __onopen(__endpoint, session=session)
            except Exception as e:
                exception(e)

    @classmethod
    def __receive_message(cls, end_point, session: Session):
        # 接收消息
        socket_client: socket.socket = session.get_socket_client()
        session_id = id(socket_client)
        data = end_point.partial_data.get(socket_client, b'') + socket_client.recv(cls.__BUFFER_SIZE)

        try:
            while len(data) >= 2:
                # 1. 解析帧头
                fin = (data[0] & 0x80) >> 7
                rsv = (data[0] & 0x70) >> 4
                opcode = data[0] & 0x0F
                masked = (data[1] & 0x80) >> 7
                payload_len = data[1] & 0x7F
                # 2. 协议合规性校验
                if rsv != 0:
                    raise ValueError("RSV位违规")
                if not masked:  # 所有客户端帧必须掩码
                    raise ValueError("未掩码数据帧")
                # 3. 解析payload长度和掩码
                mask_start = 2
                if payload_len == 126:
                    if len(data) < 4:
                        break
                    payload_len = int.from_bytes(data[2:4], 'big')
                    mask_start = 4
                elif payload_len == 127:
                    if len(data) < 10:
                        break
                    if (data[2] & 0x80) != 0:
                        raise ValueError("无效长度")
                    payload_len = int.from_bytes(data[2:10], 'big')
                    mask_start = 10
                # 4. 检查数据完整性
                if len(data) < mask_start + 4 + payload_len:
                    break
                # 5. 解码payload
                masks = data[mask_start:mask_start + 4]
                payload = data[mask_start + 4:mask_start + 4 + payload_len]
                decoded = bytes([payload[i] ^ masks[i % 4] for i in range(payload_len)])
                # 6. 处理控制帧
                if opcode in (0x8, 0x9, 0xA):
                    if not fin:
                        raise ValueError("控制帧不能分片")
                    if payload_len > 125:
                        raise ValueError("控制帧负载过长")
                    if opcode == 0x8:  # Close帧
                        return session.close()
                    elif opcode == 0x9:  # Ping帧
                        # 发送Pong帧，携带解码后的数据
                        header = bytearray([0x8A])  # FIN=1, opcode=0xA
                        if len(decoded) <= 125:
                            header.append(len(decoded))
                        elif len(decoded) <= 0xFFFF:
                            header.extend([126, *len(decoded).to_bytes(length=2, byteorder='big')])
                        else:
                            header.extend([127, *len(decoded).to_bytes(length=8, byteorder='big')])
                        socket_client.send(bytes(header) + decoded)
                        # 继续处理剩余数据
                        data = data[mask_start + 4 + payload_len:]
                        continue
                    elif opcode == 0xA:  # Pong帧，忽略
                        data = data[mask_start + 4 + payload_len:]
                        continue
                # 7. 处理数据帧
                if session_id not in end_point.message_buffer:
                    end_point.message_buffer[session_id] = {'fragments': [], 'opcode': opcode}
                else:
                    if opcode != 0x0:
                        raise ValueError("分片连续性错误")
                end_point.message_buffer[session_id]['fragments'].append(decoded)
                if fin:
                    full_data = b''.join(end_point.message_buffer[session_id]['fragments'])
                    original_opcode = end_point.message_buffer[session_id]['opcode']
                    del end_point.message_buffer[session_id]
                    # 处理数据
                    if original_opcode == 0x1:
                        try:
                            result = full_data.decode('utf-8')
                        except UnicodeDecodeError as e:
                            return session.close(close_code=60001, reason=str(e))
                    else:
                        result = full_data
                    data = data[mask_start + 4 + payload_len:]
                    end_point.partial_data[socket_client] = data
                    return result
                else:
                    data = data[mask_start + 4 + payload_len:]
        except (ConnectionResetError, ValueError) as e:
            exception(f"协议错误: {e}")
            session.close(close_code=60001, reason=str(e))
        finally:
            end_point.partial_data[socket_client] = data
        return None

    @classmethod
    def __listener_message(cls, end_point, session: Session) -> None:
        while session.get_is_connected():
            try:
                message: str = cls.__receive_message(end_point=end_point, session=session)
                if message is None:
                    continue
                elif message == "ping":
                    session.send_message("pong")
                elif message == cls.__CLOSE_SIGNATURE:
                    session.close()
                else:
                    __onmessage: callable = end_point.ws_funcs["OnMessage"]
                    if __onmessage is not None:
                        __onmessage(end_point, message=message)
            except Exception as e:
                exception(e)
                __onerror: callable = end_point.ws_funcs["OnError"]
                if __onerror is not None:
                    try:
                        __onerror(end_point, session=session, ex=e)
                    except Exception as e:
                        exception(e)


class WebSocketServer:
    def __init__(self, host: str, port: int) -> None:
        if WebSocket.CONTAINER is None:
            WebSocket.CONTAINER = SpiritApplicationContainer()
        WebSocket.CONTAINER.set_resource(signature="WebSocketServer", resource=self)
        self.__host: str = host
        self.__port: int = port

    def __call__(self, ws_app_cls: type) -> type:
        draw_websocket_banner()
        WebSocket.start_server(host=self.__host, port=self.__port)
        return ws_app_cls


class EndPoint:
    def __init__(self, signature: str) -> None:
        if not isinstance(signature, str):
            raise TypeError("the signature must be a string")
        self.__signature: str = signature

    def __call__(self, end_point_cls: type):
        end_point_cls.__decorator__ = "WebSocketServerEndPoint"
        end_point_cls.__decorator_signature__ = self.__signature
        end_point_cls.partial_data = {}
        end_point_cls.message_buffer = {}
        ws_funcs: dict[str, callable] = {
            "OnOpen": None,
            "OnMessage": None,
            "OnError": None,
            "OnClose": None
        }
        for name, method in inspect.getmembers(end_point_cls):
            if type(method) in [OnOpen, OnMessage, OnError, OnClose]:
                decorator = getattr(method, "__decorator__", None)
                if decorator in ["OnOpen", "OnMessage", "OnError", "OnClose"]:
                    ws_funcs[decorator] = method
        end_point_cls.ws_funcs = ws_funcs
        if WebSocket.WEBSOCKET_END_POINTS is None:
            WebSocket.WEBSOCKET_END_POINTS = set()
        WebSocket.WEBSOCKET_END_POINTS.add(end_point_cls)
        return end_point_cls


class OnOpen:
    def __init__(self, func: callable) -> None:
        self.__decorator__ = "OnOpen"
        self.__func = func

    def __call__(self, *args, **kwargs) -> object:
        return self.__func(*args, **kwargs)


class OnMessage:
    def __init__(self, func: callable) -> None:
        self.__decorator__ = "OnMessage"
        self.__func = func

    def __call__(self, *args, **kwargs):
        return self.__func(*args, **kwargs)


class OnError:
    def __init__(self, func: callable) -> None:
        self.__decorator__ = "OnError"
        self.__func = func

    def __call__(self, *args, **kwargs):
        return self.__func(*args, **kwargs)


class OnClose:
    def __init__(self, func: callable) -> None:
        self.__decorator__ = "OnClose"
        self.__func = func

    def __call__(self, *args, **kwargs):
        return self.__func(*args, **kwargs)


onopen: type = OnOpen
onmessage: type = OnMessage
onerror: type = OnError
onclose: type = OnClose
Session: type = WebSocket.Session
