#!/usr/bin/env python
import matplotlib
matplotlib.use('Agg')
import matplotlib.pyplot as plt
import numpy as np

from torsiondrive.tools import read_scan_xyz, find_grid_spacing

def format_2d_grid_data(grid_data, verbose=False):
    """ Take a grid_data dictionary, figure out the grid spacing of each dimension,
    then return the formatted x_array, y_array and z_mat

    Parameters
    ----------
    grid_data: Dict
        {gid: energy} generated by load_data_from_scan_xyz()

    Returns
    -------
    x_array: numpy.ndarray of shape (n_grid1,)
        All possible values for the first dihedral angle in the 2d scan
    y_array: numpy.ndarray of shape (n_grid2,)
        All possible values for the second dihedral angle in the 2d scan
    z_mat: numpy.ndarray of shape (n_grid1, n_grid2)
        The energies of the 2-D scan. Missing points have values np.nan
    """
    if not grid_data:
        raise ValueError("Empty grid data")
    # determine 2d grid dimensions
    sorted_grid_ids = sorted(grid_data)
    assert len(sorted_grid_ids[0]) == 2, 'Only 2-D scan result is supported'
    # find the smallest difference in each dimension to be the spacing
    grid_id_list_1 = sorted(set(grid_id[0] for grid_id in sorted_grid_ids))
    grid_id_list_2 = sorted(set(grid_id[1] for grid_id in sorted_grid_ids))
    grid_spacing_1 = find_grid_spacing(grid_id_list_1)
    grid_spacing_2 = find_grid_spacing(grid_id_list_2)
    grid_size_1 = int(360 / grid_spacing_1)
    grid_size_2 = int(360 / grid_spacing_2)
    if verbose:
        print(f"grid_spacing: [{grid_spacing_1}, {grid_spacing_2}]")
        print(f"grid_size:    [{grid_size_1}, {grid_size_2}]")
    # create data matrix
    x_array = np.arange(-180, 180, grid_spacing_1, dtype=int) + grid_spacing_1
    y_array = np.arange(-180, 180, grid_spacing_2, dtype=int) + grid_spacing_2
    z_mat = np.zeros((grid_size_1, grid_size_2))
    for i, x in enumerate(x_array):
        for j, y in enumerate(y_array):
            z_mat[i, j] = grid_data.get((x,y), np.nan)
    return x_array, y_array, z_mat

def plot_grid_contour(grid_data, pdf_filename, method='imshow', vmax=None):
    """ Plot grid data as a contour map """
    if not grid_data:
        print("Empty grid data, returning")
        return
    x_array, y_array, z_mat = format_2d_grid_data(grid_data)
    # convert abs energies to relative energies
    z_mat = (z_mat - np.nanmin(z_mat)) * 627.509
    # plot
    plt.Figure()
    if method == 'imshow':
        plt.imshow(z_mat, vmax=vmax, cmap='rainbow', origin='lower', extent=(-165, 180, -165, 180))
    elif method == 'contourf':
        plt.contourf(y_array, x_array, z_mat, vmax=vmax, antialiased=True, cmap='rainbow')
    plt.colorbar()
    plt.xticks(x_array[1::2], x_array[1::2])
    plt.yticks(y_array[1::2], y_array[1::2])
    cs = plt.contour(y_array, x_array, z_mat, vmax=vmax, antialiased=True, colors='black')
    plt.clabel(cs, fontsize=9, inline=1)
    plt.savefig(pdf_filename)
    plt.close()

def main():
    import argparse
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument("infile", help='scan.xyz file from torsionscan launch')
    parser.add_argument("-m", "--method", choices=['contourf', 'imshow'], default='imshow', help='method to color background')
    parser.add_argument("--vmax", type=float, help='max value of heat map')
    parser.add_argument("-o", "--output", default="torsiondrive_2D.pdf", help='Output pdf filename for plot')
    args = parser.parse_args()

    grid_data = read_scan_xyz(args.infile)
    plot_grid_contour(grid_data, args.output, method=args.method, vmax=args.vmax)
    print(f"Plot saved as {args.output}")

if __name__ == '__main__':
    main()
