"""Custom exceptions for structured-prompts."""


class StructuredPromptsError(Exception):
    """Base exception for all structured-prompts errors."""


class UnsupportedValueTypeError(StructuredPromptsError):
    """Raised when an interpolation value is neither str nor StructuredPrompt."""

    def __init__(self, key: str, value_type: type, expression: str):
        self.key = key
        self.value_type = value_type
        self.expression = expression
        super().__init__(
            f"Unsupported value type for interpolation '{expression}' (key='{key}'): "
            f"expected str or StructuredPrompt, got {value_type.__name__}"
        )


class DuplicateKeyError(StructuredPromptsError):
    """Raised when duplicate keys are found and allow_duplicate_keys=False."""

    def __init__(self, key: str):
        self.key = key
        super().__init__(
            f"Duplicate key '{key}' found. Use allow_duplicate_keys=True to allow duplicates, "
            "or use different format specs to create unique keys."
        )


class MissingKeyError(StructuredPromptsError, KeyError):
    """Raised when a key is not found during dict-like access."""

    def __init__(self, key: str, available_keys: list[str]):
        self.key = key
        self.available_keys = available_keys
        super().__init__(f"Key '{key}' not found. Available keys: {', '.join(repr(k) for k in available_keys)}")


class NotANestedPromptError(StructuredPromptsError):
    """Raised when attempting to index into a non-nested interpolation."""

    def __init__(self, key: str):
        self.key = key
        super().__init__(
            f"Cannot index into interpolation '{key}': value is not a StructuredPrompt. "
            "Only nested prompts support indexing."
        )


class EmptyExpressionError(StructuredPromptsError):
    """Raised when an empty expression {} is encountered."""

    def __init__(self):
        super().__init__(
            "Empty expression {{}} is not allowed. All interpolations must have a valid expression or format spec."
        )
