#include "astrid.h"

/**
  █████╗ ███████╗████████╗██████╗ ██╗██████╗ 
 ██╔══██╗██╔════╝╚══██╔══╝██╔══██╗██║██╔══██╗
 ███████║███████╗   ██║   ██████╔╝██║██║  ██║
 ██╔══██║╚════██║   ██║   ██╔══██╗██║██║  ██║
 ██║  ██║███████║   ██║   ██║  ██║██║██████╔╝
 ╚═╝  ╚═╝╚══════╝   ╚═╝   ╚═╝  ╚═╝╚═╝╚═════╝ 

 SECTIONS

 IPCTOOLS     inter-process (inter-instrument) communication tools
              for sharing data like control signals and buffers 
              between/within instruments
 MIDICOMM     tools for sending and receiving MIDI messages
 JACKAPI      jack process callback and init/cleanup lifecycle procs
*/

static volatile int * astrid_instrument_is_running;

void handle_instrument_shutdown(__attribute__((unused)) int sig) {
    *astrid_instrument_is_running = 0;
}

/* MIDI 
 * COMMUNICATION
 * **MIDICOMM***/
int lpmidi_setcc(astrid_session_t * session, int device_id, int channel, int cc, int value) {
    char key_name[LPMAXNAME] = {0};
    snprintf(key_name, LPMAXNAME, "midi-cc-%d-%d-%d", device_id, channel, cc);
    
    syslog(LOG_INFO, "lpmidi_setcc device_id=%d channel=%d cc=%d value=%d key=%s\n", device_id, channel, cc, value, key_name);
    
    if(astrid_session_register_shared_resource(session, key_name, &value, ASTRID_TYPE_INT, sizeof(int)) < 0) {
        syslog(LOG_ERR, "lpmidi_setcc: Could not store CC value in session\n");
        return -1;
    }
    
    return 0;
}

int lpmidi_getcc(astrid_session_t * session, int device_id, int channel, int cc) {
    char key_name[LPMAXNAME] = {0};
    int value = 0;
    snprintf(key_name, LPMAXNAME, "midi-cc-%d-%d-%d", device_id, channel, cc);
    
    if(astrid_session_get_shared_resource(session, key_name, &value) < 0) {
        return 0;
    }
    
    return value;
}

int lpmidi_setnote(astrid_session_t * session, int device_id, int channel, int note, int velocity) {
    char key_name[LPMAXNAME] = {0};
    snprintf(key_name, LPMAXNAME, "midi-note-%d-%d-%d", device_id, channel, note);
    
    syslog(LOG_INFO, "lpmidi_setnote device_id=%d channel=%d note=%d velocity=%d key=%s\n", device_id, channel, note, velocity, key_name);
    
    if(astrid_session_register_shared_resource(session, key_name, &velocity, ASTRID_TYPE_INT, sizeof(int)) < 0) {
        syslog(LOG_ERR, "lpmidi_setnote: Could not store note velocity in session\n");
        return -1;
    }
    
    return 0;
}

int lpmidi_getnote(astrid_session_t * session, int device_id, int channel, int note) {
    char key_name[LPMAXNAME] = {0};
    int velocity = 0;
    snprintf(key_name, LPMAXNAME, "midi-note-%d-%d-%d", device_id, channel, note);
    
    if(astrid_session_get_shared_resource(session, key_name, &velocity) < 0) {
        return 0;
    }
    
    return velocity;
}

void * instrument_udp_listener_thread(void * arg) {
    lpinstrument_t * instrument = (lpinstrument_t *)arg;
    int sockfd;
    struct sockaddr_in servaddr, cliaddr;
    socklen_t clilen = sizeof(cliaddr);
    char buffer[LPMAXMSG];
    int n;
    lpmsg_t msg;

    syslog(LOG_INFO, "%s UDP listener thread starting on port %d\n", 
           instrument->name, instrument->udp_port);

    // Create UDP socket
    if ((sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
        syslog(LOG_ERR, "%s UDP listener: Could not create socket. Error: %s\n", 
               instrument->name, strerror(errno));
        return NULL;
    }

    // Allow socket reuse
    int reuse = 1;
    if (setsockopt(sockfd, SOL_SOCKET, SO_REUSEADDR, &reuse, sizeof(reuse)) < 0) {
        syslog(LOG_ERR, "%s UDP listener: Could not set socket options. Error: %s\n", 
               instrument->name, strerror(errno));
        close(sockfd);
        return NULL;
    }

    // Bind socket
    memset(&servaddr, 0, sizeof(servaddr));
    servaddr.sin_family = AF_INET;
    servaddr.sin_addr.s_addr = INADDR_ANY;
    servaddr.sin_port = htons(instrument->udp_port);

    if (bind(sockfd, (struct sockaddr *)&servaddr, sizeof(servaddr)) < 0) {
        syslog(LOG_ERR, "%s UDP listener: Could not bind socket to port %d. Error: %s\n", 
               instrument->name, instrument->udp_port, strerror(errno));
        close(sockfd);
        return NULL;
    }

    // Get a producer handle for the internal message queue
    char qname[LPMAXKEY] = {0};
    snprintf(qname, LPMAXKEY, "%s-msgq", instrument->name);
    lpmsgq_t * msgq = astrid_msgq_produce(&instrument->session, qname, MAINMQ_UDP_PRODUCER);
    if(msgq == NULL) {
        syslog(LOG_ERR, "%s UDP listener: Could not get producer handle for message queue\n",
               instrument->name);
        close(sockfd);
        return NULL;
    }

    syslog(LOG_INFO, "%s UDP listener: Listening on port %d\n",
           instrument->name, instrument->udp_port);

    while (instrument->is_running) {
        memset(buffer, 0, sizeof(buffer));

        // Receive UDP message
        n = recvfrom(sockfd, buffer, sizeof(buffer) - 1, 0,
                     (struct sockaddr *)&cliaddr, &clilen);

        if (n < 0) {
            if (errno == EINTR) continue; // Interrupted, retry
            syslog(LOG_ERR, "%s UDP listener: recvfrom error: %s\n",
                   instrument->name, strerror(errno));
            continue;
        }

        buffer[n] = '\0'; // Ensure null termination

        // Log received message
        char client_ip[INET_ADDRSTRLEN];
        inet_ntop(AF_INET, &cliaddr.sin_addr, client_ip, sizeof(client_ip));

        // Initialize message structure
        init_instrument_message(&msg, instrument->name);

        // Parse the message using the external cmdline parser
        if (parse_message_from_external_cmdline(buffer, &msg) < 0) {
            syslog(LOG_ERR, "%s UDP listener: Could not parse message: %s\n",
                   instrument->name, buffer);
            continue;
        }
        if(astrid_msgq_write(msgq, &msg) < 0) {
            syslog(LOG_ERR, "Could not send play message...\n");
            continue;
        }
    }

    astrid_msgq_close(msgq);

    close(sockfd);
    syslog(LOG_INFO, "%s UDP listener thread exiting\n", instrument->name);
    return NULL;
}

void * instrument_midi_listener_thread(void * arg) {
    snd_seq_t * seq_handle;
    snd_seq_event_t * event = NULL;
    int ret, port, i;

    lpinstrument_t * instrument = (lpinstrument_t *)arg;

    syslog(LOG_ERR, "%s MIDI listener thread starting for %d devices\n", instrument->name, instrument->num_midiin_devices);

    if((ret = snd_seq_open(&seq_handle, "default", SND_SEQ_OPEN_INPUT, 0)) < 0) {
        syslog(LOG_ERR, "%s midi listener: Could not open ALSA seq. Error: (%d) %s\n", instrument->name, ret, snd_strerror(ret));
        return 0;
    }

    snd_seq_set_client_name(seq_handle, "astrid_midi_listener");

    if((port = snd_seq_create_simple_port(seq_handle, "input", 
                    SND_SEQ_PORT_CAP_WRITE | SND_SEQ_PORT_CAP_SUBS_WRITE, 
                    SND_SEQ_PORT_TYPE_APPLICATION)) < 0) {
        syslog(LOG_ERR, "%s midi listener: Could not create ALSA port. Error: (%d) %s\n", instrument->name, port, snd_strerror(port));
        snd_seq_close(seq_handle);
        return 0;
    }

    // Connect to all MIDI input devices
    for(i = 0; i < instrument->num_midiin_devices; i++) {
        if((ret = snd_seq_connect_from(seq_handle, port, instrument->midiin_device_ids[i], 0)) < 0) {
            syslog(LOG_ERR, "%s midi listener: Could not connect to ALSA port device %d. Error: (%d) %s\n", 
                   instrument->name, instrument->midiin_device_ids[i], ret, snd_strerror(ret));
            // Continue trying other devices even if one fails
        } else {
            syslog(LOG_INFO, "%s midi listener: Connected to device %d (index %d)\n", 
                   instrument->name, instrument->midiin_device_ids[i], i);
        }
    }

    // Get producer handle for the internal message queue
    char qname[LPMAXKEY] = {0};
    snprintf(qname, LPMAXKEY, "%s-msgq", instrument->name);
    lpmsgq_t * msgq = astrid_msgq_produce(&instrument->session, qname, MAINMQ_MIDI_PRODUCER);
    if(msgq == NULL) {
        syslog(LOG_ERR, "%s midi listener: Could not get producer handle for message queue\n", instrument->name);
        snd_seq_close(seq_handle);
        return 0;
    }

    while(instrument->is_running) {
        snd_seq_event_input(seq_handle, &event);

        if(event == NULL) {
            syslog(LOG_ERR, "%s midi listener: Got NULL event, skipping...\n", instrument->name);
            usleep((useconds_t)10000);
            continue;
        }

        // Get the source device ID from the event
        int source_device_id = event->source.client;

        // Find the device index for this device ID
        int device_index = -1;
        for(i = 0; i < instrument->num_midiin_devices; i++) {
            if(instrument->midiin_device_ids[i] == source_device_id) {
                device_index = i;
                break;
            }
        }

        // If we can't find the device index, skip this event
        if(device_index < 0) {
            syslog(LOG_WARNING, "%s midi listener: Could not find device index for device ID %d, skipping event\n",
                   instrument->name, source_device_id);
            continue;
        }

        switch(event->type) {
            case SND_SEQ_EVENT_NOTEON:
                // Relay to instrument for processing first
                lpmidi_relay_to_instrument(instrument, msgq, (unsigned char)device_index, NOTE_ON, event->data.note.note, event->data.note.velocity);
                // Then store note velocity in session
                lpmidi_setnote(&instrument->session, source_device_id, event->data.note.channel, event->data.note.note, event->data.note.velocity);
                break;
            case SND_SEQ_EVENT_NOTEOFF:
                // Relay to instrument for processing first
                lpmidi_relay_to_instrument(instrument, msgq, (unsigned char)device_index, NOTE_OFF, event->data.note.note, event->data.note.velocity);
                // Then store note off (velocity 0) in session
                lpmidi_setnote(&instrument->session, source_device_id, event->data.note.channel, event->data.note.note, 0);
                break;
            case SND_SEQ_EVENT_CONTROLLER:
                // Relay to instrument for processing first
                lpmidi_relay_to_instrument(instrument, msgq, (unsigned char)device_index, CONTROL_CHANGE, event->data.control.param, event->data.control.value);
                // Then store CC value in session
                lpmidi_setcc(&instrument->session, source_device_id, event->data.control.channel, event->data.control.param, event->data.control.value);
                break;
            default:
                break;
        }
    }

    astrid_msgq_close(msgq);
    snd_seq_close(seq_handle);
    return 0;
}

void * instrument_midi_output_thread(void * arg) {
    lpmsg_t msg = {0};
    snd_seq_t * seq_handle;
    snd_midi_event_t * midi_event;
    snd_seq_event_t event;
    int ret, port, device_id, device_index;

    lpmidiout_device_t * device_info = (lpmidiout_device_t *)arg;
    lpinstrument_t * instrument = device_info->instrument;
    device_id = device_info->device_id;
    device_index = device_info->device_index;

    syslog(LOG_INFO, "%s MIDI output thread starting for device ID %d with index %d\n", instrument->name, device_id, device_index);

    if((ret = snd_seq_open(&seq_handle, "default", SND_SEQ_OPEN_OUTPUT, 0)) < 0) {
        syslog(LOG_ERR, "%s midi output: Could not open ALSA seq. Error: (%d) %s\n", instrument->name, ret, snd_strerror(ret));
        return 0;
    }

    snd_seq_set_client_name(seq_handle, &instrument->midiout_device_names[device_index][1]);

    if((port = snd_seq_create_simple_port(seq_handle, "output", 
                    SND_SEQ_PORT_CAP_READ | SND_SEQ_PORT_CAP_SUBS_READ, 
                    SND_SEQ_PORT_TYPE_APPLICATION)) < 0) {
        syslog(LOG_ERR, "%s midi output: Could not create ALSA port. Error: (%d) %s\n", instrument->name, port, snd_strerror(port));
        snd_seq_close(seq_handle);
        return 0;
    }

    if((ret = snd_seq_connect_to(seq_handle, port, device_id, 0)) < 0) {
        syslog(LOG_ERR, "%s midi output: Could not connect to ALSA port. Error: (%d) %s\n", instrument->name, ret, snd_strerror(ret));
        snd_seq_close(seq_handle);
        return 0;
    }

    if(snd_midi_event_new(256, &midi_event) < 0) {
        syslog(LOG_ERR, "%s midi output: Could not create MIDI event. Error: (%d) %s\n", instrument->name, ret, snd_strerror(ret));
        snd_seq_close(seq_handle);
        return 0;
    }

    snd_midi_event_init(midi_event);

    // Get consumer handle for this device's MIDI output queue
    char qname[LPMAXKEY] = {0};
    snprintf(qname, LPMAXKEY, "%s-midiq-%d", instrument->name, device_index);
    lpmsgq_t * msgq = astrid_msgq_consume(&instrument->session, qname);
    if(msgq == NULL) {
        syslog(LOG_ERR, "%s midi output: Could not get consumer handle for MIDI queue %d\n", instrument->name, device_index);
        snd_midi_event_free(midi_event);
        snd_seq_close(seq_handle);
        return 0;
    }

    while(instrument->is_running) {
        snd_seq_ev_clear(&event);

        if(astrid_msgq_read(msgq, &msg) < 0) {
            syslog(LOG_ERR, "%s midi output: Could not read message from midi output q. Error: (%d) %s\n", instrument->name, errno, strerror(errno));
            usleep((useconds_t)10000);
            continue;
        }

        if(snd_midi_event_encode(midi_event, (unsigned char *)msg.msg, sizeof(unsigned char) * 3, &event) < 0) {
            syslog(LOG_ERR, "%s midi output: Could not decode MIDI message from lpmsg_t body. Error: (%d) %s\n", instrument->name, errno, strerror(errno));
            usleep((useconds_t)10000);
            continue;
        }

        snd_seq_ev_set_source(&event, port);
        snd_seq_ev_set_subs(&event);
        snd_seq_ev_set_direct(&event);

        if((ret = snd_seq_event_output_direct(seq_handle, &event)) < 0) {
            syslog(LOG_ERR, "%s midi output: Could not send midi event. Error: (%d) %s\n", instrument->name, errno, strerror(errno));
            usleep((useconds_t)10000);
            continue;
        }
    }

    astrid_msgq_close(msgq);
    snd_seq_close(seq_handle);

    return 0;
}

int lpmidi_relay_to_instrument(lpinstrument_t * instrument, lpmsgq_t * msgq, unsigned char device_index, unsigned char mtype, unsigned char mid, unsigned char mval) {
    lpmsg_t msg = {0};
    size_t offset = 0;
    // Add 1 to all values to avoid null bytes (which get treated as string terminators)
    // FIXME -- add a payload length field to the msg struct, and treat the msg field 
    // as a buffer with the given length in those cases
    unsigned char encoded_device_index = device_index + 1;
    unsigned char encoded_mtype = mtype + 1;
    unsigned char encoded_mid = mid + 1;
    unsigned char encoded_mval = mval + 1;

    init_instrument_message(&msg, instrument->name);
    msg.type = LPMSG_MIDI_FROM_DEVICE;

    memcpy(msg.msg, &encoded_device_index, sizeof(unsigned char));
    offset += sizeof(unsigned char);

    memcpy(msg.msg+offset, &encoded_mtype, sizeof(unsigned char));
    offset += sizeof(unsigned char);

    memcpy(msg.msg+offset, &encoded_mid, sizeof(unsigned char));
    offset += sizeof(unsigned char);

    memcpy(msg.msg+offset, &encoded_mval, sizeof(unsigned char));

    if(astrid_msgq_write(msgq, &msg) < 0) {
        syslog(LOG_ERR, "lpmidi_relay_to_instrument: Could not write to internal message queue\n");
        return -1;
    }

    return 0;
}

int lpmidi_decode_eventbytes(char * payload,
        unsigned char * device_index,
        unsigned char * mtype,
        unsigned char * mid,
        unsigned char * mval
    ) {
    int offset = 0;

    // Subtract 1 from all values to reverse encoding (avoids null byte issues)
    memcpy(device_index, payload, sizeof(unsigned char));
    *device_index = *device_index - 1;
    offset += sizeof(unsigned char);

    memcpy(mtype, payload+offset, sizeof(unsigned char));
    *mtype = *mtype - 1;
    offset += sizeof(unsigned char);

    memcpy(mid, payload+offset, sizeof(unsigned char));
    *mid = *mid - 1;
    offset += sizeof(unsigned char);

    memcpy(mval, payload+offset, sizeof(unsigned char));
    *mval = *mval - 1;

    return 0;
}

int lpmidi_encode_eventbytes(unsigned char buf[3], int channel, unsigned char message_type, int param, int value) {
    unsigned char status=0;
    memset(buf, 0, sizeof(unsigned char) * 3);

    status = message_type | (channel & 0x0F);

    buf[0] = status;
    buf[1] = param;
    buf[2] = value;

    return 0;
}

int lpmidi_encode_msg(lpmsg_t * msg, int channel, unsigned char message_type, int param, int value) {
    memset(msg->msg, 0, LPMAXMSG);
    lpmidi_encode_eventbytes((unsigned char *)msg->msg, channel, message_type, param, value);
    return 0;
}

int lpmidi_get_device_id_by_name(const char * device_name) {
    snd_seq_t * seq_handle;
    snd_seq_client_info_t * cinfo;
    snd_seq_port_info_t * pinfo;
    const char * port_name;
    int err, client_id = -1;

    if((err = snd_seq_open(&seq_handle, "default", SND_SEQ_OPEN_DUPLEX, 0)) < 0) {
        syslog(LOG_ERR, "Error opening ALSA sequencer: %s\n", snd_strerror(err));
        return -1;
    }

    snd_seq_client_info_alloca(&cinfo);
    snd_seq_port_info_alloca(&pinfo);

    snd_seq_client_info_set_client(cinfo, -1);

    while(snd_seq_query_next_client(seq_handle, cinfo) >= 0) {
        int client = snd_seq_client_info_get_client(cinfo);
        snd_seq_port_info_set_client(pinfo, client);
        snd_seq_port_info_set_port(pinfo, -1);

        while(snd_seq_query_next_port(seq_handle, pinfo) >= 0) {
            port_name = snd_seq_port_info_get_name(pinfo);
            if(strstr(port_name, device_name) != NULL) {
                client_id = snd_seq_port_info_get_client(pinfo);
                snd_seq_close(seq_handle);
                return client_id;
            }
        }
    }

    snd_seq_close(seq_handle);
    return -1;
}


/* BUFFER
 * SERIALIZATION
 * *************/
unsigned char * serialize_buffer(lpbuffer_t * buf, lpmsg_t * msg, size_t * strsize) {
    size_t audiosize, offset;
    unsigned char * str;

    audiosize = buf->length * buf->channels * sizeof(lpfloat_t);

    *strsize =  0;
    *strsize += sizeof(size_t);  /* audio size in bytes */
    *strsize += sizeof(size_t);  /* audio length in frames */
    *strsize += sizeof(int);     /* channels   */
    *strsize += sizeof(int);     /* samplerate */
    *strsize += sizeof(int);     /* is_looping */
    *strsize += sizeof(size_t);  /* onset      */
    *strsize += audiosize;       /* audio data */
    *strsize += sizeof(lpmsg_t); /* message */

    /* initialize string buffer */
    str = (unsigned char *)calloc(1, *strsize);

    offset = 0;

    memcpy(str + offset, &audiosize, sizeof(size_t));
    offset += sizeof(size_t);

    memcpy(str + offset, &buf->length, sizeof(size_t));
    offset += sizeof(size_t);

    memcpy(str + offset, &buf->channels, sizeof(int));
    offset += sizeof(int);

    memcpy(str + offset, &buf->samplerate, sizeof(int));
    offset += sizeof(int);

    memcpy(str + offset, &buf->is_looping, sizeof(int));
    offset += sizeof(int);

    memcpy(str + offset, &buf->onset, sizeof(size_t));
    offset += sizeof(size_t);

    memcpy(str + offset, buf->data, audiosize);
    offset += audiosize;

    memcpy(str + offset, msg, sizeof(lpmsg_t));
    offset += sizeof(lpmsg_t);

    return str;
}

/* MESSAGE
 * QUEUES
 * ******/
int init_instrument_message(lpmsg_t * msg, char * instrument_name) {
    memset(msg, 0, sizeof(lpmsg_t));
    strncpy(msg->instrument_name, instrument_name, LPMAXNAME-1);
    return 0; 
}

lpmsg_t * create_instrument_message(char * instrument_name) {
    lpmsg_t * msg;
    msg = LPMemoryPool.alloc(1, sizeof(lpmsg_t));
    if(init_instrument_message(msg, instrument_name) < 0) {
        return NULL;
    }

    return msg;
}

int send_udp_message(char * cmd) {
    int sockfd, port, ret;
    struct sockaddr_in servaddr;
    struct addrinfo hints, * res;
    char * msg, * space, * colon, 
         * host_port, * host, * port_str;
    char cmd_copy[50] = {0};

    //syslog(LOG_ERR, "send_udp_message: cmd=%s\n", cmd);
    strncpy(cmd_copy, cmd, sizeof(cmd_copy)-1);

    // split host:port and msg
    space = strchr(cmd_copy, ' ');
    if(space == NULL) {
        syslog(LOG_ERR, "send_udp_message: Invalid host param\n");
        return -1;
    }
    *space = '\0';
    host_port = cmd_copy;
    msg = space + 1; 

    // split host & port
    colon = strchr(host_port, ':');
    if(colon == NULL) {
        syslog(LOG_ERR, "send_udp_message: Invalid host param\n");
        return -1;
    }
    *colon = '\0';
    host = host_port;
    port_str = colon + 1;

    // validate port
    port = atoi(port_str);
    if(port <= 0 || port > 65535) {
        syslog(LOG_ERR, "send_udp_message: Invalid port value\n");
        return -1;
    }

    //syslog(LOG_ERR, "send_udp_message: host=%s port=%d msg=%s\n", host, port, msg);
    //syslog(LOG_ERR, "send_udp_message: create socket host=%s port=%d\n", host, port);

    // Create UDP socket
    if ((sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
        syslog(LOG_ERR, "send_udp_message: socket creation failed. (%d) %s\n", errno, strerror(errno));
        return -1;
    }

    // Get server IP address & prepare hints for getaddrinfo
    memset(&hints, 0, sizeof(hints));
    hints.ai_family = AF_INET;        // IPv4
    hints.ai_socktype = SOCK_DGRAM;   // UDP

    //syslog(LOG_ERR, "send_udp_message: get server ip addr host=%s port=%d\n", host, port);
    if ((ret = getaddrinfo(host, port_str, &hints, &res)) != 0) {
        syslog(LOG_ERR, "send_udp_message: getaddrinfo could not resolve host %s. (%d) %s\n", host, ret, gai_strerror(ret));
        close(sockfd);
        return -1;
    }

    memcpy(&servaddr, res->ai_addr, res->ai_addrlen);
    servaddr.sin_port = htons(port);

    char ip_str[INET_ADDRSTRLEN];
    inet_ntop(AF_INET, &((struct sockaddr_in *)res->ai_addr)->sin_addr, ip_str, sizeof(ip_str));
    //syslog(LOG_ERR, "Resolved IP: %s, Port: %d\n", ip_str, ntohs(servaddr.sin_port));

    // Send the message
    //syslog(LOG_ERR, "send_udp_message: send message! host=%s port=%d\n", host, port);
    int n = sendto(sockfd, msg, strlen(msg), 0, (struct sockaddr *)&servaddr, sizeof(servaddr));
    if (n < 0) {
        syslog(LOG_ERR, "send_udp_message: sendto failed. (%d) %s\n", errno, strerror(errno));
        freeaddrinfo(res);
        close(sockfd);
        return -1;
    }

    //syslog(LOG_ERR, "send_udp_message: done!\n");
    freeaddrinfo(res);
    close(sockfd);
    return 0;
}

int send_play_message(lpmsg_t msg) {
    mqd_t mqd;
    char qname[NAME_MAX] = {0};
    struct mq_attr attr;

    attr.mq_maxmsg = ASTRID_MQ_MAXMSG;
    attr.mq_msgsize = sizeof(lpmsg_t);

    snprintf(qname, NAME_MAX, "/%s-msgq", msg.instrument_name);

    //syslog(LOG_ERR, "sending message to %s\n", qname);

    if((mqd = mq_open(qname, O_CREAT | O_WRONLY | O_NONBLOCK, LPIPC_PERMS, &attr)) == (mqd_t) -1) {
        syslog(LOG_ERR, "send_play_message: mq_open could not open message queue. (%d) %s\n", errno, strerror(errno));
        return -1;
    }

    if(mq_send(mqd, (char *)(&msg), sizeof(lpmsg_t), 0) < 0) {
        if(errno == EAGAIN) {
            syslog(LOG_WARNING, "BUFFER_MSG_DROPPED: queue full, code=%s type=%d (%d) %s\n", msg.msg, msg.type, errno, strerror(errno));
            return 0;
        }
        syslog(LOG_ERR, "send_play_message: mq_send could not send message. (%d) %s\n", errno, strerror(errno));
        return -1;
    }

    if(mq_close(mqd) == -1) {
        syslog(LOG_ERR, "send_play_message: mq_close could not close message queue. (%d) %s\n", errno, strerror(errno));
        return -1; 
    }

    //syslog(LOG_ERR, "message sent to %s\n", qname);
    return 0;
}

int astrid_msgq_init(astrid_session_t * session, char * qname) {
    astrid_shared_resource_t ringbuf_resource;
    size_t ringbuf_size = 0, worker_size = 0;
    char ringbuf_name[LPMAXKEY] = {0};
    char buffer_name[LPMAXKEY] = {0};
    ringbuf_t * rb = NULL;

    // Get the size needed for the ringbuf_t structure (includes workers array)
    ringbuf_get_sizes(ASTRID_MAX_MSGQ_WORKERS, &ringbuf_size, &worker_size);

    // Create names for the two shared resources
    snprintf(ringbuf_name, LPMAXKEY, "%s-rb", qname);
    snprintf(buffer_name, LPMAXKEY, "%s-buf", qname);

    // Register and create the ringbuf_t shared memory
    if(astrid_session_register_shared_resource(session, ringbuf_name, NULL, ASTRID_TYPE_BYTES, ringbuf_size) < 0) {
        syslog(LOG_ERR, "astrid_msgq_init: Could not register ringbuf shared memory\n");
        return -1;
    }

    // Register and create the data buffer shared memory
    if(astrid_session_register_shared_resource(session, buffer_name, NULL, ASTRID_TYPE_BYTES, ASTRID_MSGQ_BUFSIZE) < 0) {
        syslog(LOG_ERR, "astrid_msgq_init: Could not register buffer shared memory\n");
        return -1;
    }

    // Acquire the ringbuf resource to initialize it
    if(astrid_session_aquire_shared_resource(session, &ringbuf_resource, ringbuf_name) < 0) {
        syslog(LOG_ERR, "astrid_msgq_init: Could not acquire ringbuf shared memory\n");
        return -1;
    }

    rb = (ringbuf_t *)ringbuf_resource.data;
    if(rb == NULL) {
        syslog(LOG_ERR, "astrid_msgq_init: ringbuf mapping is NULL\n");
        astrid_session_release_resource_lock(&ringbuf_resource);
        return -1;
    }

    // Setup the ringbuf with the size of the data buffer
    if(ringbuf_setup(rb, ASTRID_MAX_MSGQ_WORKERS, ASTRID_MSGQ_BUFSIZE) != 0) {
        syslog(LOG_ERR, "astrid_msgq_init: ringbuf_setup failed\n");
        astrid_session_release_resource_lock(&ringbuf_resource);
        return -1;
    }

    if(astrid_session_release_resource_lock(&ringbuf_resource) < 0) {
        syslog(LOG_ERR, "astrid_msgq_init: Could not release ringbuf resource lock\n");
        return -1;
    }

    return 0;
}

lpmsgq_t * astrid_msgq_produce(astrid_session_t * session, char * qname, int producer_id) {
    lpmsgq_t * q = NULL;
    char ringbuf_name[LPMAXKEY] = {0};
    char buffer_name[LPMAXKEY] = {0};

    // Allocate the lpmsgq_t structure
    q = (lpmsgq_t *)LPMemoryPool.alloc(1, sizeof(lpmsgq_t));
    if(q == NULL) {
        syslog(LOG_ERR, "astrid_msgq_produce: Could not allocate lpmsgq_t\n");
        return NULL;
    }

    snprintf(q->name, LPMAXKEY, "%s", qname);
    snprintf(ringbuf_name, LPMAXKEY, "%s-rb", qname);
    snprintf(buffer_name, LPMAXKEY, "%s-buf", qname);

    // Map the ringbuf resource
    if(astrid_session_aquire_shared_resource(session, &q->ringbuf_r, ringbuf_name) < 0) {
        syslog(LOG_ERR, "astrid_msgq_produce: Could not acquire ringbuf shared memory\n");
        LPMemoryPool.free(q);
        return NULL;
    }

    q->rb = (ringbuf_t *)q->ringbuf_r.data;
    if(q->rb == NULL) {
        syslog(LOG_ERR, "astrid_msgq_produce: ringbuf mapping is NULL\n");
        astrid_session_release_resource_lock(&q->ringbuf_r);
        LPMemoryPool.free(q);
        return NULL;
    }
    syslog(LOG_INFO, "astrid_msgq_produce: %s mapped ringbuf at %p, size %zu\n", qname, (void*)q->rb, q->ringbuf_r.header.size);

    if(astrid_session_release_resource_lock(&q->ringbuf_r) < 0) {
        syslog(LOG_ERR, "astrid_msgq_produce: Could not release ringbuf lock\n");
        LPMemoryPool.free(q);
        return NULL;
    }

    // Map the buffer resource
    if(astrid_session_aquire_shared_resource(session, &q->buffer_r, buffer_name) < 0) {
        syslog(LOG_ERR, "astrid_msgq_produce: Could not acquire buffer shared memory\n");
        LPMemoryPool.free(q);
        return NULL;
    }

    q->buffer = (lpmsg_t *)q->buffer_r.data;
    if(q->buffer == NULL) {
        syslog(LOG_ERR, "astrid_msgq_produce: buffer mapping is NULL\n");
        astrid_session_release_resource_lock(&q->buffer_r);
        LPMemoryPool.free(q);
        return NULL;
    }
    syslog(LOG_INFO, "astrid_msgq_produce: %s mapped buffer at %p, size %zu\n", qname, (void*)q->buffer, q->buffer_r.header.size);

    if(astrid_session_release_resource_lock(&q->buffer_r) < 0) {
        syslog(LOG_ERR, "astrid_msgq_produce: Could not release buffer lock\n");
        LPMemoryPool.free(q);
        return NULL;
    }

    // Register as a worker
    q->is_producer = 1;
    q->w = ringbuf_register(q->rb, producer_id);
    if(q->w == NULL) {
        syslog(LOG_ERR, "astrid_msgq_produce: ringbuf_register failed for producer %u\n", producer_id);
        LPMemoryPool.free(q);
        return NULL;
    }

    return q;
}

lpmsgq_t * astrid_msgq_consume(astrid_session_t * session, char * qname) {
    lpmsgq_t * q = NULL;
    char ringbuf_name[LPMAXKEY] = {0};
    char buffer_name[LPMAXKEY] = {0};

    // Allocate the lpmsgq_t structure
    q = (lpmsgq_t *)LPMemoryPool.alloc(1, sizeof(lpmsgq_t));
    if(q == NULL) {
        syslog(LOG_ERR, "astrid_msgq_consume: Could not allocate lpmsgq_t\n");
        return NULL;
    }

    snprintf(q->name, LPMAXKEY, "%s", qname);
    snprintf(ringbuf_name, LPMAXKEY, "%s-rb", qname);
    snprintf(buffer_name, LPMAXKEY, "%s-buf", qname);

    // Map the ringbuf resource
    if(astrid_session_aquire_shared_resource(session, &q->ringbuf_r, ringbuf_name) < 0) {
        syslog(LOG_ERR, "astrid_msgq_consume: Could not acquire ringbuf shared memory\n");
        LPMemoryPool.free(q);
        return NULL;
    }

    q->rb = (ringbuf_t *)q->ringbuf_r.data;
    if(q->rb == NULL) {
        syslog(LOG_ERR, "astrid_msgq_consume: ringbuf mapping is NULL\n");
        astrid_session_release_resource_lock(&q->ringbuf_r);
        LPMemoryPool.free(q);
        return NULL;
    }

    if(astrid_session_release_resource_lock(&q->ringbuf_r) < 0) {
        syslog(LOG_ERR, "astrid_msgq_consume: Could not release ringbuf lock\n");
        LPMemoryPool.free(q);
        return NULL;
    }

    // Map the buffer resource
    if(astrid_session_aquire_shared_resource(session, &q->buffer_r, buffer_name) < 0) {
        syslog(LOG_ERR, "astrid_msgq_consume: Could not acquire buffer shared memory\n");
        LPMemoryPool.free(q);
        return NULL;
    }

    q->buffer = (lpmsg_t *)q->buffer_r.data;
    if(q->buffer == NULL) {
        syslog(LOG_ERR, "astrid_msgq_consume: buffer mapping is NULL\n");
        astrid_session_release_resource_lock(&q->buffer_r);
        LPMemoryPool.free(q);
        return NULL;
    }

    if(astrid_session_release_resource_lock(&q->buffer_r) < 0) {
        syslog(LOG_ERR, "astrid_msgq_consume: Could not release buffer lock\n");
        LPMemoryPool.free(q);
        return NULL;
    }

    // Consumers don't register as workers
    q->is_producer = 0;
    q->w = NULL;

    return q;
}

void astrid_msgq_close(lpmsgq_t * q) {
    if(q->is_producer) ringbuf_unregister(q->rb, q->w);
    // Note: Shared resources (buffer_r, buffer) are cleaned up when session closes
    LPMemoryPool.free(q);
}

int astrid_msgq_read(lpmsgq_t * q, lpmsg_t * msg) {
    size_t offset, nbytes;

    if(q->rb == NULL) {
        syslog(LOG_ERR, "astrid_msgq_read: ringbuf not mapped\n");
        return -1;
    }

    if(q->buffer == NULL) {
        syslog(LOG_ERR, "astrid_msgq_read: buffer not mapped\n");
        return -1;
    }

    // Consume returns the number of bytes available and sets offset
    nbytes = ringbuf_consume(q->rb, &offset);
    if(nbytes == 0) {
        return 1; // No messages available
    }

    if(nbytes < sizeof(lpmsg_t)) {
        syslog(LOG_WARNING, "astrid_msgq_read: partial message (%zu bytes)\n", nbytes);
        ringbuf_release(q->rb, nbytes);
        return -1;
    }

    // Copy the message from the buffer at the given offset
    memcpy(msg, ((unsigned char *)q->buffer) + offset, sizeof(lpmsg_t));

    // Release the consumed bytes
    ringbuf_release(q->rb, sizeof(lpmsg_t));

    return 0;
}

int astrid_msgq_write(lpmsgq_t * q, lpmsg_t * msg) {
    ssize_t offset;

    if(q->rb == NULL) {
        syslog(LOG_ERR, "astrid_msgq_write: ringbuf not mapped\n");
        return -1;
    }

    if(q->buffer == NULL) {
        syslog(LOG_ERR, "astrid_msgq_write: buffer not mapped\n");
        return -1;
    }

    if(q->w == NULL) {
        syslog(LOG_ERR, "astrid_msgq_write: worker not registered\n");
        return -1;
    }

    // Acquire space in the ringbuf
    offset = ringbuf_acquire(q->rb, q->w, sizeof(lpmsg_t));
    if(offset == -1) {
        syslog(LOG_WARNING, "astrid_msgq_write: queue full, dropped message (type=%d)\n", msg->type);
        return -1;
    }

    // Copy the message to the buffer at the given offset
    memcpy(((unsigned char *)q->buffer) + offset, msg, sizeof(lpmsg_t));

    // Mark the space as produced
    ringbuf_produce(q->rb, q->w);

    return 0;
}

int send_posix_message(char * qname, lpmsg_t msg) {
    mqd_t mqd;
    struct mq_attr attr;

    attr.mq_maxmsg = ASTRID_MQ_MAXMSG;
    attr.mq_msgsize = sizeof(lpmsg_t);

    if((mqd = mq_open(qname, O_CREAT | O_WRONLY | O_NONBLOCK, LPIPC_PERMS, &attr)) == (mqd_t) -1) {
        syslog(LOG_ERR, "send_posix_message mq_open: Error opening message queue. Error: %s\n", strerror(errno));
        return -1;
    }

    if(mq_send(mqd, (char *)(&msg), sizeof(lpmsg_t), 0) < 0) {
        if(errno == EAGAIN) {
            syslog(LOG_INFO, "send_posix_message: dropped message. (%d) %s\n", errno, strerror(errno));
            return 0;
        }
        syslog(LOG_ERR, "send_posix_message mq_send: Error allocing during message write. Error: %s\n", strerror(errno));
        return -1;
    }

    if(mq_close(mqd) == -1) {
        syslog(LOG_ERR, "send_posix_message close: Error closing message relay queue. Error: %s\n", strerror(errno));
        return -1; 
    }

    return 0;
}

mqd_t astrid_posix_msgq_open_read(char * qname) {
    mqd_t mqd;
    struct mq_attr attr;

    attr.mq_maxmsg = ASTRID_MQ_MAXMSG;
    attr.mq_msgsize = sizeof(lpmsg_t);

    if((mqd = mq_open(qname, O_CREAT | O_RDONLY, LPIPC_PERMS, &attr)) == (mqd_t) -1) {
        syslog(LOG_ERR, "astrid_msgq_open (%s) mq_open: Error opening message queue. Error: %s\n", qname, strerror(errno));
        return (mqd_t) -1;
    }

    return mqd;
}


mqd_t astrid_posix_msgq_open(char * qname) {
    mqd_t mqd;
    struct mq_attr attr;

    attr.mq_maxmsg = ASTRID_MQ_MAXMSG;
    attr.mq_msgsize = sizeof(lpmsg_t);

    if((mqd = mq_open(qname, O_CREAT | O_RDWR | O_NONBLOCK, LPIPC_PERMS, &attr)) == (mqd_t) -1) {
        syslog(LOG_ERR, "astrid_msgq_open (%s) mq_open: Error opening message queue. Error: %s\n", qname, strerror(errno));
        return (mqd_t) -1;
    }

    return mqd;
}

int astrid_posix_msgq_close(mqd_t mqd) {
    if(mq_close(mqd) == -1) {
        syslog(LOG_ERR, "astrid_msgq_close close: Error closing msg queue FIFO. Error: %s\n", strerror(errno));
        return -1; 
    }

    return 0;
}

int astrid_posix_msgq_unlink(char * qname) {
    return mq_unlink(qname);
}

int astrid_posix_msgq_read(mqd_t mqd, lpmsg_t * msg) {
    char * msgp;
    ssize_t read_result;
    unsigned int msg_priority = 0;

    msgp = (char *)msg;

    if((read_result = mq_receive(mqd, msgp, sizeof(lpmsg_t), &msg_priority)) < 0) {
        syslog(LOG_ERR, "astrid_msgq_read mq_receive: Error reading message. (Got %ld bytes) Error: %s\n", read_result, strerror(errno));
        return -1;
    }

    return 0;
}

void * instrument_audio_slow_lane_thread(void * arg) {
    lpinstrument_t * instrument = (lpinstrument_t *)arg;

    astrid_shared_resource_t resource;
    lpbuffer_t * buf;
    lpfloat_t sample;
    size_t read_pos, write_pos, i;
    int c;

    size_t blocksize = 512;
    size_t blocksize_in_bytes = sizeof(float) * blocksize * instrument->input_channels;
    size_t available_read_space = 0;
    lpbuffer_t * block = LPBuffer.create(blocksize, instrument->input_channels, instrument->samplerate);

    if(astrid_session_aquire_shared_resource(&instrument->session, &resource, instrument->adcname) < 0) {
        syslog(LOG_ERR, 
                "astrid_instrument_audio_slow_lane_thread: astrid_session_aquire_shared_resource could not aquire ADC shm. (%d) %s\n", errno, strerror(errno));
        return 0;
    }
    if(astrid_session_release_resource_lock(&resource) < 0) {
        syslog(LOG_CRIT, "astrid_instrument_audio_slow_lane_thread: Could not release resource lock.\n");
        return 0;
    }

    while(instrument->is_running) {
        available_read_space = jack_ringbuffer_read_space(instrument->jack_input_ringbuffer);
        if(available_read_space < blocksize_in_bytes) {
            usleep((useconds_t)1000);
            continue;
        }

        // read the float block from the jack ringbuffer into a local lpbuffer block
        // FIXME this could read directly from jack instead of copying...
        if(astrid_jack_ringbuffer_read_block(instrument->jack_input_ringbuffer, block, blocksize) < 0) {
            syslog(LOG_WARNING, "astrid_instrument_audio_slow_lane_thread: could not read input block from jack ADC ringbuffer.\n");
        }

        // aquire a lock on the shared memory ringbuffer
        if(astrid_session_aquire_resource_lock(&resource, instrument->adcname) < 0) {
            syslog(LOG_ERR, "astrid_instrument_audio_slow_lane_thread: Could not aquire resource lock. (name=%s lock_name=%s)\n", instrument->adcname, resource.lock_name);
            usleep((useconds_t)1000);
            continue;
        }
        buf = (lpbuffer_t *)resource.data;

        // write the local block into the shared memory ringbuffer
        for(i=0; i < block->length; i++) {
            for(c=0; c < block->channels; c++) {
                write_pos = safe_index(i, buf->pos, c, buf->length, buf->channels);
                read_pos = safe_index(i, 0, c, block->length, block->channels);
                sample = block->data[read_pos];
                buf->data[write_pos] = sample;
            }
        }

        /* Increment the write position */
        buf->pos = (buf->pos + block->length) % buf->length;

        if(astrid_session_release_resource_lock(&resource) < 0) {
            syslog(LOG_ERR, "astrid_instrument_audio_slow_lane_thread: Could not release resource lock.\n");
            usleep((useconds_t)1000);
            continue;
        }

    }

    if(astrid_session_aquire_resource_lock(&resource, instrument->adcname) < 0) {
        syslog(LOG_ERR, "astrid_instrument_audio_slow_lane_thread: (cleaup on exit) Could not aquire resource lock. (name=%s lock_name=%s)\n", instrument->adcname, resource.lock_name);
        usleep((useconds_t)1000);
        return 0;
    }

    if(astrid_session_release_shared_resource(&instrument->session, &resource, instrument->adcname) < 0) {
        syslog(LOG_ERR, "astrid_instrument_audio_slow_lane_thread: (cleaup on exit)  could not release ADC shm.\n");
        return 0;
    }

    LPBuffer.destroy(block);

    return 0;
}

void * instrument_posix_inbox_thread(void * arg) {
    lpinstrument_t * instrument = (lpinstrument_t *)arg;
    mqd_t mqd;
    char posix_inbox_qname[LPMAXKEY];
    snprintf(posix_inbox_qname, LPMAXKEY, "/%s-posix-inbox", instrument->name);

    if((mqd = astrid_posix_msgq_open_read(posix_inbox_qname)) < 0) {
        syslog(LOG_ERR, "%s renderer: Could not open msgq. Error: (%d) %s\n", instrument->name, errno, strerror(errno));
        return 0;
    }

    // Get a producer handle for the internal message queue
    char qname[LPMAXKEY] = {0};
    snprintf(qname, LPMAXKEY, "%s-msgq", instrument->name);
    lpmsgq_t * msgq = astrid_msgq_produce(&instrument->session, qname, MAINMQ_POSIX_INBOX_PRODUCER);
    if(msgq == NULL) {
        syslog(LOG_ERR, "%s POSIX INBOX: Could not get producer handle for message queue\n", instrument->name);
        astrid_posix_msgq_close(mqd);
        return 0;
    }

    while(instrument->is_running) {
        if(astrid_posix_msgq_read(mqd, &instrument->msg) < 0) {
            syslog(LOG_ERR, "%s POSIX INBOX: Could not read message from playq. Error: (%d) %s\n", instrument->name, errno, strerror(errno));
            usleep((useconds_t)100);
            continue;
        }

        // write to internal q
        if(astrid_msgq_write(msgq, &instrument->msg) < 0) {
            syslog(LOG_ERR, "%s POSIX INBOX: Could not relay message to internal q. Error: (%d) %s\n", instrument->name, errno, strerror(errno));
            usleep((useconds_t)100);
            continue;
        }

        // Exit immediately if shutdown message received
        if(instrument->msg.type == LPMSG_SHUTDOWN) {
            break;
        }
    }

    astrid_msgq_close(msgq);

    return 0;
}

/* INSTRUMENT INBOX (LPMSGQ) */
void * instrument_message_thread(void * arg) {
    lpinstrument_t * instrument = (lpinstrument_t *)arg;
    int is_scheduled=0, ret=0, i=0;
    unsigned char device_index=0, mtype=0, mid=0, mval=0;
    static time_t last_graph_update_time = 0;
    instrument->is_waiting = 1;
    char qname[LPMAXKEY] = {0};

    // Get handles to all the outgoing message Qs we'll write to in this thread 
    // this thread is the only writer for these queues, so the producer ID is always 0
    snprintf(qname, LPMAXKEY, "%s-relayq", instrument->name);
    lpmsgq_t * relay_q = astrid_msgq_produce(&instrument->session, qname, 0);

    snprintf(qname, LPMAXKEY, "%s-gpioq", instrument->name);
    lpmsgq_t * gpio_q = astrid_msgq_produce(&instrument->session, qname, 0);

    lpmsgq_t * midi_qs[ASTRID_MAX_MIDI_DEVICES];
    for(i = 0; i < ASTRID_MAX_MIDI_DEVICES; i++) {
        snprintf(qname, LPMAXKEY, "%s-midiq-%d", instrument->name, i);
        midi_qs[i] = astrid_msgq_produce(&instrument->session, qname, 0);
    }

    // And the main Q for reading...
    snprintf(qname, LPMAXKEY, "%s-msgq", instrument->name);
    lpmsgq_t * mainq = astrid_msgq_consume(&instrument->session, qname);

    if(mainq == NULL) {
        syslog(LOG_ERR, "instrument_message_thread: Could not initialize main message queue, exiting thread\n");
        return NULL;
    }

    while(instrument->is_running) {
        ret = astrid_msgq_read(mainq, &instrument->msg);
        if(ret == 1) {
            // no messages...
            usleep((useconds_t)100);
            continue;
        } else if(ret < 0) {
            syslog(LOG_ERR, "%s message thread: Could not read message from internal playq. Error: (%d) %s\n", instrument->name, errno, strerror(errno));
            usleep((useconds_t)1000);
            continue;
        }

        //syslog(LOG_INFO, "MSG THREAD READ: type=%d\n", instrument->msg.type);

        is_scheduled = ((instrument->msg.flags & LPFLAG_IS_SCHEDULED) == LPFLAG_IS_SCHEDULED);

        if(is_scheduled) {
            // Scheduled messages get sent to the sequencer for handling later
            //syslog(LOG_ERR, "C IS SCHEDULED msg.scheduled %f\n", instrument->msg.scheduled);
            //syslog(LOG_ERR, "C IS SCHEDULED msg.initiated %ld\n", instrument->msg.initiated);
            if(astrid_schedule_message(instrument, instrument->msg) < 0) {
                syslog(LOG_ERR, "%s message thread: Could not read relay message to seq. Error: (%d) %s\n", instrument->name, errno, strerror(errno));
            }
            continue;
        } else {
            // All other messages get relayed externally, too, if the relay is enabled
            if(instrument->ext_relay_enabled) {
                if(astrid_msgq_write(relay_q, &instrument->msg) < 0) {
                    syslog(LOG_ERR, "%s message thread: Could not send message to external q. Error: (%d) %s\n", instrument->name, errno, strerror(errno));
                }
            }
        }

        // Handle shutdown
        if(instrument->msg.type == LPMSG_SHUTDOWN) {
            // send the shutdown message to the seq thread and external relay
            if(astrid_schedule_message(instrument, instrument->msg) < 0) {
                syslog(LOG_ERR, "%s message thread: Could not send shutdown message to seq. Error: (%d) %s\n", instrument->name, errno, strerror(errno));
            }
            if(instrument->ext_relay_enabled) {
                if(astrid_msgq_write(relay_q, &instrument->msg) < 0) {
                    syslog(LOG_ERR, "%s message thread: Could not send shutdown message to external q. Error: (%d) %s\n", instrument->name, errno, strerror(errno));
                }
            }
            instrument->is_running = 0;
            break;
        }

        //syslog(LOG_INFO, "C MSG (%d) %s\n", instrument->msg.type, instrument->msg.msg);

        // Now the fun stuff
        switch(instrument->msg.type) {
            case LPMSG_RENDER_COMPLETE:
                /* Schedule the buffer for playback */
                scheduler_schedule_event(&instrument->session, instrument->async_mixer, instrument->msg.msg, (size_t)(instrument->msg.scheduled * instrument->samplerate));
                break;

            case LPMSG_CONNECT:
                if(astrid_instrument_connect(instrument, instrument->msg.msg) < 0) {
                    syslog(LOG_ERR, "Could not connect %s\n", instrument->msg.msg); 
                }
                break;

            case LPMSG_DISCONNECT:
                if(astrid_instrument_disconnect(instrument, instrument->msg.msg) < 0) {
                    syslog(LOG_ERR, "Could not disconnect %s\n", instrument->msg.msg); 
                }
                break;

            case LPMSG_UPDATE:
                if(astrid_instrument_process_param_updates(instrument) < 0) {
                    syslog(LOG_ERR, "Could not encode update messages...\n");
                    continue;
                }
                break;

            case LPMSG_PLAY:
                // Schedule a C callback render if there's a callback defined
                // python renders will also be triggered at this point if we're 
                // inside a python instrument because of the message relay
                if(instrument->renderer != NULL) {
                    /* FIXME do this in another thread */
                    if(instrument->renderer(instrument) < 0) {
                        syslog(LOG_ERR, "there was a problem rendering from the C instrument\n");
                        continue;
                    }
                }
                break;

            case LPMSG_UDP:
                if(send_udp_message(instrument->msg.msg) < 0) {
                    syslog(LOG_ERR, "Could not send UDP message...\n");
                }
                break;

            case LPMSG_LOAD:
                // it would be interesting to explore live reloading of C modules
                // in the tradition of CLIVE, but at the moment only python handles these
                break;

            case LPMSG_AUTOTRIGGER_REPLACE_ALL:
            case LPMSG_AUTOTRIGGER_UPDATE:
                //astrid_autotrigger_set_from_message(&instrument->session, instrument->msg);
                // FIXME
                break;

            case LPMSG_TRIGGER:
                // maybe support raspberry pi GPIO pin toggling / triggers?
                if(instrument->trigger != NULL) instrument->trigger(instrument);
                break;

            case LPMSG_MIDI_FROM_DEVICE:
                if(instrument->midievent != NULL) {
                    lpmidi_decode_eventbytes(instrument->msg.msg, &device_index, &mtype, &mid, &mval);
                    instrument->midievent(instrument, device_index, mtype, mid, mval);
                }
                break;

            case LPMSG_MIDI_TO_DEVICE:
                // put onto the internal midiout q
                if(instrument->msg.voice_id < ASTRID_MAX_MIDI_DEVICES && midi_qs[instrument->msg.voice_id] != NULL) {
                    if(astrid_msgq_write(midi_qs[instrument->msg.voice_id], &instrument->msg) < 0) {
                        syslog(LOG_ERR, "Could not send midi message... (%d) %s\n", errno, strerror(errno));
                    }
                } else {
                    syslog(LOG_ERR, "Invalid MIDI device ID %ld or worker not registered\n", instrument->msg.voice_id);
                }
                break;

            case LPMSG_GRAPH_UPDATE_READY:
                time_t current_time = time(NULL);
                if (current_time - last_graph_update_time >= 1) {
                    if(astrid_instrument_update_graph(instrument) < 0) {
                        syslog(LOG_ERR, "Could not update ugen graph...\n");
                    } else {
                        last_graph_update_time = current_time;
                    }
                }
                break;

            default:
                // Garbage typed messages will shut 'er down
                syslog(LOG_WARNING, "C MSG: got bad msg type %d\n", instrument->msg.type);
                break;
        }
    }

    // Cleanup all the queues
    if(mainq != NULL) astrid_msgq_close(mainq);
    if(relay_q != NULL) astrid_msgq_close(relay_q);
    if(gpio_q != NULL) astrid_msgq_close(gpio_q);
    for(i = 0; i < ASTRID_MAX_MIDI_DEVICES; i++) {
        if(midi_qs[i] != NULL) astrid_msgq_close(midi_qs[i]);
    }

    instrument->is_waiting = 0;
    return 0;
}

/* UGEN GRAPH PARSING AND UPDATE
 * ******************************/

// Map ugen type string to enum
static int astrid_get_ugen_type(const char* type_str) {
    if (strcmp(type_str, "sine") == 0) return UGEN_SINE;
    if (strcmp(type_str, "adc") == 0) return UGEN_ADC;
    if (strcmp(type_str, "dac") == 0) return UGEN_DAC;
    if (strcmp(type_str, "mix") == 0) return UGEN_MIX;
    if (strcmp(type_str, "tape") == 0) return UGEN_TAPE;
    if (strcmp(type_str, "mult") == 0) return UGEN_MULT;
    if (strcmp(type_str, "pulsar") == 0) return UGEN_PULSAR;
    return -1;  // Unknown type
}

// Map parameter name to enum
static int astrid_get_param_type(const char* param_str) {
    if (strcmp(param_str, "freq") == 0) return UPARAM_FREQ;
    if (strcmp(param_str, "phase") == 0) return UPARAM_PHASE;
    if (strcmp(param_str, "amp") == 0) return UPARAM_AMP;
    if (strcmp(param_str, "gain") == 0) return UPARAM_GAIN;
    if (strcmp(param_str, "speed") == 0) return UPARAM_SPEED;
    if (strcmp(param_str, "channel") == 0) return UPARAM_CHANNEL;
    if (strcmp(param_str, "pulsewidth") == 0) return UPARAM_PULSEWIDTH;
    if (strcmp(param_str, "a") == 0) return UPARAM_A;
    if (strcmp(param_str, "b") == 0) return UPARAM_B;
    if (strcmp(param_str, "in") == 0) return UPARAM_INPUT;
    if (strcmp(param_str, "smoothing") == 0) return UPARAM_SMOOTHING;
    if (strcmp(param_str, "saturation") == 0) return UPARAM_SATURATION;
    if (strcmp(param_str, "samplerate") == 0) return UPARAM_SAMPLERATE;
    return -1;  // Unknown param
}

// Map port name to output index
#if 0
static int astrid_get_output_index(const char* port_str) {
    if (strcmp(port_str, "out") == 0) return 0;  // Main output
    if (strcmp(port_str, "level") == 0) return 1;  // Level output (for ADC)
    if (strcmp(port_str, "freq") == 0) return 1;   // Freq output (for oscillators)
    if (strcmp(port_str, "phase") == 0) return 2;  // Phase output
    if (strcmp(port_str, "peak") == 0) return 2;   // Peak output (for ADC)
    return 0;  // Default to main output
}
#endif

// Structure to hold node info during parsing
typedef struct {
    char name[64];
    ugen_t* ugen;
} astrid_graph_node_t;

// Parse a single node specification: "osc:sine:freq=440,amp=0.5"
// FIXME -- rewrite all this deserialization
static int astrid_parse_graph_node(const char* spec, size_t spec_len, char* node_name, size_t name_size, char* node_type, size_t type_size, 
                                   char param_names[][32], float* param_values, int max_params, int* param_count) {
    *param_count = 0;
    
    const char* first_colon = memchr(spec, ':', spec_len);
    if (!first_colon) {
        // Node with no type or parameters - treat as name only, assume type is same as name
        size_t len = spec_len < name_size - 1 ? spec_len : name_size - 1;
        memcpy(node_name, spec, len);
        node_name[len] = '\0';
        memcpy(node_type, node_name, type_size - 1);
        node_type[type_size - 1] = '\0';
        return 1;
    }
    
    // Extract node name (part before first colon)
    size_t name_len = first_colon - spec;
    if (name_len >= name_size) name_len = name_size - 1;
    memcpy(node_name, spec, name_len);
    node_name[name_len] = '\0';
    
    // Look for second colon to separate type from parameters
    const char* second_colon = memchr(first_colon + 1, ':', spec_len - (first_colon + 1 - spec));
    if (!second_colon) {
        // Format: name:type (no parameters)
        size_t type_len = spec_len - (first_colon + 1 - spec);
        if (type_len >= type_size) type_len = type_size - 1;
        memcpy(node_type, first_colon + 1, type_len);
        node_type[type_len] = '\0';
        return 1;
    }
    
    // Extract node type (between first and second colon)
    size_t type_len = second_colon - (first_colon + 1);
    if (type_len >= type_size) type_len = type_size - 1;
    memcpy(node_type, first_colon + 1, type_len);
    node_type[type_len] = '\0';
    
    // Parse parameters without modifying the string
    const char* params_start = second_colon + 1;
    const char* params_end = spec + spec_len;
    
    const char* current = params_start;
    while (current < params_end && *param_count < max_params) {
        // Find next comma or end
        const char* comma = memchr(current, ',', params_end - current);
        const char* param_end = comma ? comma : params_end;
        
        // Find equals sign
        const char* equals = memchr(current, '=', param_end - current);
        if (equals && equals < param_end) {
            // Extract param name
            size_t param_name_len = equals - current;
            if (param_name_len > 31) param_name_len = 31;
            memcpy(param_names[*param_count], current, param_name_len);
            param_names[*param_count][param_name_len] = '\0';
            
            // Extract param value
            float value = strtof(equals + 1, NULL);
            param_values[(*param_count)++] = value;
        }
        
        current = comma ? comma + 1 : params_end;
    }
    
    return 1;
}

// Parse a single connection: "adc.level->osc.freq*600+600"
static int astrid_parse_graph_connection(const char* spec, size_t spec_len, char* source_node, size_t source_node_size, char* source_port, size_t source_port_size, char* target_node, size_t target_node_size, char* target_port, size_t target_port_size, float* mult, float* add) {
    const char* arrow = memmem(spec, spec_len, "->", 2);
    if (!arrow) return 0;
    
    // Parse source
    size_t source_len = arrow - spec;
    const char* dot = memchr(spec, '.', source_len);
    if (dot) {
        size_t node_len = dot - spec;
        if (node_len >= source_node_size) node_len = source_node_size - 1;
        memcpy(source_node, spec, node_len);
        source_node[node_len] = '\0';
        
        size_t port_len = source_len - (dot - spec) - 1;
        if (port_len >= source_port_size) port_len = source_port_size - 1;
        memcpy(source_port, dot + 1, port_len);
        source_port[port_len] = '\0';
    }
    
    // Parse target with modifiers
    const char* target_start = arrow + 2;
    const char* target_end = spec + spec_len;
    *mult = 1.0f;
    *add = 0.0f;
    
    // Look for multiplication (*) and addition (+)
    const char* mult_pos = memchr(target_start, '*', target_end - target_start);
    const char* add_pos = memchr(target_start, '+', target_end - target_start);
    
    const char* target_node_end = target_start;
    if (mult_pos) {
        target_node_end = mult_pos;
        *mult = strtof(mult_pos + 1, NULL);
        if (add_pos && add_pos > mult_pos) {
            *add = strtof(add_pos + 1, NULL);
        }
    } else if (add_pos) {
        target_node_end = add_pos;
        *add = strtof(add_pos + 1, NULL);
    } else {
        target_node_end = target_end;
    }
    
    // Parse target node.port
    dot = memchr(target_start, '.', target_node_end - target_start);
    if (dot) {
        size_t node_len = dot - target_start;
        if (node_len >= target_node_size) node_len = target_node_size - 1;
        memcpy(target_node, target_start, node_len);
        target_node[node_len] = '\0';
        
        size_t port_len = target_node_end - dot - 1;
        if (port_len >= target_port_size) port_len = target_port_size - 1;
        memcpy(target_port, dot + 1, port_len);
        target_port[port_len] = '\0';
    }
    
    return 1;
}

astrid_graph_t * astrid_graph_create_from_message(lpmsg_t msg, astrid_session_t * session) {
    if (msg.type != LPMSG_GRAPH_UPDATE_READY) {
        syslog(LOG_ERR, "astrid_graph_create_from_message: wrong message type %d\n", msg.type);
        return NULL;
    }
    
    const char* graph_str = msg.msg;
    size_t graph_len = strlen(graph_str);
    
    if (graph_len == 0) {
        syslog(LOG_ERR, "astrid_graph_create_from_message: empty graph string\n");
        return NULL;
    }
    
    // Allocate new graph structure
    astrid_graph_t* graph = (astrid_graph_t*)calloc(1, sizeof(astrid_graph_t));
    if (!graph) {
        syslog(LOG_ERR, "astrid_graph_create_from_message: failed to allocate graph\n");
        return NULL;
    }
    
    // Node lookup table for connection phase
    astrid_graph_node_t node_table[ASTRID_GRAPH_MAX_NODES];
    size_t node_table_count = 0;
    
    const char* pipe = memchr(graph_str, '|', graph_len);
    
    if (pipe) {
        const char* nodes_start = graph_str;
        size_t nodes_len = pipe - graph_str;
        const char* connections_start = pipe + 1;
        size_t connections_len = graph_len - (pipe - graph_str) - 1;
        
        // Parse nodes - find space-separated tokens
        const char* current = nodes_start;
        const char* nodes_end = nodes_start + nodes_len;
        
        while (current < nodes_end && graph->num_nodes < ASTRID_GRAPH_MAX_NODES) {
            // Skip leading spaces
            while (current < nodes_end && *current == ' ') current++;
            if (current >= nodes_end) break;
            
            // Find end of token
            const char* token_end = current;
            while (token_end < nodes_end && *token_end != ' ') token_end++;
            
            if (token_end > current) {
                char node_name[64], node_type[32];
                char param_names[16][32];
                float param_values[16];
                int param_count;
                
                if (astrid_parse_graph_node(current, token_end - current, node_name, sizeof(node_name), node_type, sizeof(node_type), 
                                           param_names, param_values, 16, &param_count)) {
                    
                    // Get ugen type
                    int ugen_type = astrid_get_ugen_type(node_type);
                    if (ugen_type < 0) {
                        syslog(LOG_ERR, "astrid_graph_create_from_message: unknown ugen type %s\n", node_type);
                        current = token_end;
                        continue;
                    }
                    
                    // Create the ugen
                    ugen_t * u = LPUgen.create(msg.instrument_name, node_name, ugen_type);
                    if (u == NULL) {
                        syslog(LOG_ERR, "astrid_graph_create_from_message: failed to create ugen type %s\n", node_type);
                        current = token_end;
                        continue;
                    }

                    // Set parameters from parsed values and write to session
                    for (int i = 0; i < param_count; i++) {
                        int param_type = astrid_get_param_type(param_names[i]);
                        if (param_type >= 0) {
                            LPUgen.set_param(u, param_type, param_values[i]);
                            // Only write to session if the value doesn't already exist (preserves live values)
                            if (u->param_hashes[param_type] != 0) {
                                if (!AstridSession.exists_from_hash(session, u->param_hashes[param_type])) {
                                    AstridSession.set_float_from_hash(session, u->param_hashes[param_type], param_values[i]);
                                }
                            }
                        } else {
                            syslog(LOG_WARNING, "astrid_graph_create_from_message: unknown param %s for node %s\n",
                                   param_names[i], node_name);
                        }
                    }

                    // Write all other param defaults to session (params not explicitly set in graph spec)
                    for (int p = 0; p < NUM_UGEN_PARAMS; p++) {
                        if (u->param_hashes[p] != 0) {
                            // Check if this param was already set above
                            int already_set = 0;
                            for (int i = 0; i < param_count; i++) {
                                int param_type = astrid_get_param_type(param_names[i]);
                                if (param_type == p) {
                                    already_set = 1;
                                    break;
                                }
                            }
                            // Only write default if not already set in graph spec and doesn't exist in session
                            if (!already_set && !AstridSession.exists_from_hash(session, u->param_hashes[p])) {
                                AstridSession.set_float_from_hash(session, u->param_hashes[p], u->param_values[p]);
                            }
                        }
                    }

                    // Add to graph and lookup table
                    graph->nodes[graph->num_nodes++] = u;
                    strncpy(node_table[node_table_count].name, node_name, sizeof(node_table[node_table_count].name));
                    node_table[node_table_count].name[sizeof(node_table[node_table_count].name) - 1] = '\0';
                    node_table[node_table_count].ugen = u;
                    node_table_count++;
                }
            }
            
            current = token_end;
        }
        
        // Parse connections - find space-separated tokens
        current = connections_start;
        const char* connections_end = connections_start + connections_len;
        
        while (current < connections_end) {
            // Skip leading spaces
            while (current < connections_end && *current == ' ') current++;
            if (current >= connections_end) break;
            
            // Find end of token
            const char* token_end = current;
            while (token_end < connections_end && *token_end != ' ') token_end++;
            
            if (token_end > current) {
                char source_node[64], source_port[32], target_node[64], target_port[32];
                float mult, add;
                
                if (astrid_parse_graph_connection(current, token_end - current, source_node, sizeof(source_node), source_port, sizeof(source_port), 
                                                 target_node, sizeof(target_node), target_port, sizeof(target_port), &mult, &add)) {
                    
                    // Find source and target ugens
                    ugen_t* source_ugen = NULL;
                    ugen_t* target_ugen = NULL;
                    
                    for (size_t i = 0; i < node_table_count; i++) {
                        if (strcmp(node_table[i].name, source_node) == 0) {
                            source_ugen = node_table[i].ugen;
                        }
                        if (strcmp(node_table[i].name, target_node) == 0) {
                            target_ugen = node_table[i].ugen;
                        }
                    }
                    
                    // Handle special case for main output
                    if (strcmp(target_node, "main") == 0 && strcmp(target_port, "output") == 0) {
                        // This is the main output connection
                        if (source_ugen) {
                            graph->output_node = source_ugen;
                        }
                    } else if (source_ugen && target_ugen) {
                        // Regular connection
                        int param_type = astrid_get_param_type(target_port);
                        if (param_type >= 0) {
                            // Get the source output index
                            //int output_index = astrid_get_output_index(source_port);
                            
                            // For non-main outputs, we need to use a different approach
                            // Since LPUgen.connect expects a source ugen, not a specific output
                            // We'll connect the whole ugen and let the processing handle output selection
                            LPUgen.connect(target_ugen, param_type, source_ugen, mult, add);
                            
                        } else {
                            syslog(LOG_WARNING, "astrid_graph_create_from_message: unknown param %s for connection to %s\n", 
                                   target_port, target_node);
                        }
                    } else {
                        if (!source_ugen) {
                            syslog(LOG_ERR, "astrid_graph_create_from_message: source node %s not found for connection\n", source_node);
                        }
                        if (!target_ugen) {
                            syslog(LOG_ERR, "astrid_graph_create_from_message: target node %s not found for connection\n", target_node);
                        }
                    }
                }
            }
            
            current = token_end;
        }
    } else {
        // No connections, just parse nodes
        const char* current = graph_str;
        const char* graph_end = graph_str + graph_len;
        
        while (current < graph_end && graph->num_nodes < ASTRID_GRAPH_MAX_NODES) {
            // Skip leading spaces
            while (current < graph_end && *current == ' ') current++;
            if (current >= graph_end) break;
            
            // Find end of token
            const char* token_end = current;
            while (token_end < graph_end && *token_end != ' ') token_end++;
            
            if (token_end > current) {
                char node_name[64], node_type[32];
                char param_names[16][32];
                float param_values[16];
                int param_count;
                
                if (astrid_parse_graph_node(current, token_end - current, node_name, sizeof(node_name), node_type, sizeof(node_type), 
                                           param_names, param_values, 16, &param_count)) {
                    
                    // Get ugen type
                    int ugen_type = astrid_get_ugen_type(node_type);
                    if (ugen_type < 0) {
                        syslog(LOG_ERR, "astrid_graph_create_from_message: unknown ugen type %s\n", node_type);
                        current = token_end;
                        continue;
                    }
                    
                    // Create the ugen
                    ugen_t * u = LPUgen.create(msg.instrument_name, node_name, ugen_type);
                    if (u == NULL) {
                        syslog(LOG_ERR, "astrid_graph_create_from_message: failed to create ugen type %s\n", node_type);
                        current = token_end;
                        continue;
                    }

                    // Set parameters from parsed values and write to session
                    for (int i = 0; i < param_count; i++) {
                        int param_type = astrid_get_param_type(param_names[i]);
                        if (param_type >= 0) {
                            LPUgen.set_param(u, param_type, param_values[i]);
                            // Only write to session if the value doesn't already exist (preserves live values)
                            if (u->param_hashes[param_type] != 0) {
                                if (!AstridSession.exists_from_hash(session, u->param_hashes[param_type])) {
                                    AstridSession.set_float_from_hash(session, u->param_hashes[param_type], param_values[i]);
                                }
                            }
                        } else {
                            syslog(LOG_WARNING, "astrid_graph_create_from_message: unknown param %s for node %s\n",
                                   param_names[i], node_name);
                        }
                    }

                    // Write all other param defaults to session (params not explicitly set in graph spec)
                    for (int p = 0; p < NUM_UGEN_PARAMS; p++) {
                        if (u->param_hashes[p] != 0) {
                            // Check if this param was already set above
                            int already_set = 0;
                            for (int i = 0; i < param_count; i++) {
                                int param_type = astrid_get_param_type(param_names[i]);
                                if (param_type == p) {
                                    already_set = 1;
                                    break;
                                }
                            }
                            // Only write default if not already set in graph spec and doesn't exist in session
                            if (!already_set && !AstridSession.exists_from_hash(session, u->param_hashes[p])) {
                                AstridSession.set_float_from_hash(session, u->param_hashes[p], u->param_values[p]);
                            }
                        }
                    }

                    // Add to graph - if no connections, first node is output
                    graph->nodes[graph->num_nodes++] = u;
                    if (graph->output_node == NULL) {
                        graph->output_node = u;
                    }
                }
            }
            
            current = token_end;
        }
    }
    
    graph->output_gain = 0.5f;  // Default output gain
    
    return graph;
}

int astrid_instrument_update_graph(lpinstrument_t * instrument) {
    if (instrument == NULL) {
        syslog(LOG_ERR, "astrid_instrument_update_graph: instrument is NULL\n");
        return -1;
    }

    // 1) We can always replace new_graph since the JACK callback consumes it
    // If there's an old new_graph that was never swapped in, free it first
    if (instrument->new_graph != NULL) {
        astrid_graph_t * old_new_graph = (astrid_graph_t *)instrument->new_graph;
        
        // Free all ugens in the old new_graph
        for (size_t i = 0; i < old_new_graph->num_nodes; i++) {
            if (old_new_graph->nodes[i] != NULL) {
                LPUgen.destroy(old_new_graph->nodes[i]);
            }
        }
        free(old_new_graph);
    }
    
    // 3) Create the new graph from the message
    astrid_graph_t * new_graph = astrid_graph_create_from_message(instrument->msg, &instrument->session);
    if (new_graph == NULL) {
        syslog(LOG_ERR, "astrid_instrument_update_graph: failed to parse graph from message\n");
        return -1;
    }

    // Set up atomic swap pointer
    instrument->new_graph = new_graph;
    
    // Set stream callback for graph processing if not already set
    if (instrument->stream == NULL && new_graph->num_nodes > 0) {
        instrument->stream = astrid_instrument_python_graph_stream_callback;
        syslog(LOG_INFO, "astrid_instrument_update_graph: stream callback set, graph has %zu nodes\n", new_graph->num_nodes);

        // Log all nodes for debugging
        for (size_t n = 0; n < new_graph->num_nodes; n++) {
            if (new_graph->nodes[n] != NULL) {
                int channel = -1;
                lpfloat_t gain = 0.0f;

                if (new_graph->nodes[n]->type == UGEN_MIX) {
                    channel = new_graph->nodes[n]->params.mix.channel;
                    gain = new_graph->nodes[n]->params.mix.gain;
                } else if (new_graph->nodes[n]->type == UGEN_DAC) {
                    channel = new_graph->nodes[n]->params.dac.channel;
                    gain = new_graph->nodes[n]->params.dac.gain;
                } else if (new_graph->nodes[n]->type == UGEN_ADC) {
                    channel = new_graph->nodes[n]->params.adc.channel;
                    gain = 0.0f; // ADC doesn't have gain in struct, uses params
                }

                syslog(LOG_INFO, "  Node %zu: type=%d channel=%d gain=%.2f\n",
                       n, new_graph->nodes[n]->type, channel, gain);
            }
        }
    }

    // Signal ready for swap (will be picked up by JACK callback)
    instrument->graph_swap_ready = 1;
    syslog(LOG_INFO, "astrid_instrument_update_graph: graph swap ready signaled\n");
    
    return 0;
}

int astrid_split_port_names(char * cmd, char * a, char * b) {
    int i, pos=0, a_complete=0;
    int cmd_length = strnlen(cmd, LPMAXMSG);

    for(i=0; i < cmd_length; i++) {
        if(cmd[i] == SEPARATOR) {
            if(i > 0 && !a_complete) {
                a_complete = 1;
                a[pos] = '\0';
                pos = 0;
            }
        } else if(!a_complete) {
            a[pos] = cmd[i];
            pos += 1;
        } else if(a_complete) {
            b[pos] = cmd[i];
            pos += 1;
        }
    }
    b[pos] = '\0';
    return 0;
}

// FIXME it would be cool to be able to use short names, or substrings
// and look them up with a jack_get_ports call.
// also move the default auto-hookup code here and let that only run 
// from python when no route callback is defined
int astrid_instrument_connect(lpinstrument_t * instrument, char * cmd) {
    int ret = 0;
    char a[LPMAXKEY];
    char b[LPMAXKEY];
    astrid_split_port_names(cmd, a, b);
    /* port A must be an *output* port, and port B must be an *input* port */
    if((ret = jack_connect(instrument->jack_client, a, b)) != 0) {
        if(ret != EEXIST) {
            syslog(LOG_ERR, "Could not connect %s to %s\n", a, b);
        }
        return -1;
    }

    return 0;
}

int astrid_instrument_disconnect(lpinstrument_t * instrument, char * cmd) {
    char a[LPMAXKEY];
    char b[LPMAXKEY];
    astrid_split_port_names(cmd, a, b);
    if(jack_disconnect(instrument->jack_client, a, b) < 0) {
        syslog(LOG_ERR, "Could not connect %s to %s\n", a, b);
        return -1;
    }
    return 0;
}

/* COMMAND PARSING
 * & MESSAGE PREP
 * ***************/
int extract_int32_from_token(char * token, int32_t * val) {
    char * end;
    long result = 0;

    errno = 0;
    result = strtol(token, &end, 0);

    if(errno != 0) return -1;
    if(*end != '\0') return -1;

    if(result > INT_MAX) result = INT_MAX;

    *val = (int32_t)result;
    return 0;
}

int extract_float_from_token(char * token, float * val) {
    char * end;
    float result = 0;

    errno = 0;
    result = strtof(token, &end);

    if(errno != 0) return -1;
    if(*end != '\0') return -1;

    *val = result;
    return 0;
}

int extract_floatlist_from_token(char * tokenlist, lpfloat_t * val, int size) {
    char * end, * save, * token;
    char tokenlist_copy[LPMAXMSG] = {0};
    float result = 0;
    int count = 0, i = 0;

    memcpy(tokenlist_copy, tokenlist, LPMAXMSG);
    token = strtok_r(tokenlist_copy, ",", &save);

    while(token != NULL) {
        errno = 0;
        result = strtof(token, &end);

        if(errno != 0) return -1;
        if(*end != '\0') return -1;
        val[count] = result;

        count += 1;
        if(count >= size) break;

        token = strtok_r(NULL, ",", &save);
    }

    for(i=count; i < size; i++) {
        val[i] = val[0];
    }

    return count;
}

int extract_patternbuf_from_token(char * token, unsigned char * patternbuf, size_t * pattern_length) {
    size_t i;

    *pattern_length = strlen(token);
    if(*pattern_length > LPMAXPAT) *pattern_length = LPMAXPAT;

    memset(patternbuf, 0, LPMAXPAT);
    for(i=0; i < *pattern_length; i++) {
        patternbuf[i] = (token[i] == '1');
    }

    return 0;
}

int decode_update_message_param(lpmsg_t * msg, uint16_t * id, float * value) {
    memcpy(id, msg->msg, sizeof(uint16_t));
    memcpy(value, msg->msg + sizeof(uint16_t), sizeof(float));
    return 0;
}

// Takes the cmd on the instrument and runs each param through the 
// instrument's param update callback.
// TODO think about handling large / special values, like buffers...
// payload could be the shm ID like render complete messages...?
int astrid_instrument_process_param_updates(lpinstrument_t * instrument) {
    char cmdline[LPMAXMSG] = {0};
    char * paramline;
    char * keytoken;
    char * valtoken;
    char * cmdline_save;
    char * paramline_save;

    memcpy(cmdline, instrument->msg.msg, LPMAXMSG);

    paramline = strtok_r(cmdline, " ", &cmdline_save);
    while(paramline != NULL) {
        keytoken = strtok_r(paramline, "=", &paramline_save);
        if(keytoken != NULL) {
            valtoken = strtok_r(NULL, "=", &paramline_save);
            if(valtoken != NULL) {
                // process the update callback if there is one
                if(instrument->update == NULL) continue;
                if(instrument->update(instrument, keytoken, valtoken) < 0) {
                    syslog(LOG_ERR, "process_param_updates: failed to pass param (%s=%s) to update callback.\n", keytoken, valtoken);
                }
            }
        }
        paramline = strtok_r(NULL, " ", &cmdline_save);
    }
    return 0;
}

int prepare_instrument_message(lpmsg_t * msg, char msgtype) {
    //syslog(LOG_ERR, "prepare_and_send_instrument_message msgtype=%c\n", msgtype);

    /* Initialize the count and set the voice_id */
    msg->count = 0;
    msg->voice_id = 0;

    //syslog(LOG_ERR, "prepare_and_send_instrument_message voice_id=%d\n", voice_id);

    switch(msgtype) {
        case PLAY_MESSAGE:
            msg->type = LPMSG_PLAY;
            break;

        case TRIGGER_MESSAGE:
            msg->type = LPMSG_TRIGGER;
            break;

        case UDP_MESSAGE:
            msg->type = LPMSG_UDP;
            break;

        case UPDATE_MESSAGE:
            msg->type = LPMSG_UPDATE;
            break;

        case LOAD_MESSAGE:
            msg->type = LPMSG_LOAD;
            break;

        case AUTOTRIGGER_REPLACE_MESSAGE:
            msg->type = LPMSG_AUTOTRIGGER_REPLACE_ALL;
            break;

        case AUTOTRIGGER_UPDATE_MESSAGE:
            msg->type = LPMSG_AUTOTRIGGER_UPDATE;
            break;

        case SCHEDULE_MESSAGE:
            msg->type = LPMSG_SCHEDULE;
            break;

        case SHUTDOWN_MESSAGE:
            msg->type = LPMSG_SHUTDOWN;
            break;

        case SET_COUNTER_MESSAGE:
            msg->type = LPMSG_SET_COUNTER;
            break;

        default:
            syslog(LOG_CRIT, "Bad msgtype! %c\n", msgtype);
            return -1;
    }
    return 0;
}


int prepare_and_send_instrument_message(lpmsg_t * msg, char msgtype) {
    if(prepare_instrument_message(msg, msgtype) < 0) {
        return -1;
    }
    if(send_play_message(*msg) < 0) {
        syslog(LOG_ERR, "Could not send play message...\n");
        return -1;
    }
    return 0;
}

int parse_message_from_args(int argc, int arg_offset, char * argv[], lpmsg_t * msg) {
    int bytesread, a, i, length;
    char msgtype;
    char message_params[LPMAXMSG] = {0};
    char instrument_name[LPMAXNAME] = {0};
    size_t instrument_name_length;

    instrument_name_length = 0;
    msgtype = argv[arg_offset + 1][0];

    /* Prepare the message param string */
    bytesread = 0;
    for(a=arg_offset+2; a < argc; a++) {
        length = strlen(argv[a]);
        if(a==arg_offset+2) {
            instrument_name_length = (size_t)length;
            memcpy(instrument_name, argv[a], instrument_name_length);
            continue;
        }

        for(i=0; i < length; i++) {
            message_params[bytesread] = argv[a][i];
            bytesread++;
        }
        message_params[bytesread] = SPACE;
        bytesread++;
    }

    /* Set up the message struct */
    strncpy(msg->instrument_name, instrument_name, instrument_name_length);
    strncpy(msg->msg, message_params, bytesread);

    if(prepare_and_send_instrument_message(msg, msgtype) < 0) {
        syslog(LOG_ERR, "Could not prepare message of type %c: (%d) %s\n", msgtype, errno, strerror(errno));
        return -1;
    }

    return 0;
}

int parse_message_from_external_cmdline(char * cmdline, lpmsg_t * msg) {
    char * space = strchr(cmdline, ' ');
    if (!space) return -1;

    // find the external instrument name in the cmdline
    size_t name_len = space - cmdline;
    if (name_len >= LPMAXNAME) {
        name_len = LPMAXNAME - 1;
    }
    
    // first set the external instrument name on the msg
    memset(msg->instrument_name, 0, LPMAXNAME);
    memcpy(msg->instrument_name, cmdline, name_len);

    // then parse the rest normally
    return parse_message_from_cmdline(space+1, strlen(space+1), msg);
}

int parse_message_from_cmdline(char * cmdline, size_t cmdlength, lpmsg_t * msg) {
    int tokenlength;
    char *token, *save=NULL;
    char msgtype;
    char cmd_copy[LPMAXMSG] = {0};

    if(cmdline == NULL) return 0;

    // strtok clobbers input
    memcpy(cmd_copy, cmdline, cmdlength);

    // Get the message type, and ignore everything else for now...
    token = strtok_r(cmd_copy, " ", &save);
    msgtype = token[0]; // msg types only use the first char
    tokenlength = strlen(token);

    memset(msg->msg, 0, LPMAXMSG);
    strncpy(msg->msg, cmdline + tokenlength, cmdlength - tokenlength);

    if(prepare_instrument_message(msg, msgtype) < 0) {
        syslog(LOG_ERR, "Could not prepare message of type %c: (%d) %s\n", msgtype, errno, strerror(errno));
        return -1;
    }

    return 0;
}



/* SCHEDULING
 * and MIXING
 * **********/
int lpscheduler_get_now_seconds(double * now) {
    clockid_t cid;
    struct timespec ts;

#if defined(__linux__)
    cid = CLOCK_MONOTONIC_RAW;
#else
    cid = CLOCK_MONOTONIC;
#endif

    if(clock_gettime(cid, &ts) < 0) {
        syslog(LOG_ERR, "scheduler_get_now_seconds: clock_gettime error: %s\n", strerror(errno));
        return -1; 
    }

    *now = ts.tv_sec + ts.tv_nsec * 1e-9;

    return 0;
}

int lpscheduler_get_now_ticks(size_t * now, int samplerate) {
    // ticks == frames @ given samplerate
    clockid_t cid;
    struct timespec ts;

#if defined(__linux__)
    cid = CLOCK_MONOTONIC_RAW;
#else
    cid = CLOCK_MONOTONIC;
#endif

    if(clock_gettime(cid, &ts) < 0) {
        syslog(LOG_ERR, "scheduler_get_now_seconds: clock_gettime error: %s\n", strerror(errno));
        return -1; 
    }

    *now = (size_t)ts.tv_sec * samplerate;
    *now += (size_t)(((size_t)ts.tv_nsec * samplerate) * 1e-9);

    return 0;
}

void scheduler_get_now(struct timespec * now) {
    clock_gettime(CLOCK_MONOTONIC_RAW, now);
}

void scheduler_frames_to_timespec(
    size_t frames, 
    lpfloat_t samplerate, 
    struct timespec * ts
) {
    size_t ns;

    ts->tv_sec = (time_t)(frames / samplerate);

    ns = frames - (size_t)(ts->tv_sec * samplerate);
    ts->tv_nsec = (size_t)(ns / samplerate);
}

void scheduler_increment_timespec_by_ns(
    struct timespec * ts, 
    size_t ns
) {
    size_t seconds = 0;
    while(ns >= 1000000000) {
        ns -= 1000000000;
        seconds += 1;
    }

    ts->tv_sec += seconds;
    ts->tv_nsec += ns;
}

size_t scheduler_timespec_to_ticks(
    lpscheduler_t * s, 
    struct timespec * ts
) {
    size_t secs, nsecs, total_ns;     

    /* Seconds and nanoseconds since init */
    secs = ts->tv_sec - s->init->tv_sec;
    nsecs = ts->tv_nsec - s->init->tv_nsec;

    /* Convert nanoseconds since init to number of 
     * ticks since init.
     *
     * ticks advance on frame boundries as the scheduler 
     * processes each frame of output.
     *
     * `tick_ns` is the number of nanoseconds 
     * per frame at the current samplerate.
     **/
    total_ns = nsecs + (secs * 1000000000);
    return total_ns / s->tick_ns;
}

int scheduler_onset_has_elapsed(lpscheduler_t * s, struct timespec * ts) {
    printf(" s->now->tv_sec: %d\n", (int)s->now->tv_sec);
    printf("     ts->tv_sec: %d\n", (int)ts->tv_sec);
    printf("s->now->tv_nsec: %d\n", (int)s->now->tv_nsec);
    printf("    ts->tv_nsec: %d\n", (int)ts->tv_nsec);

    return ((
            s->now->tv_sec >= ts->tv_sec
        ) || (
            s->now->tv_sec == ts->tv_sec &&
            s->now->tv_nsec >= ts->tv_nsec
        )
    ) ? 1 : 0;
}

void ll_display(lpevent_t * head) {
    lpevent_t * current;
    size_t buflen = 0;
    if(head != NULL) {
        current = head;
        while(current->next != NULL) {
            if(current->buf != NULL) {
                buflen = current->buf->length;
            } else {
                buflen = 0;
            }
            syslog(LOG_INFO, "    e%ld onset: %ld pos: %ld length: %ld\n", current->id, current->onset, current->pos, buflen);
            current = (lpevent_t *)current->next;        
        }
        if(current->buf != NULL) {
            buflen = current->buf->length;
        } else {
            buflen = 0;
        }
        syslog(LOG_INFO, "    e%ld onset: %ld pos: %ld length: %ld\n", current->id, current->onset, current->pos, buflen);
    }
}

int ll_count(lpevent_t * head) {
    lpevent_t * current;
    int count;

    count = 0;

    if(head) {
        count = 1;
        current = head;
        while(current->next != NULL) {
            current = (lpevent_t *)current->next;        
            count += 1;
        }
    }

    return count;
}

static inline void start_waiting(lpscheduler_t * s, lpevent_t * e) {
    static size_t overflow_count = 0;

    // Check for waiting queue overflow
    if(s->num_waiting >= ASTRID_MAX_WAITING_BUFFERS) {
        overflow_count++;
        if((overflow_count % 100) == 0) {
            syslog(LOG_ERR, "Waiting queue FULL (%d buffers), cannot add new buffer. Dropped %zu buffers total. LEAKING BUFFER %s.\n",
                   ASTRID_MAX_WAITING_BUFFERS, overflow_count, e->buffer_code);
        }
        return; // Drop the buffer to avoid overwriting
    }

    s->waiting[s->last_waiting_added] = e;
    s->last_waiting_added += 1;
    s->num_waiting += 1;

    if(s->last_waiting_added >= ASTRID_MAX_WAITING_BUFFERS) {
        s->last_waiting_added -= ASTRID_MAX_WAITING_BUFFERS;
    }
}

static inline void start_playing(lpscheduler_t * s, lpevent_t * e) {
    lpevent_t * current;

#if 0
    syslog(LOG_INFO, "START playing event ID %ld\n", e->id);
#endif

    /* Add to the tail of the playing queue */
    if(s->playing_stack_head == NULL) {
        s->playing_stack_head = e;
    } else {
        current = s->playing_stack_head;
        while(current->next != NULL) {
            current = (lpevent_t *)current->next;
        }
        current->next = (void *)e;
    }
}

static inline void stop_playing(lpscheduler_t * s, lpevent_t * e) {
    lpevent_t * current;
    lpevent_t * prev;
    ssize_t offset;

#if 0
    syslog(LOG_INFO, "STOP playing event ID %ld\n", e->id);
#endif

    /* Remove from the playing stack */
    if(s->playing_stack_head == NULL) {
        syslog(LOG_CRIT, "Cannot move this event. There is nothing in the waiting queue!\n");
        return;
    }

    prev = NULL;
    current = s->playing_stack_head;
    while(current != e && current->next != NULL) {
        prev = current;
        current = (lpevent_t *)current->next;
    }

    if(prev) {
        prev->next = current->next;
    } else {
        s->playing_stack_head = (lpevent_t *)current->next;
    }

    current->next = NULL;

    /* Add to nursery ringbuf for lock-free cleanup */
    offset = ringbuf_acquire(s->nursery_ringbuf, s->nursery_worker, sizeof(lpevent_t *));
    if(offset == -1) {
        syslog(LOG_ERR, "Nursery ringbuf full (%d events), cannot queue event for cleanup. LEAKING BUFFER %s (NOT IDEAL).\n", ASTRID_MAX_NURSERY_EVENTS, e->buffer_code);
        // TODO: Consider immediate cleanup here or dropping the event
        // For now, we'll leak to avoid blocking JACK thread
        return;
    }

    s->nursery_buffer[offset / sizeof(lpevent_t *)] = e;
    ringbuf_produce(s->nursery_ringbuf, s->nursery_worker);
}

lpscheduler_t * scheduler_create(int realtime, int channels, lpfloat_t samplerate) {
    lpscheduler_t * s;
    size_t nursery_ringbuf_size, nursery_worker_size;

    s = (lpscheduler_t *)LPMemoryPool.alloc(1, sizeof(lpscheduler_t));
    s->now = (struct timespec *)LPMemoryPool.alloc(1, sizeof(struct timespec));

    s->realtime = realtime;

    s->last_waiting_removed = 0;
    s->last_waiting_added = 0;
    s->num_waiting = 0;
    s->playing_stack_head = NULL;

    // Initialize lock-free nursery ringbuf for finished events cleanup
    ringbuf_get_sizes(1, &nursery_ringbuf_size, &nursery_worker_size); // 1 producer (JACK thread)

    s->nursery_ringbuf = (ringbuf_t *)LPMemoryPool.alloc(1, nursery_ringbuf_size);
    if(s->nursery_ringbuf == NULL) {
        syslog(LOG_CRIT, "Could not allocate nursery ringbuf\n");
        LPMemoryPool.free(s->now);
        LPMemoryPool.free(s);
        return NULL;
    }

    s->nursery_buffer = (lpevent_t **)LPMemoryPool.alloc(ASTRID_MAX_NURSERY_EVENTS, sizeof(lpevent_t *));
    if(s->nursery_buffer == NULL) {
        syslog(LOG_CRIT, "Could not allocate nursery buffer\n");
        LPMemoryPool.free(s->nursery_ringbuf);
        LPMemoryPool.free(s->now);
        LPMemoryPool.free(s);
        return NULL;
    }

    if(ringbuf_setup(s->nursery_ringbuf, 1, ASTRID_MAX_NURSERY_EVENTS * sizeof(lpevent_t *)) < 0) {
        syslog(LOG_CRIT, "Could not setup nursery ringbuf\n");
        LPMemoryPool.free(s->nursery_buffer);
        LPMemoryPool.free(s->nursery_ringbuf);
        LPMemoryPool.free(s->now);
        LPMemoryPool.free(s);
        return NULL;
    }

    s->nursery_worker = ringbuf_register(s->nursery_ringbuf, 0);
    if(s->nursery_worker == NULL) {
        syslog(LOG_CRIT, "Could not register nursery worker\n");
        LPMemoryPool.free(s->nursery_buffer);
        LPMemoryPool.free(s->nursery_ringbuf);
        LPMemoryPool.free(s->now);
        LPMemoryPool.free(s);
        return NULL;
    }

    s->samplerate = samplerate;
    s->channels = channels;

    s->tick_ns = (size_t)(samplerate / 1000000000.f);

    if(realtime == 1) scheduler_get_now(s->now);
    s->ticks = 0;
    s->current_frame = (lpfloat_t *)LPMemoryPool.alloc(channels, sizeof(lpfloat_t));

    s->event_count = 0;

    syslog(LOG_INFO, "Scheduler created with nursery ringbuf (%zu events max)\n", (size_t)ASTRID_MAX_NURSERY_EVENTS);

    return s;
}

/* look for events waiting to be scheduled */
static inline void scheduler_update(lpscheduler_t * s) {
    size_t now_ticks = 0;
    lpevent_t * current;
    void * next;

    lpscheduler_get_now_ticks(&now_ticks, s->samplerate);

    // pop any waiting buffers into the playing queue
    // Use num_waiting counter instead of index comparison to handle circular buffer wrap-around
    while(s->num_waiting > 0) {
        start_playing(s, s->waiting[s->last_waiting_removed]);
        s->last_waiting_removed += 1;
        s->num_waiting -= 1;

        // Wrap around for circular buffer
        if(s->last_waiting_removed >= ASTRID_MAX_WAITING_BUFFERS) {
            s->last_waiting_removed -= ASTRID_MAX_WAITING_BUFFERS;
        }
    }

    /* look for events that have finished playing */
    if(s->playing_stack_head != NULL) {
        current = s->playing_stack_head;
        while(current->next != NULL) {
            next = current->next;
            if(current->buf != NULL && current->pos >= current->buf->length-1) {
                stop_playing(s, current);
            }
            current = (lpevent_t *)next;
        }

        if(current->buf != NULL && current->pos >= current->buf->length-1) {
            stop_playing(s, current);
        }
    }
}

static inline void scheduler_mix_buffers(lpscheduler_t * s) {
    lpevent_t * current;
    int c, i, output_channel;

    // Clear the output frame
    for(c=0; c < s->channels; c++) {
        s->current_frame[c] = 0.f;
    }

    if(s->playing_stack_head == NULL) {
        return;
    }

    current = s->playing_stack_head;
    while(current != NULL) {
        if(current->buf != NULL && current->pos < current->buf->length) {
            if(current->map_channels > 0) {
                // Wrap buffer channels when map requests more channels than buffer has
                for(i = 0; i < current->map_channels; i++) {
                    int buffer_channel = i % current->buf->channels;
                    output_channel = current->channel_map[i] % s->channels;
                    s->current_frame[output_channel] += current->buf->data[current->pos * current->buf->channels + buffer_channel];
                }
            } else {
                for(i = 0; i < current->buf->channels && i < s->channels; i++) {
                    s->current_frame[i] += current->buf->data[current->pos * current->buf->channels + i];
                }
            }
        }
        current = (lpevent_t *)current->next;
    }

    for(c=0; c < s->channels; c++) {
        s->current_frame[c] = lpfilternan(s->current_frame[c]);
    }
}

static inline void scheduler_advance_buffers(lpscheduler_t * s) {
    lpevent_t * current;
 
    /* loop over buffers and advance their positions */
    if(s->playing_stack_head != NULL) {
        current = s->playing_stack_head;
        while(current->next != NULL) {
            current->pos += 1;
            current = (lpevent_t *)current->next;
        }
        current->pos += 1;
    }
}

void scheduler_debug(lpscheduler_t * s) {
    if(s->playing_stack_head) {
        syslog(LOG_ERR, "%d playing\n", ll_count(s->playing_stack_head));
        ll_display(s->playing_stack_head);
    } else {
        syslog(LOG_ERR, "none playing\n");
    }
}

void lpscheduler_tick(lpscheduler_t * s) {
#if 0
    if(s->ticks % 10000 == 0) {
        syslog(LOG_ERR, "TICK10k %ld\n", s->ticks);
        scheduler_debug(s);
    }
#endif

    /* Move buffers to proper lists */
    scheduler_update(s);

    /* Mix currently playing buffers into s->current_frame */
    scheduler_mix_buffers(s);

    /* Advance the position for all playing buffers */
    scheduler_advance_buffers(s);

    /* Increment process ticks and update now timestamp */
    s->ticks += 1; // FIXME <-- just use get_now_ticks?
    if(s->realtime == 0) {
        scheduler_increment_timespec_by_ns(s->now, s->tick_ns);
    }
}

void scheduler_schedule_event(astrid_session_t * session, lpscheduler_t * s, char * buffer_code, size_t onset_delay) {
    size_t now_ticks = 0;
    lpevent_t * e;

    e = (lpevent_t *)LPMemoryPool.alloc(1, sizeof(lpevent_t));
    if(e == NULL) {
        syslog(LOG_ERR, "scheduler_schedule_event: memory allocation failed for event (buffer_code=%s)\n", buffer_code);
        return;
    }
    s->event_count += 1;
    e->id = s->event_count;
    e->onset = 0;
    e->callback_onset = 0;
    e->map_channels = 0;  // Initialize to no mapping

    memcpy(e->buffer_code, buffer_code, LPMAXKEY);

    if(astrid_session_aquire_shared_resource(session, &e->resource, buffer_code) < 0) {
        syslog(LOG_ERR, "BUFFER_SCHEDULE_FAILED: code=%s destroying orphaned buffer. (%d) %s\n", buffer_code, errno, strerror(errno));
        // Destroy the orphaned buffer that was published but can't be scheduled
        astrid_session_destroy_shared_resource_fast(session, buffer_code);
        LPMemoryPool.free(e);
        return;
    }

    e->buf = (lpbuffer_t *)e->resource.data;
    e->pos = 0;

    // Load channel map from buffer header (stored inline, no separate resource needed)
    e->map_channels = e->resource.header.map_channels;
    if(e->map_channels > 0) {
        if(e->map_channels > 256) {
            syslog(LOG_WARNING, "scheduler_schedule_event: truncating channel map from %d to 256 channels\n", e->map_channels);
            e->map_channels = 256;
        }
        memcpy(e->channel_map, e->resource.header.channel_map, e->map_channels * sizeof(int));
    }

    lpscheduler_get_now_ticks(&now_ticks, s->samplerate);
    e->onset = now_ticks + onset_delay;

    start_waiting(s, e);
}

int scheduler_count_waiting(lpscheduler_t * s) {
    return s->last_waiting_added - s->last_waiting_removed;
}

int scheduler_count_playing(lpscheduler_t * s) {
    return ll_count(s->playing_stack_head);
}

int scheduler_is_playing(lpscheduler_t * s) {
    int playing;
    playing = 0;
    if(scheduler_count_waiting(s) > 0) playing = 1;
    if(scheduler_count_playing(s) > 0) playing = 1;
    return playing;
}

void scheduler_destroy(lpscheduler_t * s) {
    /* Loop over queues and free buffers, events */
    lpevent_t * current;
    lpevent_t * next;

    if(s->playing_stack_head != NULL) {
        current = s->playing_stack_head;
        while(current->next != NULL) {
            next = (lpevent_t *)current->next;
            LPMemoryPool.free(current);
            current = (lpevent_t *)next;
        }
        LPMemoryPool.free(current);
    }

    /* Unregister and free nursery ringbuf resources */
    if(s->nursery_worker != NULL) {
        ringbuf_unregister(s->nursery_ringbuf, s->nursery_worker);
    }
    if(s->nursery_buffer != NULL) {
        LPMemoryPool.free(s->nursery_buffer);
    }
    if(s->nursery_ringbuf != NULL) {
        LPMemoryPool.free(s->nursery_ringbuf);
    }

    LPMemoryPool.free(s->now);
    LPMemoryPool.free(s->current_frame);
    LPMemoryPool.free(s);
}

int scheduler_cleanup_nursery(astrid_session_t * session, lpscheduler_t * s) {
    /* Consume finished events from lock-free nursery ringbuf and free resources */
    lpevent_t * current;
    int cleanup_errors = 0;
    size_t offset, nbytes;
    size_t num_events, start_index;
    size_t i;

    /* See if we have some events from the ringbuf */
    nbytes = ringbuf_consume(s->nursery_ringbuf, &offset);
    if(nbytes == 0) return 0;

    /* Calculate how many event pointers we received */
    num_events = nbytes / sizeof(lpevent_t *);
    start_index = offset / sizeof(lpevent_t *);

    /* Clean up each event */
    for(i = 0; i < num_events; i++) {
        current = s->nursery_buffer[start_index + i];
        if(current == NULL) {
            syslog(LOG_WARNING, "scheduler_cleanup_nursery: NULL event pointer in nursery\n");
            continue;
        }

        current->buf = NULL;

        if(astrid_session_release_shared_resource(session, &current->resource, current->buffer_code) < 0) {
            syslog(LOG_ERR, "scheduler_cleanup_nursery: could not release buffer %s\n", current->buffer_code);
            cleanup_errors++;
        }

        if(astrid_session_destroy_shared_resource(session, current->buffer_code) < 0) {
            syslog(LOG_ERR, "scheduler_cleanup_nursery: could not destroy buffer %s (release_errors=%d)\n", current->buffer_code, cleanup_errors);
            cleanup_errors++;
        }

        LPMemoryPool.free(current);
    }

    /* Release the consumed range back to the ringbuf */
    ringbuf_release(s->nursery_ringbuf, nbytes);

    return (cleanup_errors > 0) ? -1 : 0;
}

void * instrument_autotrigger_thread(void * arg) {
    lpautotrigger_table_t * table;
    size_t now=0, then=0;
    ssize_t ticks_since_last_iteration=0;
    astrid_shared_resource_t resource;
    lpinstrument_t * instrument = (lpinstrument_t *)arg;
    char name[NAME_MAX] = {0};
    size_t count=0;

    snprintf(name, NAME_MAX, "%s-autotriggers", instrument->name);

    syslog(LOG_INFO, "autotrigger thread starting (%s) sizeof(table)=%ld\n", name, sizeof(lpautotrigger_table_t));

    // register and initialize the autotrigger table
    if(astrid_session_register_shared_resource(&instrument->session, name, NULL, ASTRID_TYPE_BYTES, sizeof(lpautotrigger_table_t)) < 0) {
        syslog(LOG_ERR, "autotrigger_thread: could not register autotrigger table. Shutting down.\n");
        return NULL;
    }

    // open a long-lived reference to the autotrigger table
    if(astrid_session_aquire_shared_resource(&instrument->session, &resource, name) < 0) {
        syslog(LOG_ERR, "autotrigger thread: astrid_session_aquire_shared_resource could not aquire shm. (%d) %s\n", errno, strerror(errno));
        return 0;
    }
    if(astrid_session_release_resource_lock(&resource) < 0) {
        syslog(LOG_CRIT, "autotrigger thread: astrid_session_release_shared_resource Could not release resource lock.\n");
        return 0;
    }

    // Get producer handle for the internal message queue
    char qname[LPMAXKEY] = {0};
    snprintf(qname, LPMAXKEY, "%s-msgq", instrument->name);
    lpmsgq_t * msgq = astrid_msgq_produce(&instrument->session, qname, MAINMQ_AUTOTRIGGER_PRODUCER);
    if(msgq == NULL) {
        syslog(LOG_ERR, "%s autotrigger thread: Could not get producer handle for message queue\n", instrument->name);
        return 0;
    }

    // Initialize timing before loop starts
    lpscheduler_get_now_ticks(&then, instrument->samplerate);

    while(instrument->is_running) {
        // get now in ticks
        lpscheduler_get_now_ticks(&now, instrument->samplerate);

        // get the actual number of ticks since the last iteration
        // for the phase accumulators, and set then to now for later
        ticks_since_last_iteration = now - then;
        then = now;

        if(astrid_session_aquire_resource_lock(&resource, name) < 0) {
            syslog(LOG_WARNING, "autotrigger_thread: could not aquire lock on autotrigger table.\n");
            usleep(200000); // 200ms
            continue;
        }

        table = (lpautotrigger_table_t *)resource.data;

        // Loop over autotriggers array and print the payload field IF 
        // the autotrigger lfo has hit its interval edge
        for(int i = 0; i < table->num_active_triggers; i++) {
            lpautotrigger_t * trigger = &table->autotriggers[table->active_triggers[i]];

            if(trigger->period == 0 || trigger->speed == 0) continue;

            // check each onset time and see if phase has overlapped it,
            // then fire a message if one has not already fired
            for(int o=0; o < trigger->num_onsets; o++) {
                if(trigger->phase >= trigger->onsets[o] && trigger->has_triggered[o] == 0) {
                    trigger->messages[o].count += 1;
                    if(astrid_msgq_write(msgq, &trigger->messages[o]) < 0) {
                        syslog(LOG_ERR, "Could not send autotrigger message for slot %d-%d\n", i, o);
                        continue;
                    }
                    trigger->has_triggered[o] = 1;
                }
            }

            trigger->phase += ticks_since_last_iteration * trigger->speed;

            if(trigger->phase >= trigger->period) {
                for(int o=0; o < trigger->num_onsets; o++) {
                    trigger->has_triggered[o] = 0;
                }
            }
            while(trigger->phase >= trigger->period) {
                trigger->phase -= trigger->period;
            }
        }

        if(astrid_session_release_resource_lock(&resource) < 0) {
            syslog(LOG_WARNING, "autotrigger_thread: could not release autotrigger table lock.\n");
            usleep(200000);
            continue;
        }

        usleep(1000);
        count += 1;
    }

    astrid_msgq_close(msgq);

    if(astrid_session_aquire_resource_lock(&resource, name) < 0) {
        syslog(LOG_ERR, "astrid_session_aquire_shared_resource: Could not aquire resource lock. (name=%s lock_name=%s)\n", instrument->adcname, resource.lock_name);
        usleep((useconds_t)10);
        return 0;
    }

    if(astrid_session_release_shared_resource(&instrument->session, &resource, name) < 0) {
        syslog(LOG_ERR, "astrid_session_write_lpbuffer_to_shared_ringbuffer: astrid_session_release_shared_resource could not release ADC shm.\n");
        return 0;
    }

    return NULL;
}

/* SEQUENCING &
 * AUTOTRIGGERS
 * ************/
int astrid_autotrigger_init(lpautotrigger_t * at) {
    //memset(at, 0, sizeof(lpautotrigger_t));
    at->num_onsets = 0;
    return 0;
}

int astrid_autotrigger_add_onset(lpautotrigger_t * at, size_t onset_time_in_frames, size_t period, lpfloat_t speed, char * cmd) {
    lpmsg_t msg = {0};

    syslog(LOG_ERR, "ADD ONSET %ld cmd=\"%s\"", onset_time_in_frames, cmd);

    if(at->num_onsets == ASTRID_MAX_AUTOTRIGGER_SUBEVENTS) {
        syslog(LOG_ERR, "astrid_autotrigger_add_onset: max onsets reached\n");
        return -1;
    }

    if(parse_message_from_external_cmdline(cmd, &msg) < 0) {
        syslog(LOG_ERR, "astrid_autotrigger_add_onset: could not parse cmd string\n");
        return -1;
    }

    at->period = period;
    at->speed = speed;
    at->onsets[at->num_onsets] = onset_time_in_frames;
    at->messages[at->num_onsets] = msg;
    at->num_onsets += 1;

    return 0;
}

int astrid_autotrigger_table_get_free_slot(__attribute__((unused)) lpautotrigger_table_t * att) {
    int slot = -1;
    return slot;
}

int astrid_autotrigger_table_clear(lpautotrigger_table_t * att) {
    for(int i=0; i < att->num_active_triggers; i++) {
        att->autotriggers[att->active_triggers[i]].period = 0;
        att->autotriggers[att->active_triggers[i]].speed = 0;
        att->active_triggers[i] = 0;
    }
    att->num_active_triggers = 0;
    return 0;
}

int astrid_autotrigger_update_speed(lpautotrigger_t * at, lpfloat_t speed) {
    if(speed <= 0) {
        syslog(LOG_ERR, "astrid_autotrigger_update_speed: speed must be positive\n");
        return -1;
    }
    at->speed = speed;
    return 0;
}

int astrid_autotrigger_update_onset_value(lpautotrigger_t * at, int onset_index, size_t onset_frames) {
    if(onset_index < 0 || onset_index >= at->num_onsets) {
        syslog(LOG_ERR, "astrid_autotrigger_update_onset_value: onset_index %d out of bounds (num_onsets=%d)\n", onset_index, at->num_onsets);
        return -1;
    }

    if(onset_frames >= at->period) {
        syslog(LOG_ERR, "astrid_autotrigger_update_onset_value: onset_frames %ld >= period %ld\n", onset_frames, at->period);
        return -1;
    }

    at->onsets[onset_index] = onset_frames;
    return 0;
}

int astrid_autotrigger_update_onset_cmd(lpautotrigger_t * at, int onset_index, char * cmd) {
    lpmsg_t msg = {0};

    if(onset_index < 0 || onset_index >= at->num_onsets) {
        syslog(LOG_ERR, "astrid_autotrigger_update_onset_cmd: onset_index %d out of bounds (num_onsets=%d)\n", onset_index, at->num_onsets);
        return -1;
    }

    if(parse_message_from_external_cmdline(cmd, &msg) < 0) {
        syslog(LOG_ERR, "astrid_autotrigger_update_onset_cmd: could not parse cmd string\n");
        return -1;
    }

    at->messages[onset_index] = msg;
    return 0;
}

int astrid_autotrigger_update_all_cmds(lpautotrigger_t * at, char * cmd) {
    lpmsg_t msg = {0};

    if(parse_message_from_external_cmdline(cmd, &msg) < 0) {
        syslog(LOG_ERR, "astrid_autotrigger_update_all_cmds: could not parse cmd string\n");
        return -1;
    }

    for(int i = 0; i < at->num_onsets; i++) {
        at->messages[i] = msg;
    }

    return 0;
}

/* instrument seq priority queue callbacks */
static int msgpq_cmp_pri(size_t next, size_t curr) {
    return (next > curr);
}

static size_t msgpq_get_pri(void * a) {
    return ((lpmsgpq_node_t *)a)->timestamp;
}

static void msgpq_set_pri(void * a, size_t timestamp) {
    ((lpmsgpq_node_t *)a)->timestamp = timestamp;
}

static size_t msgpq_get_pos(void * a) {
    return ((lpmsgpq_node_t *)a)->pos;
}

static void msgpq_set_pos(void * a, size_t pos) {
    ((lpmsgpq_node_t *)a)->pos = pos;
}

int instrument_seq_remove_nodes_by_voice_id(pqueue_t * msgpq, size_t voice_id) {
    lpmsgpq_node_t * node;
    lpmsg_t * msg;
    void * d;
    size_t i;

    for(i=0; i < msgpq->size; i++) {
        d = msgpq->d[i];                
        if(d == NULL) {
            continue;
        }

        node = (lpmsgpq_node_t *)d;
        msg = &node->msg;
        if(msg->voice_id == voice_id) {
            pqueue_remove(msgpq, d);
        }
    }

    return 0;
}

void * instrument_seq_pq(void * arg) {
    lpinstrument_t * instrument = (lpinstrument_t *)arg;
    lpmsg_t * msg;
    lpmsgpq_node_t * node;
    void * d;
    size_t now = 0;

    now = 0;

    d = NULL;
    msg = NULL;
    node = NULL;

    // Get producer handle for the internal message queue
    char qname[LPMAXKEY] = {0};
    snprintf(qname, LPMAXKEY, "%s-msgq", instrument->name);
    lpmsgq_t * msgq = astrid_msgq_produce(&instrument->session, qname, MAINMQ_SCHEDULER_PRODUCER);
    if(msgq == NULL) {
        syslog(LOG_ERR, "%s seq pq thread: Could not get producer handle for message queue\n", instrument->name);
        return 0;
    }

    while(instrument->is_running) {
        /* peek into the queue */
        d = pqueue_peek(instrument->msgpq);

        /* No messages have arrived */
        if(d == NULL) {
            usleep((useconds_t)500); // 0.5 milliseconds
            continue;
        }

        /* There is a message! */
        node = (lpmsgpq_node_t *)d;
        msg = &node->msg;

        /* Get now */
        if(lpscheduler_get_now_ticks(&now, instrument->samplerate) < 0) {
            syslog(LOG_CRIT, "Error getting now in message scheduler\n");
            exit(1);
        }

        /* If msg timestamp is in the future,
         * sleep for 0.5ms and then try again */
        if(node->timestamp > now) {
            usleep((useconds_t)500);
            continue;
        }

        /* Send it along to the instrument message fifo */
        if(astrid_msgq_write(msgq, msg) < 0) {
            syslog(LOG_ERR, "Error sending play message from message priority queue\n");
            usleep((useconds_t)500);
            continue;
        }

        /* And remove it from the pq */
        if(pqueue_remove(instrument->msgpq, d) < 0) {
            syslog(LOG_ERR, "pqueue_remove: problem removing message from the pq\n");
            usleep((useconds_t)500);
        }

        // if this is a shutdown message, then shutdown
        if(msg->type == LPMSG_SHUTDOWN) {
            break;
        }
    }

    astrid_msgq_close(msgq);

    syslog(LOG_INFO, "Message scheduler pq thread shutting down...\n");
    return 0;
}

int astrid_schedule_message(lpinstrument_t * instrument, lpmsg_t msg) {
    lpmsgpq_node_t * d;
    size_t seq_delay, now=0;

    d = &instrument->pqnodes[instrument->pqnode_index];
    memcpy(&d->msg, &msg, sizeof(lpmsg_t));
    instrument->pqnode_index += 1;
    while(instrument->pqnode_index >= NUM_NODES) instrument->pqnode_index -= NUM_NODES;

    if(lpscheduler_get_now_ticks(&now, instrument->samplerate) < 0) {
        syslog(LOG_CRIT, "SEQ PQ MSG: Error getting now in seq relay\n");
        return -1;
    }

    /* Hold on to the message as long as possible while still 
     * trying to leave some time for processing before the target deadline */
    seq_delay = (msg.scheduled * instrument->samplerate) - (msg.max_processing_time * 2);
    d->timestamp = msg.initiated + seq_delay;

    // Remove the scheduled flag before relaying the message
    d->msg.flags &= ~LPFLAG_IS_SCHEDULED;

    if(pqueue_insert(instrument->msgpq, (void *)d) < 0) {
        syslog(LOG_ERR, "SEQ PQ MSG: Error while inserting message into pq during msgq loop: %s\n", strerror(errno));
        return -1;
    }

    return 0;
}

int astrid_instrument_seq_start(lpinstrument_t * instrument) {
    int i;

    /* Allocate the pq message nodes */
    if((instrument->pqnodes = (lpmsgpq_node_t *)calloc(NUM_NODES, sizeof(lpmsgpq_node_t))) == NULL) {
        syslog(LOG_ERR, "Could not initialize message priority queue nodes. Error: %s\n", strerror(errno));
        return -1;
    }
    memset(instrument->pqnodes, 0, NUM_NODES * sizeof(lpmsgpq_node_t));

    for(i=0; i < NUM_NODES; i++) {
        instrument->pqnodes[i].index = i;
    }

    /* Create the message priority queue */
    if((instrument->msgpq = pqueue_init(NUM_NODES, msgpq_cmp_pri, msgpq_get_pri, msgpq_set_pri, msgpq_get_pos, msgpq_set_pos)) == NULL) {
        syslog(LOG_ERR, "Could not initialize message priority queue. Error: %s\n", strerror(errno));
        return -1;
    }

    /* Start message pq thread */
    if(pthread_create(&instrument->message_scheduler_pq_thread, NULL, instrument_seq_pq, (void*)instrument) != 0) {
        syslog(LOG_ERR, "Could not initialize message scheduler pq thread. Error: %s\n", strerror(errno));
        return -1;
    }

    return 0;
}

//////////////////////
// SECTION: JACKAPI //
//////////////////////

/* JACK API & AUDIO I/O
 *********************/
int astrid_jack_ringbuffer_write_block(jack_ringbuffer_t * rb, lpbuffer_t * buf, size_t numframes) {
    size_t i, pos;
    int c;
    float writeval;
    float * writep;
    // write frame by frame
    for(i=0; i < numframes; i++) {
        for(c=0; c < buf->channels; c++) {
            pos = buf->pos + i;
            while(pos >= buf->length) pos -= buf->length;
            writeval = buf->data[pos * buf->channels + c];
            writep = &writeval;
            if(jack_ringbuffer_write(rb, (void*)writep, sizeof(float)) < sizeof(float)) {
                syslog(LOG_WARNING, "overrun writing to jack ringbuffer\n");
            }
        }
    }
    buf->pos += numframes;
    while(buf->pos >= buf->length) buf->pos -= buf->length;
    return 0;
}

int astrid_jack_ringbuffer_read_block(jack_ringbuffer_t * rb, lpbuffer_t * buf, size_t numframes) {
    size_t i, pos;
    int c;
    float * valp;
    float val = 0.f;
    // read frame by frame
    for(i=0; i < numframes; i++) {
        pos = i;
        while(pos >= buf->length) {
            pos -= buf->length;
        }

        for(c=0; c < buf->channels; c++) {
            valp = &val;
            jack_ringbuffer_read(rb, (void*)valp, sizeof(float));
            buf->data[pos * buf->channels + c] = (lpfloat_t)val;
        }
    }
    return 0;
}

// Stream callback for Python instruments with ugen graphs
int astrid_instrument_python_graph_stream_callback(size_t blocksize, float ** input, float ** output, void * instrument_ptr) {
    lpinstrument_t * instrument = (lpinstrument_t *)instrument_ptr;

    if (!instrument || !instrument->current_graph) {
        return 0;
    }

    volatile astrid_graph_t * graph = instrument->current_graph;

    // If no graph or empty graph, output silence
    if (graph->num_nodes == 0) return 0;

    // Update all external params from session (per-block update)
    // Check session for external parameter values and update param_values cache
    // NOTE: Default parameter values are written to session during graph creation, so
    // get_float_from_hash will return the correct initial values (e.g., gain=1.0).
    // External controllers (MIDI, etc.) can update these values via set_graph_param.
    for (size_t n = 0; n < graph->num_nodes; n++) {
        ugen_t * u = graph->nodes[n];
        if (u == NULL) continue;

        for (int p = 0; p < NUM_UGEN_PARAMS; p++) {
            // Skip if parameter is connected to a modulation source
            if (u->param_sources[p] != NULL) continue;

            // Read from session if hash exists (returns 0.0 if key not found)
            if (u->param_hashes[p] != 0) {
                u->param_values[p] = AstridSession.get_float_from_hash(&instrument->session, u->param_hashes[p]);
            }
        }
    }
    
    // Process the ugen graph frame by frame
    for (size_t i = 0; i < blocksize; i++) {
        // First pass: feed input samples to ADC ugens and output buffer to MIX ugens
        for (size_t n = 0; n < graph->num_nodes; n++) {
            if (graph->nodes[n] != NULL) {
                if (graph->nodes[n]->type == UGEN_ADC) {
                    // Update channel from parameter before reading it
                    lpfloat_t channel_param = lpugen_get_param(graph->nodes[n], UPARAM_CHANNEL);
                    graph->nodes[n]->params.adc.channel = (int)channel_param;

                    // Feed external audio input from JACK
                    int input_channel = graph->nodes[n]->params.adc.channel;
                    if (input_channel < instrument->input_channels) {
                        lpugen_adc_set_input_sample(graph->nodes[n], input[input_channel][i]);
                    }
                } else if (graph->nodes[n]->type == UGEN_MIX) {
                    // Update channel from parameter before reading it
                    lpfloat_t channel_param = lpugen_get_param(graph->nodes[n], UPARAM_CHANNEL);
                    graph->nodes[n]->params.mix.channel = (int)channel_param;

                    // Feed from output buffer (which contains async mixer output from JACK callback)
                    int mix_channel = graph->nodes[n]->params.mix.channel;
                    if (mix_channel < instrument->output_channels) {
                        lpugen_mix_set_input_sample(graph->nodes[n], output[mix_channel][i]);
                    }
                }
            }
        }
        
        // Second pass: process all ugens in the graph
        // The ugens already have their connections set up via LPUgen.connect()
        for (size_t n = 0; n < graph->num_nodes; n++) {
            if (graph->nodes[n] != NULL) {
                LPUgen.process(graph->nodes[n]);
            }
        }

        // Third pass: collect output from DAC ugens (per-channel routing)
        for (int c = 0; c < instrument->output_channels; c++) {
            lpfloat_t channel_sample = 0.0f;

            // Sum all DAC ugens targeting this output channel
            for (size_t n = 0; n < graph->num_nodes; n++) {
                if (graph->nodes[n] != NULL && graph->nodes[n]->type == UGEN_DAC) {
                    if (graph->nodes[n]->params.dac.channel == c) {
                        // DAC ugen already processed, just get its output
                        channel_sample += graph->nodes[n]->output;
                    }
                }
            }

            output[c][i] = channel_sample * graph->output_gain;
        }
    }

    return 0;
}

int astrid_instrument_jack_callback(jack_nframes_t nframes, void * arg) {
    lpinstrument_t * instrument = (lpinstrument_t *)arg;
    float * output_channels[instrument->output_channels];
    float * input_channels[instrument->input_channels];
    float * writep;
    lpfloat_t volume = instrument->initial_volume;
    size_t i, blocksize_in_bytes, rb_written, space_available, now_ticks=0;
    int c;

    if(!instrument->is_running) return 0;

    if(lpscheduler_get_now_ticks(&now_ticks, instrument->samplerate) < 0) {
        syslog(LOG_ERR, "Error getting now ticks\n");
        return 0;
    }

    if(!instrument->has_been_initialized) {
        LPRand.preseed();
        instrument->has_been_initialized = 1;
    }

    if(instrument->graph_swap_ready) {
        instrument->current_graph = instrument->new_graph;
        instrument->graph_swap_ready = 0;
        instrument->graph_cleanup_ready = 1;
        instrument->new_graph = NULL;
    }

    for(c=0; c < instrument->input_channels; c++) {
        input_channels[c] = (float *)jack_port_get_buffer(instrument->inports[c], nframes);
    }
    for(c=0; c < instrument->output_channels; c++) {
        output_channels[c] = (float *)jack_port_get_buffer(instrument->outports[c], nframes);
        memset(output_channels[c], 0, nframes * sizeof(float));
    }
    space_available = jack_ringbuffer_write_space(instrument->jack_input_ringbuffer);
    blocksize_in_bytes = nframes * sizeof(float) * instrument->input_channels;
    if(space_available >= blocksize_in_bytes) {
        for(i=0; i < (size_t)nframes; i++) {
            for(c=0; c < instrument->input_channels; c++) {
                writep = &input_channels[c][i];
                rb_written = jack_ringbuffer_write(instrument->jack_input_ringbuffer, (void*)writep, sizeof(float));
                if(rb_written < sizeof(float)) {
                    syslog(LOG_WARNING, "astrid_instrument_jack_callback: overrun writing into input ringbuffer. (%ld)\n", rb_written);
                }
            }
        }
    } 

    /* mix in async renders */
    if(instrument->async_mixer != NULL) {
        for(i=0; i < (size_t)nframes; i++) {
            lpscheduler_tick(instrument->async_mixer);
            for(c=0; c < instrument->output_channels; c++) {
                output_channels[c][i] += instrument->async_mixer->current_frame[c];
            }
        }
    }

    if(instrument->stream != NULL) {
        if(instrument->stream((size_t)nframes, input_channels, output_channels, (void *)instrument) < 0) {
            return -1;
        }
    }

    /* clamp output and apply post fader volume  */
    //volume = astrid_session_get_param_float(&instrument->session, instrument->param_volume, instrument->initial_volume);
    for(c=0; c < instrument->output_channels; c++) {
        for(i=0; i < (size_t)nframes; i++) {
            output_channels[c][i] = fmax(-1.f, fmin(output_channels[c][i] * volume, 1.f));
        }
    }

    return 0;
}

/* INSTRUMENT 
 * LIFECYCLE
 * **********/
void * instrument_cleanup_thread(void * arg) {
    lpinstrument_t * instrument = (lpinstrument_t *)arg;
    /* free buffers that are done playing */
    syslog(LOG_INFO, "%s cleanup thread: started\n", instrument->name);
    while(instrument->is_running) {
        if(scheduler_cleanup_nursery(&instrument->session, instrument->async_mixer) < 0) {
            syslog(LOG_ERR, "%s cleanup thread: Could not cleanup nursery. Error: (%d) %s\n", instrument->name, errno, strerror(errno));
        }
        usleep((useconds_t)10000);
    }

    syslog(LOG_INFO, "%s cleanup thread: final cleanup!\n", instrument->name);
    if(scheduler_cleanup_nursery(&instrument->session, instrument->async_mixer) < 0) {
        syslog(LOG_ERR, "%s cleanup thread: Could not cleanup nursery. Error: (%d) %s\n", instrument->name, errno, strerror(errno));
    }

    syslog(LOG_INFO, "%s cleanup thread: all done here\n", instrument->name);
    return NULL;
}

lpinstrument_config_t astrid_instrument_init_config(char * name) {
    lpinstrument_config_t config = {0};
    config.name = name;
    config.input_channels = 2;
    config.output_channels = 2;
    config.initial_volume = 1.f;
    config.requested_samplerate = -1.f;
    config.is_interactive = 1;
    config.adc_length = 60.f;
    config.resampler_length = 60.f;
    return config;
}

unsigned char * astrid_instrument_serialize_config(lpinstrument_config_t * config, size_t * outlen) {
    *outlen = sizeof(lpinstrument_config_t);
    unsigned char * bytes = (unsigned char *)malloc(*outlen);
    memcpy(bytes, config, *outlen);
    return bytes;
}

lpinstrument_config_t astrid_instrument_deserialize_config(unsigned char * bytes, size_t size) {
    lpinstrument_config_t config;
    memcpy(&config, bytes, size);
    
    config.name = NULL;
    for(int i = 0; i < ASTRID_MAX_MIDI_DEVICES; i++) {
        config.midiin_device_names[i] = NULL;
        config.midiout_device_names[i] = NULL;
    }
    config.num_midiin_device_names = 0;
    config.num_midiout_device_names = 0;

    config.stream_callback = NULL;
    config.renderer_callback = NULL;
    config.update_callback = NULL;
    config.trigger_callback = NULL;
    config.midi_callback = NULL;
    config.post_init_callback = NULL;
    config.ctx = NULL;
    
    return config;
}

lpinstrument_t * astrid_instrument_start_from_config(lpinstrument_config_t config) {
    lpinstrument_t * instrument;
    instrument = (lpinstrument_t *)LPMemoryPool.alloc(1, sizeof(lpinstrument_t));
    memset(instrument, 0, sizeof(lpinstrument_t));
    return astrid_instrument_start_from_config_with_ptr(config, instrument);
}

lpinstrument_t * astrid_instrument_start_from_config_with_ptr(lpinstrument_config_t config, lpinstrument_t * instrument) {
    struct sigaction shutdown_action;
    jack_status_t jack_status;
    jack_options_t jack_options = JackNullOption;
    char outport_name[50];
    char inport_name[50];
    int c=0;
    const char ** ports;
    int port_count=0;
#if 0
    struct sched_param sparam;

    /**
     * 0) PROCESS TUNING STUFF
     **************************/
    // enable realtime scheduling
    sparam.sched_priority = sched_get_priority_max(SCHED_RR);
    if(sched_setscheduler(0, SCHED_RR, &sparam) < 0) {
        syslog(LOG_WARNING, "%s Could not set realtime priority. Error: (%d) %s\n", config.name, errno, strerror(errno));
    }

    // set nice
    errno = 0; // nice can return -1 on success...
    if(nice(-11) < 0) {
        if(errno != 0) syslog(LOG_WARNING, "%s Could not set nice level. Error: (%d) %s\n", config.name, errno, strerror(errno));
    }
#endif

    // Lock memory pages to prevent swapping
    mlockall(MCL_CURRENT | MCL_FUTURE);

    /**
     * 1) SET INSTRUMENT PARAMS
     ****************************/
#ifdef DEBUG
    setlogmask(LOG_UPTO(LOG_DEBUG));
#else
    setlogmask(LOG_UPTO(LOG_INFO));
#endif
    openlog(config.name, LOG_PID, LOG_USER);
    syslog(LOG_ERR, ":: 1 :: INIT / SET PARAMS\n");
    syslog(LOG_ERR, "starting %s instrument...\n", config.name);

    instrument->name = config.name;
    instrument->initial_volume = config.initial_volume;
    instrument->input_channels = config.input_channels;
    instrument->output_channels = config.output_channels;
    instrument->context = config.ctx;
    instrument->is_interactive = config.is_interactive;

    instrument->stream = config.stream_callback;
    instrument->renderer = config.renderer_callback;
    instrument->update = config.update_callback;
    instrument->trigger = config.trigger_callback;
    instrument->midievent = config.midi_callback;
    instrument->ext_relay_enabled = config.ext_relay_enabled;
    instrument->is_ready = 1;
    instrument->is_running = 1;
    
    instrument->udp_port = config.udp_port;
    instrument->udp_is_enabled = config.udp_port > 0;

    //instrument->num_midiin_devices = config.num_midiin_device_names;
    //instrument->num_midiout_devices = config.num_midiout_device_names;

    if(pthread_mutex_init(&instrument->emergency_brake, NULL) < 0) {
        syslog(LOG_ERR, "%s Could not init emergency brake mutex. Error: %s\n", instrument->name, strerror(errno));
        exit(1);
    }

    // Seed the random number generator
    LPRand.preseed();

    // Prepare the internal message structs
    init_instrument_message(&instrument->msg, instrument->name);
    init_instrument_message(&instrument->cmd, instrument->name);

    // Initialize the built-in control param hashes
    instrument->param_volume = lphashstr("volume"); 

    /* 2) SET SIGNAL HANDLERS FOR SHUTDOWN
     ***************************************/
    syslog(LOG_ERR, ":: 2 :: SIGNAL HANDLERS\n");
    shutdown_action.sa_handler = handle_instrument_shutdown;
    sigemptyset(&shutdown_action.sa_mask);
    shutdown_action.sa_flags = SA_RESTART; // Prevent open, read, write etc from EINTR
    astrid_instrument_is_running = &instrument->is_running;

    if(sigaction(SIGINT, &shutdown_action, NULL) == -1) {
        syslog(LOG_ERR, "%s Could not init SIGINT signal handler. Error: %s\n", instrument->name, strerror(errno));
        exit(1);
    }

    if(sigaction(SIGTERM, &shutdown_action, NULL) == -1) {
        syslog(LOG_ERR, "%s Could not init SIGTERM signal handler. Error: %s\n", instrument->name, strerror(errno));
        exit(1);
    }

    /**
     * 3) GET MIDI DEVICE INFO 
     ****************************/
    syslog(LOG_ERR, ":: 3 :: MIDI DEVICE INFO\n");
    // Initialize MIDI input device arrays
    instrument->num_midiin_devices = 0;
    for(int i = 0; i < ASTRID_MAX_MIDI_DEVICES; i++) {
        instrument->midiin_device_ids[i] = -1;
        instrument->midiout_device_ids[i] = -1;
    }
    
    // Set up MIDI input devices from config
    for(int i = 0; i < config.num_midiin_device_names && i < ASTRID_MAX_MIDI_DEVICES; i++) {
        if(config.midiin_device_names[i] != NULL) {
            instrument->midiin_device_ids[i] = lpmidi_get_device_id_by_name(config.midiin_device_names[i]);
            if(instrument->midiin_device_ids[i] >= 0) {
                instrument->num_midiin_devices++;
            }
        }
    }

    // Set up MIDI ouput devices from config
    for(int i = 0; i < config.num_midiout_device_names && i < ASTRID_MAX_MIDI_DEVICES; i++) {
        if(config.midiout_device_names[i] != NULL) {
            instrument->midiout_device_ids[i] = lpmidi_get_device_id_by_name(config.midiout_device_names[i]);
            if(instrument->midiout_device_ids[i] >= 0) {
                instrument->num_midiout_devices++;
                memcpy(instrument->midiout_device_names[i], config.midiout_device_names[i], NAME_MAX);
                syslog(LOG_INFO, "set midiout device name for %s\n", instrument->midiout_device_names[i]);
                syslog(LOG_INFO, "Set MIDI output device ID (%d) for %s (index %d)\n", 
                       instrument->midiout_device_ids[i], config.midiout_device_names[i], i);
            }
        }
    }

    /**
     * 4) SET UP THE (LMDB) SESSION DB
     ***********************************/
    syslog(LOG_ERR, ":: 4 :: LMDB\n");
    astrid_session_create(&instrument->session, instrument->name);

    /**
     * 5) SET UP THE JACK CLIENT 
     *****************************/
    instrument->inports = (jack_port_t **)calloc(config.input_channels, sizeof(jack_port_t *));
    instrument->outports = (jack_port_t **)calloc(config.output_channels, sizeof(jack_port_t *));
    instrument->jack_client = jack_client_open(config.name, jack_options, &jack_status, NULL);

    syslog(LOG_ERR, ":: 5 :: JACK STATUS %d\n", (int)jack_status);
    if(instrument->jack_client == NULL) {
        syslog(LOG_ERR, "%s Could not open jack client. Client is NULL: %s\n", instrument->name, strerror(errno));
        if((jack_status & JackServerFailed) == JackServerFailed) {
            syslog(LOG_ERR, "%s Could not open jack client. Jack server failed with status %2.0x\n", instrument->name, jack_status);
        } else {
            syslog(LOG_ERR, "%s Could not open jack client. Unknown error: %s\n", instrument->name, strerror(errno));
        }
        goto astrid_instrument_shutdown_with_error;
    }
    if((jack_status & JackServerStarted) == JackServerStarted) {
        syslog(LOG_INFO, "Jack server started!\n");
    }

    // Get the samplerate from JACK 
    instrument->samplerate = (lpfloat_t)jack_get_sample_rate(instrument->jack_client);

    // 5a) SET UP THINGS THAT NEED TO KNOW THE SAMPLERATE
    // scheduler
    syslog(LOG_ERR, ":: 5a :: SCHEDULER CREATE\n");
    instrument->async_mixer = scheduler_create(1, instrument->output_channels, instrument->samplerate);

    // ringbuffers
    instrument->jack_input_ringbuffer = jack_ringbuffer_create(config.input_channels * sizeof(lpfloat_t) * instrument->samplerate * config.adc_length * 4);
    instrument->jack_output_ringbuffer = jack_ringbuffer_create(config.output_channels * sizeof(lpfloat_t) * instrument->samplerate * config.resampler_length);

    snprintf(instrument->adcname, PATH_MAX, "%s-adc", instrument->name);
    size_t adcsize = config.adc_length * instrument->input_channels * instrument->samplerate * sizeof(lpfloat_t) + sizeof(lpbuffer_t);
    lpbuffer_t * empty = LPBuffer.create(config.adc_length * instrument->samplerate, instrument->input_channels, instrument->samplerate);

    syslog(LOG_ERR, ":: 5b :: RINGBUFFER INIT name=%s size=%ld\n", instrument->adcname, adcsize);
    if(astrid_session_register_shared_resource(&instrument->session, instrument->adcname, empty, ASTRID_TYPE_BUFFER, adcsize) < 0) {
        syslog(LOG_CRIT, "Could not create ADC buffer. Shutting down.\n");
        LPBuffer.destroy(empty);
        goto astrid_instrument_shutdown_with_error;
    }
    LPBuffer.destroy(empty); // So wasteful! But the LMDB backend always copies anyway.

    // Initialize local dual graphs for atomic swapping
    syslog(LOG_ERR, ":: 5c :: GRAPH INIT\n");
    
    // Initialize graph pointers
    instrument->current_graph = NULL;
    instrument->new_graph = NULL;
    instrument->graph_swap_ready = 0;
    instrument->graph_cleanup_ready = 0;
    
 #if 0
    // resampling ringbuffer
    syslog(LOG_ERR, ":: 5c :: RESAMPLER INIT\n");
    snprintf(instrument->resamplername, PATH_MAX, "%s-resampler", instrument->name);
    size_t resamplersize = config.resampler_length * instrument->output_channels * instrument->samplerate;
    if(astrid_instrument_register_shared_resource(instrument, instrument->resamplername, NULL, ASTRID_TYPE_BUFFER, resamplersize) < 0) {
        syslog(LOG_CRIT, "Could not create resampler buffer. Shutting down.\n");
        goto astrid_instrument_shutdown_with_error;
    }
#endif
                            
    // post-init callback
    if(config.post_init_callback != NULL) {
        if(config.post_init_callback((void *)instrument) < 0) {
            syslog(LOG_ERR, "Post init callback failed. Error: %s\n", strerror(errno));
        }
    }

    /**
     * 6) OPEN MESSAGE INBOXES (TODO: network inputs, etc)
     **************************/
    syslog(LOG_ERR, ":: 6 :: OPEN MSG QUEUES\n");
    char posix_inbox_qname[LPMAXKEY];
    snprintf(posix_inbox_qname, LPMAXKEY, "/%s-posix-inbox", instrument->name);
    if((instrument->session.inbox_messages = astrid_posix_msgq_open(posix_inbox_qname)) == (mqd_t) -1) {
        syslog(LOG_CRIT, "Could not open msgq for instrument %s. Error: %s\n", instrument->name, strerror(errno));
        return NULL;
    }

    /**
     * 7) START THREADS
     ********************/
    syslog(LOG_ERR, ":: 7 :: START THREADS\n");
    // message scheduler
    if(astrid_instrument_seq_start(instrument) < 0) {
        syslog(LOG_CRIT, "Could not start message sequence threads for instrument %s. Error: %s\n", instrument->name, strerror(errno));
        return NULL;
    }

    // async / shared locking ringbuffers, things that can't happen in the audio thread
    pthread_attr_t attr;
    pthread_attr_init(&attr);
    pthread_attr_setstacksize(&attr, 16 * 1024 * 1024); // try increasing stack size to check correlation w/crashes
    if(pthread_create(&instrument->audio_slow_lane_thread, &attr, instrument_audio_slow_lane_thread, (void*)instrument) != 0) {
        syslog(LOG_ERR, "Could not initialize instrument audio slow lane thread. Error: %s\n", strerror(errno));
        return NULL;
    }

    // FIXME start a new MIXER thread, which has an arbitrary latency, and streams back to the main audiocallback
    // like the slow lane thread

    // message handler
    if(pthread_create(&instrument->message_feed_thread, NULL, instrument_message_thread, (void*)instrument) != 0) {
        syslog(LOG_ERR, "Could not initialize instrument message thread. Error: %s\n", strerror(errno));
        return NULL;
    }

    // MIDI input listener
    if(instrument->num_midiin_devices > 0) {
        if(pthread_create(&instrument->midi_listener_thread, NULL, instrument_midi_listener_thread, (void*)instrument) != 0) {
            syslog(LOG_ERR, "Could not initialize instrument midi listener thread. Error: %s\n", strerror(errno));
            return NULL;
        }
    }

    lpmidiout_device_t midiout_device = {0};
    midiout_device.instrument = instrument;

    // MIDI output relay
    if(instrument->num_midiout_devices > 0) {
        for(int i=0; i < instrument->num_midiout_devices; i++) {
            midiout_device.device_id = instrument->midiout_device_ids[i];
            midiout_device.device_index = i;
            if(pthread_create(&instrument->midi_output_threads[i], NULL, instrument_midi_output_thread, (void*)(&midiout_device)) != 0) {
                syslog(LOG_ERR, "Could not initialize instrument midi output thread. Error: %s\n", strerror(errno));
                return NULL;
            }
        }
    }

    // autotrigger
    if(pthread_create(&instrument->autotrigger_thread, NULL, instrument_autotrigger_thread, (void*)instrument) != 0) {
        syslog(LOG_ERR, "Could not initialize instrument autotrigger thread. Error: %s\n", strerror(errno));
        return NULL;
    }

    // posix inbox
    if(pthread_create(&instrument->posix_inbox_thread, NULL, instrument_posix_inbox_thread, (void*)instrument) != 0) {
        syslog(LOG_ERR, "Could not initialize instrument posix inbox thread. Error: %s\n", strerror(errno));
        return NULL;
    }

    // cleanup
    if(pthread_create(&instrument->cleanup_thread, NULL, instrument_cleanup_thread, (void*)instrument) != 0) {
        syslog(LOG_ERR, "Could not initialize instrument cleanup thread. Error: %s\n", strerror(errno));
        return NULL;
    }

    // UDP listener thread
    if(instrument->udp_is_enabled) {
        if(pthread_create(&instrument->udp_listener_thread, NULL, instrument_udp_listener_thread, (void*)instrument) != 0) {
            syslog(LOG_ERR, "Could not initialize UDP listener thread. Error: %s\n", strerror(errno));
            return NULL;
        }
    }

    /**
     * 8) ACTIVATE THE JACK CLIENT / START AUDIO / CONNECT PORTS
     *************************************************************/
    // Set the main jack callback which always runs: maybe there is an analysis-only use to support too?
    syslog(LOG_ERR, ":: 8a :: SET JACK PROCESS CALLBACK\n");
    jack_set_process_callback(instrument->jack_client, astrid_instrument_jack_callback, (void *)instrument);
    for(c=0; c < instrument->output_channels; c++) {
        snprintf(outport_name, sizeof(outport_name), "out%d", c);
        instrument->outports[c] = jack_port_register(instrument->jack_client, outport_name, JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0);
    }

    for(c=0; c < instrument->input_channels; c++) {
        snprintf(inport_name, sizeof(inport_name), "in%d", c);
        instrument->inports[c] = jack_port_register(instrument->jack_client, inport_name, JACK_DEFAULT_AUDIO_TYPE, JackPortIsInput, 0);
    }

    // Request some ports from jack
    for(c=0; c < instrument->output_channels; c++) {
        if(instrument->outports[c] == NULL) {
            syslog(LOG_ERR, "No more JACK output ports available, shutting down...\n");
            goto astrid_instrument_shutdown_with_error;
        }
    }

    for(c=0; c < instrument->input_channels; c++) {
        if(instrument->inports[c] == NULL) {
            syslog(LOG_ERR, "No more JACK input ports available, shutting down...\n");
            goto astrid_instrument_shutdown_with_error;
        }
    }

    syslog(LOG_ERR, ":: 8b :: ACTIVATE JACK & CONNECT PORTS\n");
    if(jack_activate(instrument->jack_client) != 0) {
        syslog(LOG_ERR, "%s Could not activate JACK client, shutting down...\n", instrument->name);
        goto astrid_instrument_shutdown_with_error;
    }

    // connect the jack ports FIXME would be nice to optionally do routing in the instrument callback
    if((ports = jack_get_ports(instrument->jack_client, NULL, NULL, JackPortIsPhysical|JackPortIsOutput)) == NULL) {
		syslog(LOG_CRIT, "Cannot find any physical capture ports");
        goto astrid_instrument_shutdown_with_error;
	}

    port_count = 0;
    while(ports[port_count] != NULL) {
        if(port_count < instrument->input_channels) {
            syslog(LOG_ERR, "CONNECTING INPUT CHANNEL %d\n", port_count);
            if(jack_connect(instrument->jack_client, ports[port_count], jack_port_name(instrument->inports[port_count]))) {
                syslog(LOG_NOTICE, "Could not connect input port %d. (There probably aren't enough available inputs on this device.)\n", port_count);
            }
        }
        port_count += 1;
    }
	free(ports);
	
	if((ports = jack_get_ports(instrument->jack_client, NULL, NULL, JackPortIsPhysical|JackPortIsInput)) == NULL) {
		syslog(LOG_CRIT, "Cannot find any physical playback ports");
        goto astrid_instrument_shutdown_with_error;
	}

    port_count = 0;
    while(ports[port_count] != NULL) {
        if(port_count < instrument->output_channels) {
            syslog(LOG_ERR, "CONNECTING OUTPUT CHANNEL %d\n", port_count);
            if(jack_connect(instrument->jack_client, jack_port_name(instrument->outports[port_count]), ports[port_count])) {
                syslog(LOG_NOTICE, "Could not connect output port %d. (There probably aren't enough available outputs on this device.)\n", c);
            }
        }
        port_count += 1;
    }
	free(ports);

    /**
     * 9) SET UP REPL AND/OR EXEC POST INIT CALLBACK
     ************************************************/
    syslog(LOG_ERR, ":: 9 :: FINALIZE AND POST INIT CALLBACK\n");
    if(instrument->is_interactive) {
        linenoiseHistoryLoad("history.txt"); // FIXME this goes in the instrument config dir / or share?
    }

    syslog(LOG_ERR, ":: 10 :: STARTED\n");
    return instrument;

astrid_instrument_shutdown_with_error:
    instrument->is_running = 0;
    jack_client_close(instrument->jack_client);
    closelog();
    return NULL;
}

lpinstrument_t * astrid_instrument_start(
    char * name, 
    int input_channels, 
    int output_channels, 
    int ext_relay_enabled,
    double adc_length,
    double resampler_length,
    void * ctx,
    char ** midiin_device_names,
    int num_midiin_device_names,
    char ** midiout_device_names,
    int num_midiout_device_names,
    int (*stream)(size_t blocksize, float ** input, float ** output, void * instrument),
    int (*renderer)(void * instrument),
    int (*update)(void * instrument, char * key, char * val),
    int (*trigger)(void * instrument)
) {
    syslog(LOG_WARNING, "astrid_instrument_start is deprecated.\nuse astrid_instrument_start_from_config instead.");
    lpinstrument_config_t config = astrid_instrument_init_config(name);
    config.input_channels = input_channels;
    config.output_channels = output_channels;
    config.ext_relay_enabled = ext_relay_enabled;
    config.adc_length = adc_length;
    config.resampler_length = resampler_length;
    config.ctx = ctx;
    // Set up MIDI input device names array
    config.num_midiin_device_names = num_midiin_device_names;
    for(int i = 0; i < num_midiin_device_names && i < ASTRID_MAX_MIDI_DEVICES; i++) {
        config.midiin_device_names[i] = midiin_device_names[i];
    }
    config.num_midiout_device_names = num_midiout_device_names;
    for(int i = 0; i < num_midiout_device_names && i < ASTRID_MAX_MIDI_DEVICES; i++) {
        config.midiout_device_names[i] = midiout_device_names[i];
    }
    config.stream_callback = stream;
    config.renderer_callback = renderer;
    config.update_callback = update;
    config.trigger_callback = trigger;
    return astrid_instrument_start_from_config(config);
}

int astrid_instrument_stop(lpinstrument_t * instrument) {
    int c, ret;

    syslog(LOG_INFO, "%s instrument shutting down and cleaning up...\n", instrument->name);

    if(instrument->is_interactive) {
        linenoiseEditStop(&instrument->cmdstate);
    }

    syslog(LOG_INFO, "Sending shutdown message to threads and queues...\n");
    if(instrument->is_waiting) {
        instrument->msg.type = LPMSG_SHUTDOWN;

        // Send shutdown message to internal queue
        char qname[LPMAXKEY] = {0};
        snprintf(qname, LPMAXKEY, "%s-msgq", instrument->name);
        lpmsgq_t * msgq = astrid_msgq_produce(&instrument->session, qname, MAINMQ_MAIN_PRODUCER);
        if(msgq != NULL) {
            if(astrid_msgq_write(msgq, &instrument->msg) < 0) {
                syslog(LOG_ERR, "astrid instrument message thread cleanup: Could not send shutdown message...\n");
            }
            astrid_msgq_close(msgq);
        }

        // Also send shutdown to POSIX inbox to unblock posix_inbox_thread
        char posix_inbox_qname[LPMAXKEY];
        snprintf(posix_inbox_qname, LPMAXKEY, "/%s-posix-inbox", instrument->name);
        send_posix_message(posix_inbox_qname, instrument->msg);
    }

    if((ret = pthread_join(instrument->message_feed_thread, NULL)) != 0) {
        if(ret == EINVAL) syslog(LOG_ERR, "EINVAL\n");
        if(ret == EDEADLK) syslog(LOG_ERR, "DEADLOCK\n");
        if(ret == ESRCH) syslog(LOG_ERR, "ESRCH\n");
        syslog(LOG_ERR, "Error while attempting to join with message feed thread. Ret: %d Errno: %d (%s)\n", ret, errno, strerror(ret));
    }

    if((ret = pthread_join(instrument->audio_slow_lane_thread, NULL)) != 0) {
        if(ret == EINVAL) syslog(LOG_ERR, "EINVAL\n");
        if(ret == EDEADLK) syslog(LOG_ERR, "DEADLOCK\n");
        if(ret == ESRCH) syslog(LOG_ERR, "ESRCH\n");
        syslog(LOG_ERR, "Error while attempting to join with audio slow lane thread. Ret: %d Errno: %d (%s)\n", ret, errno, strerror(ret));
    }

    if((ret = pthread_join(instrument->message_scheduler_pq_thread, NULL)) != 0) {
        if(ret == EINVAL) syslog(LOG_ERR, "EINVAL\n");
        if(ret == EDEADLK) syslog(LOG_ERR, "DEADLOCK\n");
        if(ret == ESRCH) syslog(LOG_ERR, "ESRCH\n");
        syslog(LOG_ERR, "Error while attempting to join with message scheduler pq thread. Ret: %d Errno: %d (%s)\n", ret, errno, strerror(ret));
    }

    if((ret = pthread_join(instrument->autotrigger_thread, NULL)) != 0) {
        if(ret == EINVAL) syslog(LOG_ERR, "EINVAL\n");
        if(ret == EDEADLK) syslog(LOG_ERR, "DEADLOCK\n");
        if(ret == ESRCH) syslog(LOG_ERR, "ESRCH\n");
        syslog(LOG_ERR, "Error while attempting to join with autotrigger thread. Ret: %d Errno: %d (%s)\n", ret, errno, strerror(ret));
    }

    if((ret = pthread_join(instrument->posix_inbox_thread, NULL)) != 0) {
        if(ret == EINVAL) syslog(LOG_ERR, "EINVAL\n");
        if(ret == EDEADLK) syslog(LOG_ERR, "DEADLOCK\n");
        if(ret == ESRCH) syslog(LOG_ERR, "ESRCH\n");
        syslog(LOG_ERR, "Error while attempting to join with posix_inbox thread. Ret: %d Errno: %d (%s)\n", ret, errno, strerror(ret));
    }

    if((ret = pthread_join(instrument->cleanup_thread, NULL)) != 0) {
        if(ret == EINVAL) syslog(LOG_ERR, "EINVAL\n");
        if(ret == EDEADLK) syslog(LOG_ERR, "DEADLOCK\n");
        if(ret == ESRCH) syslog(LOG_ERR, "ESRCH\n");
        syslog(LOG_ERR, "Error while attempting to join with cleanup thread. Ret: %d Errno: %d (%s)\n", ret, errno, strerror(ret));
    }

    if(instrument->num_midiin_devices > 0) {
        if((ret = pthread_join(instrument->midi_listener_thread, NULL)) != 0) {
            if(ret == EINVAL) syslog(LOG_ERR, "EINVAL\n");
            if(ret == EDEADLK) syslog(LOG_ERR, "DEADLOCK\n");
            if(ret == ESRCH) syslog(LOG_ERR, "ESRCH\n");
            syslog(LOG_ERR, "Error while attempting to join with midi listener thread. Ret: %d Errno: %d (%s)\n", ret, errno, strerror(ret));
        }
    }

    if(instrument->num_midiout_devices > 0) {
        for(int i=0; i < instrument->num_midiout_devices; i++) {
            if((ret = pthread_join(instrument->midi_output_threads[i], NULL)) != 0) {
                if(ret == EINVAL) syslog(LOG_ERR, "EINVAL\n");
                if(ret == EDEADLK) syslog(LOG_ERR, "DEADLOCK\n");
                if(ret == ESRCH) syslog(LOG_ERR, "ESRCH\n");
                syslog(LOG_ERR, "Error while attempting to join with midi output thread. [i=%d] Ret: %d Errno: %d (%s)\n", i, ret, errno, strerror(ret));
            }
        }
    }

    if(instrument->udp_is_enabled) {
        if((ret = pthread_join(instrument->udp_listener_thread, NULL)) != 0) {
            if(ret == EINVAL) syslog(LOG_ERR, "EINVAL\n");
            if(ret == EDEADLK) syslog(LOG_ERR, "DEADLOCK\n");
            if(ret == ESRCH) syslog(LOG_ERR, "ESRCH\n");
            syslog(LOG_ERR, "Error while attempting to join with UDP listener thread. Ret: %d Errno: %d (%s)\n", ret, errno, strerror(ret));
        }
    }

    if(instrument->session.inbox_messages != (mqd_t) -1) astrid_posix_msgq_close(instrument->session.inbox_messages);
    char posix_inbox_qname[LPMAXKEY];
    snprintf(posix_inbox_qname, LPMAXKEY, "/%s-posix-inbox", instrument->name);
    astrid_posix_msgq_unlink(posix_inbox_qname);

    for(c=0; c < instrument->output_channels; c++) {
        jack_port_unregister(instrument->jack_client, instrument->outports[c]);
    }
    for(c=0; c < instrument->input_channels; c++) {
        jack_port_unregister(instrument->jack_client, instrument->inports[c]);
    }

    jack_ringbuffer_free(instrument->jack_input_ringbuffer);
    jack_ringbuffer_free(instrument->jack_output_ringbuffer);

    jack_client_close(instrument->jack_client);

    /* and the jack i/o buffers */
    free(instrument->inports);
    free(instrument->outports);

    astrid_session_close(&instrument->session);

    /* save the history FIXME save the path on the instrument */
    linenoiseHistorySave("history.txt");
    linenoiseEditStop(&instrument->cmdstate);

    if(instrument->async_mixer != NULL) scheduler_destroy(instrument->async_mixer);

    /* cleanup the pq memory */
    pqueue_free(instrument->msgpq);
    free(instrument->pqnodes);

    pthread_mutex_destroy(&instrument->emergency_brake);

    /* poof! */
    free(instrument);

    closelog();
    return 0;
}

int astrid_session_get_or_create_datadir_path(char * dbpath) {
   int ret;
   char xdg_data_home[LPMAXPATH];
   char * user_home;
   char * _xdg_data_home = getenv("XDG_DATA_HOME");
   
   if(_xdg_data_home == NULL) {
       user_home = getenv("HOME");
       ret = snprintf(xdg_data_home, LPMAXPATH, "%s/.local/share", user_home);
       if(ret < 0) return -1;
   } else {
       memcpy(xdg_data_home, _xdg_data_home, LPMAXPATH);
   }

   if(access(xdg_data_home, F_OK) != 0) {
       if(mkdir(xdg_data_home, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH) != 0) {
           return -1;
       }
   }

   ret = snprintf(dbpath, LPMAXPATH, "%s/astrid", xdg_data_home);
   if(ret < 0 || ret >= LPMAXPATH) return -1;
   if(access(dbpath, F_OK) != 0) {
       if(mkdir(dbpath, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH) != 0) {
           return -1;
       }
   }

   return 0;
}

int send_render_to_mixer(lpinstrument_t * instrument, lpbuffer_t * buf) {
    char qname[LPMAXKEY] = {0};
    lpmsgq_t * msgq = NULL;

    //syslog(LOG_INFO, "SEND RENDER serializing buffer with value 10 %f\n", buf->data[10]);

    // Get producer handle for the main message queue
    snprintf(qname, LPMAXKEY, "%s-msgq", instrument->name);
    msgq = astrid_msgq_produce(&instrument->session, qname, MAINMQ_RENDER_CALLBACK_PRODUCER);
    if(msgq == NULL) {
        syslog(LOG_ERR, "send_render_to_mixer: Could not get producer handle for message queue\n");
        return -1;
    }

    if(astrid_session_publish_buffer(&instrument->session, msgq, buf, 0, NULL) < 0) {
        astrid_msgq_close(msgq);
        return -1;
    }

    astrid_msgq_close(msgq);
    return 0;
}

int astrid_instrument_process_command_tick(lpinstrument_t * instrument) {
    char * cmdline;
    size_t cmdlength;

    cmdline = linenoiseEditFeed(&instrument->cmdstate);

    if(cmdline != linenoiseEditMore) {
        // done editing, now process the command
        linenoiseEditStop(&instrument->cmdstate);

        /* NULL means user pressed CTRL-C or CTRL-D with empty line */
        if(cmdline == NULL) {
            return 0;
        }

        /* add the command to the history */
        linenoiseHistoryAdd(cmdline);
        linenoiseHistorySave("history.txt"); // FIXME put this in .local or etc

        cmdlength = strnlen(cmdline, ASTRID_MAX_CMDLINE);

        if(parse_message_from_cmdline(cmdline, cmdlength, &instrument->cmd) < 0) {
            syslog(LOG_ERR, "Could not parse message from cmdline %s\n", cmdline);
            return -1;
        }

        // Send command to internal queue
        char qname[LPMAXKEY] = {0};
        snprintf(qname, LPMAXKEY, "%s-msgq", instrument->name);
        lpmsgq_t * msgq = astrid_msgq_produce(&instrument->session, qname, MAINMQ_MAIN_PRODUCER);
        if(msgq == NULL) {
            syslog(LOG_ERR, "Could not get producer handle for message queue\n");
            return -1;
        }

        if(astrid_msgq_write(msgq, &instrument->cmd) < 0) {
            syslog(LOG_ERR, "Could not send play message...\n");
            astrid_msgq_close(msgq);
            return -1;
        }

        astrid_msgq_close(msgq);

        free(cmdline);
        instrument->is_ready = 1;
    }

    return 0;
}

int astrid_instrument_tick(lpinstrument_t * instrument) {
    struct timeval timeout;
    fd_set readfds;
    int ret;
    size_t prompt_len = strlen("^_- ") + strlen(instrument->name) + strlen(": ") + 1;
    char prompt[LPMAXNAME+10] = {0};

#if 0
    syslog(LOG_INFO, "%s CPU: %f\n", instrument->name, jack_cpu_load(instrument->jack_client));
#endif

    if(instrument->is_interactive == 0) {
        usleep((useconds_t)500);
        return 0;
    }

    snprintf(prompt, prompt_len, "^_- %s: ", instrument->name);

    if(instrument->is_running == 0) {
        linenoiseEditStop(&instrument->cmdstate);
        return 0;
    }

    if(instrument->is_ready) {
        linenoiseEditStart(&instrument->cmdstate, -1, -1, instrument->cmdbuf, sizeof(instrument->cmdbuf), prompt);
        instrument->is_ready = 0;
    }

    FD_ZERO(&readfds);
    FD_SET(instrument->cmdstate.ifd, &readfds);
    timeout.tv_sec = 0;
    timeout.tv_usec = 1000;

    if((ret = select(instrument->cmdstate.ifd+1, &readfds, NULL, NULL, &timeout)) < 0) {
        syslog(LOG_ERR, "astrid_instrument_tick: select error (%d) %s\n", errno, strerror(errno));
        return -1;
    } 

    //syslog(LOG_INFO, "TICK %ld\n", instrument->async_mixer->ticks);
    // Yield for foreground bookkeeping on timeout
    if(ret == 0) return 0;

    return astrid_instrument_process_command_tick(instrument);
}

/* GUI LIFECYCLE
 ***************/
#ifdef ASTRID_GUI_ENABLED
void astrid_gui_draw_text(lpgui_t * gui, cairo_t * cr, int x, int y, const char * text) {
    char buffer[128];
    snprintf(buffer, sizeof(buffer), "%s", text);

    cairo_set_source_rgb(cr, 0, 0, 0);
    cairo_select_font_face(cr, "Sans", CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_NORMAL);
    cairo_set_font_size(cr, gui->font_size);
    cairo_move_to(cr, x, y);
    cairo_show_text(cr, buffer);
}

int astrid_gui_draw_block(cairo_t * cr, int x, int y, int width) {
    cairo_set_source_rgb(cr, 0, 0, 1);
    cairo_set_line_width(cr, 10);
    cairo_move_to(cr, x, y);
    cairo_line_to(cr, x + width, y);
    cairo_stroke(cr);

    return 0;
}

static gboolean astrid_gui_mouse_move_callback(
        GtkWidget * widget, 
        __attribute__((unused)) GdkEventMotion * event, 
        gpointer user_data
    ) {
    lpgui_t * gui = (lpgui_t *)user_data;
    //scale = (float)event->x / 50.f;
    //offset = (float)event->y * 2;
    gui->mouse_x = event->x;
    gui->mouse_y = event->y;
    printf("Mouse moved to: %f, %f\n", gui->mouse_x, gui->mouse_y);
    printf("width %d height: %d\n", gui->width, gui->height);

    gtk_widget_queue_draw(widget);
    return FALSE;
}

static gboolean astrid_gui_waveform_on_draw_event(__attribute__((unused)) GtkWidget * widget, cairo_t * cr, gpointer user_data) {
    lpfloat_t sample, avg;
    int c;
    size_t i, j;
    double x, y;
    size_t binsize = 64;
    lpgui_t * gui = (lpgui_t *)user_data;

    // on init, this is mapped into memory, aquire, draw, then release
    if(astrid_session_aquire_shared_resource(&gui->session, &gui->waveform.resource, gui->waveform.name) < 0) {
        syslog(LOG_ERR, "astrid_gui_config_waveform: could not aquire buffer (%s)\n", gui->waveform.name);
        return -1;
    }

    lpbuffer_t * snd = (lpbuffer_t *)gui->waveform.resource.data;

    // display mouse position
    char mouse_pos[50];
    sprintf(mouse_pos, "X: %d, Y: %d", (int)gui->mouse_x, (int)gui->mouse_y);
    syslog(LOG_ERR, "mouse pos yall!!!!! X: %d, Y: %d\n", (int)gui->mouse_x, (int)gui->mouse_y);
    astrid_gui_draw_text(gui, cr, 50, 50, mouse_pos);

    // average the bin data and draw the point
    for(i=0; i < snd->length-binsize; i += binsize) {
        avg = 0;
        for(j=0; j < binsize; j++) {
            sample = 0.f;
            for(c=0; c < snd->channels; c++) {
                sample += snd->data[(i+j) * snd->channels + c];
            }

            avg += sample;
        }

        avg = avg / binsize;

        x = (i / (double)snd->length) * gui->width;
        y = (avg * 0.5f + 0.5f) * gui->height;

        //printf("width %f height %f\n", width, height);
        //printf(" avg %f, x %f, y %f\n", avg, x, y);

        if (avg > 0.6 || avg < -0.6) {
            cairo_set_source_rgb(cr, 1, 0, 0);
        } else {
            cairo_set_source_rgb(cr, 0, 0, 0);
        }

        cairo_arc(cr, x, y, 1.f, 0.f, PI2);
        cairo_fill(cr);
    }

    // draw a line from the point to zero
    cairo_set_line_width(cr, 10);
    cairo_move_to(cr, x, y);
    cairo_line_to(cr, x, gui->height * 0.5f);
    cairo_stroke(cr);

    if(astrid_session_release_shared_resource(&gui->session, &gui->waveform.resource, gui->waveform.name) < 0) {
        syslog(LOG_ERR, "astrid_gui_config_waveform: could not release buffer (%s)\n", gui->waveform.name);
        return -1;
    }

    return FALSE;
}

static void astrid_gui_get_current_size(GtkWidget * widget, GtkAllocation * allocation, gpointer user_data) {
    lpgui_t * gui = (lpgui_t *)user_data;
    gtk_window_get_size(GTK_WINDOW(widget), &gui->width, &gui->height);
    gui->width = allocation->width;
    gui->height = allocation->height;
    gui->font_size = gui->width * 0.01;  // Adjust font size based on window width
}

static void astrid_gui_on_quit_activate(
        __attribute__((unused)) GtkMenuItem * menuitem, 
        __attribute__((unused)) gpointer user_data
    ) {
    // Exit the GTK main loop which will quit the program
    gtk_main_quit();
}

lpgui_t * astrid_gui_init(char * name, int width, int height) {
    lpgui_t * gui = LPMemoryPool.alloc(1, sizeof(lpgui_t));
    int argc = 0;

    // Initialize GTK
    gtk_init(&argc, NULL);

    // Create a new window
    gui->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(gui->window), name);
    gtk_window_set_default_size(GTK_WINDOW(gui->window), width, height);

    // Create a container to pack menu and canvas
    gui->container = gtk_box_new(GTK_ORIENTATION_VERTICAL, 0);
    gtk_container_add(GTK_CONTAINER(gui->window), gui->container);

    // Create a menu bar and add it to the container
    gui->menubar = gtk_menu_bar_new();
    gui->fileMenu = gtk_menu_new();
    gui->fileMenuItem = gtk_menu_item_new_with_label("File");
    gui->quitMenuItem = gtk_menu_item_new_with_label("Quit");

    // Assemble the menu
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(gui->fileMenuItem), gui->fileMenu);
    gtk_menu_shell_append(GTK_MENU_SHELL(gui->fileMenu), gui->quitMenuItem);
    gtk_menu_shell_append(GTK_MENU_SHELL(gui->menubar), gui->fileMenuItem);
    gtk_box_pack_start(GTK_BOX(gui->container), gui->menubar, FALSE, FALSE, 0);

    // Create a drawing area (canvas) and pack it after the menu
    gui->canvas = gtk_drawing_area_new();
    gtk_box_pack_start(GTK_BOX(gui->container), gui->canvas, TRUE, TRUE, 0);

    // Tell the canvas track mouse movement
    gtk_widget_set_events(gui->canvas, gtk_widget_get_events(gui->canvas) | GDK_POINTER_MOTION_MASK);

    g_signal_connect(G_OBJECT(gui->canvas), "motion-notify-event", G_CALLBACK(astrid_gui_mouse_move_callback), (void *)gui);
    g_signal_connect(G_OBJECT(gui->canvas), "size-allocate", G_CALLBACK(astrid_gui_get_current_size), (void *)gui);
    g_signal_connect(G_OBJECT(gui->quitMenuItem), "activate", G_CALLBACK(astrid_gui_on_quit_activate), NULL);
    g_signal_connect(G_OBJECT(gui->window), "destroy", G_CALLBACK(gtk_main_quit), NULL);

    return gui;
}

int astrid_gui_config_waveform(lpgui_t * gui, char * buffer_name) {
    memset(&gui->waveform, 0, sizeof(lpgui_waveform_widget_t));

    snprintf(gui->waveform.name, LPMAXKEY, "%s", buffer_name);

    if(astrid_session_aquire_shared_resource(&gui->session, &gui->waveform.resource, gui->waveform.name) < 0) {
        syslog(LOG_ERR, "astrid_gui_config_waveform: could not aquire buffer (%s)\n", gui->waveform.name);
        return -1;
    }

    if(astrid_session_release_shared_resource(&gui->session, &gui->waveform.resource, gui->waveform.name) < 0) {
        syslog(LOG_ERR, "astrid_gui_config_waveform: could not release buffer (%s)\n", gui->waveform.name);
        return -1;
    }

    g_signal_connect(G_OBJECT(gui->canvas), "draw", G_CALLBACK(astrid_gui_waveform_on_draw_event), (void *)gui);

    return 0;
}

int astrid_gui_run_forever(lpgui_t * gui) {
    printf("SHOW ALL\n");
    gtk_widget_show_all(gui->window);
    printf("MAIN\n");
    gtk_main();
    return 0;
}

int astrid_gui_destroy(lpgui_t * gui) {
    /*
    if(gui->type == ASTRID_GUI_WAVEFORM) {
        LPBuffer.destroy(gui->waveform.snd);
    }
    */
    LPMemoryPool.free(gui);
    return 0;
}
#endif


/* UTILS & MISC
 * ************/
void log_byte_buffer(unsigned char * buf, size_t size) {
    size_t i;
    for(i=0; i < size; i++) {
        if(buf[i] == '\n') {
            syslog(LOG_INFO, "%ld: \\n\n", i);
        } else if(buf[i] == '\r') {
            syslog(LOG_INFO, "%ld: \\r\n", i);
        } else if(buf[i] == '\t') {
            syslog(LOG_INFO, "%ld: \\t\n", i);
        } else if(isprint(buf[i])) {
            syslog(LOG_INFO, "%ld: %c\n", i, buf[i]);
        } else {
            syslog(LOG_INFO, "%ld: \\x%02X\n", i, buf[i]);
        }
    }
}

int lpencode_with_prefix(char * prefix, size_t val, char * encoded) {
    size_t size = 0;
    size = snprintf(NULL, 0,  "%s-%d-buf-%020ld", prefix, (int)getpid(), val) + 1;
    if(size == 0) {
        syslog(LOG_ERR, "Could not estimate space for key. (%d) %s\n", errno, strerror(errno));
        return -1;
    }

    if(size > LPKEY_MAXLENGTH) {
        syslog(LOG_ERR, "key's too big. (%d) %s\n", errno, strerror(errno));
        return -1;
    }

    if(snprintf(encoded, size, "%s-%d-buf-%020ld", prefix, (int)getpid(), val) < 0) {
        syslog(LOG_ERR, "Could not encode key. (%d) %s\n", errno, strerror(errno));
        return -1;
    }

    return 0;
}

void lptimeit_since(struct timespec * start) {
    struct timespec now;
    long long elapsed;

    clock_gettime(CLOCK_REALTIME, &now);
    elapsed = (now.tv_sec - start->tv_sec) * 1e9 + (now.tv_nsec - start->tv_nsec);
    printf("%lld nanoseconds\n", elapsed);

    start->tv_sec = now.tv_sec;
    start->tv_nsec = now.tv_nsec;
}

/* INTERPROCESS  (IPC)
 * COMMUNICATION TOOLS
 * *****IPCTOOLS******/

/** Session interface ********************/
const astrid_session_factory_t AstridSession = {
    astrid_session_create,
    astrid_session_open,
    astrid_session_close,
    astrid_session_get_info,
    astrid_session_get_size,
    astrid_session_get_type,
    astrid_session_set_float,
    astrid_session_get_float,
    astrid_session_get_float_from_hash,
    astrid_session_set_float_from_hash,
    astrid_session_exists,
    astrid_session_exists_from_hash,
    astrid_session_set_int,
    astrid_session_get_int,
    astrid_session_set_string,
    astrid_session_get_string,
    astrid_session_set_buffer,
    astrid_session_get_buffer
};

size_t astrid_get_resource_type_size(int resource_type, size_t size) {
    if(resource_type >= 0 && resource_type < ASTRID_TYPE_BUFFER) {
        size = astrid_shared_type_sizes[resource_type];
    }

    if(size == 0) {
        syslog(LOG_ERR, "astrid_get_resource_type_size: resource_type=%d size=%ld\n", resource_type, size);
    }

    return size;
}

int astrid_session_aquire_resource_lock(astrid_shared_resource_t * resource, char * name) {
    int ret = 0;
    struct timespec timeout = {0};
    struct timespec start_time, end_time;
    long elapsed_ms;

    if(resource == NULL || name == NULL) {
        syslog(LOG_ERR, "astrid_session_aquire_resource_lock: (%s) Invalid parameters.\n", name);
        return -1;
    }

    // resource structs aren't usually long-lived, but we could have the name already in some cases
    astrid_make_sem_name(name, resource->lock_name);

    if((resource->lock = sem_open(resource->lock_name, 0)) == SEM_FAILED) {
        syslog(LOG_ERR, "astrid_session_aquire_resource_lock: (name=%s | lock_name=%s) Could not open semaphore. (%d) %s\n", name, resource->lock_name, errno, strerror(errno));
        return -1;
    }
    //if(sem_wait(resource->lock) < 0) {
    if(clock_gettime(CLOCK_REALTIME, &timeout) < 0) {
        syslog(LOG_ERR, "astrid_session_aquire_resource_lock: (name=%s | lock_name=%s) Could not get time. (%d) %s\n", name, resource->lock_name, errno, strerror(errno));
        sem_close(resource->lock);
        return -1;
    }

    // Track elapsed time for lock acquisition
    clock_gettime(CLOCK_MONOTONIC, &start_time);

    timeout.tv_sec += 5;
    while((ret = sem_timedwait(resource->lock, &timeout)) < 0 && errno == EINTR) {
        continue; // retry if a signal interrupts with EINTR
    }

    // Log if lock acquisition took significant time
    clock_gettime(CLOCK_MONOTONIC, &end_time);
    elapsed_ms = (end_time.tv_sec - start_time.tv_sec) * 1000 + (end_time.tv_nsec - start_time.tv_nsec) / 1000000;
    if(elapsed_ms > 100) {
        syslog(LOG_WARNING, "astrid_session_aquire_resource_lock: lock acquisition for %s took %ldms\n", name, elapsed_ms);
    }

    if(ret < 0) {
        if(errno == ETIMEDOUT) {
            syslog(LOG_ERR, "astrid_session_aquire_resource_lock: (%s) Semaphore timed out. (%d) %s\n", name, errno, strerror(errno));
        } else {
            syslog(LOG_ERR, "astrid_session_aquire_resource_lock: (%s) Could not aquire semaphore. (%d) %s\n", name, errno, strerror(errno));
        }
        sem_close(resource->lock);
        return -1;
    }

    return 0;
}

int astrid_session_release_resource_lock(astrid_shared_resource_t * resource) {
    if(resource == NULL || resource->lock == NULL) {
        syslog(LOG_ERR, "astrid_session_release_resource_lock: Invalid parameters.\n");
        return -1;
    }

    if(sem_post(resource->lock) < 0) {
        syslog(LOG_ERR, "astrid_session_release_resource_lock: Could not unlock semaphore. (%d) %s\n", errno, strerror(errno));
        sem_close(resource->lock);
        return -1;
    }
    if(sem_close(resource->lock) < 0) {
        syslog(LOG_ERR, "astrid_session_release_resource_lock: sem_close FAILED handle=%p errno=%d %s\n", (void*)resource->lock, errno, strerror(errno));
        return -1;
    }

    return 0;
}


ssize_t astrid_session_increment_buffer_count(astrid_session_t * session) {
    return (ssize_t)(session->buffer_count++);
#if 0
    int rc;
    MDB_txn * txn;
    MDB_val key, data;
    uint32_t key_hash;
    ssize_t count=0, new_count=0;

    key_hash = lphashstr("astrid-session-buffer-count");
    key.mv_size = sizeof(uint32_t);
    key.mv_data = &key_hash;
    data.mv_size = sizeof(ssize_t);

    // read and increment the count from lmdb
    rc = mdb_txn_begin(session->dbenv, NULL, MDB_RDONLY, &txn);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "astrid_session_read_shared_resource_header mdb_txn_begin: read (%d) %s\n", rc, mdb_strerror(rc));
        return -1;
    }

    rc = mdb_get(txn, session->dbi, &key, &data);
    if(rc == MDB_SUCCESS) {
        memcpy(&count, data.mv_data, sizeof(ssize_t));
    } else {
        syslog(LOG_ERR, "astrid_session_increment mdb_get: read (%d) %s\n", rc, mdb_strerror(rc));
        mdb_txn_reset(txn);
        return -1;
    }

    mdb_txn_commit(txn);

    new_count = count + 1;
    key.mv_size = sizeof(uint32_t);
    key.mv_data = &key_hash;
    data.mv_size = sizeof(ssize_t);
    data.mv_data = &new_count;

    rc = mdb_txn_begin(session->dbenv, NULL, 0, &txn);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "astrid_session_increment_buffer_count: mdb_txn_begin write (%d) %s\n", errno, strerror(errno));
        return -1;
    }

    rc = mdb_put(txn, session->dbi, &key, &data, 0);
    if(rc != MDB_SUCCESS) {
        mdb_txn_abort(txn);
        syslog(LOG_ERR, "astrid_session_increment_buffer_count: mdb_put write (%d) %s\n", errno, strerror(errno));
        return -1;
    }

    rc = mdb_txn_commit(txn);
    if(rc != MDB_SUCCESS) {
        mdb_txn_abort(txn);
        syslog(LOG_ERR, "astrid_session_increment_buffer_count: mdb_txn_commit write (%d) %s\n", errno, strerror(errno));
        return -1;
    }

    return count;
#endif
}

int astrid_session_register_shared_resource_header(
        astrid_session_t * session, 
        astrid_shared_resource_header_t * header,
        char * name
    ) {
    MDB_txn * txn;
    MDB_val key, data;
    uint32_t key_hash = lphashstr(name);

    // Prepare the data
    key.mv_size = sizeof(uint32_t);
    key.mv_data = &key_hash;
    data.mv_size = sizeof(astrid_shared_resource_header_t);
    data.mv_data = (void *)header;

    // Begin a new write transaction
    int rc = mdb_txn_begin(session->dbenv, NULL, 0, &txn);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "astrid_session_register_shared_resource_header: mdb_txn_begin (%d) %s\n", rc, mdb_strerror(rc));
        return -1;
    }

    // Put the data into the key storage
    rc = mdb_put(txn, session->dbi, &key, &data, 0);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "astrid_session_register_shared_resource_header: mdb_put (%d) %s\n", rc, mdb_strerror(rc));
        mdb_txn_abort(txn);
        return -1;
    }

    // Commit the transaction
    rc = mdb_txn_commit(txn);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "astrid_session_register_shared_resource_header: mdb_txn_commit (%d) %s\n", rc, mdb_strerror(rc));
        mdb_txn_abort(txn);
        return -1;
    }

    return 0;
}

int astrid_session_read_shared_resource_header(
        astrid_session_t * session, 
        astrid_shared_resource_header_t * header,
        char * name
    ) {
    MDB_txn * txn;
    MDB_val key, data;
    uint32_t key_hash = lphashstr(name);

    key.mv_size = sizeof(uint32_t);
    key.mv_data = &key_hash;
    data.mv_size = sizeof(astrid_shared_resource_header_t);

    int rc = mdb_txn_begin(session->dbenv, NULL, MDB_RDONLY, &txn);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "astrid_session_read_shared_resource_header mdb_txn_begin: read (%d) %s\n", rc, mdb_strerror(rc));
        return -1;
    }

    rc = mdb_get(txn, session->dbi, &key, &data);
    if(rc == MDB_SUCCESS) {
        memcpy(header, data.mv_data, sizeof(astrid_shared_resource_header_t));
        mdb_txn_commit(txn);
    } else {
        // MDB_NOTFOUND is normal - key may not exist yet
        if(rc != MDB_NOTFOUND) {
            syslog(LOG_ERR, "astrid_session_get_shared_resource mdb_get: read (%d) %s\n", rc, mdb_strerror(rc));
        }
        mdb_txn_commit(txn);
        return -1;
    }

    return 0;
}

int astrid_session_write_shared_resource_header(
        astrid_session_t * session, 
        astrid_shared_resource_header_t * header,
        char * name
    ) {
    int rc;
    MDB_val key, data;
    MDB_txn * txn;
    uint32_t key_hash;

    // Write the changes back to LMDB
    key_hash = lphashstr(name);
    key.mv_size = sizeof(uint32_t);
    key.mv_data = &key_hash;
    data.mv_size = sizeof(astrid_shared_resource_header_t);
    data.mv_data = header;

    // Create a write transaction
    rc = mdb_txn_begin(session->dbenv, NULL, 0, &txn);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "astrid_session_write_shared_resource_header: mdb_txn_begin (%d) %s\n", rc, mdb_strerror(rc));
        return -1;
    }

    // Put the data
    rc = mdb_put(txn, session->dbi, &key, &data, 0);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "astrid_session_write_shared_resource_header: mdb_put (%d) %s\n", rc, mdb_strerror(rc));
        mdb_txn_abort(txn);
        return -1;
    }

    // Commit the transaction
    rc = mdb_txn_commit(txn);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_WARNING, "astrid_session_write_shared_resource_header: mdb_txn_commit (%d) %s\n", rc, mdb_strerror(rc));
        return -1;
    }

    return 0;
}

int astrid_resize_aquired_shared_resource(astrid_shared_resource_t * resource, size_t new_size) {
    void * shmp;
    int shmfd;

    if((shmfd = shm_open(resource->lock_name, O_RDWR, 0)) < 0) {
        syslog(LOG_ERR, "astrid_resize_aquired_shared_resource: Could not open shared memory segment. (%d) %s\n", errno, strerror(errno));
        return -1;
    }

    if(ftruncate(shmfd, new_size) < 0) {
        close(shmfd);
        syslog(LOG_ERR, "astrid_resize_aquired_shared_resource: ftruncate (%d) %s\n", errno, strerror(errno));
        return -1;
    }

    shmp = mremap(resource->data, resource->mapped_size, new_size, MREMAP_MAYMOVE);
    if(shmp == (void *)-1) {
        close(shmfd);
        syslog(LOG_ERR, "astrid_resize_aquired_shared_resource: mremap (%d) %s\n", errno, strerror(errno));
        return -1;
    }

    close(shmfd);
    resource->data = shmp;
    resource->header.size = new_size;
    resource->mapped_size = new_size; // Update mapped size for correct munmap later

    return 0;
}

int astrid_session_aquire_shared_resource(
        astrid_session_t * session, 
        astrid_shared_resource_t * resource,
        char * name
    ) {
    //syslog(LOG_ERR, "AQUIRE RESOURCE LOCK (PID %d) name=%s\n", (int)getpid(), name);
    struct stat statbuf;
    int shmfd;

    if(!name || resource == NULL || session == NULL) {
        syslog(LOG_ERR, "astrid_session_aquire_shared_resource: invalid or missing params (%s)\n", name);
        return -1;
    }

    //syslog(LOG_ERR, "init resource bag name=%s\n", name);
    // initialize the resource bag
    memset(resource, 0, sizeof(astrid_shared_resource_t));

    // Read the header
    //syslog(LOG_ERR, "read header name=%s\n", name);
    if(astrid_session_read_shared_resource_header(session, &resource->header, name) < 0) {
        syslog(LOG_ERR, "astrid_session_aquire_shared_resource: Could not get header. (%s) (%d) %s\n", name, errno, strerror(errno));
        return -1;
    }

    if(resource->header.flags & ASTRID_FLAG_VALUE_STORED_IN_HEADER) {
        //syslog(LOG_ERR, "header storing value name=%s\n", name);
        // Use the value we've already got in the union field and skip POSIX shm
        // Small values fit nicely into LMDB directly...
        resource->data = resource->header.value;
        resource->mapped_size = 0; // No mmap for header-stored values
    } else {
        //syslog(LOG_ERR, "shm backed value name=%s\n", name);
        /* POSIX shared memory storage for large objects
         ************************************************/
        // Aquire the resource lock
        //syslog(LOG_ERR, "aquire shm lock name=%s\n", name);
        if(astrid_session_aquire_resource_lock(resource, name) < 0) {
            syslog(LOG_ERR, "astrid_session_aquire_shared_resource: Could not aquire resource lock. (name=%s lock_name=%s)\n", name, resource->lock_name);
            return -1;
        }

        // Map the shared memory and truncate it to the desired size
        //syslog(LOG_ERR, "open shm name=%s\n", name);
        if((shmfd = shm_open(resource->lock_name, O_RDWR, LPIPC_PERMS)) < 0) {
            syslog(LOG_ERR, "astrid_session_aquire_shared_resource: Could not open shared memory segment. (%d) %s\n", errno, strerror(errno));
            astrid_session_release_resource_lock(resource);
            return -1;
        }

        /* Get the real size of the segment */
        if(fstat(shmfd, &statbuf) < 0) {
            syslog(LOG_ERR, "astrid_session_aquire_shared_resource: fstat (%d) %s\n", errno, strerror(errno));
            close(shmfd);
            shm_unlink(resource->lock_name);
            astrid_session_release_resource_lock(resource);
            return -1;
        }

        //syslog(LOG_ERR, "mmap shm name=%s\n", name);
        if((resource->data = mmap(NULL, statbuf.st_size, PROT_READ | PROT_WRITE, MAP_SHARED, shmfd, 0)) == MAP_FAILED) {
            syslog(LOG_ERR, "astrid_session_aquire_shared_resource: mmap (%d) %s\n", errno, strerror(errno));
            close(shmfd);
            shm_unlink(resource->lock_name);
            astrid_session_release_resource_lock(resource);
            return -1;
        }

        // Store the actual mapped size for correct munmap later
        resource->mapped_size = statbuf.st_size;

        close(shmfd);
    }

    return 0;
}

int astrid_session_release_shared_resource(
        astrid_session_t * session,
        astrid_shared_resource_t * resource,
        char * name
    ) {

    if(resource == NULL) {
        syslog(LOG_ERR, "astrid_session_release_shared_resource: resource is NULL. (%d) %s\n", errno, strerror(errno));
        return -1;
    }

    // Write the header back to LMDB
    if(resource->header.flags & ASTRID_FLAG_VALUE_STORED_IN_HEADER) {
        if(astrid_session_write_shared_resource_header(session, &resource->header, name) < 0) {
            syslog(LOG_ERR, "astrid_session_release_shared_resource: Failed to write resource header. (%d) %s\n", errno, strerror(errno));
            return -1;
        }
    } else {
        // Clean up - use mapped_size instead of header.size for correct munmap
        munmap(resource->data, resource->mapped_size);
        if(astrid_session_release_resource_lock(resource) < 0) {
            syslog(LOG_ERR, "astrid_session_release_shared_resource: Could not release resource lock.\n");
            return -1;
        }
    }

    return 0;
}

int astrid_make_sem_name(char * name, char * sem_name) {
    return snprintf(sem_name, LPMAXPATH, "/%s", name);
}

// Fast destroy that skips LMDB - use for rollback/error paths
int astrid_session_destroy_shared_resource_fast(
        astrid_session_t * session,
        char * name
    ) {
    char sem_name[LPMAXPATH] = {0};

    // Skip LMDB operations - just unlink sem/shm
    astrid_make_sem_name(name, sem_name);

    if(sem_unlink(sem_name) < 0) {
        if(errno != ENOENT) { // Don't log if resource doesn't exist
            syslog(LOG_WARNING, "fast sem_unlink failed for %s (sem_name=%s): (%d) %s\n", name, sem_name, errno, strerror(errno));
        }
    }
    if(shm_unlink(sem_name) < 0) {
        if(errno != ENOENT) {
            syslog(LOG_WARNING, "fast shm_unlink failed for %s (sem_name=%s): (%d) %s\n", name, sem_name, errno, strerror(errno));
        }
    }
    return 0;
}

int astrid_session_destroy_shared_resource(
        astrid_session_t * session,
        char * name
    ) {
    int rc;
    char sem_name[LPMAXPATH] = {0};
    MDB_txn * txn;
    MDB_val key;
    uint32_t key_hash = lphashstr(name);

    // Delete the LMDB entry
    key.mv_size = sizeof(uint32_t);
    key.mv_data = &key_hash;

    rc = mdb_txn_begin(session->dbenv, NULL, 0, &txn);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_WARNING, "astrid_session_destroy_shared_resource: mdb_txn_begin failed for %s: (%d) %s\n", name, rc, mdb_strerror(rc));
    } else {
        rc = mdb_del(txn, session->dbi, &key, NULL);
        if(rc != MDB_SUCCESS && rc != MDB_NOTFOUND) {
            syslog(LOG_WARNING, "astrid_session_destroy_shared_resource: mdb_del failed for %s: (%d) %s\n", name, rc, mdb_strerror(rc));
            mdb_txn_abort(txn);
        } else {
            rc = mdb_txn_commit(txn);
            if(rc != MDB_SUCCESS) {
                syslog(LOG_WARNING, "astrid_session_destroy_shared_resource: mdb_txn_commit failed for %s: (%d) %s\n", name, rc, mdb_strerror(rc));
            }
        }
    }

    // Unlink the semaphore and shared memory resources
    astrid_make_sem_name(name, sem_name);

    if(sem_unlink(sem_name) < 0) {
        // ENOENT is expected - resource may not exist (e.g., header-only values, or already cleaned up)
        if(errno != ENOENT) {
            syslog(LOG_ERR, "sem_unlink failed for %s (sem_name=%s): (%d) %s\n", name, sem_name, errno, strerror(errno));
        }
    }
    if(shm_unlink(sem_name) < 0) {
        // ENOENT is expected - resource may not exist (e.g., header-only values, or already cleaned up)
        if(errno != ENOENT) {
            syslog(LOG_ERR, "shm_unlink failed for %s (sem_name=%s): (%d) %s\n", name, sem_name, errno, strerror(errno));
        }
    }
    return 0;
}

int astrid_session_register_shared_resource(
        astrid_session_t * session, 
        char * name, 
        void * value, 
        int resource_type, 
        size_t size
    ) {
    int rc, shmfd;
    char sem_name[LPMAXPATH] = {0};
    MDB_txn * txn;
    MDB_val key, data;
    uint32_t key_hash = lphashstr(name);
    astrid_shared_resource_t resource = {0};

    size = astrid_get_resource_type_size(resource_type, size);

    // Prepare the header
    resource.header.type = resource_type;
    resource.header.size = size;
    if(resource_type < ASTRID_TYPE_MUSTARD) {
        syslog(LOG_ERR, "!!! REGISTER: value in header (%s) size=%ld resource_type=%d\n", name, size, resource_type);
        resource.header.flags = ASTRID_FLAG_VALUE_STORED_IN_HEADER;
        memcpy(resource.header.value, value, size);
    }

    key.mv_size = sizeof(uint32_t);
    key.mv_data = &key_hash;
    data.mv_size = sizeof(astrid_shared_resource_header_t);
    data.mv_data = &resource.header;

    // Begin a new write transaction
    rc = mdb_txn_begin(session->dbenv, NULL, 0, &txn);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "astrid_session_register_shared_resource mdb_txn_begin: (%d) %s\n", rc, mdb_strerror(rc));
        return -1;
    }

    // Put the data into the key storage
    rc = mdb_put(txn, session->dbi, &key, &data, 0);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "astrid_session_register_shared_resource mdb_put: (%d) %s\n", rc, mdb_strerror(rc));
        mdb_txn_abort(txn);
        return -1;
    }

    // Commit the transaction
    rc = mdb_txn_commit(txn);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "astrid_session_register_shared_resource mdb_txn_commit: (%d) %s\n", rc, mdb_strerror(rc));
        mdb_txn_abort(txn);
        return -1;
    }

    if(resource_type < ASTRID_TYPE_MUSTARD) goto shared_register_close;

    /* POSIX shared memory storage for large objects
     ************************************************/
    // Create and aquire the access lock
    astrid_make_sem_name(name, sem_name);
    sem_unlink(sem_name);
    if((resource.lock = sem_open(sem_name, O_CREAT | O_EXCL, LPIPC_PERMS, 1)) == SEM_FAILED) {
        syslog(LOG_ERR, "astrid_session_register_shared_resource: Could not create shared resource lock semaphore. (%d) %s\n", errno, strerror(errno));
        return -1;
    }
    sem_close(resource.lock); // FIXME maybe add a create_and_aquire path to avoid opening twice here...
    if(astrid_session_aquire_resource_lock(&resource, name) < 0) {
        syslog(LOG_ERR, "astrid_session_aquire_shared_resource: Could not aquire resource lock.\n");
        return -1;
    }

    // Remove any existing segment with the same name
    shm_unlink(sem_name);

    // Map the shared memory and truncate it to the desired size
    if((shmfd = shm_open(sem_name, O_CREAT | O_RDWR | O_EXCL, LPIPC_PERMS)) < 0) {
        syslog(LOG_ERR, "astrid_session_register_shared_resource: Could not create shared memory segment (%d) %s\n", errno, strerror(errno));
        sem_post(resource.lock);
        sem_close(resource.lock);
        return -1;
    }
    if(ftruncate(shmfd, size) < 0) {
        syslog(LOG_ERR, "astrid_session_register_shared_resource: ftruncate (%d) %s\n", errno, strerror(errno));
        close(shmfd);
        shm_unlink(sem_name);
        sem_post(resource.lock);
        sem_close(resource.lock);
        return -1;
    }
    if((resource.data = mmap(NULL, size, PROT_READ | PROT_WRITE, MAP_SHARED, shmfd, 0)) == MAP_FAILED) {
        syslog(LOG_ERR, "astrid_session_register_shared_resource: mmap (%d) %s\n", errno, strerror(errno));
        close(shmfd);
        shm_unlink(sem_name);
        sem_post(resource.lock);
        sem_close(resource.lock);
        return -1;
    }

    // Copy the value into the shared memory segment: 0 sizes and NULL values, 
    // just act like a buffer clear and memset zeros
    if(value == NULL) {
        syslog(LOG_ERR, "ZEROING shm %s on registration to %ld bytes\n", name, size);
        memset(resource.data, 0, size);
    } else if(size > 0) {
        //syslog(LOG_ERR, "INIT shm %s on registration with %ld byte value\n", name, size);
        memcpy(resource.data, value, size);
    } else {
        syslog(LOG_ERR, "astrid_session_register_shared_resource: ignoring an attempt to set shared resource %s with size 0\n", name);
    }

    // Clean up and release the lock
    munmap(resource.data, size);
    close(shmfd);
    if(astrid_session_release_resource_lock(&resource) < 0) {
        syslog(LOG_ERR, "astrid_session_register_shared_resource: Could not release resource lock.\n");
        return -1;
    }

shared_register_close:
    return 0;
}

int astrid_session_set_shared_resource(astrid_session_t * session, char * name, void * value) {
    astrid_shared_resource_t resource;

    if(astrid_session_aquire_shared_resource(session, &resource, name) < 0) {
        syslog(LOG_ERR, "astrid_session_write_lpbuffer_to_shared_ringbuffer: could not aquire ADC shm.\n");
        return -1;
    }

    memcpy(resource.data, value, resource.header.size);

    if(astrid_session_release_shared_resource(session, &resource, name) < 0) {
        syslog(LOG_ERR, "astrid_session_write_lpbuffer_to_shared_ringbuffer: could not release ADC shm.\n");
        return -1;
    }

    return 0;
}

int astrid_session_get_shared_resource(astrid_session_t * session, char * name, void * value) {
    astrid_shared_resource_t resource;

    if(astrid_session_aquire_shared_resource(session, &resource, name) < 0) {
        syslog(LOG_ERR, "astrid_session_write_lpbuffer_to_shared_ringbuffer: could not aquire ADC shm.\n");
        return -1;
    }

    memcpy(value, resource.data, resource.header.size);

    if(astrid_session_release_shared_resource(session, &resource, name) < 0) {
        syslog(LOG_ERR, "astrid_session_write_lpbuffer_to_shared_ringbuffer: could not release ADC shm.\n");
        return -1;
    }

    return 0;
}

int astrid_session_publish_buffer(
        astrid_session_t * session,
        lpmsgq_t * msgq,
        lpbuffer_t * buf,
        int map_channels,
        int * channel_map
    ) {
    char buffer_code[LPMAXKEY] = {0};
    ssize_t buffer_id, size;
    lpmsg_t msg = {0};

    if(msgq == NULL) {
        syslog(LOG_ERR, "astrid_session_publish_buffer: message queue is NULL\n");
        return -1;
    }

    buffer_id = astrid_session_increment_buffer_count(session);

    // generate the buffer code from the buffer id
    if(lpencode_with_prefix(session->instrument_name, buffer_id, buffer_code) < 0) {
        return -1;
    }

    size = sizeof(lpbuffer_t) + (buf->length * buf->channels * sizeof(lpfloat_t));
    if(astrid_session_register_shared_resource(session, buffer_code, buf, ASTRID_TYPE_BUFFER, size) < 0) {
        syslog(LOG_ERR, "astrid_session_publish_buffer: could not register buffer in shared memory\n");
        return -1;
    }

    // Store channel map inline in buffer header (eliminates separate cmap resource)
    if(map_channels > 0 && channel_map != NULL) {
        astrid_shared_resource_header_t header = {0};

        // Read the buffer header we just created
        if(astrid_session_read_shared_resource_header(session, &header, buffer_code) < 0) {
            syslog(LOG_ERR, "astrid_session_publish_buffer: could not read buffer header to add channel map\n");
            astrid_session_destroy_shared_resource_fast(session, buffer_code);
            return -1;
        }

        // Add channel map to header
        header.map_channels = map_channels;
        if(map_channels > 256) {
            syslog(LOG_WARNING, "astrid_session_publish_buffer: truncating channel map from %d to 256 channels\n", map_channels);
            header.map_channels = 256;
        }
        memcpy(header.channel_map, channel_map, header.map_channels * sizeof(int));

        // Write the updated header back
        if(astrid_session_write_shared_resource_header(session, &header, buffer_code) < 0) {
            syslog(LOG_ERR, "astrid_session_publish_buffer: could not write updated buffer header with channel map\n");
            astrid_session_destroy_shared_resource_fast(session, buffer_code);
            return -1;
        }
    }

    // prepare the render complete message
    memcpy(msg.instrument_name, session->instrument_name, strlen(session->instrument_name));
    memcpy(msg.msg, buffer_code, strlen(buffer_code));
    msg.type = LPMSG_RENDER_COMPLETE;

    // send the render complete message to signal the buffer is ready to be read
    if(astrid_msgq_write(msgq, &msg) < 0) {
        syslog(LOG_ERR, "astrid_session_publish_buffer: could not send render complete message, rolling back\n");
        astrid_session_destroy_shared_resource_fast(session, buffer_code);
        return -1;
    }

    // free the original buffer, which has been copied into shared memory at this point
    // FIXME let scripts create the shared memory output buffer directly from the ctx
    LPBuffer.destroy(buf);

    return 0;
}


int astrid_session_publish_bufstr(
        astrid_session_t * session,
        lpmsgq_t * msgq,
        unsigned char * bufstr,
        size_t size
    ) {
    /* FIXME is this still useful somehow? */
    char buffer_code[LPMAXKEY] = {0};
    ssize_t buffer_id = 0;
    lpmsg_t msg = {0};
    //syslog(LOG_ERR, "PUBLISH BUFSTR session->name=%s\n", session->instrument_name);

    if(msgq == NULL) {
        syslog(LOG_ERR, "astrid_session_publish_bufstr: message queue is NULL\n");
        return -1;
    }

    buffer_id = getpid() + astrid_session_increment_buffer_count(session);

    // generate the buffer code from the buffer id
    if(lpencode_with_prefix(session->instrument_name, buffer_id, buffer_code) < 0) {
        return -1;
    }
    //syslog(LOG_ERR, "PUBLISH BUFSTR buffer code=%s\n", buffer_code);

    if(astrid_session_register_shared_resource(session,
                buffer_code, bufstr, ASTRID_TYPE_BYTES, size) < 0) {
        syslog(LOG_ERR, "astrid_session_publish_bufstr: astrid_session_register_shared_resource could not set bufstr in LMDB\n");
        return -1;
    }

    // prepare the render complete message
    memcpy(msg.instrument_name, session->instrument_name, strlen(session->instrument_name));
    memcpy(msg.msg, buffer_code, strlen(buffer_code));
    msg.type = LPMSG_RENDER_COMPLETE;

    // send the render complete message to signal the bufstr is ready to be read
    if(astrid_msgq_write(msgq, &msg) < 0) {
        syslog(LOG_ERR, "astrid_session_publish_bufstr: astrid_msgq_write could not send render complete message on bufstr publish\n");
        return -1;
    }

    return 0;
}

lpbuffer_t * astrid_session_deserialize_buffer(astrid_session_t * session, char * buffer_code, lpmsg_t * msg) {
    /* FIXME is this still useful somehow? */
    astrid_shared_resource_t resource;
    lpbuffer_t * buf;
    unsigned char * str;
    size_t audiosize, offset, length, onset;
    int channels, samplerate, is_looping;

    if(astrid_session_aquire_shared_resource(session, &resource, buffer_code) < 0) {
        syslog(LOG_ERR, "astrid_session_deserialize_buffer: astrid_session_aquire_shared_resource could not aquire buffer.\n");
        return NULL;
    }

    // since we don't know the size, have LMDB fill in the size for us
    str = (unsigned char *)LPMemoryPool.alloc(1, resource.header.size);
    memcpy(str, resource.data, resource.header.size);

    if(astrid_session_release_shared_resource(session, &resource, buffer_code) < 0) {
        syslog(LOG_ERR, "astrid_session_deserialize_buffer: astrid_session_release_shared_resource could not release buffer.\n");
        return NULL;
    }

    if(astrid_session_destroy_shared_resource(session, buffer_code) < 0) {
        syslog(LOG_ERR, "astrid_session_deserialize_buffer: astrid_session_destroy_shared_resource could not destroy buffer.\n");
        return NULL;
    }

    offset = 0;
    memcpy(&audiosize, str + offset, sizeof(size_t));
    offset += sizeof(size_t);
    memcpy(&length, str + offset, sizeof(size_t));
    offset += sizeof(size_t);
    memcpy(&channels, str + offset, sizeof(int));
    offset += sizeof(int);
    memcpy(&samplerate, str + offset, sizeof(int));
    offset += sizeof(int);
    memcpy(&is_looping, str + offset, sizeof(int));
    offset += sizeof(int);
    memcpy(&onset, str + offset, sizeof(size_t));
    offset += sizeof(size_t);

    buf = (lpbuffer_t *)LPMemoryPool.alloc(1, sizeof(lpbuffer_t) + audiosize);
    memcpy(buf->data, str + offset, audiosize);
    offset += audiosize;
    memcpy(msg, str + offset, sizeof(lpmsg_t));

    LPMemoryPool.free(str);

    buf->length = length;
    buf->channels = channels;
    buf->samplerate = samplerate;
    buf->is_looping = is_looping;
    buf->onset = onset;
    buf->phase = 0.f;
    buf->pos = 0;
    buf->boundary = length-1;
    buf->range = length;

    return buf;
}

int astrid_session_write_lpbuffer_to_shared_ringbuffer(
        astrid_session_t * session,
        char * name, // name of the shared resource containing the ringbuffer
        lpbuffer_t * src // the source block to write
    ) {
    astrid_shared_resource_t resource;
    lpbuffer_t * buf;
    size_t write_pos, read_pos, length=src->length;
    int c, channels=src->channels;


    // aquire the shared resource and get a pointer to the ringbuffer
    if(astrid_session_aquire_shared_resource(session, &resource, name) < 0) {
        syslog(LOG_ERR, 
                "astrid_session_write_lpbuffer_to_shared_ringbuffer: astrid_session_aquire_shared_resource could not aquire ADC shm. (%d) %s\n", errno, strerror(errno));
        return -1;
    }
    buf = (lpbuffer_t *)resource.data;

    if(buf->length <= 0) {
        syslog(LOG_ERR, "astrid_session_write_lpbuffer_to_shared_ringbuffer: buf->length == %ld\n", buf->length);
        return -1;
    }

    if(buf->channels < channels) channels = buf->channels;
    if(buf->length < length) length = buf->length;

    for(read_pos=0; read_pos < length; read_pos++) {
        write_pos = buf->pos + read_pos;
        while(write_pos >= buf->length) {
            write_pos -= buf->length;
        }

        for(c=0; c < channels; c++) {
            buf->data[write_pos * channels + c] = src->data[read_pos * channels + c];
        }
    }

    /* Increment the write position */
    buf->pos += src->length;
    while(buf->pos >= buf->length) {
        buf->pos -= buf->length;
    }

    if(astrid_session_release_shared_resource(session, &resource, name) < 0) {
        syslog(LOG_ERR, "astrid_session_write_lpbuffer_to_shared_ringbuffer: astrid_session_release_shared_resource could not release ADC shm.\n");
    }

    return 0;
}

#if 0
int astrid_overdub_ringbuffer_block(
        lpbuffer_t * buf,
        float ** block, 
        int channels, 
        lpfloat_t volume,
        lpfloat_t feedback,
        size_t blocksize_in_frames
    ) {
    size_t insert_pos, i, boundary;
    int c;
    float * channelp;
    float sample = 0;

    boundary = buf->length * channels;

    /* Copy the block */
    for(c=0; c < channels; c++) {
        channelp = block[c];
        for(i=0; i < blocksize_in_frames; i++) {
            insert_pos = ((buf->pos+i) * channels + c) % boundary;
            sample = *channelp++;
            buf->data[insert_pos] += (lpfilternan(sample) * volume) + (buf->data[insert_pos] * feedback);
        }
    }

    /* Increment the write position */
    buf->pos += blocksize_in_frames;
    while(buf->pos >= buf->length) {
        buf->pos -= buf->length;
    }

    return 0;
}
#endif

int astrid_session_read_shared_ringbuffer_block(
        astrid_session_t * session,
        char * name,
        size_t offset_in_frames,
        lpbuffer_t * out
    ) {

    astrid_shared_resource_t resource;

    //syslog(LOG_ERR, "A (PID %d) ADC name=%s\n", (int)getpid(), name);
    if(astrid_session_aquire_shared_resource(session, &resource, name) < 0) {
        syslog(LOG_ERR, "astrid_session_read_shared_ringbuffer_block: astrid_session_aquire_shared_resource could not aquire shm.\n");
        return -1;
    }

    //syslog(LOG_ERR, "read ringbuffer block name=%s\n", name);
    if(astrid_read_ringbuffer_block((lpbuffer_t *)resource.data, offset_in_frames, out) < 0) {
        syslog(LOG_ERR, "astrid_session_read_shared_ringbuffer_block: astrid_read_ringbuffer_block could not read data.\n");
        return -1;
    }

    if(astrid_session_release_shared_resource(session, &resource, name) < 0) {
        syslog(LOG_ERR, "astrid_session_read_shared_ringbuffer_block: astrid_session_release_shared_resource could not release shm.\n");
        return -1;
    }

    return 0;
}

size_t safe_index(size_t i, ssize_t offset, int c, size_t length, int channels) {
    size_t index = 0;
    size_t max_index = 0;

    if(channels <= 0 || c < 0 || length == 0) return 0;

    while(offset < 0) offset += (ssize_t)length;

    index = ((size_t)offset + i) % length;
    c = c % channels;
    index = index * channels + c;

    max_index = length * channels;
    if(index >= max_index) {
        syslog(LOG_ERR, "safe_index: calculated index %zu >= max %zu (i=%zu offset=%zd c=%d length=%zu channels=%d)\n",
               index, max_index, i, offset, c, length, channels);
        return 0;
    }

    return index;
}

int astrid_read_ringbuffer_block(
        lpbuffer_t * buf,
        size_t offset_in_frames, 
        lpbuffer_t * block 
    ) {
    size_t i, read_index, write_index;
    ssize_t offset;
    lpfloat_t sample;
    int c;

    if(buf == NULL) {
        syslog(LOG_ERR, "read ringbuffer block buffer is null: (%d) %s\n", errno, strerror(errno));
        return -1;
    }

    offset = buf->pos - block->length - offset_in_frames;
    while(offset < 0) offset += buf->length;

    for(i=0; i < block->length; i++) {
        for(c=0; c < block->channels; c++) {
            read_index = safe_index(i, offset, c, buf->length, buf->channels);
            write_index = safe_index(i, 0, c, block->length, block->channels);
            sample = buf->data[read_index];
            block->data[write_index] = sample;
        }
    }

    return 0;
}

int astrid_session_create(astrid_session_t * session, char * instrument_name) {
    int rc, i, dead;
    char datapath[LPMAXPATH];
    MDB_txn * txn;

    snprintf(session->instrument_name, LPMAXNAME, "%s", instrument_name);
    syslog(LOG_INFO, "astrid_session_create: PID %d opening session for %s\n", (int)getpid(), instrument_name);
    if(astrid_session_get_or_create_datadir_path(datapath) < 0) {
        syslog(LOG_ERR, "session data path (%s) mkdir: (%d) %s\n", session->datapath, errno, strerror(errno));
        return -1;
    }

    session->buffer_count = 0;

    /* Create the sempahore used to guard write access */
    snprintf(session->lock_name, LPMAXPATH, "/%s-session-write-lock", session->instrument_name);

    sem_unlink(session->lock_name); // on create recreate the session lock
    session->lock = sem_open(session->lock_name, O_CREAT, LPIPC_PERMS, 1);
    if(session->lock == SEM_FAILED) {
        syslog(LOG_ERR, "Could not create session write lock semaphore. (%d) %s\n", errno, strerror(errno));
        return -1;
    }

    /* get a handle to the environment */
    rc = mdb_env_create(&session->dbenv);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "mdb_env_create: (%d) %s\n", rc, mdb_strerror(rc));
        return -1;
    }

    /* set db params */
    rc = mdb_env_set_mapsize(session->dbenv, LMDB_MAPSIZE);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "mdb_env_set_mapsize(%ld): (%d) %s\n", LMDB_MAPSIZE, rc, mdb_strerror(rc));
        mdb_env_close(session->dbenv);
        return -1;
    }

    /* set max readers to support 30 render processes + relay + main + headroom */
    rc = mdb_env_set_maxreaders(session->dbenv, 256);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "mdb_env_set_maxreaders(256): (%d) %s\n", rc, mdb_strerror(rc));
        mdb_env_close(session->dbenv);
        return -1;
    }

    /* open or create it at the db directory */
    /* Use WRITEMAP for better space reclamation, NOSYNC for performance, NOMEMINIT to reduce memory pressure */
    rc = mdb_env_open(session->dbenv, datapath, MDB_WRITEMAP | MDB_MAPASYNC | MDB_NOSYNC | MDB_NOMEMINIT, 0664);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "mdb_env_open: (%d) %s\n", rc, mdb_strerror(rc));
        return -1;
    }

    /* clean up any stale reader slots from crashed processes */
    rc = mdb_reader_check(session->dbenv, &dead);
    if(rc == MDB_SUCCESS && dead > 0) {
        syslog(LOG_INFO, "Cleaned %d stale LMDB reader slots\n", dead);
    }

    /* create a transaction to init the dbi handle */
    rc = mdb_txn_begin(session->dbenv, NULL, 0, &txn);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "mdb_txn_begin: init (%d) %s\n", rc, mdb_strerror(rc));
        return -1;
    }

    /* open the dbi handle -- this is kept open for all future transactions */
    rc = mdb_dbi_open(txn, NULL, 0, &session->dbi);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "mdb_dbi_open: (%d) %s\n", rc, mdb_strerror(rc));
        mdb_txn_abort(txn);
        return -1;
    }
    mdb_txn_abort(txn);

    /* initialize the read transaction to be reused later -- see the LMDB docs 
     * for this optimization on resetting read transactions */
    rc = mdb_txn_begin(session->dbenv, NULL, MDB_RDONLY, &session->dbtxn_read);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "mdb_txn_begin: read (%d) %s\n", rc, mdb_strerror(rc));
        return -1;
    }
    mdb_txn_reset(session->dbtxn_read);

    /* now register the buffer counter... */
    if(astrid_session_register_shared_resource(session,
                "astrid-session-buffer-count", NULL, ASTRID_TYPE_BYTES, sizeof(ssize_t)) < 0) {
        syslog(LOG_ERR, "could not set buffer counter in LMDB\n");
        return -1;
    }

    /* init instrument message queues */
    char queue_name[LPMAXKEY];

    // Main message queue
    snprintf(queue_name, LPMAXKEY, "%s-msgq", instrument_name);
    if(astrid_msgq_init(session, queue_name) < 0) {
        syslog(LOG_ERR, "Could not initialize main message queue\n");
        return -1;
    }

    // Relay queue
    snprintf(queue_name, LPMAXKEY, "%s-relayq", instrument_name);
    if(astrid_msgq_init(session, queue_name) < 0) {
        syslog(LOG_ERR, "Could not initialize relay message queue\n");
        return -1;
    }

    // MIDI output queues
    for(i = 0; i < ASTRID_MAX_MIDI_DEVICES; i++) {
        snprintf(queue_name, LPMAXKEY, "%s-midiq-%d", instrument_name, i);
        if(astrid_msgq_init(session, queue_name) < 0) {
            syslog(LOG_ERR, "Could not initialize MIDI queue %d\n", i);
            return -1;
        }
    }

    // GPIO queue
    snprintf(queue_name, LPMAXKEY, "%s-gpioq", instrument_name);
    if(astrid_msgq_init(session, queue_name) < 0) {
        syslog(LOG_ERR, "Could not initialize GPIO message queue\n");
        return -1;
    }

	return 0;
}

int astrid_session_open(astrid_session_t * session, char * instrument_name) {
    int rc, dead;
    char datapath[LPMAXPATH];
    MDB_txn * txn;

    snprintf(session->instrument_name, LPMAXNAME, "%s", instrument_name);
    syslog(LOG_INFO, "astrid_session_open: PID %d opening session for %s\n", (int)getpid(), instrument_name);
    if(astrid_session_get_or_create_datadir_path(datapath) < 0) {
        syslog(LOG_ERR, "session data path (%s) mkdir: (%d) %s\n", session->datapath, errno, strerror(errno));
        return -1;
    }

    session->buffer_count = 0;

    /* open the sempahore used to guard write access */
    snprintf(session->lock_name, LPMAXPATH, "/%s-session-write-lock", session->instrument_name);

    session->lock = sem_open(session->lock_name, 0);
    if(session->lock == SEM_FAILED) {
        syslog(LOG_ERR, "Could not create session write lock semaphore. (%d) %s\n", errno, strerror(errno));
        return -1;
    }

    /* get a handle to the environment */
    rc = mdb_env_create(&session->dbenv);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "mdb_env_create: (%d) %s\n", rc, mdb_strerror(rc));
        return -1;
    }

    /* set db params */
    rc = mdb_env_set_mapsize(session->dbenv, LMDB_MAPSIZE);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "mdb_env_set_mapsize(%ld): (%d) %s\n", LMDB_MAPSIZE, rc, mdb_strerror(rc));
        mdb_env_close(session->dbenv);
        return -1;
    }

    /* set max readers to support 30 render processes + relay + main + headroom */
    rc = mdb_env_set_maxreaders(session->dbenv, 256);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "mdb_env_set_maxreaders(256): (%d) %s\n", rc, mdb_strerror(rc));
        mdb_env_close(session->dbenv);
        return -1;
    }

    /* open or create it at the db directory */
    /* Use WRITEMAP for better space reclamation, NOSYNC for performance, NOMEMINIT to reduce memory pressure */
    rc = mdb_env_open(session->dbenv, datapath, MDB_WRITEMAP | MDB_MAPASYNC | MDB_NOSYNC | MDB_NOMEMINIT, 0664);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "mdb_env_open: (%d) %s\n", rc, mdb_strerror(rc));
        return -1;
    }

    /* clean up any stale reader slots from crashed processes */
    rc = mdb_reader_check(session->dbenv, &dead);
    if(rc == MDB_SUCCESS && dead > 0) {
        syslog(LOG_INFO, "Cleaned %d stale LMDB reader slots\n", dead);
    }

    /* create a transaction to init the dbi handle */
    rc = mdb_txn_begin(session->dbenv, NULL, 0, &txn);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "mdb_txn_begin: init (%d) %s\n", rc, mdb_strerror(rc));
        return -1;
    }

    /* open the dbi handle -- this is kept open for all future transactions */
    rc = mdb_dbi_open(txn, NULL, 0, &session->dbi);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "mdb_dbi_open: (%d) %s\n", rc, mdb_strerror(rc));
        mdb_txn_abort(txn);
        return -1;
    }
    mdb_txn_abort(txn);

    /* initialize the read transaction to be reused later -- see the LMDB docs
     * for this optimization on resetting read transactions */
    rc = mdb_txn_begin(session->dbenv, NULL, MDB_RDONLY, &session->dbtxn_read);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "mdb_txn_begin: read (%d) %s\n", rc, mdb_strerror(rc));
        return -1;
    }
    mdb_txn_reset(session->dbtxn_read);

	return 0;
}

int astrid_session_close(astrid_session_t * session) {
    syslog(LOG_INFO, "astrid_session_close: PID %d closing session for %s\n", (int)getpid(), session->instrument_name);
    mdb_txn_abort(session->dbtxn_read);
	mdb_dbi_close(session->dbenv, session->dbi);
	mdb_env_close(session->dbenv);

    if(sem_close(session->lock) < 0) {
        syslog(LOG_ERR, "astrid_session_cleanup: sem_close could not close semaphore. (%d) %s\n", errno, strerror(errno));
        return -1;
    }

    if(sem_unlink(session->lock_name) < 0) {
        syslog(LOG_ERR, "astrid_session_cleanup: sem_unlink could not unlink semaphore. (%d) %s\n", errno, strerror(errno));
    }
    return 0;
}

int astrid_session_get_info(astrid_session_t * session, char * key, astrid_shared_resource_header_t * header) {
    if(astrid_session_read_shared_resource_header(session, header, key) < 0) {
        if(errno != 0) syslog(LOG_WARNING, "astrid_session_get_info (%d) %s\n", errno, strerror(errno));
        return -1;
    }
    return 0;
}

size_t astrid_session_get_size(astrid_session_t * session, char * key) {
    astrid_shared_resource_header_t header = {0};
    if(astrid_session_read_shared_resource_header(session, &header, key) < 0) {
        syslog(LOG_WARNING, "astrid_session_get_size (%d) %s\n", errno, strerror(errno));
        return 0;
    }
    return (size_t)header.size;
}

uint32_t astrid_session_get_type(astrid_session_t * session, char * key) {
    astrid_shared_resource_header_t header = {0};
    if(astrid_session_read_shared_resource_header(session, &header, key) < 0) {
        syslog(LOG_WARNING, "astrid_session_get_type (%d) %s\n", errno, strerror(errno));
        return 0;
    }
    return header.type;
}

int astrid_session_set_float(astrid_session_t * session, char * key, lpfloat_t val) {
    void * valp = &val;
    astrid_shared_resource_header_t header = {0};
    if(astrid_session_get_info(session, key, &header) < 0) {
        if(astrid_session_register_shared_resource(session, key, valp, ASTRID_TYPE_FLOAT, sizeof(lpfloat_t)) < 0) {
            syslog(LOG_ERR, "astrid_session_set_float: register shared resource (%d) %s\n", errno, strerror(errno));
            return -1;
        }
        return 0;
    }
    if(astrid_session_set_shared_resource(session, key, valp) < 0) {
        syslog(LOG_ERR, "astrid_session_set_float: set shared resource (%d) %s\n", errno, strerror(errno));
        return -1;
    }
    return 0;
}

lpfloat_t astrid_session_get_float(astrid_session_t * session, char * key) {
    lpfloat_t val = 0;
    if(astrid_session_get_shared_resource(session, key, &val) < 0) {
        syslog(LOG_WARNING, "astrid_session_get_float (%d) %s\n", errno, strerror(errno));
    }
    return val;
}

lpfloat_t astrid_session_get_float_from_hash(astrid_session_t * session, uint32_t key_hash) {
    MDB_txn * txn;
    MDB_val key, data;
    astrid_shared_resource_header_t header = {0};
    lpfloat_t val = 0;

    key.mv_size = sizeof(uint32_t);
    key.mv_data = &key_hash;
    data.mv_size = sizeof(astrid_shared_resource_header_t);

    int rc = mdb_txn_begin(session->dbenv, NULL, MDB_RDONLY, &txn);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "astrid_session_read_shared_resource_header mdb_txn_begin: read (%d) %s\n", rc, mdb_strerror(rc));
        return -1;
    }

    rc = mdb_get(txn, session->dbi, &key, &data);
    if(rc == MDB_SUCCESS) {
        memcpy(&header, data.mv_data, sizeof(astrid_shared_resource_header_t));
        mdb_txn_commit(txn);
    } else {
        // MDB_NOTFOUND is normal - key may not exist yet
        if(rc != MDB_NOTFOUND) {
            syslog(LOG_ERR, "astrid_session_get_shared_resource mdb_get: read (%d) %s\n", rc, mdb_strerror(rc));
        }
        mdb_txn_commit(txn);
        return 0.0f;  // Return 0 for missing keys (will use param default)
    }

    memcpy(&val, header.value, sizeof(lpfloat_t));

    return val;
}

int astrid_session_set_float_from_hash(astrid_session_t * session, uint32_t key_hash, lpfloat_t val) {
    MDB_txn * txn;
    MDB_val key, data;
    astrid_shared_resource_header_t header = {0};

    key.mv_size = sizeof(uint32_t);
    key.mv_data = &key_hash;

    // First, check if the key exists
    int rc = mdb_txn_begin(session->dbenv, NULL, MDB_RDONLY, &txn);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "astrid_session_set_float_from_hash mdb_txn_begin: (%d) %s\n", rc, mdb_strerror(rc));
        return -1;
    }

    rc = mdb_get(txn, session->dbi, &key, &data);
    if(rc == MDB_SUCCESS) {
        // Key exists, get the header
        memcpy(&header, data.mv_data, sizeof(astrid_shared_resource_header_t));
        mdb_txn_commit(txn);

        // Update the value in the header
        memcpy(header.value, &val, sizeof(lpfloat_t));

        // Write back
        rc = mdb_txn_begin(session->dbenv, NULL, 0, &txn);
        if(rc != MDB_SUCCESS) {
            syslog(LOG_ERR, "astrid_session_set_float_from_hash mdb_txn_begin write: (%d) %s\n", rc, mdb_strerror(rc));
            return -1;
        }

        data.mv_size = sizeof(astrid_shared_resource_header_t);
        data.mv_data = &header;

        rc = mdb_put(txn, session->dbi, &key, &data, 0);
        if(rc != MDB_SUCCESS) {
            syslog(LOG_ERR, "astrid_session_set_float_from_hash mdb_put: (%d) %s\n", rc, mdb_strerror(rc));
            mdb_txn_abort(txn);
            return -1;
        }

        rc = mdb_txn_commit(txn);
        if(rc != MDB_SUCCESS) {
            syslog(LOG_ERR, "astrid_session_set_float_from_hash mdb_txn_commit: (%d) %s\n", rc, mdb_strerror(rc));
            return -1;
        }

        return 0;
    } else if(rc == MDB_NOTFOUND) {
        // Key doesn't exist yet, need to register it
        mdb_txn_commit(txn);

        // Create a new header
        header.type = ASTRID_TYPE_FLOAT;
        header.size = sizeof(lpfloat_t);
        memcpy(header.value, &val, sizeof(lpfloat_t));

        // Register the new resource
        rc = mdb_txn_begin(session->dbenv, NULL, 0, &txn);
        if(rc != MDB_SUCCESS) {
            syslog(LOG_ERR, "astrid_session_set_float_from_hash mdb_txn_begin register: (%d) %s\n", rc, mdb_strerror(rc));
            return -1;
        }

        data.mv_size = sizeof(astrid_shared_resource_header_t);
        data.mv_data = &header;

        rc = mdb_put(txn, session->dbi, &key, &data, 0);
        if(rc != MDB_SUCCESS) {
            syslog(LOG_ERR, "astrid_session_set_float_from_hash mdb_put register: (%d) %s\n", rc, mdb_strerror(rc));
            mdb_txn_abort(txn);
            return -1;
        }

        rc = mdb_txn_commit(txn);
        if(rc != MDB_SUCCESS) {
            syslog(LOG_ERR, "astrid_session_set_float_from_hash mdb_txn_commit register: (%d) %s\n", rc, mdb_strerror(rc));
            return -1;
        }

        return 0;
    } else {
        syslog(LOG_ERR, "astrid_session_set_float_from_hash mdb_get: (%d) %s\n", rc, mdb_strerror(rc));
        mdb_txn_commit(txn);
        return -1;
    }
}

int astrid_session_exists(astrid_session_t * session, char * key) {
    // Compute hash and delegate to hash-based version
    return astrid_session_exists_from_hash(session, lphashstr(key));
}

int astrid_session_exists_from_hash(astrid_session_t * session, uint32_t key_hash) {
    MDB_txn * txn;
    MDB_val mdb_key, data;

    mdb_key.mv_size = sizeof(uint32_t);
    mdb_key.mv_data = &key_hash;

    int rc = mdb_txn_begin(session->dbenv, NULL, MDB_RDONLY, &txn);
    if(rc != MDB_SUCCESS) {
        syslog(LOG_ERR, "astrid_session_exists_from_hash mdb_txn_begin: (%d) %s\n", rc, mdb_strerror(rc));
        return 0;  // Assume doesn't exist on error
    }

    rc = mdb_get(txn, session->dbi, &mdb_key, &data);
    mdb_txn_commit(txn);

    // Return 1 if exists, 0 if not found
    return (rc == MDB_SUCCESS) ? 1 : 0;
}

int astrid_session_set_int(astrid_session_t * session, char * key, lpint_t val) {
    void * valp = &val;
    astrid_shared_resource_header_t header = {0};
    if(astrid_session_get_info(session, key, &header) < 0) {
        if(astrid_session_register_shared_resource(session, key, valp, ASTRID_TYPE_INT, sizeof(lpint_t)) < 0) {
            syslog(LOG_ERR, "astrid_session_set_int: register shared resource (%d) %s\n", errno, strerror(errno));
            return -1;
        }
        return 0;
    }
    if(astrid_session_set_shared_resource(session, key, valp) < 0) {
        syslog(LOG_ERR, "astrid_session_set_int (%d) %s\n", errno, strerror(errno));
        return -1;
    }
    return 0;
}

lpint_t astrid_session_get_int(astrid_session_t * session, char * key) {
    lpint_t val = 0;
    if(astrid_session_get_shared_resource(session, key, &val) < 0) {
        syslog(LOG_WARNING, "astrid_session_get_int (%d) %s\n", errno, strerror(errno));
    }
    return val;
}

int astrid_session_set_string(astrid_session_t * session, char * key, char * str) {
    astrid_shared_resource_header_t header = {0};
    if(astrid_session_get_info(session, key, &header) < 0) {
        if(astrid_session_register_shared_resource(session, key, str, ASTRID_TYPE_STRING, sizeof(char) * LPMAXMSG) < 0) {
            syslog(LOG_ERR, "astrid_session_set_string: register shared resource (%d) %s\n", errno, strerror(errno));
            return -1;
        }
        return 0;
    }
    if(astrid_session_set_shared_resource(session, key, str) < 0) {
        syslog(LOG_ERR, "astrid_session_set_string (%d) %s\n", errno, strerror(errno));
        return -1;
    }
    return 0;
}

int astrid_session_get_string(astrid_session_t * session, char * key, char * str) {
    if(astrid_session_get_shared_resource(session, key, &str) < 0) {
        syslog(LOG_WARNING, "astrid_session_get_string (%d) %s\n", errno, strerror(errno));
    }
    return 0;
}

int astrid_session_set_buffer(astrid_session_t * session, char * key, lpbuffer_t * buf) {
    astrid_shared_resource_header_t header = {0};
    size_t size = sizeof(lpbuffer_t) + (buf->length * buf->channels * sizeof(lpfloat_t));
    if(astrid_session_get_info(session, key, &header) < 0) {
        syslog(LOG_WARNING, "astrid_session_get_buffer (%d) %s\n", errno, strerror(errno));
    }

    // FIXME will this cause a race condition? maybe no need to destroy before registering
    if(header.size > 0) {
        syslog(LOG_INFO, "astrid_session_set_buffer: header.size=%ld destroying shared buffer\n", header.size);
        if(astrid_session_destroy_shared_resource(session, key) < 0) {
            syslog(LOG_ERR, "astrid_session_set_buffer: destroy shared resource (%d) %s\n", errno, strerror(errno));
            return -1;
        }
    }

    if(astrid_session_register_shared_resource(session, key, buf, ASTRID_TYPE_BUFFER, size) < 0) {
        syslog(LOG_ERR, "astrid_session_set_buffer: register shared resource (%d) %s\n", errno, strerror(errno));
        return -1;
    }
    return 0;
}

lpbuffer_t * astrid_session_get_buffer(astrid_session_t * session, char * key) {
    astrid_shared_resource_t resource;
    lpbuffer_t * buf;
    lpbuffer_t * out;

    if(astrid_session_aquire_shared_resource(session, &resource, key) < 0) {
        syslog(LOG_ERR, "astrid_session_get_buffer: aquire shared resource (%d) %s\n", errno, strerror(errno));
        return NULL;
    }

    buf = (lpbuffer_t *)resource.data;
    out = LPBuffer.clone(buf);

    if(astrid_session_release_shared_resource(session, &resource, key) < 0) {
        syslog(LOG_ERR, "astrid_session_get_buffer: release shared resource (%d) %s\n", errno, strerror(errno));
        return NULL;
    }

    return out;
}


