#!/usr/bin/env python
from setuptools import setup
from setuptools.extension import Extension
from setuptools.command.build_ext import build_ext
from Cython.Build import cythonize
import numpy as np
import os
import subprocess
import sys

# Debug mode can be enabled with DEBUG environment variable
# By default, build in production mode for wheel distribution
dev = 'DEBUG' in os.environ

INCLUDES = ['libpippi/vendor', 'libpippi/src', '/usr/local/include', np.get_include()]
MACROS = [("NPY_NO_DEPRECATED_API", "NPY_1_7_API_VERSION")]
DIRECTIVES = {}
LIBPIPPI_STATIC = 'libpippi/libpippi.a'


class BuildExtWithLibpippi(build_ext):
    """Build libpippi.a first (once) as a static shared module for all cython extensions to use"""

    def run(self):
        if not os.path.exists(LIBPIPPI_STATIC):
            print(f"Building {LIBPIPPI_STATIC}...")
            try:
                import multiprocessing
                nproc = multiprocessing.cpu_count()
            except:
                nproc = 1

            # Build libpippi.a using make
            result = subprocess.run(
                ['make', 'lib', f'-j{nproc}'],
                cwd='libpippi',
                check=False,
                capture_output=True,
                text=True
            )

            if result.returncode != 0:
                print("Error building libpippi.a:", file=sys.stderr)
                print(result.stdout, file=sys.stderr)
                print(result.stderr, file=sys.stderr)
                sys.exit(1)

            print(f"Successfully built {LIBPIPPI_STATIC}")
        else:
            print(f"{LIBPIPPI_STATIC} already exists, skipping build")

        super().run()

if dev:
    MACROS += [("CYTHON_TRACE_NOGIL", "1"), ("DEBUG", "1")]
    DIRECTIVES['profile'] = True
    DIRECTIVES['linetrace'] = True
    DIRECTIVES['binding'] = True

ext_modules = cythonize([
        Extension('astrid', [
                'astrid.pyx',
                'libpippi/vendor/linenoise/linenoise.c',
                'libpippi/vendor/libpqueue/src/pqueue.c',
                'libpippi/vendor/lmdb/libraries/liblmdb/mdb.c',
                'libpippi/vendor/lmdb/libraries/liblmdb/midl.c',
                'libpippi/vendor/ringbuf/src/ringbuf.c',
                'astrid/src/astrid.c',
            ],
            libraries=['jack', 'rt', 'asound'],
            include_dirs=INCLUDES+[
                'libpippi/vendor/libpqueue/src',
                'libpippi/vendor/linenoise',
                'libpippi/vendor/lmdb/libraries/liblmdb',
                'libpippi/vendor/ringbuf/src',
                'astrid/src'
            ],           
            define_macros=MACROS,
            extra_compile_args=['-g'],
            extra_link_args=['-g'],

            extra_objects=[LIBPIPPI_STATIC],
        ),

        Extension('pippi.events', ['pippi/events.pyx'], include_dirs=INCLUDES, define_macros=MACROS),

        Extension('pippi.defaults', ['pippi/defaults.pyx']), 
        Extension('pippi.dsp', ['pippi/dsp.pyx'], 
            include_dirs=INCLUDES + ['libpippi/vendor/fft'],
            define_macros=MACROS,
            extra_objects=[LIBPIPPI_STATIC],
        ), 
        Extension('pippi.fx', [
                'pippi/fx.pyx'
            ],
            libraries=['soundpipe'], 
            library_dirs=['/usr/local/lib'],
            include_dirs=INCLUDES + ['libpippi/vendor/fft'],
            define_macros=MACROS,
            extra_objects=[LIBPIPPI_STATIC],
        ),
        Extension('pippi.microsound', [
                'pippi/microsound.pyx'
            ],
            include_dirs=INCLUDES,
            define_macros=MACROS,
            extra_objects=[LIBPIPPI_STATIC],
        ),
        Extension('pippi.lists', ['pippi/lists.pyx'],
            include_dirs=INCLUDES, 
            define_macros=MACROS
        ),
        Extension('pippi.fft', [
                'pippi/fft.pyx'
            ], 
            include_dirs=INCLUDES + ['libpippi/vendor'],
            define_macros=MACROS,
            extra_objects=[LIBPIPPI_STATIC],
        ), 

        Extension('pippi.graph', ['pippi/graph.pyx'], 
            include_dirs=INCLUDES, 
            define_macros=MACROS,
            extra_objects=[LIBPIPPI_STATIC],
        ), 

        Extension('pippi.noise', [
                'pippi/noise/noise.pyx'
            ], 
            include_dirs=INCLUDES,
            define_macros=MACROS,
            extra_objects=[LIBPIPPI_STATIC],
        ), 
        Extension('pippi.shapes', ['pippi/shapes.pyx'], 
            include_dirs=INCLUDES, 
            define_macros=MACROS,
            extra_objects=[LIBPIPPI_STATIC],
        ), 
        Extension('pippi.mir', [
                'pippi/mir.pyx'
            ],
            include_dirs=INCLUDES, 
            define_macros=MACROS,
            extra_objects=[LIBPIPPI_STATIC],
        ),

        Extension('pippi.oscs', ['pippi/oscs.pyx'], 
            include_dirs=INCLUDES,
            define_macros=MACROS,
            extra_objects=[LIBPIPPI_STATIC],
        ), 

        Extension('pippi.midi', ['pippi/midi.pyx'], 
            include_dirs=INCLUDES,
            define_macros=MACROS
        ), 

        Extension('pippi.rand', [
                'pippi/rand.pyx',
            ], 
            include_dirs=INCLUDES, 
            define_macros=MACROS,
            extra_objects=[LIBPIPPI_STATIC],
        ), 

        Extension('pippi.soundbuffer', [
                'pippi/soundbuffer.pyx'
            ], 
            include_dirs=INCLUDES + ['libpippi/vendor/fft'],
            define_macros=MACROS,
            extra_objects=[LIBPIPPI_STATIC],
        ), 
        Extension('pippi.soundfont', ['pippi/soundfont.pyx'], 
            include_dirs= INCLUDES + ['modules/TinySoundFont'], 
            define_macros=MACROS
        ), 
        Extension('pippi.wavesets', ['pippi/wavesets.pyx'], 
            include_dirs=INCLUDES, 
            define_macros=MACROS,
            extra_objects=[LIBPIPPI_STATIC],
        ), 

        # Tuning / scales / harmony / melody
        Extension('pippi.scales', ['pippi/tune/scales.pyx'], 
            include_dirs=INCLUDES,
            define_macros=MACROS
        ), 
        Extension('pippi.intervals', ['pippi/tune/intervals.pyx'], 
            include_dirs=INCLUDES,
            define_macros=MACROS
        ), 
        Extension('pippi.frequtils', ['pippi/tune/frequtils.pyx'], 
            include_dirs=INCLUDES,
            define_macros=MACROS
        ), 
        Extension('pippi.chords', ['pippi/tune/chords.pyx'], 
            include_dirs=INCLUDES,
            define_macros=MACROS
        ), 
        Extension('pippi.tune', ['pippi/tune/tune.pyx'], 
            include_dirs=INCLUDES,
            define_macros=MACROS
        ), 
        Extension('pippi.ugens', [
                'pippi/ugens.pyx'
            ],
            include_dirs=INCLUDES, 
            define_macros=MACROS,
            extra_objects=[LIBPIPPI_STATIC],
        ), 
    ], 
    annotate=dev, 
    compiler_directives=DIRECTIVES,
    gdb_debug=dev,
) 

setup(
    ext_modules=ext_modules,
    cmdclass={'build_ext': BuildExtWithLibpippi}
)
