try:
    import pygame
    import winsound
    import platform
    import threading
    import time
    import shutil
    import subprocess
    import simpleaudio as sa
    
except ImportError as e:
    raise ImportError(f"Required module not found: {e.name}")
class ImageAPI:
    def __init__(self, app):
        self.app = app

    def LoadImage(self, image_path):
        # Load an image from the given path
        pass

    def DisplayImage(self, image):
        # Display the image
        pass

    def ResizeImage(self, image, size):
        # Resize the image to the given size
        pass

    def SaveImage(self, image, save_path):
        # Save the image to the given path
        pass

    #? ################  SOUND API #####################
class SoundAPI:
    """
    Cross-platform, best-effort Sound API.

    - Tries to use pygame.mixer (preferred) for full control (play/pause/resume/stop/volume).
    - Falls back to platform players via subprocess (afplay/aplay/paplay/mpg123) or winsound on Windows.
    - Provides PlaySound, StopSound, PauseSound, ResumeSound, SetVolume, GetVolume, IsPlaying.
    - Keeps internal state for volume and playback status even when platform player doesn't support all features.
    """

    def __init__(self, app):

        self.app = app
        self._player_type = "none"
        self._proc = None
        self._is_paused = False
        self._current_file = None
        self._volume = 1.0  # 0.0 .. 1.0
        self._loop = False

        # Try pygame mixer first (best feature set)
        try:
            pygame.mixer.init()
            self._pygame = pygame
            self._player_type = "pygame"
        except Exception:
            self._pygame = None
            # Try windows winsound (only wav, minimal control)
            try:
                self._winsound = winsound
                self._player_type = "winsound"
            except Exception:
                self._winsound = None
                # fallback to subprocess based players
                plt = platform.system().lower()
                if plt == "darwin":
                    # afplay is commonly available on macOS
                    self._player_type = "subprocess"
                    self._preferred_cmd = ["afplay"]
                elif plt == "linux":
                    # try a list of common linux players; we'll try them at runtime
                    self._player_type = "subprocess"
                    self._preferred_cmd = ["paplay", "aplay", "mpg123", "ffplay", "play"]
                else:
                    self._player_type = "subprocess"
                    self._preferred_cmd = ["afplay", "paplay", "aplay", "mpg123", "ffplay", "play"]

    def _log(self, msg):
        try:
            if getattr(self.app, "Log", None):
                self.app.Log.WriteLog("sound.log", msg)
        except Exception:
            pass

    def PlaySound(self, sound_file, loop: bool = False, blocking: bool = False):
        """
        Play a sound file.
        - sound_file: path to file
        - loop: if True, attempt to loop indefinitely (pygame supports).
        - blocking: if True, block until playback finishes (where supported).
        Returns True on success, False otherwise.
        """
        self.StopSound()  # stop current if any
        self._current_file = sound_file
        self._loop = loop
        self._is_paused = False

        try:
            if self._player_type == "pygame" and self._pygame:
                try:
                    self._pygame.mixer.music.load(sound_file)
                    self._pygame.mixer.music.set_volume(self._volume)
                    loops = -1 if loop else 0
                    self._pygame.mixer.music.play(loops=loops)
                    if blocking:
                        while self._pygame.mixer.music.get_busy():
                            self._pygame.time.wait(100)
                    return True
                except Exception as e:
                    self._log(f"pygame play error: {e}")
                    return False

            elif self._player_type == "winsound" and self._winsound:
                # winsound only supports WAV and async play; no pause/volume
                flags = self._winsound.SND_FILENAME | self._winsound.SND_ASYNC
                if not loop:
                    self._winsound.PlaySound(sound_file, flags)
                else:
                    # manual loop: play async then re-play in a thread if needed

                    def _loop_play():
                        try:
                            while True:
                                self._winsound.PlaySound(sound_file, flags)
                                # no reliable blocking call to detect end, so sleep a bit
                                time.sleep(0.5)
                                if not self._loop:
                                    break
                        except Exception:
                            pass

                    t = threading.Thread(target=_loop_play, daemon=True)
                    self._proc = t
                    t.start()
                return True

            else:
                # subprocess-based playback
                cmd = None
                if isinstance(self._preferred_cmd, (list, tuple)):
                    for c in self._preferred_cmd:
                        if shutil.which(c):
                            cmd = c
                            break
                else:
                    cmd = self._preferred_cmd if shutil.which(self._preferred_cmd) else None

                if not cmd:
                    # try python built-in simpleaudio if installed
                    try:
                        wave_obj = sa.WaveObject.from_wave_file(sound_file)
                        play_obj = wave_obj.play()
                        self._proc = play_obj
                        # simpleaudio play_obj has is_playing and stop
                        if blocking:
                            play_obj.wait_done()
                        return True
                    except Exception:
                        self._log("No playback utility found.")
                        return False

                # build command
                args = [cmd, sound_file]
                # ffplay/ffmpeg/sox variants may need flags to suppress output and auto-exit
                if cmd in ("ffplay", "ffmpeg", "play"):
                    # ffplay: -nodisp -autoexit -loglevel quiet
                    if cmd == "ffplay":
                        args = [cmd, "-nodisp", "-autoexit", "-loglevel", "quiet", sound_file]
                    elif cmd == "ffmpeg":
                        args = [cmd, "-i", sound_file, "-nodisp", "-loglevel", "quiet"]
                    elif cmd == "play":
                        args = [cmd, sound_file, ">/dev/null", "2>&1"]
                # start subprocess
                # for blocking we wait
                popen = subprocess.Popen(args, stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
                self._proc = popen
                if blocking:
                    popen.wait()
                return True

        except Exception as e:
            self._log(f"PlaySound error: {e}")
            return False

    def StopSound(self):
        """Stop playback."""
        try:
            if self._player_type == "pygame" and self._pygame:
                self._pygame.mixer.music.stop()
                self._is_paused = False
                self._current_file = None
                return True

            elif self._player_type == "winsound" and self._winsound:
                # stop winsound
                try:
                    self._winsound.PlaySound(None, self._winsound.SND_PURGE)
                except Exception:
                    try:
                        self._winsound.PlaySound(None, 0)
                    except Exception:
                        pass
                self._current_file = None
                self._is_paused = False
                return True

            else:
                if self._proc:
                    try:
                        # simpleaudio.PlayObject
                        if hasattr(self._proc, "stop"):
                            self._proc.stop()
                        else:
                            # subprocess.Popen
                            self._proc.terminate()
                        self._proc = None
                        self._current_file = None
                        self._is_paused = False
                        return True
                    except Exception:
                        try:
                            self._proc.kill()
                            self._proc = None
                            return True
                        except Exception:
                            return False
                return True
        except Exception as e:
            self._log(f"StopSound error: {e}")
            return False

    def SetVolume(self, volume: float):
        """
        Set volume 0.0 .. 1.0. Not all backends support volume; in that case value is stored.
        Returns True on success (or stored), False on error.
        """
        try:
            self._volume = max(0.0, min(1.0, float(volume)))
            if self._player_type == "pygame" and self._pygame:
                self._pygame.mixer.music.set_volume(self._volume)
            else:
                # other backends: can't set volume programmatically here
                pass
            return True
        except Exception as e:
            self._log(f"SetVolume error: {e}")
            return False

    def GetVolume(self):
        return self._volume

    def PauseSound(self):
        """Pause playback (best-effort)."""
        try:
            if self._player_type == "pygame" and self._pygame:
                self._pygame.mixer.music.pause()
                self._is_paused = True
                return True
            # winsound and subprocess fallbacks: not supported reliably
            return False
        except Exception as e:
            self._log(f"PauseSound error: {e}")
            return False

    def ResumeSound(self):
        """Resume previously paused playback (best-effort)."""
        try:
            if self._player_type == "pygame" and self._pygame:
                self._pygame.mixer.music.unpause()
                self._is_paused = False
                return True
            return False
        except Exception as e:
            self._log(f"ResumeSound error: {e}")
            return False

    def IsPlaying(self):
        """Return True if a sound is playing."""
        try:
            if self._player_type == "pygame" and self._pygame:
                return self._pygame.mixer.music.get_busy() and not self._is_paused
            elif self._player_type == "winsound" and self._winsound:
                # winsound has no query; assume if current_file set and not paused it's playing
                return bool(self._current_file and not self._is_paused)
            else:
                if not self._proc:
                    return False
                # simpleaudio.PlayObject
                if hasattr(self._proc, "is_playing"):
                    return self._proc.is_playing()
                # subprocess
                if hasattr(self._proc, "poll"):
                    return self._proc.poll() is None
                return False
        except Exception:
            return False

    def ConnectSequenceAPI(self, sequence_api):
        # Connect to a SequenceAPI instance
        self.Sequence = sequence_api

    def CreateSoundSequence(self, sound_file, sequence_name):
        # Create a sound sequence that can be triggered in SequenceAPI
        if hasattr(self, 'Sequence'):
            sound = {
                "sequence": sequence_name,
                "meta": [
                    {'instance': self, 'method': 'PlaySound', 'args': [sound_file], 'kwargs': {}}
                ]
            }
            self.Sequence.AddSequence(sound)
            return sound
        return False

class SettingsAPI:

    def __init__(self, app, settings_path: str=None):
        self.USE_SETTINGS_DICT = False
        try:
            self.app = app
            self.SETTINGSPATH = self.app.SDK.SDK_SETTINGS if not settings_path else settings_path
            self.SETTINGS = self.LoadSettings()
            self.VERSION = self.SETTINGS.get("version") if self.SETTINGS.get("version") else None
            self.LANGUAGE = self.SETTINGS.get("language") if self.SETTINGS.get("language") else None
            self.PACKAGEPATH = self.SETTINGS.get("packagepath") if self.SETTINGS.get("packagepath") else None
            self.CACHEPATH = self.SETTINGS.get("cachepath") if self.SETTINGS.get("cachepath") else None
            self.TEMPPATH = self.SETTINGS.get("temppath") if self.SETTINGS.get("temppath") else None
            self.LOGPATH = self.SETTINGS.get("logpath") if self.SETTINGS.get("logpath") else None
            self.APIPATH = self.SETTINGS.get("apipath") if self.SETTINGS.get("apipath") else None
            self.LANGUAGEPATH = self.SETTINGS.get("languagepath") if self.SETTINGS.get("languagepath") else None
            self.MODPATH = self.SETTINGS.get("modpath") if self.SETTINGS.get("modpath") else None
            self.MODS_ENABLED = self.SETTINGS.get("mods_enabled") if self.SETTINGS.get ("mods_enabled") else False
        except Exception:
            pass
        
    def AddSetting(self, key, value):
        if self.USE_SETTINGS_DICT:
            self.DICT_SETTINGS[key] = value
            return True
        try:
            self.SETTINGS[key] = value
            import json
            with open(self.SETTINGSPATH, 'w', encoding='utf-8') as f:
                json.dump(self.SETTINGS, f, indent=4)
            return True
        except Exception:
            return False

    def LoadSettings(self, own=False, settings: dict=None):
        if self.USE_SETTINGS_DICT:
            return self.DICT_SETTINGS
        try:
            import json
            if own and settings:
                return settings
            with open(self.SETTINGSPATH, 'r', encoding='utf-8') as f:
                return json.load(f)
        except Exception:
            raise FileNotFoundError(f"Einstellungsdatei nicht gefunden: {self.SETTINGSPATH}")
        
    def Global(self, key):
        if self.USE_SETTINGS_DICT:
            return self.DICT_SETTINGS.get(key, None)
        return self.SETTINGS.get(key, None)
    
    def SetUpdate(self):
        try:
            self.SETTINGS["update"] = True
            import json
            with open(self.SETTINGSPATH, 'w', encoding='utf-8') as f:
                json.dump(self.SETTINGS, f, indent=4)
        except Exception:
            return False
            
    def CheckIfUpdate(self):
        return self.SETTINGS.get("update", False)
    
    def SetSettingsPath(self, path):
        self.SETTINGSPATH = path
        self.Update()
        
    def SetSettings(self, settings: dict):
        if not isinstance(settings, dict):
            return False
        self.USE_SETTINGS_DICT = True
        self.DICT_SETTINGS = settings
        self.LoadSettings(own=True, settings=settings)
    
    
    def Update(self):
        try:
            import json
            with open(self.SETTINGSPATH, 'r', encoding='utf-8') as f:
                self.SETTINGS = json.load(f)
            self.VERSION = self.SETTINGS.get("version") if self.SETTINGS.get("version") else None
            self.LANGUAGE = self.SETTINGS.get("language") if self.SETTINGS.get("language") else None
            self.PACKAGEPATH = self.SETTINGS.get("packagepath") if self.SETTINGS.get("packagepath") else None
            self.CACHEPATH = self.SETTINGS.get("cachepath") if self.SETTINGS.get("cachepath") else None
            self.TEMPPATH = self.SETTINGS.get("temppath") if self.SETTINGS.get("temppath") else None
            self.LOGPATH = self.SETTINGS.get("logpath") if self.SETTINGS.get("logpath") else None
            self.APIPATH = self.SETTINGS.get("apipath") if self.SETTINGS.get("apipath") else None
            self.LANGUAGEPATH = self.SETTINGS.get("languagepath") if self.SETTINGS.get("languagepath") else None
            self.MODPATH = self.SETTINGS.get("modpath") if self.SETTINGS.get("modpath") else None
            self.MODS_ENABLED = self.SETTINGS.get("mods_enabled") if self.SETTINGS.get ("mods_enabled") else False
        except Exception:
            return False

    #? ################  StateMachine API #####################
    
class StateMachineAPI:
    STEP_1 = "step_1"
    STEP_2 = "step_2"
    STEP_3 = "step_3"
    STEP_4 = "step_4"
    STEP_5 = "step_5"
    EXIT = "exit"
    MAINMENU = "main_menu"
    FIRST_ENTRY = "first_entry"
    LOGIN = "login"
    VERIFIED = "verified"
    

    def __init__(self):
        """beginning with first_entry state"""
        
        self.sequenceapi = None
        self.sSTATE = self.FIRST_ENTRY
        self.STATES = []
        
   #? Single State Functions
    
    def sSetState(self, new_state):
        self.sSTATE = new_state

    def sGetState(self):
        return self.sSTATE

    def sIsState(self, check_state):
        return self.sSTATE == check_state

    def sStateIsNot(self, state: str):
        return self.sSTATE != state

   #? Single State Functions with Key (format: 'state:key')

    def sSetStateKey(self, state: str, key: str):
        self.sSTATE = f"{state}:{key}"

    def sGetStateKey(self):
        if ":" in self.sSTATE:
            return self.sSTATE.split(":")[1]
        return None
    
    def sStateKeyIs(self, key: str):
        if ":" in self.sSTATE:
            return self.sSTATE.split(":")[1] == key
        return False
    
    def sIsStateKey(self, state: str, key: str):
        if ":" in self.sSTATE:
            s, k = self.sSTATE.split(":")
            return s == state and k == key
        return False
    
    
   #? Multi State Functions #########################################
   
   
   #? Core
    def mAddState(self, statename: str, data: dict=None):
        if not any(s['state'] == statename for s in self.STATES):
            self.STATES.append({"state": statename, "data": data if data else {}})
            return True
        return False
    
    def mRemoveState(self, statename: str):
        for state in self.STATES:
            if state['state'] == statename:
                self.STATES.remove(state)
                return True
        return False
    
    def mStateExists(self, statename: str):
        return any(s['state'] == statename for s in self.STATES)
    
   #? Checks 
    
    def mStateHasKey(self, statename: str, key: str):
        for state in self.STATES:
            if state['state'] == statename:
                return key in state['data']
        return False
    
    def mStateHasValueInKey(self, statename: str, key: str, value):
        for state in self.STATES:
            if state['state'] == statename:
                return state['data'].get(key) == value
        return False
    
    
    #? Options
    def mAddKeyToState(self, statename: str, key: str, value):
        """ Adding a New Key to a State function:
        ```python
        {
            'state': 'name',
            'data': {
                'newkey', ''
            }
        }
        ```
        
        """
        for state in self.STATES:
            if state['state'] == statename:
                state['data'][key] = value
                return True
        return False
    
    
    def mVarIs(self, statename: str, key: str, var: str):
        for state in self.STATES:
            if state['state'] == statename:
                if state['data'][key] == var:
                    return True
                return False
            return False
    
    
    def mRemoveKeyFromState(self, statename: str, key: str):
        for state in self.STATES:
            if state['state'] == statename:
                if key in state['data']:
                    del state['data'][key]
                    return True
        return False
    
    def mAddValueToKey(self, statename: str, key: str, value):
        for state in self.STATES:
            if state['state'] == statename:
                if key in state['data']:
                    if isinstance(state['data'][key], list):
                        state['data'][key].append(value)
                    else:
                        state['data'][key] = [state['data'][key], value]
                else:
                    state['data'][key] = [value]
                return True
        return False
    
    def mEditValueOnKey(self, statename: str, key: str, new_value):
        for state in self.STATES:
            if state['state'] == statename:
                state['data'][key] = new_value
                return True
        return False

    
    def mGetStates(self):
        return self.STATES
    

    
    def mReadVar(self, statename: str, key: str):
        for state in self.STATES:
            if state['state'] == statename:
                return state['data'].get(key, None)
        return None
    
    def mDeleteKey(self, statename: str, key: str):
        for state in self.STATES:
            if state['state'] == statename:
                if key in state['data']:
                    del state['data'][key]
                    return True
        return False
    
    
    def mFreezeState(self, statename: str):
        for state in self.STATES:
            if state['state'] == statename:
                for i in state['data']:
                    if not i.get('frezze'):
                        i['frezze'] = True
                    else:
                        i['frezze'] = True
                return True
        return False
    
    def mUnfreezeState(self, statename: str):
        for state in self.STATES:
            if state['state'] == statename:
                for i in state['data']:
                    if i.get('frezze'):
                        i['frezze'] = False
                    else:
                        i['frezze'] = False
                return True
        return False
    
    def mIsFrozen(self, statename: str):
        for state in self.STATES:
            if state['state'] == statename:
                for i in state['data']:
                    if i.get('frezze'):
                        return True
        return False
    
    def mGetStateData(self, statename: str):
        for state in self.STATES:
            if state['state'] == statename:
                return state['data']
        return None
    
   #? ################  Sequence Functions #####################
   
    def Connect(self, sequence):
        self.sequenceapi = sequence
        
    def mLinkSequenceToState(self, statename: str, sequence_name: str, sequence_data: dict = None):
        if self.sequenceapi and self.mStateExists(statename):
            if not sequence_data:
                sequence = self.sequenceapi.ReadSequence(sequence_name)
            else:
                sequence = sequence_data

            self.mAddKeyToState(statename, "sequence", sequence)
            return True
        return False
    
    def mUnlinkSequenceFromState(self, statename: str):
        if self.mStateExists(statename):
            return self.mRemoveKeyFromState(statename, "sequence")
        return False
    
    def mRunSequenceIf(self, statename: str, if_key: str, if_value, allow_clear: bool=False, enable_header: bool=False, library: str="standard"):
        if self.sequenceapi and self.mStateExists(statename):
            if self.mVarIs(statename, if_key, if_value):
                sequence = self.mReadVar(statename, "sequence")
                if sequence:
                    seq_name = sequence['sequence'] if isinstance(sequence, dict) else sequence
                    return self.sequenceapi.DoSequence(seq_name, allow_clear=allow_clear, enable_header=enable_header, library=library)
        return False
    

class CacheAPI:
    
    def __init__(self, cache_path=None):
        try:
            self.CACHEPATH = cache_path
            if not self.CacheExists():
                import os
                os.makedirs(cache_path)
        except Exception:
            pass
    
    def SetCachePath(self, path):
        self.CACHEPATH = path
        if not self.CacheExists():
            import os
            os.makedirs(path)
        
        
    def WriteCacheFile(self, filename, content):
        with open(f"{self.CACHEPATH}/{filename}", 'w', encoding='utf-8') as f:
            f.write(content)
            
    def ReadCacheFile(self, filename):
        with open(f"{self.CACHEPATH}/{filename}", 'r', encoding='utf-8') as f:
            return f.read()
    
    def AddContent(self, filename, content):
        with open(f"{self.CACHEPATH}/{filename}", 'a', encoding='utf-8') as f:
            f.write(content + "\n")
            
    def RemoveCacheFile(self, filename):
        import os
        os.remove(f"{self.CACHEPATH}/{filename}")
        
    def CacheExists(self, filename=None):
        try:
            import os
            if filename:
                return os.path.exists(f"{self.CACHEPATH}/{filename}")
            return os.path.exists(self.CACHEPATH)
        except Exception:
            return False

    #? ################  TEMP API #####################

class TempAPI:
    
    def __init__(self, temp_path=None):
        try:
            self.TEMPPATH = temp_path
            if not self.TempExists():
                import os
                os.makedirs(temp_path)
        except Exception:
            pass
        
    def SetTempPath(self, path):
        self.TEMPPATH = path
        if not self.TempExists():
            import os
            os.makedirs(path)
        
    def WriteTempFile(self, filename, content):
        with open(f"{self.TEMPPATH}/{filename}", 'w', encoding='utf-8') as f:
            f.write(content)
            
    def ReadTempFile(self, filename):
        with open(f"{self.TEMPPATH}/{filename}", 'r', encoding='utf-8') as f:
            return f.read()
        
    def AddContent(self, filename, content):
        with open(f"{self.TEMPPATH}/{filename}", 'a', encoding='utf-8') as f:
            f.write(content + "\n")
    
    def TempExists(self, filename=None):
        try:
            import os
            if filename:
                return os.path.exists(f"{self.TEMPPATH}/{filename}")
            return os.path.exists(self.TEMPPATH)
        except Exception:
            return False

    def RemoveTempFile(self, filename=None):
        if not filename: # leere Temp ordner
            import os
            for file in os.listdir(self.TEMPPATH):
                file_path = os.path.join(self.TEMPPATH, file)
                try:
                    if os.path.isfile(file_path):
                        os.remove(file_path)
                except Exception:
                    pass
            return True
        try:
            import os
            os.remove(f"{self.TEMPPATH}/{filename}")
        except Exception:
            return False

    #? ################  PACKAGE API #####################

class PackageAPI:
    
    def __init__(self, package_path=None):
        self.PACKAGEPATH = package_path
        self.isLoggedIn = False
        self.USERNAME = None
        
    def SetPackagePath(self, path):
        self.PACKAGEPATH = path
        if not self.PackageExists():
            import os
            os.makedirs(path)
        
    def Login(self, username, password):
        if username == "admin" and password == "password":
            self.isLoggedIn = True
            self.USERNAME = username
            return True
        return False
    
    def Logout(self):
        self.isLoggedIn = False
        self.USERNAME = None
        
    def WritePackageFile(self, filename, content):
        with open(f"{self.PACKAGEPATH}/{filename}", 'w', encoding='utf-8') as f:
            f.write(content)
            
    def ReadPackageFile(self, filename):
        with open(f"{self.PACKAGEPATH}/{filename}", 'r', encoding='utf-8') as f:
            return f.read()
        
    def AddContent(self, filename, content):
        with open(f"{self.PACKAGEPATH}/{filename}", 'a', encoding='utf-8') as f:
            f.write(content + "\n")
    
    def RemovePackageFile(self, filename):
        import os
        os.remove(f"{self.PACKAGEPATH}/{filename}")
        
    #? ################  LOG API #####################
        
class LogAPI:
    
    def __init__(self, log_path=None):
        try:
            self.LOGPATH = log_path
            if not self.LogExists():
                import os
                os.makedirs(log_path)
        except Exception:
            pass
            
    def SetLogPath(self, path):
        self.LOGPATH = path
        if not self.LogExists():
            import os
            os.makedirs(path)
        
    def WriteLog(self, filename, message):
        import datetime
        timestamp = datetime.datetime.now().isoformat()
        with open(f"{self.LOGPATH}/{filename}", 'a', encoding='utf-8') as f:
            f.write(f"[{timestamp}] {message}\n")
            
    def ReadLog(self, filename):
        with open(f"{self.LOGPATH}/{filename}", 'r', encoding='utf-8') as f:
            return f.read()
        
    def DeleteLog(self, filename):
        import os
        os.remove(f"{self.LOGPATH}/{filename}")
        
    def ClearLog(self, filename):
        with open(f"{self.LOGPATH}/{filename}", 'w') as f:
            f.write("")
               
    def LogExists(self, filename=None):
        try:
            import os
            if filename:
                return os.path.exists(f"{self.LOGPATH}/{filename}")
            return os.path.exists(self.LOGPATH)
        except Exception:
            return False
        
    
    #? ################  Animation API #####################
    
class SequenceAPICreator:
    def __init__(self, sequence_api):
        self.sequence_api = sequence_api

    def CreateStateSequence(self, sequence_name: str, instance = None, method = None, lambda_func = None, args: list=None, kwargs: dict=None):
        sequence = {
            "sequence": sequence_name,
            "meta": []
        }
        if instance and method:
            sequence["meta"].append({
                "instance": instance,
                "method": method,
                "args": args if args else [],
                "kwargs": kwargs if kwargs else {}
            })
        if lambda_func:
            sequence["meta"].append({
                "instance": None,
                "method": lambda_func,
                "args": args if args else [],
                "kwargs": kwargs if kwargs else {}
            })
        added = self.sequence_api.AddSequence(sequence)
        if added:
            return sequence
        return None
    
    def CreateUrsinaSequence(self, sequence_name: str, ursina_entity, animation_type: str, duration: float, target_value, loop: bool=False):
        sequence = {
            "sequence": sequence_name,
            "meta": [
                {
                    "instance": ursina_entity,
                    "method": f"animate_{animation_type}",
                    "args": [target_value, duration],
                    "kwargs": {"loop": loop}
                }
            ]
        }
        added = self.sequence_api.AddSequence(sequence)
        if added:
            return sequence
        return None
    
    def CreateSequence(self, sequence_name: str, actions: list):
        sequence = {
            "sequence": sequence_name,
            "meta": actions
        }
        added = self.sequence_api.AddSequence(sequence)
        if added:
            return sequence
        return None


class SequenceAPI:

    def __init__(self, app):
        self.app = app
        self.Sequences = []
        self.Creator = SequenceAPICreator(self)
        
    def AddSequence(self, sequence: dict):
        """A Sequence is in this Format:
        ```python
        your_sequence = {
            "sequence": "my_sequence_name",
            "meta": [
                {'instance': your_instance, 'method': 'method_name', 'args': [arg1, arg2], 'kwargs': {'key': value}},
                {'instance': None, 'method': lambda x: print(x), 'args': ['Hello'], 'kwargs': {}}
            ]
        ```
        """
        if not any(s['sequence'] == sequence['sequence'] for s in self.Sequences):
            self.Sequences.append(sequence)
            return True
        return False
    
    def RemoveSequence(self, sequence_name: str):
        for seq in self.Sequences:
            if seq['sequence'] == sequence_name:
                self.Sequences.remove(seq)
                return True
        return False
    
    def SequenceExists(self, sequence_name: str):
        return any(s['sequence'] == sequence_name for s in self.Sequences)
    
    def ReadSequence(self, sequence_name: str):
        for seq in self.Sequences:
            if seq['sequence'] == sequence_name:
                return seq
        return None
    
    def EditSequenceInstance(self, sequence_name: str, new_instance):
        for seq in self.Sequences:
            if seq['sequence'] == sequence_name:
                for action in seq['meta']:
                    action['instance'] = new_instance
                return True
        return False
    
    def EditSequenceMethod(self, sequence_name: str, action_index: int, new_method):
        for seq in self.Sequences:
            if seq['sequence'] == sequence_name:
                if 0 <= action_index < len(seq['meta']):
                    seq['meta'][action_index]['method'] = new_method
                    return True
        return False
    
    def EditSequenceArgs(self, sequence_name: str, action_index: int, new_args: list):
        for seq in self.Sequences:
            if seq['sequence'] == sequence_name:
                if 0 <= action_index < len(seq['meta']):
                    seq['meta'][action_index]['args'] = new_args
                    return True
        return False
    
    def EditSequenceKwargs(self, sequence_name: str, action_index: int, new_kwargs: dict):
        for seq in self.Sequences:
            if seq['sequence'] == sequence_name:
                if 0 <= action_index < len(seq['meta']):
                    seq['meta'][action_index]['kwargs'] = new_kwargs
                    return True
        return False
    
    def DoSequence(self, sequence: str, allow_clear: bool=False, enable_header: bool=False, library: str="standard"):
        if library == "standard":
            for seq in self.Sequences:
                if seq['sequence'] == sequence:
                    if enable_header:
                        self.app.Log.WriteLog(f"{self.app.Settings.LOGPATH}/sequence.log", f"Starting sequence: {sequence}")
                    for action in seq['meta']:
                        instance = action.get('instance')
                        method = action.get('method')
                        args = action.get('args', [])
                        kwargs = action.get('kwargs', {})
                        if instance and method:
                            func = getattr(instance, method, None)
                            if callable(func):
                                func(*args, **kwargs)
                        elif callable(method):  # if method is a lambda or function
                            method(*args, **kwargs)
                    if enable_header:
                        self.app.Log.WriteLog(f"{self.app.Settings.LOGPATH}/sequence.log", f"Finished sequence: {sequence}")
                    if allow_clear:
                        self.RemoveSequence(sequence)
                    return True
            return False
        else:
            raise NotImplementedError(f"Library '{library}' not implemented in SequencesAPI Use 'standard' instead.")
        

        
        
        
    #? ################  MANAGER API #####################

class ManagerAPI:
    
    def __init__(self):
        pass
        
        
        
    #? ################  GUI API #####################
    
class GuiAPI:
    
    def __init__(self):
        pass
        
    #? ################  HELPER API #####################

class HelperAPI:
    
    def __init__(self, app):
        self.app = app

        self.ui = GuiAPI()
        self.command = CommandAPI(app)
        self.Sound = SoundAPI(app)
        self.Image = ImageAPI(app)


    def GetVersion(self):
        return self.app.Settings.VERSION

    def GetLanguage(self):
        return self.app.Settings.LANGUAGE
    
    #? ################  COMMAND API #####################

class CommandAPI:

    def __init__(self, app):
        try:
            self.app = app
        except Exception:
            pass

    def Execute(self, command):
        import subprocess
        result = subprocess.run(command, shell=True, capture_output=True, text=True)
        return result.stdout, result.stderr, result.returncode
    
    
    
    #? #################### Developer API #####################
    
    
class DeveloperAPI:
    
    def __init__(self):
        self.DEBUG = False
    
    def DebugInfo(self, message):
        if self.DEBUG:
            print(f"[DEBUG] {message}")
        else:
            return None
        
    def EnableDebug(self):
        self.DEBUG = True
        
    def DisableDebug(self):
        self.DEBUG = False
    
    
    
    #? ####################  AI API #####################
    
    
class AiAPI:
    def __init__(self, api_key=None, model="gpt-4", temperature=0.7):
        self.api_key = api_key
        self.model = model
        self.temperature = temperature
        
    def SetApiKey(self, api_key):
        self.api_key = api_key
        
    def GenerateText(self, prompt):
        if not self.api_key:
            raise ValueError("API key is not set.")
        import openai
        openai.api_key = self.api_key
        response = openai.Completion.create(
            engine=self.model,
            prompt=prompt,
            temperature=self.temperature,
            max_tokens=150
        )
        return response.choices[0].text.strip()



    #? ################  LANGUAGE API #################


class LanguageAPI:

    def __init__(self, settings, standard_library=True):
        try:
            self.Settings = settings
            self.LANGUAGE = self.Settings.Global("language")
            self.LANGUAGEPATH = self.Settings.Global("LANGUAGEPATH")
            self.PACKAGES = []
            if standard_library:
                import os
                package_dir = os.path.dirname(os.path.abspath(__file__))
                self.LANGUAGEPATH = os.path.join(package_dir, "data", "lang")
            self.language_data = self.LoadLanguageData(self.LANGUAGE)
        except Exception:
            pass
        
    #? Core Functions

    # Reloading language data (e.g. after changing language in settings or adding new language-packs)
    def Reload(self):
        """Reloading Language-Data and applied Language-Packages"""
        self.LANGUAGE = self.Settings.Global("language")
        self.language_data = self.LoadLanguageData(self.LANGUAGE)
        if self.PACKAGES:
            for package in self.PACKAGES:
                if package["language"] == self.LANGUAGE:
                    self.language_data.update(package["data"])

    def SetLanguageData(self, keys: dict=None, prefered_lang_reference=False):
        if prefered_lang_reference:
            # Verwende toolos package data/lang Verzeichnis
            import os
            package_dir = os.path.dirname(os.path.abspath(__file__))
            self.LANGUAGEPATH = os.path.join(package_dir, "data", "lang")
            self.language_data = self.LoadLanguageData(self.LANGUAGE)
        elif keys:
            self.language_data = keys
    
    # Loading Original Language-Data json formats from /assets/manager/lang/{'de', 'en', 'ru',..}.json    
    def LoadLanguageData(self, language):
        """Loading Language-Data by parameter: language"""
        import json
        try:
            with open(f"{self.LANGUAGEPATH}/{language}.json", 'r', encoding='utf-8') as f:
                return json.load(f)
        except FileNotFoundError:
            try:
                with open(f"{self.LANGUAGEPATH}/de.json", 'r', encoding='utf-8') as f:
                    return json.load(f)
            except FileNotFoundError:
                return {}

    #? Interaction Functions
    
    def Translate(self, key):
        """Translating Keyword by key with current language-data"""
        return self.language_data.get(key, key)
    
    def GetAllTranslationKeys(self):
        """Returning all translation keys"""
        return list(self.language_data.keys())
    
    def GetAvailableLanguages(self):
        """Returning all available languages from {self.LANGUAGEPATH}"""
        import os
        files = os.listdir(self.LANGUAGEPATH)
        languages = [f.split('.')[0] for f in files if f.endswith('.json')]
        return languages
    
    def AddLanguagePackage(self, language, datapath):
        import json
        with open(datapath, 'r', encoding='utf-8') as f:
            data = json.load(f)
        self.PACKAGES.append({"language": language, "data": data})

   
   

    #? ################ Plugin API #####################
    
    
class PluginAPI:
    
    def __init__(self):
        self.plugins = []
        
    def AddPlugin(self, plugin, call):
        self.plugins.append({"plugin": plugin, "call": call})

    def RemovePlugin(self, call):
        for i in self.plugins:
            if i.get('call') == call:
                self.plugins.remove(i)
                return True
        return False

    def GetPlugin(self, call):
        for i in self.plugins:
            if i.get('call') == call:
                return i.get('plugin', None)
        return None
    
    def ListPlugins(self):
        return self.plugins
    
    def GetPluginInstances(self):
        return [p['plugin'] for p in self.plugins if 'plugin' in p]



    #? ################  APP API #####################
    
    
    
class AppAPI:
    
    def __init__(self, app):
        self.app = app
        self.MENU = []
        self.IMENU = []
        
    
    def BuildMenu(self, menus: list=None, start=0):
        if not menus:
            menu = self.MENU if not None else []
        else:
            menu = menus
        for i, key in enumerate(menu, start=start):
            self.InteractiveMenu = {
                "index": i,
                "name": key,
                "lambda": None
            }
            self.IMENU.append(self.InteractiveMenu)
            
    def AddLambdaToMenu(self, index, func):
        for item in self.IMENU:
            if item["index"] == index:
                item["lambda"] = func
                return True
        return False
    
    def ClearMenu(self):
        self.MENU = []
        self.IMENU = []
            
    def ShowMenu(self, menus: list=None):
        if menus:
            for i, key in enumerate(menus):
                print(f"{i}: {key}")
        else:
            for item in self.IMENU:
                print(f"{item['index']}: {item['name']}")

    def SelectMenuLambda(self, index):
        for item in self.IMENU:
            if item["index"] == index and item["lambda"]:
                return item["lambda"]
                    
                
    def SelectMenu(self, index, use_imenu: bool=False):
        if use_imenu:
            for item in self.IMENU:
                if item["index"] == index:
                    return item["name"]
        else:
            if index < len(self.MENU):
                return self.MENU[index]
        return None
    
    def GetIndexAndKey(self, index):
        for item in self.IMENU:
            if item["index"] == index:
                return item["name"], item["lambda"] if item["lambda"] else None
        return None, None
    
    def AskInput(self, input_style=None):
        if input_style == "terminal":
            return input("$ ")
        return input("> ")
            
        
        

    #? ################  TOOL API #####################

# class ToolAPI:

    # def __init__(self, sdk: dict=None, settings_path: str=None, enable_languages: bool=True):
    #     """Requires sdk{version, name}. Build for ToolOS
        
    #     # OUTDATED - use Api class instead!"""
    #     self.SDK = SDK(sdk)
    #     self.Settings = SettingsAPI(self)
    #     if self.CheckCompatibility(self.Settings.VERSION, self.SDK.SDK_VERSION):
    #         self.Cache = CacheAPI(self.Settings.CACHEPATH)
    #         self.Temp = TempAPI(self.Settings.TEMPPATH)
    #         self.Package = PackageAPI(self.Settings.PACKAGEPATH)
    #         self.Log = LogAPI(self.Settings.LOGPATH)
    #         self.manager = ManagerAPI()
    #         self.helper = HelperAPI(self)
    #         self.language = LanguageAPI(self.Settings, standard_library=self.SDK.SDK_LangLib)
    #         self.state_machine = StateMachineAPI()
    #         self.app = AppAPI(self)

    # def CheckCompatibility(self, api_version, sdk_version: str):
    #     major, minor, patch = sdk_version.split(".")
    #     if major != api_version.split(".")[0]:
    #         raise ValueError(f"Inkompatible Versionen: API {api_version} != SDK {sdk_version}")
    #     return True

    #? ################  Global API #####################
    
class Api:
    def __init__(self, sdk: dict=None, settings_path: str=None, enable_languages: bool=True, settings: dict=None, crack_bot: bool=False):
        """
            ![ClayTech](file:///C:/Users/hatte/Downloads/py_claytech_badge.svg)
            ![Python](https://img.shields.io/badge/python-3670A0?style=for-the-badge&logo=python&logoColor=ffdd54)
            ## ToolAPI's API-SDK. made for general use.
        
            ## Parameters:
            ```python
            sdk: dict = None
            settings_path: str = None
            enable_languages: bool = True
            settings: dict = None
            crack_bot: bool = False
            ```
            
            ## API
            ```python
            self.Settings  # SettingsAPI
            self.Cache  # CacheAPI
            self.Temp  # TempAPI
            self.Package  # PackageAPI
            self.Log  # LogAPI
            self.Manager  # ManagerAPI
            self.Helper  # HelperAPI
            self.Language  # LanguageAPI
            self.StateMachine  # StateMachineAPI
            self.App  # AppAPI
            self.SDK  # SDK
            ```
            # Language Api (Mini Reference)
            
            add language package:
            
            ```python
            self.Language.AddLanguagePackage(language, datapath)
            ```
            
            translate key:
            ```python
            self.Language.Translate("hello_world")
            ```
            
            get all translation keys:
            ```python
            self.Language.GetAllTranslationKeys()
            ```
            
            get available languages:
            ```python
            self.Language.GetAvailableLanguages()
            ```
            
            reload language data:
            ```python
            self.Language.Reload()
            ```
            
            ## StateMachine Api (Mini Reference)
            
            ### Single State Functions:
            ```python
            self.StateMachine.sSetState()
            self.StateMachine.sGetState()
            ```
            
            ### Single State Functions with Key ('state:key')
            
            ```python
            self.StateMachine.sSetStateKey()
            self.StateMachine.sGetStateKey()
            ```
            
            ### Multi State Functions:
            
            ```python
            self.StateMachine.mAddState(state="state_name", key={"key1": "value1"})
            self.StateMachine.mGetStates()
            ```
            # Details
            Last Updated: 04.10.25
            Version: v2.6.3
            Api-Reference: https://pypi.org/project/toolos/
            Author: Lilias Hatterscheidt
            Copyright © 2025 ClayTechnologie. All rights reserved.
            
            
            """
        self.OwnSettings = settings
        
        self.CACHEPATH = None
        self.TEMPPATH = None
        self.PACKAGEPATH = None
        self.LOGPATH = None
        
        self.SDK = SDK(sdk)
        self.Settings = SettingsAPI(self, settings_path=settings_path if settings_path else None)
        if not self.SDK.SDK_AVAILABLE:
            settings_path = settings_path
        if self.OwnSettings:
            self.Settings.SetSettings(settings=self.OwnSettings)
            if not self.Settings.USE_SETTINGS_DICT:
                self.CACHEPATH = None
                self.TEMPPATH = None
                self.PACKAGEPATH = None
                self.LOGPATH = None
            
            self.CACHEPATH = self.Settings.Global("CACHEPATH")
            self.TEMPPATH = self.Settings.Global("TEMPPATH")
            self.PACKAGEPATH = self.Settings.Global("PACKAGEPATH")
            self.LOGPATH = self.Settings.Global("LOGPATH")
        
        self.Cache = CacheAPI(self.CACHEPATH)
        self.Temp = TempAPI(self.TEMPPATH)
        self.Package = PackageAPI(self.PACKAGEPATH)
        self.Log = LogAPI(self.LOGPATH)
        self.Manager = ManagerAPI()
        self.Helper = HelperAPI(self)
        self.Language = LanguageAPI(self.Settings, standard_library=self.SDK.SDK_LangLib if not enable_languages else False)
        self.StateMachine = StateMachineAPI()
        self.App = AppAPI(self)
        self.Plugin = PluginAPI()
        self.Dev = DeveloperAPI()
        self.Sequence = SequenceAPI(self)
        self.Ai = AiAPI()
        
        if crack_bot:
            self.Bot = BotDriver(self, self.Settings, settings_path if settings_path else None)
        
    #? ################  SDK #####################

class SDK:

    def __init__(self, sdk: dict):
        """ToolAPI's SDK. made for developers."""
        try:
            self.SDK = sdk
            self.SDK_VERSION = sdk.get("version", "2.4.7")
            self.SDK_SETTINGS = sdk.get("settings_path")
            self.SDK_NAME = sdk.get("name")
            self.SDK_LangLib = sdk.get("standard_language_library")
            self.SDK_AVAILABLE = True
            self.SDK_SOURCETAR = self.GetSDKSuperManifest()
        except Exception:
                self.SDK_AVAILABLE = False
                
    def GetSDKSuperManifest(self):
        import secrets
        import hashlib
        token = secrets.token_hex(len(self.SDK_VERSION))
        return hashlib.sha256(token.encode()).hexdigest()


class BotDriver:

    def __init__(self, settings: dict, config_path: str, metadata: list[dict, dict, dict]=None, app = None):
        """Provides a full Discord-Bot Development Environment with several helper functions and Api-based Working
        The settings dict must contain at least the following keys: BOT_APIKEY, GUILD_ID
        # Use built-ins!:
         
        - self.discord
        - self.commands
        - self.tasks
        - self.app_commands
        - self.asyncio
        - self.os
        # Use Settings: 
        - self.Settings
        - self.Config
        - self.BotMetadata
        - self.ApiKey
        - self.GuildId"""
        import discord
        import discord.ext.commands as commands
        import discord.ext.tasks as tasks
        import discord.app_commands as app_commands
        import asyncio
        import os

        self.BotMetadata = metadata if metadata else None
        self.Settings = settings
        self.Config = config_path
        self.discord = discord
        self.commands = commands
        self.tasks = tasks
        self.app_commands = app_commands
        self.asyncio = asyncio
        self.os = os
        self.app = app
        try:
            self.Settings.SetSettings(settings=self.Settings)
            self.Settings.update()
            self.ApiKey = self.Settings.Global("BOT_APIKEY")
            self.GuildId = self.Settings.Global("GUILD_ID")
            
        except Exception:
            self.SETTINGS_LOADED = None
        finally:
            self.SETTINGS_LOADED = True
            self.Config = config_path
            
            
    def GetApiKey(self):
        return self.ApiKey
    
    def GetGuildId(self):
        return self.GuildId
    
    def GetConfigPath(self):
        return self.Config
    
    
    def RunBot(self, key: str = None):
        self.bot = self.commands.Bot(command_prefix="!", intents=self.discord.Intents.all())
        self.bot.run(key if not None else self.ApiKey)
        