from .templates import autotest, utils, main, run, nonpies
from .templates import requirements
from pathlib import Path
import textwrap as wrap
from . import helper
import shutil
import os

class Constructor:
    def __init__(self, pro_name:str, author:str, lic:str, 
                 path: str, dual_lic: str | None = None, 
                 no_lic:bool = False, minimal: bool = False, 
                 force: bool = False):
        self.name        = pro_name
        self.author      = author
        self.license     = lic
        self.dual_lic    = dual_lic
        self.no_license  = no_lic
        self.minimal     = minimal
        self.force       = force
        self.DRT_DIR     = Path(path)/self.name
        self.SRC_DIR     = self.DRT_DIR/"src"
        self.PRO_DIR     = self.SRC_DIR/self.name.lower()
        self.setup_project()
    
    def create_folders(self) -> None:
        if self.force and os.path.exists(self.DRT_DIR):
            otp = helper.gen_otp()
            pin = input(utils.style_text(f"\nEnter {otp!r}"
                + " to overwrite directory:\n"
                +f"'{self.DRT_DIR!s}': ", "yellow"))
            if pin == otp: shutil.rmtree(self.DRT_DIR)
            else: helper.err(otp=True)
            
        dirs = [self.DRT_DIR, self.SRC_DIR, self.PRO_DIR, 
                self.SRC_DIR/"tests"]
                
        for i, path in enumerate(dirs):
            if i == len(dirs) - 1 and self.minimal: continue
            try: path.mkdir(exist_ok=self.force)
            except FileExistsError: continue

    def create_drt_files(self) -> None:
        LICENSES = []
        if not self.no_license:
            LICENSES.append(nonpies.get_license(
                self.license, self.author))
            if self.dual_lic: 
                LICENSES.append(nonpies.get_license(
                    self.dual_lic, self.author))

        toml      = nonpies.create_toml(self.name, 
                    self.author)
        readme    = nonpies.readme
        gitignore = nonpies.gitignore
        inpy      = [run, requirements]
        nonpy     = LICENSES + [toml, readme, gitignore]
        files     = ["LICENSE" if i==0 else f"LICENSE-{i+1}" 
                for i in range(len(LICENSES))] + [
                    "pyproject.toml", "README.md", 
                    ".gitignore"]
        paths     = ["run.py", "requirements.txt"]

        for module, path in zip(inpy, paths):
            with open(module.__file__) as template:
                with open(self.DRT_DIR/path, "w") as file:
                    file.write(template.read())

        for info, path in zip(nonpy, files):
            with open(self.DRT_DIR/path, "w") as file:
                file.write(info)

    def create_src_files(self) -> None:
        n, inpy = self.name.lower(), [autotest, utils, main]
        paths = ["autotests.py", "utils.py", f"{n}/main.py"]
        if self.minimal:
            inpy, paths = [main], [f"{n}/main.py"]

        for module, path in zip(inpy, paths):
            with open(module.__file__) as template:
                with open(self.SRC_DIR/path, "w") as file:
                    file.write(template.read())

        with open(self.PRO_DIR/"__init__.py", "w") as file:
            file.write(f"# Project: {self.name}\n")
            file.write(f"# Author: {self.author}\n")
            file.write(f"# License: {self.license}")
            if self.dual_lic:
                file.write(f", {self.dual_lic}")
            file.write("\n# Generated by Droot\n")
    
    def setup_project(self) -> None:
        name = self.name
        self.create_folders()
        self.create_drt_files()
        self.create_src_files()
        done_msg =(f"Created project {name!r} in "
                 + f"{self.DRT_DIR.parent}")
        print(f"\n{utils.style_text(done_msg, 'green')}\n")