
from .elements import DrawableLine

from ... import config

class UserRenderer:

    # sad global
    _global_ui = None
    _global_robot = None
    _renderers = []
    
    @classmethod
    def _attach_ui(cls, ui, robot):
        cls._global_ui = ui
        cls._global_robot = robot
        for renderer in cls._renderers:
            for deferred in renderer._deferred:
                deferred()
            del renderer._deferred
    
    def __init__(self):
        self._elements = []
        self._deferred = []
        self._renderers.append(self)

    def clear(self):
        for e in self._elements:
            e.delete()
            
    # draw a line in robot coordinates
    
    # draw a line in field coordinates
    # -> maybe just convert?
    
    # draw pathfinder?
    def draw_pathfinder_trajectory(self, trajectory, color='#ff0000', offset=None, **kwargs):
        '''
            Special helper function for drawing trajectories generated by
            robotpy-pathfinder
            
            :param trajectory: A list of pathfinder segment objects
            :param offset: If specified, should be x/y tuple to add to the path
                           relative to the robot coordinates
            :param kwargs: Keyword options to pass to tkinter.create_line
        '''
        # pathfinder x/y coordinates are switched
        pts = [(pt.x, -pt.y) for pt in trajectory]
        robot_coordinates = offset if offset else True
        self.draw_line(pts, color=color, robot_coordinates=robot_coordinates,
                      relative_to_first=True, arrow=True)
        
    def draw_line(self, line_pts, color='#ff0000',
                  robot_coordinates=False, relative_to_first=False,
                  arrow=True, **kwargs):
        '''
            :param line_pts: A list of (x,y) pairs to draw. (x,y) are in field units
                             which are measured in feet
            :param color: The color of the line, expressed as a 6-digit hex color
            :param robot_coordinates: If True, the pts will be adjusted such that
                                      the first point starts at the center
                                      of the robot and that x and y coordinates
                                      are rotated according to the robot's
                                      current heading. If a tuple, then the pts
                                      are adjusted relative to the robot center
                                      AND the x,y in the tuple
            :param relative_to_first: If True, the points will be adjusted such
                                      that the first point is considered to be
                                      (0,0)
            :param arrow: If True, draw the line with an arrow at the end
            :param kwargs: Keyword options to pass to tkinter.create_line
        '''
        def _defer(): # called later because the field might not exist yet
            px_per_ft = UserRenderer._global_ui.field.px_per_ft
            if arrow:
                kwargs['arrow'] = 'last'
            line = DrawableLine([(x*px_per_ft, y*px_per_ft) for x,y in line_pts], color, kwargs)
            
            # if relative to first, create object, then move relative to the first
            if relative_to_first:
                line.move((-line.pts[0][0], -line.pts[0][1]))
            
            if robot_coordinates:
                x, y = UserRenderer._global_robot.center
                angle = UserRenderer._global_robot.angle
                line.move((x,y))
                
                if isinstance(robot_coordinates, (list, tuple)):
                    line.move((px_per_ft*robot_coordinates[1], px_per_ft*robot_coordinates[0]))
                
                # for rotation to work, rotate it about the front center of the robot
                line.center = (x,y)
                line.rotate(angle)
                
            line.update_coordinates()
            UserRenderer._global_ui.field.add_moving_element(line)
            
        self._run(_defer)
        
    def _run(self, d):
        # TODO: need to idle_add this
        if UserRenderer._global_ui:
            UserRenderer._global_ui.idle_add(d)
        else:
            self._deferred.append(d)


def get_user_renderer():
    '''
        This retrieves an object that can be used to draw on the simulated field
        when running the robot in simulation.
    
        :returns: None if no renderers are available, else a :class:`.UserRenderer` object
    '''
    if config.mode != 'sim':
        return

    return UserRenderer()
  
