import asyncio
import random

from auditora.decorators.sentinel import sentinel
from auditora.core.proxies import session, monitor, report


# --- 1. Sync Data Processing Step ---
@sentinel(session_id="pipeline-session")
def preprocess_data(data: list[int]) -> list[int]:
    report.info("Starting preprocessing", data_size=len(data))
    session.add_tag("preprocessing")

    start = monitor.start_timer("preprocess")
    result = [x * 2 for x in data]  # mock transformation
    duration = monitor.stop_timer("preprocess", start, records=len(result))

    report.info("Preprocessing complete", duration=duration, result_preview=result[:3])
    return result


# --- 2. Async LLM Agent 1 ---
@sentinel()
async def agent_one_call(query: str) -> str:
    session.add_tag("agent_one")
    report.debug("Agent 1 received query", query=query)

    await asyncio.sleep(random.uniform(0.3, 0.6))  # simulate latency
    response = f"[AgentOne response to: {query}]"

    report.log_llm_call(
        model="agent-one-llm",
        prompt_tokens=len(query),
        completion_tokens=42,
        response_time=0.5,
    )
    return response


# --- 3. Async LLM Agent 2 ---
@sentinel()
async def agent_two_call(query: str) -> str:
    session.add_tag("agent_two")
    report.debug("Agent 2 received query", query=query)

    await asyncio.sleep(random.uniform(0.4, 0.7))  # simulate latency
    response = f"[AgentTwo response to: {query}]"

    report.log_llm_call(
        model="agent-two-llm",
        prompt_tokens=len(query),
        completion_tokens=36,
        response_time=0.6,
    )
    return response


# --- Orchestration ---
async def main():
    processed = preprocess_data([1, 2, 3, 4])

    results = await asyncio.gather(
        agent_one_call("Summarize dataset stats"),
        agent_two_call("Generate visualization hints"),
    )

    print("\n=== SESSION SNAPSHOT ===")
    print(preprocess_data._session.get_state_snapshot())

    print("\n=== MONITOR SUMMARY ===")
    print(preprocess_data._monitor.get_summary())

    print("\n=== REPORT INSTANCE ===")
    print(preprocess_data._report)

    print("\n=== Pipeline Output ===")
    print("Processed Data:", processed)
    print("Agent Responses:", results)


if __name__ == "__main__":
    asyncio.run(main())
