"""
Usage Example: Thread-Local Context Demonstration with Auditora

- Two mock functions: data processing and LLM call
- Both decorated with @sentinel
- Run in separate threads to illustrate context isolation
"""
from auditora import sentinel, session, monitor, report

import threading
import time
import random

@sentinel(session_id="data-processing-session")
def mock_process_documents(documents):
    report.info(f"Starting processing of {len(documents)} documents")
    session.add_tag("data_pipeline")

    results = []
    for i, doc in enumerate(documents):
        time.sleep(random.uniform(0.05, 0.1))
        results.append(doc.upper())
        monitor.track("document_processed", index=i, original_doc=doc)
        report.info(f"Processed document '{doc}'")

    session.set("processed_docs", results)
    return results

@sentinel(session_id="llm-call-session")
def mock_llm_call(documents):
    report.info(f"Starting LLM processing of {len(documents)} documents")
    session.add_tag("llm_pipeline")

    summaries = []
    for i, doc in enumerate(documents):
        time.sleep(random.uniform(0.05, 0.1))
        summary = f"llm_summary_of_{doc}"
        summaries.append(summary)
        monitor.track("llm_document_processed", index=i, doc=doc)
        report.info(f"LLM processed document '{doc}'", response=summary)

    session.set("llm_summaries", summaries)
    return summaries

def run_in_thread(func, args):
    result = func(*args)
    print(f"\n=== {func.__name__} SESSION SNAPSHOT ===")
    print(func._session.get_state_snapshot())
    print(f"\n=== {func.__name__} MONITOR SUMMARY ===")
    print(func._monitor.get_summary())
    print(f"\n=== {func.__name__} REPORT INSTANCE ===")
    print(func._report)
    print(f"\n=== {func.__name__} OUTPUT ===")
    print(result)
    print("="*50)

if __name__ == "__main__":
    documents = ["alpha", "beta", "gamma", "delta"]

    # Launch each function in a separate thread
    threads = [
        threading.Thread(target=run_in_thread, args=(mock_process_documents, (documents,))),
        threading.Thread(target=run_in_thread, args=(mock_llm_call, (documents,))),
    ]

    for t in threads:
        t.start()
    for t in threads:
        t.join()

