"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CognitoToApiGatewayToLambda = void 0;
const api = require("@aws-cdk/aws-apigateway");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
class CognitoToApiGatewayToLambda extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the CognitoToApiGatewayToLambda class.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        // This Construct requires that the auth type be COGNITO regardless of what is specified in the props
        if (props.apiGatewayProps) {
            if (props.apiGatewayProps.defaultMethodOptions === undefined) {
                props.apiGatewayProps.defaultMethodOptions = {
                    authorizationType: api.AuthorizationType.COGNITO,
                };
            }
            else if (((_a = props.apiGatewayProps) === null || _a === void 0 ? void 0 : _a.defaultMethodOptions.authorizationType) === undefined) {
                props.apiGatewayProps.defaultMethodOptions.authorizationType = api.AuthorizationType.COGNITO;
            }
            else if (((_b = props.apiGatewayProps) === null || _b === void 0 ? void 0 : _b.defaultMethodOptions.authorizationType) !== 'COGNITO_USER_POOLS') {
                defaults.printWarning('Overriding Authorization type to be AuthorizationType.COGNITO');
                props.apiGatewayProps.defaultMethodOptions.authorizationType = api.AuthorizationType.COGNITO;
            }
        }
        if (props.apiGatewayProps && (typeof props.apiGatewayProps.proxy !== 'undefined') && (props.apiGatewayProps.proxy === false)) {
            defaults.printWarning('For non-proxy API, addAuthorizers() method must be called after all the resources and methods for API are fuly defined. Not calling addAuthorizers() will result in API methods NOT protected by Cognito.');
        }
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] =
            defaults.GlobalLambdaRestApi(this, this.lambdaFunction, props.apiGatewayProps);
        this.userPool = defaults.buildUserPool(this, props.cognitoUserPoolProps);
        this.userPoolClient = defaults.buildUserPoolClient(this, this.userPool, props.cognitoUserPoolClientProps);
        this.apiGatewayAuthorizer = new api.CfnAuthorizer(this, 'CognitoAuthorizer', {
            restApiId: this.apiGateway.restApiId,
            type: 'COGNITO_USER_POOLS',
            providerArns: [this.userPool.userPoolArn],
            identitySource: "method.request.header.Authorization",
            name: "authorizer"
        });
        this.addAuthorizers();
    }
    addAuthorizers() {
        this.apiGateway.methods.forEach((apiMethod) => {
            // Leave the authorizer NONE for HTTP OPTIONS method to support CORS, for the rest set it to COGNITO
            const child = apiMethod.node.findChild('Resource');
            if (apiMethod.httpMethod === 'OPTIONS') {
                child.addPropertyOverride('AuthorizationType', 'NONE');
            }
            else {
                child.addPropertyOverride('AuthorizationType', 'COGNITO_USER_POOLS');
                child.addPropertyOverride('AuthorizerId', { Ref: this.apiGatewayAuthorizer.logicalId });
            }
        });
    }
}
exports.CognitoToApiGatewayToLambda = CognitoToApiGatewayToLambda;
//# sourceMappingURL=data:application/json;base64,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