# -*- coding: utf-8 -*-

# PLEASE DO NOT EDIT THIS FILE, IT IS GENERATED AND WILL BE OVERWRITTEN:
# https://github.com/ccxt/ccxt/blob/master/CONTRIBUTING.md#how-to-contribute-code

import ccxt.async_support
from ccxt.async_support.base.ws.cache import ArrayCache, ArrayCacheBySymbolById, ArrayCacheByTimestamp
from ccxt.base.types import Any, Balances, Bool, Int, Num, Order, OrderBook, OrderSide, OrderType, Str, Strings, Ticker, Tickers, Trade
from ccxt.async_support.base.ws.client import Client
from typing import List
from ccxt.base.errors import ExchangeError
from ccxt.base.errors import AuthenticationError
from ccxt.base.errors import PermissionDenied
from ccxt.base.errors import AccountSuspended
from ccxt.base.errors import ArgumentsRequired
from ccxt.base.errors import BadRequest
from ccxt.base.errors import BadSymbol
from ccxt.base.errors import InsufficientFunds
from ccxt.base.errors import InvalidOrder
from ccxt.base.errors import OrderNotFound
from ccxt.base.errors import NotSupported
from ccxt.base.errors import RateLimitExceeded
from ccxt.base.errors import ExchangeNotAvailable
from ccxt.base.errors import ChecksumError
from ccxt.base.precise import Precise


class kraken(ccxt.async_support.kraken):

    def describe(self) -> Any:
        return self.deep_extend(super(kraken, self).describe(), {
            'has': {
                'ws': True,
                'watchBalance': True,
                'watchMyTrades': True,
                'watchOHLCV': True,
                'watchOrderBook': True,
                'watchOrderBookForSymbols': True,
                'watchOrders': True,
                'watchTicker': True,
                'watchTickers': True,
                'watchBidsAsks': True,
                'watchTrades': True,
                'watchTradesForSymbols': True,
                'createOrderWs': True,
                'editOrderWs': True,
                'cancelOrderWs': True,
                'cancelOrdersWs': True,
                'cancelAllOrdersWs': True,
                # 'watchHeartbeat': True,
                # 'watchStatus': True,
            },
            'urls': {
                'api': {
                    'ws': {
                        'public': 'wss://ws.kraken.com',
                        'private': 'wss://ws-auth.kraken.com',
                        'privateV2': 'wss://ws-auth.kraken.com/v2',
                        'publicV2': 'wss://ws.kraken.com/v2',
                        'beta': 'wss://beta-ws.kraken.com',
                        'beta-private': 'wss://beta-ws-auth.kraken.com',
                    },
                },
            },
            # 'versions': {
            #     'ws': '0.2.0',
            # },
            'options': {
                'tradesLimit': 1000,
                'OHLCVLimit': 1000,
                'ordersLimit': 1000,
                'symbolsByOrderId': {},
                'watchOrderBook': {
                    'checksum': False,
                },
            },
            'streaming': {
                'ping': self.ping,
                'keepAlive': 6000,
            },
            'exceptions': {
                'ws': {
                    'exact': {
                        'Event(s) not found': BadRequest,
                    },
                    'broad': {
                        'Already subscribed': BadRequest,
                        'Currency pair not in ISO 4217-A3 format': BadSymbol,
                        'Currency pair not supported': BadSymbol,
                        'Malformed request': BadRequest,
                        'Pair field must be an array': BadRequest,
                        'Pair field unsupported for self subscription type': BadRequest,
                        'Pair(s) not found': BadSymbol,
                        'Subscription book depth must be an integer': BadRequest,
                        'Subscription depth not supported': BadRequest,
                        'Subscription field must be an object': BadRequest,
                        'Subscription name invalid': BadRequest,
                        'Subscription object unsupported field': BadRequest,
                        'Subscription ohlc interval must be an integer': BadRequest,
                        'Subscription ohlc interval not supported': BadRequest,
                        'Subscription ohlc requires interval': BadRequest,
                        'EAccount:Invalid permissions': PermissionDenied,
                        'EAuth:Account temporary disabled': AccountSuspended,
                        'EAuth:Account unconfirmed': AuthenticationError,
                        'EAuth:Rate limit exceeded': RateLimitExceeded,
                        'EAuth:Too many requests': RateLimitExceeded,
                        'EDatabase: Internal error(to be deprecated)': ExchangeError,
                        'EGeneral:Internal error[:<code>]': ExchangeError,
                        'EGeneral:Invalid arguments': BadRequest,
                        'EOrder:Cannot open opposing position': InvalidOrder,
                        'EOrder:Cannot open position': InvalidOrder,
                        'EOrder:Insufficient funds(insufficient user funds)': InsufficientFunds,
                        'EOrder:Insufficient margin(exchange does not have sufficient funds to allow margin trading)': InsufficientFunds,
                        'EOrder:Invalid price': InvalidOrder,
                        'EOrder:Margin allowance exceeded': InvalidOrder,
                        'EOrder:Margin level too low': InvalidOrder,
                        'EOrder:Margin position size exceeded(client would exceed the maximum position size for self pair)': InvalidOrder,
                        'EOrder:Order minimum not met(volume too low)': InvalidOrder,
                        'EOrder:Orders limit exceeded': InvalidOrder,
                        'EOrder:Positions limit exceeded': InvalidOrder,
                        'EOrder:Rate limit exceeded': RateLimitExceeded,
                        'EOrder:Scheduled orders limit exceeded': InvalidOrder,
                        'EOrder:Unknown position': OrderNotFound,
                        'EOrder:Unknown order': OrderNotFound,
                        'EOrder:Invalid order': InvalidOrder,
                        'EService:Deadline elapsed': ExchangeNotAvailable,
                        'EService:Market in cancel_only mode': NotSupported,
                        'EService:Market in limit_only mode': NotSupported,
                        'EService:Market in post_only mode': NotSupported,
                        'EService:Unavailable': ExchangeNotAvailable,
                        'ETrade:Invalid request': BadRequest,
                        'ESession:Invalid session': AuthenticationError,
                    },
                },
            },
        })

    def order_request_ws(self, method: str, symbol: str, type: str, request: dict, amount: Num, price: Num = None, params={}):
        isLimitOrder = type.endswith('limit')  # supporting limit, stop-loss-limit, take-profit-limit, etc
        if isLimitOrder:
            if price is None:
                raise ArgumentsRequired(self.id + ' limit orders require a price argument')
            request['params']['limit_price'] = self.parse_to_numeric(self.price_to_precision(symbol, price))
        isMarket = (type == 'market')
        postOnly = None
        postOnly, params = self.handle_post_only(isMarket, False, params)
        if postOnly:
            request['params']['post_only'] = True
        clientOrderId = self.safe_string(params, 'clientOrderId')
        if clientOrderId is not None:
            request['params']['cl_ord_id'] = clientOrderId
        cost = self.safe_string(params, 'cost')
        if cost is not None:
            request['params']['order_qty'] = self.parse_to_numeric(self.cost_to_precision(symbol, cost))
        stopLoss = self.safe_dict(params, 'stopLoss', {})
        takeProfit = self.safe_dict(params, 'takeProfit', {})
        presetStopLoss = self.safe_string(stopLoss, 'triggerPrice')
        presetTakeProfit = self.safe_string(takeProfit, 'triggerPrice')
        presetStopLossLimit = self.safe_string(stopLoss, 'price')
        presetTakeProfitLimit = self.safe_string(takeProfit, 'price')
        isPresetStopLoss = presetStopLoss is not None
        isPresetTakeProfit = presetTakeProfit is not None
        stopLossPrice = self.safe_string(params, 'stopLossPrice')
        takeProfitPrice = self.safe_string(params, 'takeProfitPrice')
        isStopLossPriceOrder = stopLossPrice is not None
        isTakeProfitPriceOrder = takeProfitPrice is not None
        trailingAmount = self.safe_string(params, 'trailingAmount')
        trailingPercent = self.safe_string(params, 'trailingPercent')
        trailingLimitAmount = self.safe_string(params, 'trailingLimitAmount')
        trailingLimitPercent = self.safe_string(params, 'trailingLimitPercent')
        isTrailingAmountOrder = trailingAmount is not None
        isTrailingPercentOrder = trailingPercent is not None
        isTrailingLimitAmountOrder = trailingLimitAmount is not None
        isTrailingLimitPercentOrder = trailingLimitPercent is not None
        offset = self.safe_string(params, 'offset', '')  # can set self to - for minus
        trailingAmountString = offset + self.number_to_string(trailingAmount) if (trailingAmount is not None) else None
        trailingPercentString = offset + self.number_to_string(trailingPercent) if (trailingPercent is not None) else None
        trailingLimitAmountString = offset + self.number_to_string(trailingLimitAmount) if (trailingLimitAmount is not None) else None
        trailingLimitPercentString = offset + self.number_to_string(trailingLimitPercent) if (trailingLimitPercent is not None) else None
        priceType = 'pct' if (isTrailingPercentOrder or isTrailingLimitPercentOrder) else 'quote'
        if method == 'createOrderWs':
            reduceOnly = self.safe_bool(params, 'reduceOnly')
            if reduceOnly:
                request['params']['reduce_only'] = True
            timeInForce = self.safe_string_lower(params, 'timeInForce')
            if timeInForce is not None:
                request['params']['time_in_force'] = timeInForce
            params = self.omit(params, ['reduceOnly', 'timeInForce'])
            if isStopLossPriceOrder or isTakeProfitPriceOrder or isTrailingAmountOrder or isTrailingPercentOrder or isTrailingLimitAmountOrder or isTrailingLimitPercentOrder:
                request['params']['triggers'] = {}
            if isPresetStopLoss or isPresetTakeProfit:
                request['params']['conditional'] = {}
                if isPresetStopLoss:
                    request['params']['conditional']['order_type'] = 'stop-loss'
                    request['params']['conditional']['trigger_price'] = self.parse_to_numeric(self.price_to_precision(symbol, presetStopLoss))
                elif isPresetTakeProfit:
                    request['params']['conditional']['order_type'] = 'take-profit'
                    request['params']['conditional']['trigger_price'] = self.parse_to_numeric(self.price_to_precision(symbol, presetTakeProfit))
                if presetStopLossLimit is not None:
                    request['params']['conditional']['order_type'] = 'stop-loss-limit'
                    request['params']['conditional']['limit_price'] = self.parse_to_numeric(self.price_to_precision(symbol, presetStopLossLimit))
                elif presetTakeProfitLimit is not None:
                    request['params']['conditional']['order_type'] = 'take-profit-limit'
                    request['params']['conditional']['limit_price'] = self.parse_to_numeric(self.price_to_precision(symbol, presetTakeProfitLimit))
                params = self.omit(params, ['stopLoss', 'takeProfit'])
            elif isStopLossPriceOrder or isTakeProfitPriceOrder:
                if isStopLossPriceOrder:
                    request['params']['triggers']['price'] = self.parse_to_numeric(self.price_to_precision(symbol, stopLossPrice))
                    if isLimitOrder:
                        request['params']['order_type'] = 'stop-loss-limit'
                    else:
                        request['params']['order_type'] = 'stop-loss'
                else:
                    request['params']['triggers']['price'] = self.parse_to_numeric(self.price_to_precision(symbol, takeProfitPrice))
                    if isLimitOrder:
                        request['params']['order_type'] = 'take-profit-limit'
                    else:
                        request['params']['order_type'] = 'take-profit'
            elif isTrailingAmountOrder or isTrailingPercentOrder or isTrailingLimitAmountOrder or isTrailingLimitPercentOrder:
                request['params']['triggers']['price_type'] = priceType
                if not isLimitOrder and (isTrailingAmountOrder or isTrailingPercentOrder):
                    request['params']['order_type'] = 'trailing-stop'
                    if isTrailingAmountOrder:
                        request['params']['triggers']['price'] = self.parse_to_numeric(trailingAmountString)
                    else:
                        request['params']['triggers']['price'] = self.parse_to_numeric(trailingPercentString)
                else:
                    # trailing limit orders are not conventionally supported because the static limit_price_type param is not available for trailing-stop-limit orders
                    request['params']['limit_price_type'] = priceType
                    request['params']['order_type'] = 'trailing-stop-limit'
                    if isTrailingLimitAmountOrder:
                        request['params']['triggers']['price'] = self.parse_to_numeric(trailingLimitAmountString)
                    else:
                        request['params']['triggers']['price'] = self.parse_to_numeric(trailingLimitPercentString)
        elif method == 'editOrderWs':
            if isPresetStopLoss or isPresetTakeProfit:
                raise NotSupported(self.id + ' editing the stopLoss and takeProfit on existing orders is currently not supported')
            if isStopLossPriceOrder or isTakeProfitPriceOrder:
                if isStopLossPriceOrder:
                    request['params']['trigger_price'] = self.parse_to_numeric(self.price_to_precision(symbol, stopLossPrice))
                else:
                    request['params']['trigger_price'] = self.parse_to_numeric(self.price_to_precision(symbol, takeProfitPrice))
            elif isTrailingAmountOrder or isTrailingPercentOrder or isTrailingLimitAmountOrder or isTrailingLimitPercentOrder:
                request['params']['trigger_price_type'] = priceType
                if not isLimitOrder and (isTrailingAmountOrder or isTrailingPercentOrder):
                    if isTrailingAmountOrder:
                        request['params']['trigger_price'] = self.parse_to_numeric(trailingAmountString)
                    else:
                        request['params']['trigger_price'] = self.parse_to_numeric(trailingPercentString)
                else:
                    request['params']['limit_price_type'] = priceType
                    if isTrailingLimitAmountOrder:
                        request['params']['trigger_price'] = self.parse_to_numeric(trailingLimitAmountString)
                    else:
                        request['params']['trigger_price'] = self.parse_to_numeric(trailingLimitPercentString)
        params = self.omit(params, ['clientOrderId', 'cost', 'offset', 'stopLossPrice', 'takeProfitPrice', 'trailingAmount', 'trailingPercent', 'trailingLimitAmount', 'trailingLimitPercent'])
        return [request, params]

    async def create_order_ws(self, symbol: str, type: OrderType, side: OrderSide, amount: float, price: Num = None, params={}) -> Order:
        """
        create a trade order

        https://docs.kraken.com/api/docs/websocket-v2/add_order

        :param str symbol: unified symbol of the market to create an order in
        :param str type: 'market' or 'limit'
        :param str side: 'buy' or 'sell'
        :param float amount: how much of currency you want to trade in units of base currency
        :param float [price]: the price at which the order is to be fulfilled, in units of the quote currency, ignored in market orders
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: an `order structure <https://docs.ccxt.com/#/?id=order-structure>`
        """
        await self.load_markets()
        token = await self.authenticate()
        market = self.market(symbol)
        url = self.urls['api']['ws']['privateV2']
        requestId = self.request_id()
        messageHash = requestId
        request: dict = {
            'method': 'add_order',
            'params': {
                'order_type': type,
                'side': side,
                'order_qty': self.parse_to_numeric(self.amount_to_precision(symbol, amount)),
                'symbol': market['symbol'],
                'token': token,
            },
            'req_id': requestId,
        }
        request, params = self.order_request_ws('createOrderWs', symbol, type, request, amount, price, params)
        return await self.watch(url, messageHash, self.extend(request, params), messageHash)

    def handle_create_edit_order(self, client, message):
        #
        #  createOrder
        #     {
        #         "method": "add_order",
        #         "req_id": 1,
        #         "result": {
        #             "order_id": "OXM2QD-EALR2-YBAVEU"
        #         },
        #         "success": True,
        #         "time_in": "2025-05-13T10:12:13.876173Z",
        #         "time_out": "2025-05-13T10:12:13.890137Z"
        #     }
        #
        #  editOrder
        #     {
        #         "method": "amend_order",
        #         "req_id": 1,
        #         "result": {
        #             "amend_id": "TYDLSQ-OYNYU-3MNRER",
        #             "order_id": "OGL7HR-SWFO4-NRQTHO"
        #         },
        #         "success": True,
        #         "time_in": "2025-05-14T13:54:10.840342Z",
        #         "time_out": "2025-05-14T13:54:10.855046Z"
        #     }
        #
        result = self.safe_dict(message, 'result', {})
        order = self.parse_order(result)
        messageHash = self.safe_value_2(message, 'reqid', 'req_id')
        client.resolve(order, messageHash)

    async def edit_order_ws(self, id: str, symbol: str, type: OrderType, side: OrderSide, amount: Num = None, price: Num = None, params={}) -> Order:
        """
        edit a trade order

        https://docs.kraken.com/api/docs/websocket-v2/amend_order

        :param str id: order id
        :param str symbol: unified symbol of the market to create an order in
        :param str type: 'market' or 'limit'
        :param str side: 'buy' or 'sell'
        :param float amount: how much of the currency you want to trade in units of the base currency
        :param float [price]: the price at which the order is to be fulfilled, in units of the quote currency, ignored in market orders
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: an `order structure <https://docs.ccxt.com/#/?id=order-structure>`
        """
        await self.load_markets()
        token = await self.authenticate()
        url = self.urls['api']['ws']['privateV2']
        requestId = self.request_id()
        messageHash = requestId
        request: dict = {
            'method': 'amend_order',
            'params': {
                'order_id': id,
                'order_qty': self.parse_to_numeric(self.amount_to_precision(symbol, amount)),
                'token': token,
            },
            'req_id': requestId,
        }
        request, params = self.order_request_ws('editOrderWs', symbol, type, request, amount, price, params)
        return await self.watch(url, messageHash, self.extend(request, params), messageHash)

    async def cancel_orders_ws(self, ids: List[str], symbol: Str = None, params={}):
        """
        cancel multiple orders

        https://docs.kraken.com/api/docs/websocket-v2/cancel_order

        :param str[] ids: order ids
        :param str [symbol]: unified market symbol, default is None
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: an list of `order structures <https://docs.ccxt.com/#/?id=order-structure>`
        """
        if symbol is not None:
            raise NotSupported(self.id + ' cancelOrdersWs() does not support cancelling orders for a specific symbol.')
        await self.load_markets()
        token = await self.authenticate()
        url = self.urls['api']['ws']['privateV2']
        requestId = self.request_id()
        messageHash = requestId
        request: dict = {
            'method': 'cancel_order',
            'params': {
                'order_id': ids,
                'token': token,
            },
            'req_id': requestId,
        }
        return await self.watch(url, messageHash, self.extend(request, params), messageHash)

    async def cancel_order_ws(self, id: str, symbol: Str = None, params={}) -> Order:
        """
        cancels an open order

        https://docs.kraken.com/api/docs/websocket-v2/cancel_order

        :param str id: order id
        :param str [symbol]: unified symbol of the market the order was made in
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: An `order structure <https://docs.ccxt.com/#/?id=order-structure>`
        """
        if symbol is not None:
            raise NotSupported(self.id + ' cancelOrderWs() does not support cancelling orders for a specific symbol.')
        await self.load_markets()
        token = await self.authenticate()
        url = self.urls['api']['ws']['privateV2']
        requestId = self.request_id()
        messageHash = requestId
        request: dict = {
            'method': 'cancel_order',
            'params': {
                'order_id': [id],
                'token': token,
            },
            'req_id': requestId,
        }
        return await self.watch(url, messageHash, self.extend(request, params), messageHash)

    def handle_cancel_order(self, client, message):
        #
        #     {
        #         "method": "cancel_order",
        #         "req_id": 123456789,
        #         "result": {
        #             "order_id": "OKAGJC-YHIWK-WIOZWG"
        #         },
        #         "success": True,
        #         "time_in": "2023-09-21T14:36:57.428972Z",
        #         "time_out": "2023-09-21T14:36:57.437952Z"
        #     }
        #
        reqId = self.safe_value(message, 'req_id')
        client.resolve(message, reqId)

    async def cancel_all_orders_ws(self, symbol: Str = None, params={}):
        """
        cancel all open orders

        https://docs.kraken.com/api/docs/websocket-v2/cancel_all

        :param str [symbol]: unified market symbol, only orders in the market of self symbol are cancelled when symbol is not None
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict[]: a list of `order structures <https://docs.ccxt.com/#/?id=order-structure>`
        """
        if symbol is not None:
            raise NotSupported(self.id + ' cancelAllOrdersWs() does not support cancelling orders in a specific market.')
        await self.load_markets()
        token = await self.authenticate()
        url = self.urls['api']['ws']['privateV2']
        requestId = self.request_id()
        messageHash = requestId
        request: dict = {
            'method': 'cancel_all',
            'params': {
                'token': token,
            },
            'req_id': requestId,
        }
        return await self.watch(url, messageHash, self.extend(request, params), messageHash)

    def handle_cancel_all_orders(self, client, message):
        #
        #     {
        #         "method": "cancel_all",
        #         "req_id": 123456789,
        #         "result": {
        #             "count": 1
        #         },
        #         "success": True,
        #         "time_in": "2023-09-21T14:36:57.428972Z",
        #         "time_out": "2023-09-21T14:36:57.437952Z"
        #     }
        #
        reqId = self.safe_value(message, 'req_id')
        client.resolve(message, reqId)

    def handle_ticker(self, client, message):
        #
        #     {
        #         "channel": "ticker",
        #         "type": "snapshot",
        #         "data": [
        #             {
        #                 "symbol": "BTC/USD",
        #                 "bid": 108359.8,
        #                 "bid_qty": 0.01362603,
        #                 "ask": 108359.9,
        #                 "ask_qty": 17.17988863,
        #                 "last": 108359.8,
        #                 "volume": 2158.32346723,
        #                 "vwap": 108894.5,
        #                 "low": 106824,
        #                 "high": 111300,
        #                 "change": -2679.9,
        #                 "change_pct": -2.41
        #             }
        #         ]
        #     }
        #
        data = self.safe_list(message, 'data', [])
        ticker = data[0]
        symbol = self.safe_string(ticker, 'symbol')
        messageHash = self.get_message_hash('ticker', None, symbol)
        vwap = self.safe_string(ticker, 'vwap')
        quoteVolume = None
        baseVolume = self.safe_string(ticker, 'volume')
        if baseVolume is not None and vwap is not None:
            quoteVolume = Precise.string_mul(baseVolume, vwap)
        last = self.safe_string(ticker, 'last')
        result = self.safe_ticker({
            'symbol': symbol,
            'timestamp': None,
            'datetime': None,
            'high': self.safe_string(ticker, 'high'),
            'low': self.safe_string(ticker, 'low'),
            'bid': self.safe_string(ticker, 'bid'),
            'bidVolume': self.safe_string(ticker, 'bid_qty'),
            'ask': self.safe_string(ticker, 'ask'),
            'askVolume': self.safe_string(ticker, 'ask_qty'),
            'vwap': vwap,
            'open': None,
            'close': last,
            'last': last,
            'previousClose': None,
            'change': self.safe_string(ticker, 'change'),
            'percentage': self.safe_string(ticker, 'change_pct'),
            'average': None,
            'baseVolume': baseVolume,
            'quoteVolume': quoteVolume,
            'info': ticker,
        })
        self.tickers[symbol] = result
        client.resolve(result, messageHash)

    def handle_trades(self, client: Client, message):
        #
        #     {
        #         "channel": "trade",
        #         "type": "update",
        #         "data": [
        #             {
        #                 "symbol": "MATIC/USD",
        #                 "side": "sell",
        #                 "price": 0.5117,
        #                 "qty": 40.0,
        #                 "ord_type": "market",
        #                 "trade_id": 4665906,
        #                 "timestamp": "2023-09-25T07:49:37.708706Z"
        #             }
        #         ]
        #     }
        #
        data = self.safe_list(message, 'data', [])
        trade = data[0]
        symbol = self.safe_string(trade, 'symbol')
        messageHash = self.get_message_hash('trade', None, symbol)
        stored = self.safe_value(self.trades, symbol)
        if stored is None:
            limit = self.safe_integer(self.options, 'tradesLimit', 1000)
            stored = ArrayCache(limit)
            self.trades[symbol] = stored
        market = self.market(symbol)
        parsed = self.parse_trades(data, market)
        for i in range(0, len(parsed)):
            stored.append(parsed[i])
        client.resolve(stored, messageHash)

    def handle_ohlcv(self, client: Client, message, subscription):
        #
        #     [
        #         216,  # channelID
        #         [
        #             "1574454214.962096",  # Time, seconds since epoch
        #             "1574454240.000000",  # End timestamp of the interval
        #             "0.020970",  # Open price at midnight UTC
        #             "0.020970",  # Intraday high price
        #             "0.020970",  # Intraday low price
        #             "0.020970",  # Closing price at midnight UTC
        #             "0.020970",  # Volume weighted average price
        #             "0.08636138",  # Accumulated volume today
        #             1,  # Number of trades today
        #         ],
        #         "ohlc-1",  # Channel Name of subscription
        #         "ETH/XBT",  # Asset pair
        #     ]
        #
        info = self.safe_value(subscription, 'subscription', {})
        interval = self.safe_integer(info, 'interval')
        name = self.safe_string(info, 'name')
        wsName = self.safe_string(message, 3)
        market = self.safe_value(self.options['marketsByWsName'], wsName)
        symbol = market['symbol']
        timeframe = self.find_timeframe(interval)
        duration = self.parse_timeframe(timeframe)
        if timeframe is not None:
            candle = self.safe_value(message, 1)
            messageHash = name + ':' + timeframe + ':' + wsName
            timestamp = self.safe_float(candle, 1)
            timestamp -= duration
            ts = self.parse_to_int(timestamp * 1000)
            result = [
                ts,
                self.safe_float(candle, 2),
                self.safe_float(candle, 3),
                self.safe_float(candle, 4),
                self.safe_float(candle, 5),
                self.safe_float(candle, 7),
            ]
            self.ohlcvs[symbol] = self.safe_value(self.ohlcvs, symbol, {})
            stored = self.safe_value(self.ohlcvs[symbol], timeframe)
            if stored is None:
                limit = self.safe_integer(self.options, 'OHLCVLimit', 1000)
                stored = ArrayCacheByTimestamp(limit)
                self.ohlcvs[symbol][timeframe] = stored
            stored.append(result)
            client.resolve(stored, messageHash)

    def request_id(self):
        # their support said that reqid must be an int32, not documented
        reqid = self.sum(self.safe_integer(self.options, 'reqid', 0), 1)
        self.options['reqid'] = reqid
        return reqid

    async def watch_public(self, name, symbol, params={}):
        await self.load_markets()
        market = self.market(symbol)
        wsName = self.safe_value(market['info'], 'wsname')
        messageHash = name + ':' + wsName
        url = self.urls['api']['ws']['public']
        requestId = self.request_id()
        subscribe: dict = {
            'event': 'subscribe',
            'reqid': requestId,
            'pair': [
                wsName,
            ],
            'subscription': {
                'name': name,
            },
        }
        request = self.deep_extend(subscribe, params)
        return await self.watch(url, messageHash, request, messageHash)

    async def watch_ticker(self, symbol: str, params={}) -> Ticker:
        """
        watches a price ticker, a statistical calculation with the information calculated over the past 24 hours for a specific market

        https://docs.kraken.com/api/docs/websocket-v2/ticker

        :param str symbol: unified symbol of the market to fetch the ticker for
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: a `ticker structure <https://docs.ccxt.com/#/?id=ticker-structure>`
        """
        await self.load_markets()
        symbol = self.symbol(symbol)
        tickers = await self.watch_tickers([symbol], params)
        return tickers[symbol]

    async def watch_tickers(self, symbols: Strings = None, params={}) -> Tickers:
        """
        watches a price ticker, a statistical calculation with the information calculated over the past 24 hours for a specific market

        https://docs.kraken.com/api/docs/websocket-v2/ticker

        :param str[] symbols:
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: a `ticker structure <https://docs.ccxt.com/#/?id=ticker-structure>`
        """
        await self.load_markets()
        symbols = self.market_symbols(symbols, None, False)
        ticker = await self.watch_multi_helper('ticker', 'ticker', symbols, None, params)
        if self.newUpdates:
            result: dict = {}
            result[ticker['symbol']] = ticker
            return result
        return self.filter_by_array(self.tickers, 'symbol', symbols)

    async def watch_bids_asks(self, symbols: Strings = None, params={}) -> Tickers:
        """
        watches best bid & ask for symbols

        https://docs.kraken.com/api/docs/websocket-v2/ticker

        :param str[] symbols: unified symbol of the market to fetch the ticker for
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: a `ticker structure <https://docs.ccxt.com/#/?id=ticker-structure>`
        """
        await self.load_markets()
        symbols = self.market_symbols(symbols, None, False)
        params['event_trigger'] = 'bbo'
        ticker = await self.watch_multi_helper('bidask', 'ticker', symbols, None, params)
        if self.newUpdates:
            result: dict = {}
            result[ticker['symbol']] = ticker
            return result
        return self.filter_by_array(self.bidsasks, 'symbol', symbols)

    async def watch_trades(self, symbol: str, since: Int = None, limit: Int = None, params={}) -> List[Trade]:
        """
        get the list of most recent trades for a particular symbol

        https://docs.kraken.com/api/docs/websocket-v2/trade

        :param str symbol: unified symbol of the market to fetch trades for
        :param int [since]: timestamp in ms of the earliest trade to fetch
        :param int [limit]: the maximum amount of trades to fetch
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict[]: a list of `trade structures <https://docs.ccxt.com/#/?id=public-trades>`
        """
        return await self.watch_trades_for_symbols([symbol], since, limit, params)

    async def watch_trades_for_symbols(self, symbols: List[str], since: Int = None, limit: Int = None, params={}) -> List[Trade]:
        """
        get the list of most recent trades for a list of symbols

        https://docs.kraken.com/api/docs/websocket-v2/trade

        :param str[] symbols: unified symbol of the market to fetch trades for
        :param int [since]: timestamp in ms of the earliest trade to fetch
        :param int [limit]: the maximum amount of trades to fetch
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict[]: a list of `trade structures <https://docs.ccxt.com/#/?id=public-trades>`
        """
        trades = await self.watch_multi_helper('trade', 'trade', symbols, None, params)
        if self.newUpdates:
            first = self.safe_list(trades, 0)
            tradeSymbol = self.safe_string(first, 'symbol')
            limit = trades.getLimit(tradeSymbol, limit)
        return self.filter_by_since_limit(trades, since, limit, 'timestamp', True)

    async def watch_order_book(self, symbol: str, limit: Int = None, params={}) -> OrderBook:
        """
        watches information on open orders with bid(buy) and ask(sell) prices, volumes and other data

        https://docs.kraken.com/api/docs/websocket-v2/book

        :param str symbol: unified symbol of the market to fetch the order book for
        :param int [limit]: the maximum amount of order book entries to return
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: A dictionary of `order book structures <https://docs.ccxt.com/#/?id=order-book-structure>` indexed by market symbols
        """
        return await self.watch_order_book_for_symbols([symbol], limit, params)

    async def watch_order_book_for_symbols(self, symbols: List[str], limit: Int = None, params={}) -> OrderBook:
        """
        watches information on open orders with bid(buy) and ask(sell) prices, volumes and other data

        https://docs.kraken.com/api/docs/websocket-v2/book

        :param str[] symbols: unified array of symbols
        :param int [limit]: the maximum amount of order book entries to return
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: A dictionary of `order book structures <https://docs.ccxt.com/#/?id=order-book-structure>` indexed by market symbols
        """
        requiredParams: dict = {}
        if limit is not None:
            if self.in_array(limit, [10, 25, 100, 500, 1000]):
                requiredParams['depth'] = limit  # default 10, valid options 10, 25, 100, 500, 1000
            else:
                raise NotSupported(self.id + ' watchOrderBook accepts limit values of 10, 25, 100, 500 and 1000 only')
        orderbook = await self.watch_multi_helper('orderbook', 'book', symbols, {'limit': limit}, self.extend(requiredParams, params))
        return orderbook.limit()

    async def watch_ohlcv(self, symbol: str, timeframe='1m', since: Int = None, limit: Int = None, params={}) -> List[list]:
        """
        watches historical candlestick data containing the open, high, low, and close price, and the volume of a market

        https://docs.kraken.com/api/docs/websocket-v1/ohlc

        :param str symbol: unified symbol of the market to fetch OHLCV data for
        :param str timeframe: the length of time each candle represents
        :param int [since]: timestamp in ms of the earliest candle to fetch
        :param int [limit]: the maximum amount of candles to fetch
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns int[][]: A list of candles ordered, open, high, low, close, volume
        """
        await self.load_markets()
        name = 'ohlc'
        market = self.market(symbol)
        symbol = market['symbol']
        wsName = self.safe_value(market['info'], 'wsname')
        messageHash = name + ':' + timeframe + ':' + wsName
        url = self.urls['api']['ws']['public']
        requestId = self.request_id()
        subscribe: dict = {
            'event': 'subscribe',
            'reqid': requestId,
            'pair': [
                wsName,
            ],
            'subscription': {
                'name': name,
                'interval': self.safe_value(self.timeframes, timeframe, timeframe),
            },
        }
        request = self.deep_extend(subscribe, params)
        ohlcv = await self.watch(url, messageHash, request, messageHash)
        if self.newUpdates:
            limit = ohlcv.getLimit(symbol, limit)
        return self.filter_by_since_limit(ohlcv, since, limit, 0, True)

    async def load_markets(self, reload=False, params={}):
        markets = await super(kraken, self).load_markets(reload, params)
        marketsByWsName = self.safe_value(self.options, 'marketsByWsName')
        if (marketsByWsName is None) or reload:
            marketsByWsName = {}
            for i in range(0, len(self.symbols)):
                symbol = self.symbols[i]
                market = self.markets[symbol]
                info = self.safe_value(market, 'info', {})
                wsName = self.safe_string(info, 'wsname')
                marketsByWsName[wsName] = market
            self.options['marketsByWsName'] = marketsByWsName
        return markets

    def ping(self, client: Client):
        url = client.url
        request = {}
        if url.find('v2') >= 0:
            request['method'] = 'ping'
        else:
            request['event'] = 'ping'
        return request

    def handle_pong(self, client: Client, message):
        client.lastPong = self.milliseconds()
        return message

    async def watch_heartbeat(self, params={}):
        await self.load_markets()
        event = 'heartbeat'
        url = self.urls['api']['ws']['public']
        return await self.watch(url, event)

    def handle_heartbeat(self, client: Client, message):
        #
        # every second(approx) if no other updates are sent
        #
        #     {"event": "heartbeat"}
        #
        event = self.safe_string(message, 'event')
        client.resolve(message, event)

    def handle_order_book(self, client: Client, message):
        #
        # first message(snapshot)
        #
        #     {
        #         "channel": "book",
        #         "type": "snapshot",
        #         "data": [
        #             {
        #                 "symbol": "MATIC/USD",
        #                 "bids": [
        #                     {
        #                         "price": 0.5666,
        #                         "qty": 4831.75496356
        #                     },
        #                     {
        #                         "price": 0.5665,
        #                         "qty": 6658.22734739
        #                     }
        #                 ],
        #                 "asks": [
        #                     {
        #                         "price": 0.5668,
        #                         "qty": 4410.79769741
        #                     },
        #                     {
        #                         "price": 0.5669,
        #                         "qty": 4655.40412487
        #                     }
        #                 ],
        #                 "checksum": 2439117997
        #             }
        #         ]
        #     }
        #
        # subsequent updates
        #
        #     {
        #         "channel": "book",
        #         "type": "update",
        #         "data": [
        #             {
        #                 "symbol": "MATIC/USD",
        #                 "bids": [
        #                     {
        #                         "price": 0.5657,
        #                         "qty": 1098.3947558
        #                     }
        #                 ],
        #                 "asks": [],
        #                 "checksum": 2114181697,
        #                 "timestamp": "2023-10-06T17:35:55.440295Z"
        #             }
        #         ]
        #     }
        #
        type = self.safe_string(message, 'type')
        data = self.safe_list(message, 'data', [])
        first = self.safe_dict(data, 0, {})
        symbol = self.safe_string(first, 'symbol')
        a = self.safe_value(first, 'asks', [])
        b = self.safe_value(first, 'bids', [])
        c = self.safe_integer(first, 'checksum')
        messageHash = self.get_message_hash('orderbook', None, symbol)
        orderbook = None
        if type == 'update':
            orderbook = self.orderbooks[symbol]
            storedAsks = orderbook['asks']
            storedBids = orderbook['bids']
            if a is not None:
                self.custom_handle_deltas(storedAsks, a)
            if b is not None:
                self.custom_handle_deltas(storedBids, b)
            datetime = self.safe_string(first, 'timestamp')
            orderbook['symbol'] = symbol
            orderbook['timestamp'] = self.parse8601(datetime)
            orderbook['datetime'] = datetime
        else:
            # snapshot
            depth = len(a)
            self.orderbooks[symbol] = self.order_book({}, depth)
            orderbook = self.orderbooks[symbol]
            keys = ['asks', 'bids']
            for i in range(0, len(keys)):
                key = keys[i]
                bookside = orderbook[key]
                deltas = self.safe_value(first, key, [])
                if len(deltas) > 0:
                    self.custom_handle_deltas(bookside, deltas)
            orderbook['symbol'] = symbol
        orderbook.limit()
        # checksum temporarily disabled because the exchange checksum was not reliable
        checksum = self.handle_option('watchOrderBook', 'checksum', False)
        if checksum:
            payloadArray = []
            if c is not None:
                checkAsks = orderbook['asks']
                checkBids = orderbook['bids']
                # checkAsks = asks.map((elem) => [elem['price'], elem['qty']])
                # checkBids = bids.map((elem) => [elem['price'], elem['qty']])
                for i in range(0, 10):
                    currentAsk = self.safe_value(checkAsks, i, {})
                    formattedAsk = self.format_number(currentAsk[0]) + self.format_number(currentAsk[1])
                    payloadArray.append(formattedAsk)
                for i in range(0, 10):
                    currentBid = self.safe_value(checkBids, i, {})
                    formattedBid = self.format_number(currentBid[0]) + self.format_number(currentBid[1])
                    payloadArray.append(formattedBid)
            payload = ''.join(payloadArray)
            localChecksum = self.crc32(payload, False)
            if localChecksum != c:
                error = ChecksumError(self.id + ' ' + self.orderbook_checksum_message(symbol))
                del client.subscriptions[messageHash]
                del self.orderbooks[symbol]
                client.reject(error, messageHash)
                return
        client.resolve(orderbook, messageHash)

    def custom_handle_deltas(self, bookside, deltas):
        # sortOrder = True if (key == 'bids') else False
        for j in range(0, len(deltas)):
            delta = deltas[j]
            price = self.safe_number(delta, 'price')
            amount = self.safe_number(delta, 'qty')
            bookside.store(price, amount)
            # if amount == 0:
            #     index = bookside.findIndex((x: Int) => x[0] == price)
            #     bookside.splice(index, 1)
            # else:
            #     bookside.store(price, amount)
            # }
            # bookside = self.sort_by(bookside, 0, sortOrder)
            # bookside[0:9]

    def format_number(self, data):
        parts = data.split('.')
        integer = self.safe_string(parts, 0)
        decimals = self.safe_string(parts, 1, '')
        joinedResult = integer + decimals
        i = 0
        while(joinedResult[i] == '0'):
            i += 1
        if i > 0:
            joinedResult = joinedResult[i:]
        return joinedResult

    def handle_system_status(self, client: Client, message):
        #
        # todo: answer the question whether handleSystemStatus should be renamed
        # and unified for any usage pattern that
        # involves system status and maintenance updates
        #
        #     {
        #         "connectionID": 15527282728335292000,
        #         "event": "systemStatus",
        #         "status": "online",  # online|maintenance|(custom status tbd)
        #         "version": "0.2.0"
        #     }
        #
        # v2
        #     {
        #         channel: 'status',
        #         type: 'update',
        #         data: [
        #             {
        #                 version: '2.0.10',
        #                 system: 'online',
        #                 api_version: 'v2',
        #                 connection_id: 6447481662169813000
        #             }
        #         ]
        #     }
        #
        return message

    async def authenticate(self, params={}):
        url = self.urls['api']['ws']['private']
        client = self.client(url)
        authenticated = 'authenticated'
        subscription = self.safe_value(client.subscriptions, authenticated)
        now = self.seconds()
        start = self.safe_integer(subscription, 'start')
        expires = self.safe_integer(subscription, 'expires')
        if (subscription is None) or ((subscription is not None) and (start + expires) <= now):
            # https://docs.kraken.com/api/docs/rest-api/get-websockets-token
            response = await self.privatePostGetWebSocketsToken(params)
            #
            #     {
            #         "error":[],
            #         "result":{
            #             "token":"xeAQ\/RCChBYNVh53sTv1yZ5H4wIbwDF20PiHtTF+4UI",
            #             "expires":900
            #         }
            #     }
            #
            subscription = self.safe_dict(response, 'result')
            subscription['start'] = now
            client.subscriptions[authenticated] = subscription
        return self.safe_string(subscription, 'token')

    async def watch_private(self, name, symbol: Str = None, since: Int = None, limit: Int = None, params={}):
        await self.load_markets()
        token = await self.authenticate()
        subscriptionHash = name
        messageHash = name
        if symbol is not None:
            symbol = self.symbol(symbol)
            messageHash += ':' + symbol
        url = self.urls['api']['ws']['private']
        requestId = self.request_id()
        subscribe: dict = {
            'event': 'subscribe',
            'reqid': requestId,
            'subscription': {
                'name': name,
                'token': token,
            },
        }
        request = self.deep_extend(subscribe, params)
        result = await self.watch(url, messageHash, request, subscriptionHash)
        if self.newUpdates:
            limit = result.getLimit(symbol, limit)
        return self.filter_by_symbol_since_limit(result, symbol, since, limit)

    async def watch_my_trades(self, symbol: Str = None, since: Int = None, limit: Int = None, params={}) -> List[Trade]:
        """
        watches information on multiple trades made by the user

        https://docs.kraken.com/api/docs/websocket-v1/owntrades

        :param str symbol: unified market symbol of the market trades were made in
        :param int [since]: the earliest time in ms to fetch trades for
        :param int [limit]: the maximum number of trade structures to retrieve
        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict[]: a list of `trade structures <https://docs.ccxt.com/#/?id=trade-structure>`
        """
        return await self.watch_private('ownTrades', symbol, since, limit, params)

    def handle_my_trades(self, client: Client, message, subscription=None):
        #
        #     [
        #         [
        #             {
        #                 "TT5UC3-GOIRW-6AZZ6R": {
        #                     "cost": "1493.90107",
        #                     "fee": "3.88415",
        #                     "margin": "0.00000",
        #                     "ordertxid": "OTLAS3-RRHUF-NDWH5A",
        #                     "ordertype": "market",
        #                     "pair": "XBT/USDT",
        #                     "postxid": "TKH2SE-M7IF5-CFI7LT",
        #                     "price": "6851.50005",
        #                     "time": "1586822919.335498",
        #                     "type": "sell",
        #                     "vol": "0.21804000"
        #                 }
        #             },
        #             {
        #                 "TIY6G4-LKLAI-Y3GD4A": {
        #                     "cost": "22.17134",
        #                     "fee": "0.05765",
        #                     "margin": "0.00000",
        #                     "ordertxid": "ODQXS7-MOLK6-ICXKAA",
        #                     "ordertype": "market",
        #                     "pair": "ETH/USD",
        #                     "postxid": "TKH2SE-M7IF5-CFI7LT",
        #                     "price": "169.97999",
        #                     "time": "1586340530.895739",
        #                     "type": "buy",
        #                     "vol": "0.13043500"
        #                 }
        #             },
        #         ],
        #         "ownTrades",
        #         {sequence: 1}
        #     ]
        #
        allTrades = self.safe_value(message, 0, [])
        allTradesLength = len(allTrades)
        if allTradesLength > 0:
            if self.myTrades is None:
                limit = self.safe_integer(self.options, 'tradesLimit', 1000)
                self.myTrades = ArrayCache(limit)
            stored = self.myTrades
            symbols: dict = {}
            for i in range(0, len(allTrades)):
                trades = self.safe_value(allTrades, i, {})
                ids = list(trades.keys())
                for j in range(0, len(ids)):
                    id = ids[j]
                    trade = trades[id]
                    parsed = self.parse_ws_trade(self.extend({'id': id}, trade))
                    stored.append(parsed)
                    symbol = parsed['symbol']
                    symbols[symbol] = True
            name = 'ownTrades'
            client.resolve(self.myTrades, name)
            keys = list(symbols.keys())
            for i in range(0, len(keys)):
                messageHash = name + ':' + keys[i]
                client.resolve(self.myTrades, messageHash)

    def parse_ws_trade(self, trade, market=None):
        #
        #     {
        #         "id": "TIMIRG-WUNNE-RRJ6GT",  # injected from outside
        #         "ordertxid": "OQRPN2-LRHFY-HIFA7D",
        #         "postxid": "TKH2SE-M7IF5-CFI7LT",
        #         "pair": "USDCUSDT",
        #         "time": 1586340086.457,
        #         "type": "sell",
        #         "ordertype": "market",
        #         "price": "0.99860000",
        #         "cost": "22.16892001",
        #         "fee": "0.04433784",
        #         "vol": "22.20000000",
        #         "margin": "0.00000000",
        #         "misc": ''
        #     }
        #
        #     {
        #         "id": "TIY6G4-LKLAI-Y3GD4A",
        #         "cost": "22.17134",
        #         "fee": "0.05765",
        #         "margin": "0.00000",
        #         "ordertxid": "ODQXS7-MOLK6-ICXKAA",
        #         "ordertype": "market",
        #         "pair": "ETH/USD",
        #         "postxid": "TKH2SE-M7IF5-CFI7LT",
        #         "price": "169.97999",
        #         "time": "1586340530.895739",
        #         "type": "buy",
        #         "vol": "0.13043500"
        #     }
        #
        wsName = self.safe_string(trade, 'pair')
        market = self.safe_value(self.options['marketsByWsName'], wsName, market)
        symbol = None
        orderId = self.safe_string(trade, 'ordertxid')
        id = self.safe_string_2(trade, 'id', 'postxid')
        timestamp = self.safe_timestamp(trade, 'time')
        side = self.safe_string(trade, 'type')
        type = self.safe_string(trade, 'ordertype')
        price = self.safe_float(trade, 'price')
        amount = self.safe_float(trade, 'vol')
        cost = None
        fee = None
        if 'fee' in trade:
            currency = None
            if market is not None:
                currency = market['quote']
            fee = {
                'cost': self.safe_float(trade, 'fee'),
                'currency': currency,
            }
        if market is not None:
            symbol = market['symbol']
        if price is not None:
            if amount is not None:
                cost = price * amount
        return {
            'id': id,
            'order': orderId,
            'info': trade,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'symbol': symbol,
            'type': type,
            'side': side,
            'takerOrMaker': None,
            'price': price,
            'amount': amount,
            'cost': cost,
            'fee': fee,
        }

    async def watch_orders(self, symbol: Str = None, since: Int = None, limit: Int = None, params={}) -> List[Order]:
        """

        https://docs.kraken.com/api/docs/websocket-v1/openorders

        watches information on multiple orders made by the user
        :param str symbol: unified market symbol of the market orders were made in
        :param int [since]: the earliest time in ms to fetch orders for
        :param int [limit]: the maximum number of  orde structures to retrieve
        :param dict [params]: maximum number of orderic to the exchange API endpoint
        :returns dict[]: a list of `order structures <https://docs.ccxt.com/#/?id=order-structure>`
        """
        return await self.watch_private('openOrders', symbol, since, limit, params)

    def handle_orders(self, client: Client, message, subscription=None):
        #
        #     [
        #         [
        #             {
        #                 "OGTT3Y-C6I3P-XRI6HX": {
        #                     "cost": "0.00000",
        #                     "descr": {
        #                         "close": "",
        #                         "leverage": "0:1",
        #                         "order": "sell 10.00345345 XBT/EUR @ limit 34.50000 with 0:1 leverage",
        #                         "ordertype": "limit",
        #                         "pair": "XBT/EUR",
        #                         "price": "34.50000",
        #                         "price2": "0.00000",
        #                         "type": "sell"
        #                     },
        #                     "expiretm": "0.000000",
        #                     "fee": "0.00000",
        #                     "limitprice": "34.50000",
        #                     "misc": "",
        #                     "oflags": "fcib",
        #                     "opentm": "0.000000",
        #                     "price": "34.50000",
        #                     "refid": "OKIVMP-5GVZN-Z2D2UA",
        #                     "starttm": "0.000000",
        #                     "status": "open",
        #                     "stopprice": "0.000000",
        #                     "userref": 0,
        #                     "vol": "10.00345345",
        #                     "vol_exec": "0.00000000"
        #                 }
        #             },
        #             {
        #                 "OGTT3Y-C6I3P-XRI6HX": {
        #                     "cost": "0.00000",
        #                     "descr": {
        #                         "close": "",
        #                         "leverage": "0:1",
        #                         "order": "sell 0.00000010 XBT/EUR @ limit 5334.60000 with 0:1 leverage",
        #                         "ordertype": "limit",
        #                         "pair": "XBT/EUR",
        #                         "price": "5334.60000",
        #                         "price2": "0.00000",
        #                         "type": "sell"
        #                     },
        #                     "expiretm": "0.000000",
        #                     "fee": "0.00000",
        #                     "limitprice": "5334.60000",
        #                     "misc": "",
        #                     "oflags": "fcib",
        #                     "opentm": "0.000000",
        #                     "price": "5334.60000",
        #                     "refid": "OKIVMP-5GVZN-Z2D2UA",
        #                     "starttm": "0.000000",
        #                     "status": "open",
        #                     "stopprice": "0.000000",
        #                     "userref": 0,
        #                     "vol": "0.00000010",
        #                     "vol_exec": "0.00000000"
        #                 }
        #             },
        #         ],
        #         "openOrders",
        #         {"sequence": 234}
        #     ]
        #
        # status-change
        #
        #     [
        #         [
        #             {"OGTT3Y-C6I3P-XRI6HX": {"status": "closed"}},
        #             {"OGTT3Y-C6I3P-XRI6HX": {"status": "closed"}},
        #         ],
        #         "openOrders",
        #         {"sequence": 59342}
        #     ]
        #
        allOrders = self.safe_value(message, 0, [])
        allOrdersLength = len(allOrders)
        if allOrdersLength > 0:
            limit = self.safe_integer(self.options, 'ordersLimit', 1000)
            if self.orders is None:
                self.orders = ArrayCacheBySymbolById(limit)
            stored = self.orders
            symbols: dict = {}
            for i in range(0, len(allOrders)):
                orders = self.safe_value(allOrders, i, {})
                ids = list(orders.keys())
                for j in range(0, len(ids)):
                    id = ids[j]
                    order = orders[id]
                    parsed = self.parse_ws_order(order)
                    parsed['id'] = id
                    symbol = None
                    symbolsByOrderId = self.safe_value(self.options, 'symbolsByOrderId', {})
                    if parsed['symbol'] is not None:
                        symbol = parsed['symbol']
                        symbolsByOrderId[id] = symbol
                        self.options['symbolsByOrderId'] = symbolsByOrderId
                    else:
                        symbol = self.safe_string(symbolsByOrderId, id)
                    previousOrders = self.safe_value(stored.hashmap, symbol)
                    previousOrder = self.safe_value(previousOrders, id)
                    newOrder = parsed
                    if previousOrder is not None:
                        newRawOrder = self.extend(previousOrder['info'], newOrder['info'])
                        newOrder = self.parse_ws_order(newRawOrder)
                        newOrder['id'] = id
                    length = len(stored)
                    if length == limit and (previousOrder is None):
                        first = stored[0]
                        if first['id'] in symbolsByOrderId:
                            del symbolsByOrderId[first['id']]
                    stored.append(newOrder)
                    symbols[symbol] = True
            name = 'openOrders'
            client.resolve(self.orders, name)
            keys = list(symbols.keys())
            for i in range(0, len(keys)):
                messageHash = name + ':' + keys[i]
                client.resolve(self.orders, messageHash)

    def parse_ws_order(self, order, market=None):
        #
        # createOrder
        #    {
        #        "avg_price": "0.00000",
        #        "cost": "0.00000",
        #        "descr": {
        #            "close": null,
        #            "leverage": null,
        #            "order": "sell 0.01000000 ETH/USDT @ limit 1900.00000",
        #            "ordertype": "limit",
        #            "pair": "ETH/USDT",
        #            "price": "1900.00000",
        #            "price2": "0.00000",
        #            "type": "sell"
        #        },
        #        "expiretm": null,
        #        "fee": "0.00000",
        #        "limitprice": "0.00000",
        #        "misc": '',
        #        "oflags": "fciq",
        #        "opentm": "1667522705.757622",
        #        "refid": null,
        #        "starttm": null,
        #        "status": "open",
        #        "stopprice": "0.00000",
        #        "timeinforce": "GTC",
        #        "userref": 0,
        #        "vol": "0.01000000",
        #        "vol_exec": "0.00000000"
        #    }
        #
        description = self.safe_value(order, 'descr', {})
        orderDescription = self.safe_string(description, 'order')
        side = None
        type = None
        wsName = None
        price = None
        amount = None
        if orderDescription is not None:
            parts = orderDescription.split(' ')
            side = self.safe_string(parts, 0)
            amount = self.safe_string(parts, 1)
            wsName = self.safe_string(parts, 2)
            type = self.safe_string(parts, 4)
            price = self.safe_string(parts, 5)
        side = self.safe_string(description, 'type', side)
        type = self.safe_string(description, 'ordertype', type)
        wsName = self.safe_string(description, 'pair', wsName)
        market = self.safe_value(self.options['marketsByWsName'], wsName, market)
        symbol = None
        timestamp = self.safe_timestamp(order, 'opentm')
        amount = self.safe_string(order, 'vol', amount)
        filled = self.safe_string(order, 'vol_exec')
        fee = None
        cost = self.safe_string(order, 'cost')
        price = self.safe_string(description, 'price', price)
        if (price is None) or (Precise.string_eq(price, '0.0')):
            price = self.safe_string(description, 'price2')
        if (price is None) or (Precise.string_eq(price, '0.0')):
            price = self.safe_string(order, 'price', price)
        average = self.safe_string_2(order, 'avg_price', 'price')
        if market is not None:
            symbol = market['symbol']
            if 'fee' in order:
                flags = order['oflags']
                feeCost = self.safe_string(order, 'fee')
                fee = {
                    'cost': feeCost,
                    'rate': None,
                }
                if flags.find('fciq') >= 0:
                    fee['currency'] = market['quote']
                elif flags.find('fcib') >= 0:
                    fee['currency'] = market['base']
        status = self.parse_order_status(self.safe_string(order, 'status'))
        id = self.safe_string(order, 'id')
        if id is None:
            txid = self.safe_value(order, 'txid')
            id = self.safe_string(txid, 0)
        clientOrderId = self.safe_string(order, 'userref')
        rawTrades = self.safe_value(order, 'trades')
        trades = None
        if rawTrades is not None:
            trades = self.parse_trades(rawTrades, market, None, None, {'order': id})
        stopPrice = self.safe_number(order, 'stopprice')
        return self.safe_order({
            'id': id,
            'clientOrderId': clientOrderId,
            'info': order,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'lastTradeTimestamp': None,
            'status': status,
            'symbol': symbol,
            'type': type,
            'timeInForce': None,
            'postOnly': None,
            'side': side,
            'price': price,
            'stopPrice': stopPrice,
            'triggerPrice': stopPrice,
            'cost': cost,
            'amount': amount,
            'filled': filled,
            'average': average,
            'remaining': None,
            'fee': fee,
            'trades': trades,
        })

    async def watch_multi_helper(self, unifiedName: str, channelName: str, symbols: Strings = None, subscriptionArgs=None, params={}):
        await self.load_markets()
        # symbols are required
        symbols = self.market_symbols(symbols, None, False, True, False)
        messageHashes = []
        for i in range(0, len(symbols)):
            eventTrigger = self.safe_string(params, 'event_trigger')
            if eventTrigger is not None:
                messageHashes.append(self.get_message_hash(channelName, None, self.symbol(symbols[i])))
            else:
                messageHashes.append(self.get_message_hash(unifiedName, None, self.symbol(symbols[i])))
        request: dict = {
            'method': 'subscribe',
            'params': {
                'channel': channelName,
                'symbol': symbols,
            },
            'req_id': self.request_id(),
        }
        request['params'] = self.deep_extend(request['params'], params)
        url = self.urls['api']['ws']['publicV2']
        return await self.watch_multiple(url, messageHashes, request, messageHashes, subscriptionArgs)

    async def watch_balance(self, params={}) -> Balances:
        """
        watch balance and get the amount of funds available for trading or funds locked in orders

        https://docs.kraken.com/api/docs/websocket-v2/balances

        :param dict [params]: extra parameters specific to the exchange API endpoint
        :returns dict: a `balance structure <https://docs.ccxt.com/#/?id=balance-structure>`
        """
        await self.load_markets()
        token = await self.authenticate()
        messageHash = 'balances'
        url = self.urls['api']['ws']['privateV2']
        requestId = self.request_id()
        subscribe: dict = {
            'method': 'subscribe',
            'req_id': requestId,
            'params': {
                'channel': 'balances',
                'token': token,
            },
        }
        request = self.deep_extend(subscribe, params)
        return await self.watch(url, messageHash, request, messageHash)

    def handle_balance(self, client: Client, message):
        #
        #     {
        #         "channel": "balances",
        #         "data": [
        #             {
        #                 "asset": "BTC",
        #                 "asset_class": "currency",
        #                 "balance": 1.2,
        #                 "wallets": [
        #                     {
        #                         "type": "spot",
        #                         "id": "main",
        #                         "balance": 1.2
        #                     }
        #                 ]
        #             }
        #         ],
        #         "type": "snapshot",
        #         "sequence": 1
        #     }
        #
        data = self.safe_list(message, 'data', [])
        result: dict = {'info': message}
        for i in range(0, len(data)):
            currencyId = self.safe_string(data[i], 'asset')
            code = self.safe_currency_code(currencyId)
            account = self.account()
            eq = self.safe_string(data[i], 'balance')
            account['total'] = eq
            result[code] = account
        type = 'spot'
        balance = self.safe_balance(result)
        oldBalance = self.safe_value(self.balance, type, {})
        newBalance = self.deep_extend(oldBalance, balance)
        self.balance[type] = self.safe_balance(newBalance)
        channel = self.safe_string(message, 'channel')
        client.resolve(self.balance[type], channel)

    def get_message_hash(self, unifiedElementName: str, subChannelName: Str = None, symbol: Str = None):
        # unifiedElementName can be : orderbook, trade, ticker, bidask ...
        # subChannelName only applies to channel that needs specific variation(i.e. depth_50, depth_100..) to be selected
        withSymbol = symbol is not None
        messageHash = unifiedElementName
        if not withSymbol:
            messageHash += 's'
        else:
            messageHash += '@' + symbol
        if subChannelName is not None:
            messageHash += '#' + subChannelName
        return messageHash

    def handle_subscription_status(self, client: Client, message):
        #
        # public
        #
        #     {
        #         "channelID": 210,
        #         "channelName": "book-10",
        #         "event": "subscriptionStatus",
        #         "reqid": 1574146735269,
        #         "pair": "ETH/XBT",
        #         "status": "subscribed",
        #         "subscription": {depth: 10, name: "book"}
        #     }
        #
        # private
        #
        #     {
        #         "channelName": "openOrders",
        #         "event": "subscriptionStatus",
        #         "reqid": 1,
        #         "status": "subscribed",
        #         "subscription": {maxratecount: 125, name: "openOrders"}
        #     }
        #
        channelId = self.safe_string(message, 'channelID')
        if channelId is not None:
            client.subscriptions[channelId] = message
        # requestId = self.safe_string(message, "reqid")
        # if requestId in client.futures:
        #     del client.futures[requestId]
        # }

    def handle_error_message(self, client: Client, message) -> Bool:
        #
        #     {
        #         "errorMessage": "Currency pair not in ISO 4217-A3 format foobar",
        #         "event": "subscriptionStatus",
        #         "pair": "foobar",
        #         "reqid": 1574146735269,
        #         "status": "error",
        #         "subscription": {name: "ticker"}
        #     }
        #
        # v2
        #     {
        #         "error": "Unsupported field: 'price' for the given msg type: add order",
        #         "method": "add_order",
        #         "success": False,
        #         "time_in": "2025-05-13T08:59:44.803511Z",
        #         "time_out": "2025-05-13T08:59:44.803542Z'
        #     }
        #
        errorMessage = self.safe_string_2(message, 'errorMessage', 'error')
        if errorMessage is not None:
            # requestId = self.safe_value_2(message, 'reqid', 'req_id')
            broad = self.exceptions['ws']['broad']
            broadKey = self.find_broadly_matched_key(broad, errorMessage)
            exception = None
            if broadKey is None:
                exception = ExchangeError(errorMessage)  # c# requirement to convert the errorMessage to string
            else:
                exception = broad[broadKey](errorMessage)
            # if requestId is not None:
            #     client.reject(exception, requestId)
            # else:
            client.reject(exception)
            # }
            return False
        return True

    def handle_message(self, client: Client, message):
        if isinstance(message, list):
            channelId = self.safe_string(message, 0)
            subscription = self.safe_value(client.subscriptions, channelId, {})
            info = self.safe_value(subscription, 'subscription', {})
            messageLength = len(message)
            channelName = self.safe_string(message, messageLength - 2)
            name = self.safe_string(info, 'name')
            methods: dict = {
                # public
                'ohlc': self.handle_ohlcv,
                # private
                'openOrders': self.handle_orders,
                'ownTrades': self.handle_my_trades,
            }
            method = self.safe_value_2(methods, name, channelName)
            if method is not None:
                method(client, message, subscription)
        else:
            channel = self.safe_string(message, 'channel')
            if channel is not None:
                methods: dict = {
                    'balances': self.handle_balance,
                    'book': self.handle_order_book,
                    'ticker': self.handle_ticker,
                    'trade': self.handle_trades,
                }
                method = self.safe_value(methods, channel)
                if method is not None:
                    method(client, message)
            if self.handle_error_message(client, message):
                event = self.safe_string_2(message, 'event', 'method')
                methods: dict = {
                    'heartbeat': self.handle_heartbeat,
                    'systemStatus': self.handle_system_status,
                    'subscriptionStatus': self.handle_subscription_status,
                    'add_order': self.handle_create_edit_order,
                    'amend_order': self.handle_create_edit_order,
                    'cancel_order': self.handle_cancel_order,
                    'cancel_all': self.handle_cancel_all_orders,
                    'pong': self.handle_pong,
                }
                method = self.safe_value(methods, event)
                if method is not None:
                    method(client, message)
