"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToLambda = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const api = require("aws-cdk-lib/aws-apigateway");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const route53 = require("aws-cdk-lib/aws-route53");
const targets = require("aws-cdk-lib/aws-route53-targets");
const constructs_1 = require("constructs");
const utils_1 = require("./utils");
class ApiGatewayToLambda extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaFunction = props.lambdaFunction;
        // Determine if we should use proxy mode
        const useProxy = props.proxy !== false && !props.customRoutes;
        if (useProxy) {
            this.apiGateway = new api.LambdaRestApi(this, "ApiGateway", {
                handler: props.lambdaFunction,
                restApiName: props.apiName,
                proxy: true,
                ...props.lambdaApiProps,
            });
        }
        else {
            this.apiGateway = new api.RestApi(this, "ApiGateway", {
                restApiName: props.apiName,
                ...props.restApiProps,
            });
            // Add custom routes if provided
            if (props.customRoutes) {
                this.addCustomRoutes(props.customRoutes);
            }
            // If no custom routes but proxy is disabled, add default integration
            if (!props.customRoutes && props.proxy === false) {
                this.apiGateway.root.addMethod("ANY", new api.LambdaIntegration(props.lambdaFunction));
            }
        }
        // Setup CloudWatch logging
        if (props.enableLogging) {
            const loggingResources = (0, utils_1.setupLogging)(this, props.apiName, props.logGroupProps);
            this.apiGatewayLogGroup = loggingResources.logGroup;
        }
        // Setup custom domain
        if (props.customDomainName) {
            const domainResources = this.setupCustomDomain(props);
            this.certificate = domainResources.certificate;
            this.domain = domainResources.domain;
            this.aRecord = domainResources.aRecord;
        }
        // Setup usage plan
        if (props.createUsagePlan) {
            this.usagePlan = this.setupUsagePlan();
        }
    }
    /**
     * Add custom routes to the API
     */
    addCustomRoutes(routes) {
        const resourceMap = new Map();
        routes.forEach((route) => {
            const pathParts = route.path
                .split("/")
                .filter((part) => part !== "");
            let currentResource = this.apiGateway.root;
            // Build nested resources
            let currentPath = "";
            for (const part of pathParts) {
                currentPath += `/${part}`;
                if (!resourceMap.has(currentPath)) {
                    currentResource = currentResource.addResource(part);
                    resourceMap.set(currentPath, currentResource);
                }
                else {
                    currentResource = resourceMap.get(currentPath);
                }
            }
            // Add method to the final resource
            currentResource.addMethod(route.method, new api.LambdaIntegration(route.handler), route.methodOptions);
        });
    }
    /**
     * Setup custom domain with certificate and Route53 record
     */
    setupCustomDomain(props) {
        let certificate;
        // Use existing certificate or create new one
        if (props.existingCertificate) {
            certificate = props.existingCertificate;
        }
        else if (props.hostedZone) {
            certificate = new acm.Certificate(this, "Certificate", {
                domainName: props.customDomainName,
                validation: acm.CertificateValidation.fromDns(props.hostedZone),
            });
        }
        else {
            throw new Error("Either certificateArn or hostedZone must be provided for custom domain");
        }
        // Create custom domain
        const domain = new api.DomainName(this, "CustomDomain", {
            domainName: props.customDomainName,
            certificate: certificate,
        });
        // Create base path mapping
        new api.BasePathMapping(this, "BasePathMapping", {
            domainName: domain,
            restApi: this.apiGateway,
        });
        let aRecord;
        // Create Route53 alias record if hosted zone provided
        if (props.hostedZone) {
            aRecord = new route53.ARecord(this, "CustomDomainAliasRecord", {
                zone: props.hostedZone,
                recordName: props.customDomainName,
                target: route53.RecordTarget.fromAlias(new targets.ApiGatewayDomain(domain)),
            });
        }
        return { certificate, domain, aRecord };
    }
    /**
     * Setup usage plan
     */
    setupUsagePlan() {
        return new api.UsagePlan(this, "UsagePlan", {
            name: `${this.apiGateway.restApiName}-usage-plan`,
            apiStages: [
                {
                    api: this.apiGateway,
                    stage: this.apiGateway.deploymentStage,
                },
            ],
        });
    }
    /**
     * Add a custom route after construction (for dynamic route addition)
     */
    addRoute(route) {
        const pathParts = route.path
            .split("/")
            .filter((part) => part !== "");
        let currentResource = this.apiGateway.root;
        // Navigate/create resource path
        for (const part of pathParts) {
            const existingResource = currentResource.getResource(part);
            if (existingResource) {
                currentResource = existingResource;
            }
            else {
                currentResource = currentResource.addResource(part);
            }
        }
        return currentResource.addMethod(route.method, new api.LambdaIntegration(route.handler), route.methodOptions);
    }
    get apiUrl() {
        return this.domain?.domainNameAliasDomainName ?? this.apiGateway.url;
    }
}
exports.ApiGatewayToLambda = ApiGatewayToLambda;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayToLambda[_a] = { fqn: "must-cdk.ApiGatewayToLambda", version: "0.0.52" };
//# sourceMappingURL=data:application/json;base64,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