"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontToOrigins = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const route53 = require("aws-cdk-lib/aws-route53");
const targets = require("aws-cdk-lib/aws-route53-targets");
const s3 = require("aws-cdk-lib/aws-s3");
const constructs_1 = require("constructs");
class CloudFrontToOrigins extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.aRecords = [];
        // Private maps for internal use
        this.s3BucketsMap = new Map();
        this.httpOriginsMap = new Map();
        this.s3OriginsMap = new Map();
        // Validate that at least one origin is provided
        if ((!props.s3Origins || props.s3Origins.length === 0) &&
            (!props.httpOrigins || props.httpOrigins.length === 0)) {
            throw new Error("At least one origin (s3Origins or httpOrigins) must be provided");
        }
        // Validate unique origin IDs
        this.validateOriginIds(props);
        // Apply intelligent defaults (unless disabled)
        const enhancedProps = props.disableIntelligentDefaults
            ? props
            : this.applyDefaults(props);
        let certificate;
        let logBucket;
        // Setup all S3 origins
        if (enhancedProps.s3Origins) {
            for (const s3Config of enhancedProps.s3Origins) {
                this.s3BucketsMap.set(s3Config.id, s3Config.bucket);
                const s3Origin = this.setupS3Origin(s3Config);
                this.s3OriginsMap.set(s3Config.id, s3Origin);
            }
        }
        // Setup all HTTP origins
        if (enhancedProps.httpOrigins) {
            for (const httpConfig of enhancedProps.httpOrigins) {
                const httpOrigin = this.setupHttpOrigin(httpConfig);
                this.httpOriginsMap.set(httpConfig.id, httpOrigin);
            }
        }
        // Setup custom domain and certificate
        if (enhancedProps.customDomainName) {
            certificate = this.setupCustomDomain(enhancedProps);
        }
        // Setup logging (enabled by default unless explicitly disabled)
        if (enhancedProps.enableLogging !== false) {
            logBucket = this.setupLogging(enhancedProps);
        }
        // Create CloudFront distribution
        this.distribution = this.createDistribution(enhancedProps, {
            certificate,
            logBucket,
        });
        // Setup Route53 records for all domains
        if (enhancedProps.createRoute53Records !== false &&
            enhancedProps.hostedZone) {
            this.setupRoute53Records(enhancedProps);
        }
        // Assign readonly properties
        this.logBucket = logBucket;
        this.certificate = certificate;
        this.domainNames = this.getAllDomainNames(enhancedProps);
    }
    /**
     * Get all domain names (primary + additional)
     */
    getAllDomainNames(props) {
        const domains = [];
        if (props.customDomainName) {
            domains.push(props.customDomainName);
        }
        if (props.additionalDomainNames) {
            domains.push(...props.additionalDomainNames);
        }
        return domains.length > 0 ? domains : undefined;
    }
    /**
     * Validate that all origin IDs are unique
     */
    validateOriginIds(props) {
        const allIds = new Set();
        // Check S3 origin IDs
        if (props.s3Origins) {
            for (const s3Origin of props.s3Origins) {
                if (allIds.has(s3Origin.id)) {
                    throw new Error(`Duplicate origin ID: ${s3Origin.id}`);
                }
                allIds.add(s3Origin.id);
            }
        }
        // Check HTTP origin IDs
        if (props.httpOrigins) {
            for (const httpOrigin of props.httpOrigins) {
                if (allIds.has(httpOrigin.id)) {
                    throw new Error(`Duplicate origin ID: ${httpOrigin.id}`);
                }
                allIds.add(httpOrigin.id);
            }
        }
        // Validate default origin ID exists
        if (props.defaultOriginId && !allIds.has(props.defaultOriginId)) {
            throw new Error(`Default origin ID '${props.defaultOriginId}' not found in origins`);
        }
        // Validate cache behavior origin IDs
        if (props.cacheBehaviors) {
            for (const behavior of props.cacheBehaviors) {
                if (!allIds.has(behavior.originId)) {
                    throw new Error(`Cache behavior origin ID '${behavior.originId}' not found in origins`);
                }
            }
        }
    }
    /**
     * Apply intelligent defaults based on origin configuration
     */
    applyDefaults(props) {
        const hasS3 = props.s3Origins && props.s3Origins.length > 0;
        const hasHttp = props.httpOrigins && props.httpOrigins.length > 0;
        let errorPages;
        // Only apply error page defaults if not disabled and not provided
        if (!props.disableDefaultErrorPages && !props.errorPages) {
            if (hasS3 && hasHttp) {
                // Hybrid: SPA-friendly error pages
                errorPages = this.getHybridErrorPages();
            }
            else if (hasS3) {
                // S3 only: SPA-friendly error pages
                errorPages = this.getS3ErrorPages();
            }
            else {
                // HTTP only: Basic error pages
                errorPages = this.getHttpErrorPages();
            }
        }
        return {
            // Apply base defaults
            enableLogging: true,
            createRoute53Records: true,
            defaultRootObject: "index.html",
            priceClass: cloudfront.PriceClass.PRICE_CLASS_100,
            enabled: true,
            // Override with user props
            ...props,
            // Apply error pages if generated
            ...(errorPages && { errorPages }),
            // Apply HTTP origin protocol defaults
            httpOrigins: props.httpOrigins?.map((httpOrigin) => ({
                protocolPolicy: cloudfront.OriginProtocolPolicy.HTTPS_ONLY,
                ...httpOrigin,
            })),
            // Merge cache behaviors
            cacheBehaviors: [...(props.cacheBehaviors || [])],
        };
    }
    /**
     * Default error pages for S3-only distributions
     */
    getS3ErrorPages() {
        return [
            {
                httpStatus: 404,
                responseHttpStatus: 200,
                responsePagePath: "/index.html",
                ttl: aws_cdk_lib_1.Duration.minutes(5),
            },
            {
                httpStatus: 403,
                responseHttpStatus: 200,
                responsePagePath: "/index.html",
                ttl: aws_cdk_lib_1.Duration.minutes(5),
            },
        ];
    }
    /**
     * Default error pages for HTTP-only distributions
     */
    getHttpErrorPages() {
        return [
            {
                httpStatus: 504,
                responseHttpStatus: 504,
                responsePagePath: "/error.html",
                ttl: aws_cdk_lib_1.Duration.seconds(30),
            },
        ];
    }
    /**
     * Default error pages for hybrid distributions
     */
    getHybridErrorPages() {
        return [
            {
                httpStatus: 404,
                responseHttpStatus: 200,
                responsePagePath: "/index.html",
                ttl: aws_cdk_lib_1.Duration.minutes(5),
            },
        ];
    }
    /**
     * Setup S3 origin
     */
    setupS3Origin(s3Config) {
        if (s3Config.useLegacyOAI) {
            // Use legacy Origin Access Identity
            const originAccessIdentity = s3Config.originAccessIdentity ||
                new cloudfront.OriginAccessIdentity(this, `OAI-${s3Config.id}`, {
                    comment: `OAI for ${s3Config.bucket.bucketName}`,
                });
            // Grant CloudFront access to S3 bucket
            s3Config.bucket.grantRead(originAccessIdentity);
            return new origins.S3Origin(s3Config.bucket, {
                originAccessIdentity,
                originPath: s3Config.originPath,
                ...s3Config.s3OriginProps,
            });
        }
        else {
            // Use modern Origin Access Control (recommended)
            return origins.S3BucketOrigin.withOriginAccessControl(s3Config.bucket, {
                originPath: s3Config.originPath,
                ...s3Config.s3OriginProps,
            });
        }
    }
    /**
     * Setup HTTP origin
     */
    setupHttpOrigin(httpConfig) {
        return new origins.HttpOrigin(httpConfig.domainName, {
            originPath: httpConfig.originPath,
            protocolPolicy: httpConfig.protocolPolicy || cloudfront.OriginProtocolPolicy.HTTPS_ONLY,
            httpPort: httpConfig.httpPort,
            httpsPort: httpConfig.httpsPort,
            ...httpConfig.httpOriginProps,
        });
    }
    /**
     * Setup custom domain and certificate
     */
    setupCustomDomain(props) {
        // If certificate ARN is provided, use it directly
        if (props.certificateArn) {
            // Validate that certificate ARN is from us-east-1
            if (!props.certificateArn.includes("us-east-1")) {
                throw new Error("Certificate must be in us-east-1 region for CloudFront");
            }
            return acm.Certificate.fromCertificateArn(this, "Certificate", props.certificateArn);
        }
        // Auto-create certificate in us-east-1 if hostedZone is provided
        if (props.hostedZone && props.customDomainName) {
            return this.createCrossRegionCertificate(props);
        }
        throw new Error("Either certificateArn or hostedZone must be provided for custom domain. " +
            "If hostedZone is provided, certificate will be automatically created in us-east-1.");
    }
    /**
     * Create certificate in us-east-1 using CDK cross-region support
     */
    createCrossRegionCertificate(props) {
        const allDomains = this.getAllDomainNames(props) || [
            props.customDomainName,
        ];
        // Create certificate directly in us-east-1 using CDK
        return new acm.Certificate(this, "Certificate", {
            domainName: props.customDomainName,
            subjectAlternativeNames: allDomains.length > 1 ? allDomains.slice(1) : undefined,
            validation: acm.CertificateValidation.fromDns(props.hostedZone),
        });
    }
    /**
     * Setup CloudFront logging
     */
    setupLogging(props) {
        return (props.logBucket ||
            new s3.Bucket(this, "LogBucket", {
                blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                versioned: false,
                lifecycleRules: [
                    {
                        expiration: aws_cdk_lib_1.Duration.days(90),
                        transitions: [
                            {
                                storageClass: s3.StorageClass.INFREQUENT_ACCESS,
                                transitionAfter: aws_cdk_lib_1.Duration.days(30),
                            },
                        ],
                    },
                ],
            }));
    }
    /**
     * Get origin by ID
     */
    getOriginById(originId) {
        // Check S3 origins first
        const s3Origin = this.s3OriginsMap.get(originId);
        if (s3Origin) {
            return s3Origin;
        }
        // Check HTTP origins
        const httpOrigin = this.httpOriginsMap.get(originId);
        if (httpOrigin) {
            return httpOrigin;
        }
        throw new Error(`Origin with ID '${originId}' not found`);
    }
    /**
     * Determine default origin ID
     */
    getDefaultOriginId(props) {
        if (props.defaultOriginId) {
            return props.defaultOriginId;
        }
        // Default to first S3 origin, then first HTTP origin
        if (props.s3Origins && props.s3Origins.length > 0) {
            return props.s3Origins[0].id;
        }
        if (props.httpOrigins && props.httpOrigins.length > 0) {
            return props.httpOrigins[0].id;
        }
        throw new Error("No origins configured");
    }
    /**
     * Create CloudFront distribution
     */
    createDistribution(props, resources) {
        // Determine default origin
        const defaultOriginId = this.getDefaultOriginId(props);
        const defaultOrigin = this.getOriginById(defaultOriginId);
        const defaultBehavior = {
            origin: defaultOrigin,
            viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            compress: true,
        };
        // Build additional behaviors
        const additionalBehaviors = {};
        if (props.cacheBehaviors) {
            for (const behavior of props.cacheBehaviors) {
                const origin = this.getOriginById(behavior.originId);
                // Build behavior options object properly
                const behaviorConfig = {
                    origin,
                    viewerProtocolPolicy: behavior.viewerProtocolPolicy ||
                        cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                    allowedMethods: behavior.allowedMethods,
                    cachedMethods: behavior.cachedMethods,
                    compress: behavior.compress ?? true,
                };
                // Add cache policy (prioritize direct policy over ID)
                if (behavior.cachePolicy) {
                    behaviorConfig.cachePolicy = behavior.cachePolicy;
                }
                else if (behavior.cachePolicyId) {
                    behaviorConfig.cachePolicy = cloudfront.CachePolicy.fromCachePolicyId(this, `CachePolicy-${behavior.pathPattern.replace(/[^a-zA-Z0-9]/g, "")}`, behavior.cachePolicyId);
                }
                // Add origin request policy
                if (behavior.originRequestPolicy) {
                    behaviorConfig.originRequestPolicy = behavior.originRequestPolicy;
                }
                else if (behavior.originRequestPolicyId) {
                    behaviorConfig.originRequestPolicy =
                        cloudfront.OriginRequestPolicy.fromOriginRequestPolicyId(this, `OriginRequestPolicy-${behavior.pathPattern.replace(/[^a-zA-Z0-9]/g, "")}`, behavior.originRequestPolicyId);
                }
                // Add response headers policy
                if (behavior.responseHeadersPolicy) {
                    behaviorConfig.responseHeadersPolicy = behavior.responseHeadersPolicy;
                }
                else if (behavior.responseHeadersPolicyId) {
                    behaviorConfig.responseHeadersPolicy =
                        cloudfront.ResponseHeadersPolicy.fromResponseHeadersPolicyId(this, `ResponseHeadersPolicy-${behavior.pathPattern.replace(/[^a-zA-Z0-9]/g, "")}`, behavior.responseHeadersPolicyId);
                }
                additionalBehaviors[behavior.pathPattern] =
                    behaviorConfig;
            }
        }
        // Prepare distribution props
        const distributionProps = {
            comment: props.comment || `CloudFront distribution for ${this.node.id}`,
            defaultBehavior,
            additionalBehaviors,
            domainNames: this.getAllDomainNames(props),
            certificate: resources.certificate,
            defaultRootObject: props.defaultRootObject,
            enabled: props.enabled,
            priceClass: props.priceClass,
            geoRestriction: props.geoRestriction,
            enableLogging: props.enableLogging !== false,
            logBucket: resources.logBucket,
            logFilePrefix: props.logPrefix,
            logIncludesCookies: props.logIncludeCookies,
            errorResponses: props.errorPages,
            webAclId: props.webAclId,
            enableIpv6: props.enableIpv6,
            httpVersion: props.httpVersion,
        };
        return new cloudfront.Distribution(this, "Distribution", distributionProps);
    }
    /**
     * Setup Route53 alias records for all domains
     */
    setupRoute53Records(props) {
        if (!props.hostedZone) {
            return;
        }
        const domains = this.getAllDomainNames(props);
        if (!domains) {
            return;
        }
        // Create A record for each domain
        domains.forEach((domain, index) => {
            const recordId = index === 0 ? "AliasRecord" : `AliasRecord${index + 1}`;
            const aRecord = new route53.ARecord(this, recordId, {
                zone: props.hostedZone,
                recordName: domain,
                target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(this.distribution)),
            });
            this.aRecords.push(aRecord);
        });
    }
    /**
     * Get S3 bucket by origin ID
     */
    getS3Bucket(originId) {
        return this.s3BucketsMap.get(originId);
    }
    /**
     * Get HTTP origin by origin ID
     */
    getHttpOrigin(originId) {
        return this.httpOriginsMap.get(originId);
    }
    /**
     * Get all S3 bucket origin IDs
     */
    get s3OriginIds() {
        return Array.from(this.s3BucketsMap.keys());
    }
    /**
     * Get all HTTP origin IDs
     */
    get httpOriginIds() {
        return Array.from(this.httpOriginsMap.keys());
    }
    /**
     * Get all S3 buckets as an array of objects with ID and bucket
     */
    get s3Origins() {
        return Array.from(this.s3BucketsMap.entries()).map(([id, bucket]) => ({
            id,
            bucket,
        }));
    }
    /**
     * Get all HTTP origins as an array of objects with ID and origin
     */
    get httpOrigins() {
        return Array.from(this.httpOriginsMap.entries()).map(([id, origin]) => ({
            id,
            origin,
        }));
    }
    /**
     * Get the CloudFront distribution domain name
     */
    get distributionDomainName() {
        return this.distribution.distributionDomainName;
    }
    /**
     * Get the CloudFront distribution URL with protocol
     */
    get distributionUrl() {
        return `https://${this.distribution.distributionDomainName}`;
    }
    /**
     * Get the custom domain URL (if configured)
     */
    get customDomainUrl() {
        return this.domainNames?.[0] ? `https://${this.domainNames[0]}` : undefined;
    }
}
exports.CloudFrontToOrigins = CloudFrontToOrigins;
_a = JSII_RTTI_SYMBOL_1;
CloudFrontToOrigins[_a] = { fqn: "must-cdk.CloudFrontToOrigins", version: "0.0.52" };
//# sourceMappingURL=data:application/json;base64,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