# Binary Builds

This document describes how to build standalone binary distributions of Riveter using PyInstaller.

## Overview

Riveter can be built as standalone executables that don't require Python to be installed on the target system. This is useful for:

- Distribution via package managers (e.g., Homebrew)
- Deployment in environments where Python setup is complex
- Simplified installation for end users
- CI/CD environments with minimal dependencies

## Requirements

### Build Dependencies

The following dependencies are required to build binaries:

- Python 3.12+
- PyInstaller 6.0.0+
- All Riveter runtime dependencies

Install build dependencies:

```bash
pip install -e ".[dev]"
```

This will install PyInstaller along with other development dependencies.

### Supported Platforms

Binary builds are supported for the following platforms:

- **macOS Intel** (`macos-intel`): x86_64 architecture
- **macOS Apple Silicon** (`macos-arm64`): ARM64 architecture
- **Linux x86_64** (`linux-x86_64`): x86_64 architecture

## Building Binaries

### Basic Usage

Build a binary for the current platform:

```bash
python scripts/build_binary.py
```

### Platform-Specific Builds

Build for a specific target platform:

```bash
# macOS Intel
python scripts/build_binary.py --target macos-intel

# macOS Apple Silicon
python scripts/build_binary.py --target macos-arm64

# Linux x86_64
python scripts/build_binary.py --target linux-x86_64
```

### Debug Mode

Enable debug mode for verbose output and debugging information:

```bash
python scripts/build_binary.py --debug
```

### Clean Build Artifacts

Clean previous build artifacts without building:

```bash
python scripts/build_binary.py --clean-only
```

## Build Process

The build process consists of the following steps:

1. **Environment Validation**: Checks that PyInstaller and required dependencies are available
2. **Artifact Cleanup**: Removes previous build artifacts
3. **Spec Generation**: Creates a PyInstaller spec file with platform-specific configurations
4. **Binary Creation**: Runs PyInstaller to create the standalone executable
5. **Validation**: Tests the built binary to ensure it works correctly

## Build Output

Successful builds produce:

- **Binary**: `dist/riveter` - The standalone executable
- **Build artifacts**: `build/` directory (can be cleaned up)
- **Temporary spec file**: Automatically cleaned up after build

## Binary Features

The built binaries include:

### Included Components

- All Riveter Python modules
- All runtime dependencies (PyYAML, Click, Rich, HCL2, etc.)
- Rule packs from the `rule_packs/` directory
- Static assets and configuration templates

### Excluded Components

To minimize binary size, the following are excluded:

- Test frameworks (pytest, unittest, etc.)
- Development tools (black, mypy, ruff, etc.)
- Documentation tools (sphinx, etc.)
- Unused standard library modules
- GUI frameworks (tkinter, etc.)

### Performance Optimizations

- **UPX Compression**: Enabled on supported platforms to reduce binary size
- **Strip Debug Symbols**: Removes debug information in release builds
- **Module Exclusion**: Excludes unnecessary modules to reduce size
- **Import Optimization**: Uses hidden imports to ensure all required modules are included

## Binary Size

Typical binary sizes:

- **macOS**: ~50-70 MB
- **Linux**: ~45-65 MB

Actual size depends on:
- Number of rule packs included
- Platform-specific dependencies
- Compression settings

## Troubleshooting

### Common Issues

#### PyInstaller Not Found

```
Error: PyInstaller is not installed
```

**Solution**: Install development dependencies:
```bash
pip install -e ".[dev]"
```

#### Missing Modules at Runtime

If the binary fails with import errors:

1. Check the hidden imports in `scripts/build_spec.py`
2. Add missing modules to the `get_all_hidden_imports()` function
3. Rebuild the binary

#### Binary Too Large

To reduce binary size:

1. Review excluded modules in `get_platform_config()`
2. Remove unnecessary rule packs
3. Enable UPX compression (if not already enabled)

#### Permission Denied (Linux/macOS)

The build script automatically makes binaries executable, but if you encounter permission issues:

```bash
chmod +x dist/riveter
```

### Debug Mode

Use debug mode to troubleshoot build issues:

```bash
python scripts/build_binary.py --debug
```

This provides:
- Verbose PyInstaller output
- Detailed logging of build steps
- Module inclusion/exclusion information

### Manual Testing

Test the binary manually:

```bash
# Basic functionality
./dist/riveter --version
./dist/riveter --help

# Rule pack listing
./dist/riveter list-rule-packs

# Sample scan (requires Terraform file)
./dist/riveter scan -p aws-security -t main.tf
```

## Integration with CI/CD

### GitHub Actions

Example workflow for building binaries:

```yaml
name: Build Binaries
on:
  release:
    types: [published]

jobs:
  build:
    strategy:
      matrix:
        include:
          - os: macos-latest
            target: macos-intel
          - os: macos-latest-xlarge
            target: macos-arm64
          - os: ubuntu-latest
            target: linux-x86_64

    runs-on: ${{ matrix.os }}
    steps:
      - uses: actions/checkout@v4
      - uses: actions/setup-python@v4
        with:
          python-version: '3.12'

      - name: Install dependencies
        run: pip install -e ".[dev]"

      - name: Build binary
        run: python scripts/build_binary.py --target ${{ matrix.target }}

      - name: Upload binary
        uses: actions/upload-artifact@v3
        with:
          name: riveter-${{ matrix.target }}
          path: dist/riveter
```

## Security Considerations

### Code Signing

Currently, binaries are not code-signed. For production distribution:

- **macOS**: Consider code signing with Apple Developer certificates
- **Windows**: Consider Authenticode signing (when Windows support is added)
- **Linux**: Consider GPG signing of distribution packages

### Checksums

Always generate and verify checksums for distributed binaries:

```bash
# Generate SHA256 checksum
shasum -a 256 dist/riveter > riveter.sha256

# Verify checksum
shasum -a 256 -c riveter.sha256
```

### Supply Chain Security

- Pin PyInstaller version for reproducible builds
- Verify all dependencies before building
- Use isolated build environments
- Scan binaries for vulnerabilities before distribution

## Future Enhancements

Planned improvements:

- **Windows Support**: Add Windows binary builds
- **Code Signing**: Implement automatic code signing
- **Size Optimization**: Further reduce binary size
- **Cross-compilation**: Build for multiple platforms from single host
- **Reproducible Builds**: Ensure bit-for-bit reproducible builds
