# Checksum Verification Guide

This guide explains how to verify the integrity of Riveter binary distributions using SHA256 checksums.

## Overview

All Riveter binary releases include SHA256 checksums to ensure file integrity and authenticity. This helps protect against:

- Corrupted downloads
- Tampered files
- Man-in-the-middle attacks
- Storage corruption

## Automatic Verification (Recommended)

### Using the Verification Script

Riveter includes a built-in verification script that can automatically verify binaries against official release checksums:

```bash
# Verify against official release checksum
python scripts/verify_checksum.py riveter-1.2.3-macos-intel.tar.gz --verify-release 1.2.3 --platform macos-intel

# Verify with local checksum file
python scripts/verify_checksum.py riveter-1.2.3-macos-intel.tar.gz --checksum-file riveter-1.2.3-macos-intel.tar.gz.sha256

# Verify with explicit checksum
python scripts/verify_checksum.py riveter-1.2.3-macos-intel.tar.gz abc123def456...
```

### Supported Platforms

- `macos-intel` - macOS Intel (x86_64)
- `macos-arm64` - macOS Apple Silicon (ARM64)
- `linux-x86_64` - Linux x86_64

## Manual Verification

### Step 1: Download Files

Download both the binary archive and its corresponding checksum file from the [GitHub releases page](https://github.com/riveter/riveter/releases):

```bash
# Example for version 1.2.3 on macOS Intel
wget https://github.com/riveter/riveter/releases/download/v1.2.3/riveter-1.2.3-macos-intel.tar.gz
wget https://github.com/riveter/riveter/releases/download/v1.2.3/riveter-1.2.3-macos-intel.tar.gz.sha256
```

### Step 2: Verify Checksum

#### On macOS and Linux

```bash
# Using sha256sum (Linux)
sha256sum -c riveter-1.2.3-macos-intel.tar.gz.sha256

# Using shasum (macOS)
shasum -a 256 -c riveter-1.2.3-macos-intel.tar.gz.sha256

# Manual comparison
expected=$(cat riveter-1.2.3-macos-intel.tar.gz.sha256 | cut -d' ' -f1)
actual=$(sha256sum riveter-1.2.3-macos-intel.tar.gz | cut -d' ' -f1)
if [ "$expected" = "$actual" ]; then
    echo "✅ Checksum verification passed"
else
    echo "❌ Checksum verification failed"
fi
```

#### Using Python (Cross-platform)

```python
import hashlib

def verify_checksum(file_path, checksum_file_path):
    # Read expected checksum
    with open(checksum_file_path, 'r') as f:
        expected_checksum = f.read().strip().split()[0]

    # Calculate actual checksum
    sha256_hash = hashlib.sha256()
    with open(file_path, 'rb') as f:
        for chunk in iter(lambda: f.read(4096), b""):
            sha256_hash.update(chunk)
    actual_checksum = sha256_hash.hexdigest()

    # Compare
    if actual_checksum.lower() == expected_checksum.lower():
        print("✅ Checksum verification passed")
        return True
    else:
        print("❌ Checksum verification failed")
        print(f"Expected: {expected_checksum}")
        print(f"Actual:   {actual_checksum}")
        return False

# Usage
verify_checksum("riveter-1.2.3-macos-intel.tar.gz", "riveter-1.2.3-macos-intel.tar.gz.sha256")
```

## Homebrew Installation Verification

When installing via Homebrew, checksums are automatically verified by the Homebrew system. The Homebrew formula includes the official SHA256 checksums for each platform.

If you want to manually verify before installation:

```bash
# Download the binary that Homebrew would install
brew --cache scottryanhoward/homebrew-riveter/riveter

# Find the cached file and verify it
cache_file=$(brew --cache scottryanhoward/homebrew-riveter/riveter)
python scripts/verify_checksum.py "$cache_file" --verify-release <version> --platform <platform>
```

## Checksum File Format

Checksum files follow the standard format used by `sha256sum`:

```
abc123def456789...  riveter-1.2.3-macos-intel.tar.gz
```

- First field: 64-character hexadecimal SHA256 checksum
- Second field: Filename (separated by two spaces)

## Security Considerations

### What Checksums Protect Against

✅ **File corruption** during download or storage
✅ **Accidental modifications** to the binary
✅ **Basic tampering** attempts
✅ **Storage medium errors**

### What Checksums Don't Protect Against

❌ **Compromised release infrastructure** (if the official checksums are also compromised)
❌ **Supply chain attacks** at build time
❌ **Advanced persistent threats** with access to release systems

### Additional Security Measures

For maximum security, consider:

1. **Verify checksums from multiple sources** (GitHub releases page, official documentation)
2. **Use HTTPS** for all downloads
3. **Verify GPG signatures** (when available in future releases)
4. **Build from source** for critical deployments
5. **Use official package managers** (Homebrew, pip) when possible

## Troubleshooting

### Common Issues

#### "Checksum verification failed"

1. **Re-download the file** - the download may have been corrupted
2. **Check file integrity** - ensure the file wasn't modified after download
3. **Verify you have the correct version** - ensure the checksum file matches the binary version
4. **Check for typos** - ensure you're using the correct filenames

#### "Checksum file not found"

1. **Download the checksum file** from the same release page as the binary
2. **Check filename spelling** - ensure the checksum file name matches exactly
3. **Verify release exists** - ensure you're downloading from a valid release

#### "Invalid checksum format"

1. **Re-download checksum file** - the file may have been corrupted
2. **Check file encoding** - ensure the file is plain text, not HTML
3. **Verify source** - ensure you downloaded from the official GitHub releases page

### Getting Help

If you encounter issues with checksum verification:

1. **Check the troubleshooting steps** above
2. **Verify you're using the latest verification script** from the main branch
3. **Open an issue** on the [GitHub repository](https://github.com/riveter/riveter/issues) with:
   - The exact command you ran
   - The error message received
   - Your operating system and version
   - The Riveter version you're trying to verify

## Examples

### Complete Verification Workflow

```bash
# 1. Download binary and checksum
wget https://github.com/riveter/riveter/releases/download/v1.2.3/riveter-1.2.3-linux-x86_64.tar.gz
wget https://github.com/riveter/riveter/releases/download/v1.2.3/riveter-1.2.3-linux-x86_64.tar.gz.sha256

# 2. Verify checksum
python scripts/verify_checksum.py riveter-1.2.3-linux-x86_64.tar.gz --checksum-file riveter-1.2.3-linux-x86_64.tar.gz.sha256

# 3. Extract and install (if verification passed)
tar -xzf riveter-1.2.3-linux-x86_64.tar.gz
sudo cp riveter-1.2.3-linux-x86_64/riveter /usr/local/bin/

# 4. Verify installation
riveter --version
```

### Automated Verification in CI/CD

```yaml
# GitHub Actions example
- name: Download and verify Riveter
  run: |
    VERSION="1.2.3"
    PLATFORM="linux-x86_64"

    # Download files
    wget "https://github.com/riveter/riveter/releases/download/v${VERSION}/riveter-${VERSION}-${PLATFORM}.tar.gz"
    wget "https://github.com/riveter/riveter/releases/download/v${VERSION}/riveter-${VERSION}-${PLATFORM}.tar.gz.sha256"

    # Verify checksum
    sha256sum -c "riveter-${VERSION}-${PLATFORM}.tar.gz.sha256"

    # Extract and use
    tar -xzf "riveter-${VERSION}-${PLATFORM}.tar.gz"
    ./riveter-${VERSION}-${PLATFORM}/riveter --version
```

## Best Practices

1. **Always verify checksums** before using downloaded binaries
2. **Use the verification script** when available for consistency
3. **Store checksums securely** if you need to verify files later
4. **Automate verification** in CI/CD pipelines
5. **Report suspicious checksums** immediately to the maintainers
6. **Keep verification tools updated** to ensure compatibility

## Related Documentation

- [Installation Guide](../README.md#installation)
- [Security Setup](SECURITY_SETUP.md)
- [Release Process](RELEASE_WORKFLOW.md)
- [Homebrew Installation](HOMEBREW_TAP_SETUP.md)
