# Version Management Workflow

This document explains how version management works in the Riveter project to ensure consistency across all distribution methods.

## Overview

Riveter uses `pyproject.toml` as the **single source of truth** for version information. All other version references (CLI, binary, Homebrew formula) are synchronized from this source.

## Version Synchronization Flow

### 1. Development Phase

- **Pre-commit Hook**: Validates version consistency only when version-related files are modified
- **Feature Branches**: Version validation is advisory only (warnings, not failures)
- **Main Branch**: Strict version validation to catch any inconsistencies

### 2. Release Process

The release workflow follows this sequence:

1. **Validation**: Check current version state
2. **Version Bump**: Update version in `pyproject.toml` (e.g., 0.9.0 → 0.9.1)
3. **Synchronization**: Run `sync_versions.py --sync` to update:
   - CLI version import (`src/riveter/cli.py`)
   - Binary build configuration
   - Homebrew formula template
4. **Validation**: Confirm all versions are now consistent
5. **Commit & Tag**: Commit all synchronized files and create release tag
6. **Build & Publish**: Build binaries and publish to PyPI
7. **Formula Update**: Update Homebrew formula with real checksums

### 3. Post-Release

- **Homebrew Formula**: Automatically updated with correct checksums from release assets
- **Binary Validation**: Built binaries report the correct version from `pyproject.toml`

## Tools and Scripts

### `scripts/sync_versions.py`

Main version synchronization tool:

```bash
# Validate version consistency
python scripts/sync_versions.py --validate

# Synchronize all versions from pyproject.toml
python scripts/sync_versions.py --sync

# Update only Homebrew formula
python scripts/sync_versions.py --update-formula

# Update formula with checksums
python scripts/sync_versions.py --update-formula --checksums "macos-intel:abc123,linux-x86_64:def456"
```

### Pre-commit Hook

Install the pre-commit hook to validate version consistency:

```bash
python scripts/install_hooks.py
```

The hook is intelligent and only validates when:
- Version-related files are being modified
- Changes don't appear to be version bumps

## CI/CD Integration

### GitHub Workflows

1. **Test Workflow** (`.github/workflows/test.yml`):
   - Validates version consistency on main branch (strict)
   - Advisory validation on feature branches

2. **Release Workflow** (`.github/workflows/release.yml`):
   - Handles complete version bump and synchronization
   - Validates consistency at each step

3. **Homebrew Formula Update** (`.github/workflows/update-homebrew-formula.yml`):
   - Automatically updates formula after releases
   - Calculates and updates checksums

### Version Validation Workflow

Dedicated workflow for version consistency checks:
- Runs on all pushes and PRs
- Tests version synchronization script
- Validates CLI version import

## Manual Operations

### Making a Release

1. **Trigger Release Workflow**:
   ```bash
   # Via GitHub UI or CLI
   gh workflow run release.yml -f version_type=patch
   ```

2. **The workflow automatically**:
   - Bumps version in `pyproject.toml`
   - Synchronizes all version references
   - Creates git tag
   - Builds and publishes package
   - Updates Homebrew formula

### Fixing Version Inconsistencies

If versions get out of sync:

```bash
# Check current state
python scripts/sync_versions.py --validate

# Fix inconsistencies
python scripts/sync_versions.py --sync

# Commit the fixes
git add -A
git commit -m "Synchronize version references"
```

## Repository Connection

### Cross-Repository Synchronization

The Riveter project spans two GitHub repositories:

1. **Main Repository**: `ScottRyanHoward/riveter`
   - Contains the Python package source code
   - Holds the single source of truth for versions (`pyproject.toml`)
   - Contains the version synchronization scripts

2. **Homebrew Repository**: `ScottRyanHoward/homebrew-riveter`
   - Contains the Homebrew formula (`Formula/riveter.rb`)
   - Updated automatically via GitHub Actions from the main repo

### How They Connect

1. **Release Trigger**: When a release is published in the main repo
2. **Cross-Repo Checkout**: GitHub Actions checks out both repositories
3. **Version Sync**: The sync script updates the formula in the homebrew repo
4. **Pull Request**: Creates a PR in the homebrew repo with the updates

```yaml
# In .github/workflows/update-homebrew-formula.yml
- name: Checkout homebrew-riveter repository
  uses: actions/checkout@v4
  with:
    repository: ScottRyanHoward/homebrew-riveter
    path: homebrew-riveter
    token: ${{ secrets.GITHUB_TOKEN }}
```

## Version Sources

### Single Source of Truth
- **`pyproject.toml`**: `project.version` field (in main repo)

### Synchronized Targets
- **CLI**: `src/riveter/cli.py` - uses `importlib.metadata.version("riveter")`
- **Binary**: Built with correct version from package metadata
- **Homebrew Formula**: `homebrew-riveter/Formula/riveter.rb` (in separate repo)

## Troubleshooting

### Common Issues

1. **Pre-commit Hook Fails**:
   - Check if you're modifying version files without updating `pyproject.toml`
   - Run `python scripts/sync_versions.py --sync` to fix

2. **CI Version Validation Fails**:
   - Usually indicates version drift on main branch
   - Run sync script and commit the fixes

3. **Release Workflow Fails**:
   - Check version consistency before release
   - Ensure all required secrets are configured

### Debug Commands

```bash
# Check version consistency with debug output
python scripts/sync_versions.py --validate --debug

# Check current versions across components
python -c "import tomllib; print('pyproject:', tomllib.load(open('pyproject.toml', 'rb'))['project']['version'])"
riveter --version
```

## Best Practices

1. **Always use the release workflow** for version bumps
2. **Don't manually edit version numbers** in multiple places
3. **Run version sync after any manual version changes**
4. **Use the pre-commit hook** to catch issues early
5. **Test version consistency** before important releases

## Authentication & Permissions

### GitHub Token Requirements

The workflow is configured to work automatically with minimal setup:

1. **GITHUB_TOKEN** (automatic - recommended):
   - Works for same-user public repositories
   - No additional setup required
   - Used as fallback in the workflow

2. **HOMEBREW_UPDATE_TOKEN** (optional):
   - Only needed for private repositories or restricted permissions
   - Personal Access Token with `repo` and `workflow` scopes
   - Store as repository secret if needed

**Setup Check**: Run `python scripts/setup_cross_repo_access.py --check` to verify your configuration.

### Repository Access

The automation assumes:
- Both repositories are owned by the same GitHub user/organization
- The homebrew repository is public or accessible with the provided token
- The main repository has Actions enabled with appropriate permissions

## Security Considerations

- Version synchronization scripts validate file integrity
- Homebrew formula updates include checksum validation
- Release workflow includes comprehensive validation steps
- All version changes are tracked in git history
- Cross-repository updates use pull requests (not direct pushes) for safety
- Checksums are calculated from published release assets to ensure integrity
