from django.contrib.auth import get_user_model
from django.db.models import F, Value as V, Max
from django.core.serializers.json import DjangoJSONEncoder

from NearBeach.models import (
    ChangeTask,
    Customer,
    Group,
    KanbanBoard,
    KanbanCard,
    Organisation,
    Project,
    RequestForChange,
    Requirement,
    RequirementItem,
    Task,
    UserGroup,
)

import json

OBJECT_DICT = {
    "change_task": ChangeTask.objects,
    "customer": Customer.objects,
    "project": Project.objects,
    "task": Task.objects,
    "requirement": Requirement.objects,
    "requirement_item": RequirementItem.objects,
    "kanban_board": KanbanBoard.objects,
    "kanban_card": KanbanCard.objects,
    "organisation": Organisation.objects,
    "request_for_change": RequestForChange.objects,
    "user": get_user_model().objects,
}


# Internal function
def get_all_groups():
    group_results = Group.objects.filter(
        is_deleted=False,
    ).values(
        "group_id",
        "group_name",
        "parent_group_id",
    )

    return json.dumps(list(group_results), cls=DjangoJSONEncoder)


# Internal function
def get_object_from_destination(input_object, destination, location_id):
    """
    To stop the repeat code of finding specific objects using destination and location_id - we will import
    the object filter for it here - before returning it.
    :param object: The object we want to filter
    :param destination: The destination we are interested in
    :param location_id: The location_id
    :return:
    """
    input_object = input_object.filter(**{destination: location_id})

    # Just send back the array
    return input_object


# Internal Function
def get_user_group_permission(username, object_types):
    """
    Function will get a list of ALL current groups assigned to the user. It will then gather the permissions for each
    of the object types and groups.

    :param username: This is the username for the user. Please use request.user
    :param object_types: This is an array of types that you want to look up. Example. ["project"] or ["project", "task"]

    :return: returns json ready to be placed within the CONTEXT
    """
    results = UserGroup.objects.none()

    for single_type in object_types:
        results = results.union(UserGroup.objects.filter(
            is_deleted=False,
            username_id=username,
        ).annotate(
            object_type=V(single_type),
            group_name=F("group__group_name"),
        ).values(
            "group_id",
            "group_name",
            "object_type",
        ).annotate(
            object_permission_value=Max(F"permission_set__{single_type}")
        ))

    return json.dumps(list(results), cls=DjangoJSONEncoder)


# Internal Function
def get_user_permissions(field, value):
    return (
        UserGroup.objects.filter(
            is_deleted=False,
            **{field: value},
        )
        .values(
            "user_group_id",
            "username",
            "username__first_name",
            "username__last_name",
            "username__email",
            "group",
            "group__group_name",
            "group_leader",
            "permission_set",
            "permission_set__permission_set_name",
        )
        .order_by(
            "username__first_name",
            "username__last_name",
            "group__group_name",
            "permission_set__permission_set_name",
        )
    )


# Internal function
def lookup_choice_from_key(choices, key):
    for k, v in choices:
        if k == key:
            return v

    raise KeyError(f"Missing entry for {key}")


# Internal function
def set_object_from_destination(input_object, destination, location_id):
    """
    This function is used to set data against an object using the destination and location data.
    :param input_object: The input object that we are setting data for
    :param destination: The destination we are interested in
    :param location_id: The location we are interested in
    :return:
    """
    setattr(input_object, destination, OBJECT_DICT[destination].get(pk=location_id))

    return input_object


