#  Pyrogram - Telegram MTProto API Client Library for Python
#  Copyright (C) 2017-present Dan <https://github.com/delivrance>
#
#  This file is part of Pyrogram.
#
#  Pyrogram is free software: you can redistribute it and/or modify
#  it under the terms of the GNU Lesser General Public License as published
#  by the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  Pyrogram is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU Lesser General Public License for more details.
#
#  You should have received a copy of the GNU Lesser General Public License
#  along with Pyrogram.  If not, see <http://www.gnu.org/licenses/>.

from datetime import datetime
from typing import List

import pyrogram
from pyrogram import raw, utils
from pyrogram import types
from pyrogram.file_id import (
    FileId,
    FileType,
    FileUniqueId,
    FileUniqueType,
    ThumbnailSource,
)
from ..object import Object


class AlternativeVideo(Object):
    """Describes an alternative reencoded quality of a video file.

    Parameters:
        file_id (``str``):
            Identifier for this file, which can be used to download or reuse the file.

        file_unique_id (``str``):
            Unique identifier for this file, which is supposed to be the same over time and for different accounts.
            Can't be used to download or reuse the file.

        width (``int``):
            Video width as defined by sender.

        height (``int``):
            Video height as defined by sender.

        codec (``str``):
            Codec used for video file encoding, for example, "h264", "h265", or "av1".

        duration (``int``):
            Duration of the video in seconds as defined by sender.

        file_name (``str``, *optional*):
            Video file name.

        mime_type (``str``, *optional*):
            Mime type of a file as defined by sender.

        file_size (``int``, *optional*):
            File size.

        supports_streaming (``bool``, *optional*):
            True, if the video was uploaded with streaming support.

        date (:py:obj:`~datetime.datetime`, *optional*):
            Date the video was sent.

        thumbs (List of :obj:`~pyrogram.types.Thumbnail`, *optional*):
            Video thumbnails.

    """

    def __init__(
        self,
        *,
        client: "pyrogram.Client" = None,
        file_id: str,
        file_unique_id: str,
        width: int,
        height: int,
        codec: str,
        duration: int,
        file_name: str = None,
        mime_type: str = None,
        file_size: int = None,
        supports_streaming: bool = None,
        date: datetime = None,
        thumbs: List["types.Thumbnail"] = None,
    ):
        super().__init__(client)

        self.file_id = file_id
        self.file_unique_id = file_unique_id
        self.width = width
        self.height = height
        self.codec = codec
        self.duration = duration
        self.file_name = file_name
        self.mime_type = mime_type
        self.file_size = file_size
        self.supports_streaming = supports_streaming
        self.date = date
        self.thumbs = thumbs

    @staticmethod
    def _parse(
        client,
        video: "raw.types.Document",
        video_attributes: "raw.types.DocumentAttributeVideo",
        file_name: str,
    ) -> "AlternativeVideo":
        return AlternativeVideo(
            file_id=(
                FileId(
                    file_type=FileType.VIDEO,
                    dc_id=video.dc_id,
                    media_id=video.id,
                    access_hash=video.access_hash,
                    file_reference=video.file_reference,
                ).encode()
                if video
                else None
            ),
            file_unique_id=(
                FileUniqueId(
                    file_unique_type=FileUniqueType.DOCUMENT, media_id=video.id
                ).encode()
                if video
                else None
            ),
            width=video_attributes.w if video_attributes else None,
            height=video_attributes.h if video_attributes else None,
            codec=video_attributes.video_codec if video_attributes else None,
            duration=video_attributes.duration if video_attributes else None,
            file_name=file_name,
            mime_type=video.mime_type if video else None,
            supports_streaming=(
                video_attributes.supports_streaming if video_attributes else None
            ),
            file_size=video.size if video else None,
            date=utils.timestamp_to_datetime(video.date) if video else None,
            thumbs=types.Thumbnail._parse(client, video) if video else None,
            client=client,
        )
