import json

import requests
from .signer import Signer
from .exceptions import Bank131Error


class Bank131Client:
    def __init__(self, project_id: str, private_key_path: str,
                 base_url: str = "https://api.bank131.ru/api/v1"):
        self.project_id = project_id
        self.base_url = base_url
        self.signer = Signer(private_key_path)

    def _request(self, endpoint: str, payload: dict) -> dict:
        # Убираем None-поля, как в PHP array_filter
        clean_payload = {k: v for k, v in payload.items() if v is not None}

        # Формируем JSON идентичный PHP json_encode(JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)
        json_body = json.dumps(clean_payload, ensure_ascii=False, separators=(",", ":"))

        signature = self.signer.sign(clean_payload)

        headers = {
            "Content-Type": "application/json",
            "X-PARTNER-PROJECT": self.project_id,
            "X-PARTNER-SIGN": signature,
        }

        print("SIGNED JSON:", json_body)
        print("DEBUG SIGNATURE:", signature)

        response = requests.post(
            f"{self.base_url}/{endpoint}",
            headers=headers,
            data=json_body  # ✅ используем data, не json
        )

        print("URL:", f"{self.base_url}/{endpoint}")
        print("HEADERS:", headers)
        print("BODY:", json_body)
        print("RESPONSE:", response.text)

        if not response.ok:
            raise Bank131Error(f"Ошибка {response.status_code}: {response.text}")
        return response.json()

    def create_session(self, amount: int, currency: str, description: str) -> dict:
        return self._request("session/create", {
            "amount": amount,
            "currency": currency,
            "description": description
        })

    def init_payout(self, amount: int, currency: str, account: str, description: str) -> dict:
        return self._request("session/init/payout", {
            "amount": amount,
            "currency": currency,
            "account": account,
            "description": description
        })

    def get_payment_status(self, session_id: str) -> dict:
        return self._request("session/status", {"session": session_id})

    def create_payment_form(self, amount: int, currency: str, description: str, order_id: str, success_url: str, fail_url: str) -> dict:
        payload = {
            "amount": amount,
            "currency": currency,
            "description": description,
            "order_id": order_id,
            "success_url": success_url,
            "fail_url": fail_url
        }
        return self._request("payment/form/init", payload)

    def create_session_v2(self, customer_ref: str, email: str, amount: int, currency: str, metadata: str) -> dict:
        payload = {
            "customer": {
                "reference": customer_ref,
                "contacts": [{"email": email}]
            },
            "amount_details": {
                "amount": amount,
                "currency": currency
            },
            "metadata": metadata
        }
        return self._request("session/create", payload)

    def get_token(self, session_id: str, show_recurrent_checkbox: bool = False) -> dict:
        payload = {
            "acquiring_widget": {"session_id": session_id}
        }

        if show_recurrent_checkbox:
            payload["show_recurrent_checkbox"] = True

        return self._request("token", payload)

    def confirm_session(self, session_id: str) -> dict:
        payload = {"session_id": session_id}
        return self._request("session/confirm", payload)