import logging
from typing import Any, Dict, Annotated
import click
import mcp.types as types
from mcp.server.fastmcp import FastMCP
from mcp.server.fastmcp.exceptions import ToolError
from pydantic import BaseModel, Field

logger = logging.getLogger(__name__)

app = FastMCP(
    "monocle-mcp-2",
)
code_snippets = {
    "python": "from monocle_apptrace import setup_monocle_telemetry"
    "setup_monocle_telemetry(workflow_name='my_workflow')\n",
    "typescript": "import { trace } from 'monocle';\n\ntrace();\n",
}
setup_instructions = {
    "python": "pip install monocle_apptrace package and add monocle_apptrace at the end of the requirement.txt for future. This code snippet provided below should be just below other imports in your application code.",
    "typescript": "Install monocle package and add the trace function call at the top of your application. This code snippet provided below should be just below other imports in your application code.",
}

description = {
    "python": "This will automatically trace your Python application, capturing telemetry data for your genAI operations.",
    "typescript": "This will automatically trace your TypeScript application, capturing telemetry data for your genAI operations.",
}


# Use the existing dictionaries to get language-specific information
# code_snippet = code_snippets.get(app_language, code_snippets['python'])
# instructions = setup_instructions.get(app_language, setup_instructions['python'])
# desc = description.get(app_language, description['python'])
enable_monocle_prompt = """Do following changes to the project:
1. run command 'pip install monocle_apptrace'
2. Add following line to  requirements.txt in current directory 'monocle_apptrace' if it's not there already
3. Add following lines just below other imports in the file if it's not there already:
from monocle_apptrace import setup_monocle_telemetry
setup_monocle_telemetry(workflow_name=<module-name>, monomonocle_exporters_list='file'))
"""

monocle_trace_prompt: str = """
You are a helpful assistant to analyze given Monocle trace data. Review the trace data and provide a summary of whether the execution was successful or if there were any errors. Your job is to analyze the trace data and provide a summary. Be very concise in reporting. Check for errors or problem situations. First provide a headline that indicates status of the trace and if there are problems. If there are problems, then provide a brief summary of the problems. If there are problems, then provide a brief summary of the root causes.
Use the trace details provided below to understand the data.
 
Monocle trace has recorded the traced the execution of a genAI application. Each step of the genAI operation is captured as span. The trace file is a JSON structure that is an array of span JSON elements. There's a root span that represents the overall execution context.
The important span JSON elements are as follows:
- context.trace_id is the unique id common for alls spans.
- context.span_id is the unique id for each span.
- parent_id indicates the parent span of that span. The span that has no parent is the root span.
- status.status_code indicate the status of span. It can be OK or ERROR or UNSET. If the span status is ERROR, then it generally propogated to it's parent and all the way to the root span. This means the entire trace is in error. However, sometimes the parent span will handle the error (eg. if parent name indicagtes 'retry') and the trace will be successful, even though an individual span has error.
- attributes captures various attributes involved in that span's operation
    - attribute."span.type" indicates the type of operation
        - When the span.type is "workflow", that span captures the summary of the entire application
    - entity.[NUMBER] are the different entties involved in the operation, for example large language model, agent, tool etc
    - entity.type indicates the type of that entity
- events is an array that captures the details of the operation such as input, output and metadata
    - events array element with name=data.input captures the input provided to the operation
    - events array element with name=data.output captures the output generated by the operation
    - events array element with name=metaata captures the metadata of the operation
        - total_tokens is the count of tokens
        - completion_tokens is the count of ouput tokens
        - prompt_tokens is the count of input tokens
        - finish_type indicates how the operation finished, for example stop sequence or max tokens etc. If the finish_type is not success and indicates problems such as max tokens, then the operation is not successful. Check the status of it's parent span and check if it's due to the finish type of this child span. In that case callout the root cause as the child span error for the parent span error.
"""

prompt_map = {
    "enable_monocle_tracing": enable_monocle_prompt,
    "analyze_monocle_tracing": monocle_trace_prompt,
}


@app.tool(
    name="add_monocle",
    description="This tool provides step by step guidlines to verify and add Monocle tracing to your genAI application. "
    "This should be the first tool to use for adding Monocle tracing. "
    "Note that you don't need to do anything specific to Open Telemetry or Azure Monitor setup when adding Monocle.",
)
async def add_monocle(
    app_language: Annotated[
        str,
        Field(
            description="The programming language of your application (e.g., 'python', 'javascript', 'java')"
        ),
    ],
) -> Dict[str, Any]:
    """
    Args:
        app_language: The programming language of the application (e.g., 'Python', 'TypeScript')
    """
    if not app_language:
        raise ToolError("App language cannot be empty")

    code_snippet = code_snippets.get(app_language.lower())
    if not code_snippet:
        raise ToolError("App language {} not supported".format(app_language))

    return {
        "Step1 - Check if Monocle is enabled": "Use check_monocle tool to verify if monocle tracing is already enabled in your application.",
        "Step2 - Get steps to enable Monocle tracing": "Use enable_monocle_tracing tool to get the code snippet and setup instructions for enabling Monocle tracing.",
        #        'Step3 - Check if there are Azure Function route methods': 'Use find_Azure_function_route tool to identify Azure Function route methods in your code.',
        #        'Step4 - Add Monocle annotation to Azure Function route methods': 'Use add_monocle_annotation_to_azure_function tool to add Monocle annotations to your Azure Function route methods.'
    }


@app.tool(
    name="check_monocle",
    description="This tool provides instruction for checking if the application is already instrumented with Monocle tracing.",
)
async def check_monocle(
    app_language: Annotated[
        str,
        Field(
            description="The programming language of your application (e.g., 'python', 'javascript', 'java')"
        ),
    ],
) -> Dict[str, Any]:
    """
    Args:
        app_language: The programming language of the application (e.g., 'Python', 'TypeScript')
    """
    if not app_language:
        raise ToolError("App language cannot be empty")

    if app_language.lower() == "python":
        return {
            "instruction": 'check if the code contains "from monocle_apptrace import setup_monocle_telemetry"'
        }
    elif app_language.lower() == "typescript":
        return {
            "instruction": "check if the code contains \"import { trace } from 'monocle'\""
        }
    else:
        raise ToolError("Unsupported app language: {}".format(app_language))


@app.tool(
    name="enable_monocle_tracing",
    description="This tool provides setup instructions to install monocle tracing library and provides a code snippet to be included in the code.",
)
async def enable_monocle_tracing(
    app_language: Annotated[
        str,
        Field(
            description="The programming language of your application (e.g., 'python', 'javascript', 'java')"
        ),
    ],
) -> Dict[str, Any]:
    """
    Args:
        app_language: The programming language of the application (e.g., 'Python', 'TypeScript')
    """
    if not app_language:
        raise ToolError("App language cannot be empty")

    # Generate code snippets
    code_snippet = code_snippets.get(app_language.lower())
    if not code_snippet:
        raise ToolError("App language {} not supported".format(app_language))

    return {
        "Step 1 - setup_instructions": setup_instructions.get(app_language.lower()),
        "Step 2 - code_snippet": code_snippet,
    }


@app.tool(
    name="find_azure_function_route",
    description="This tool helps identify Azure Function route methods in your application code.",
)
async def find_azure_function_route(
    app_language: Annotated[
        str,
        Field(
            description="The programming language of your application (e.g., 'python', 'javascript', 'java')"
        ),
    ],
) -> Dict[str, Any]:
    """
    Args:
        app_language: The programming language of the application (e.g., 'Python', 'TypeScript')
    """
    if not app_language:
        raise ToolError("App language cannot be empty")
    return {
        "search_azure_function_route": "If the file contains 'import azure.functions' and a function is annotated with @app.route, it is likely an Azure Function route method."
    }


@app.tool(
    name="add_monocle_annotation_to_azure_function",
    description="This tool provides code snippets and setup instructions for integrating Monocle annotations "
    "to Azure Function route methods in your code. This requires monocle tracing already enabled in your code. "
    "You should first check if the application is already instrumented with Monocle tracing using check_monocle tool. "
    "If it's not instrumented, you should first add Monocle tracing to your application, using the add_monocle tool",
)
async def add_monocle_annotation_to_azure_function(
    app_language: Annotated[
        str,
        Field(
            description="The programming language of your application (e.g., 'python', 'javascript', 'java')"
        ),
    ],
) -> Dict[str, Any]:
    """
    Args:
        app_language: The programming language of the application (e.g., 'Python', 'TypeScript')
    """
    if not app_language:
        raise ToolError("App language cannot be empty")

    if app_language.lower() == "python":
        return {
            "recommendation": "The annotation should be placed immediately in the line above the function definition. The declaration snippet should be added right after import monocle_apptrace line",
            "setup_instructions": "Ensure you have the monocle_apptrace package installed and imported in your Azure Function file.",
            "declaration_snippet": "from monocle_apptrace import monocle_trace_azure_function_route",
            "code_snippet": "@monocle_apptrace.trace_azure_function",
        }
    else:
        raise ToolError(
            "Monocle annotation for Azure Functions is only supported in Python at this time.",
        )


@app.prompt()
def enable_monocle_tracing_prompt(name: str, app_language: str = "python") -> str:
    """Generate a prompt for enabling monocle tracing"""
    prompt = prompt_map.get(name, enable_monocle_prompt)
    return prompt.format(app_language=app_language)


@app.prompt()
def analyze_monocle_tracing_prompt(name: str) -> str:
    """Generate a prompt for analyzing monocle tracing"""
    return prompt_map.get(name, monocle_trace_prompt)


@click.command()
@click.option(
    "--log-level",
    default="info",
    help="Logging level (DEBUG, INFO, WARNING, ERROR, CRITICAL)",
)
def main(log_level: str) -> int:
    # Configure logging
    logging.basicConfig(
        level=getattr(logging, log_level.upper()),
        format="%(asctime)s - %(name)s - %(levelname)s - %(message)s",
    )

    # Run the FastMCP server using stdio transport
    app.run(transport="stdio")
    return 0


if __name__ == "__main__":
    main()  # type: ignore
