# Contributing to WCG

Thank you for your interest in contributing to WCG! This document provides guidelines and instructions for contributing.

## Getting Started

### Prerequisites

- Python 3.12 or later
- Git
- `uv` package manager (recommended) or `pip`

### Development Setup

1. Fork and clone the repository:

```bash
git clone https://github.com/YOUR_USERNAME/WCG.git
cd WCG
```

2. Create a virtual environment and install dependencies:

```bash
# Using uv (recommended)
uv venv
source .venv/bin/activate  # On Windows: .venv\Scripts\activate
uv pip install -e ".[dev]"

# Or using pip
python -m venv .venv
source .venv/bin/activate
pip install -e ".[dev]"
```

3. Verify the installation:

```bash
pytest tests/
```

## Development Workflow

### Making Changes

1. Create a new branch for your feature or fix:

```bash
git checkout -b feature/your-feature-name
# or
git checkout -b fix/your-bug-fix
```

2. Make your changes following the code style guidelines

3. Add or update tests as needed

4. Run tests and linting:

```bash
# Run tests
pytest tests/ -v

# Check code style
ruff check src/

# Format code
ruff format src/
```

5. Commit your changes with a clear message:

```bash
git commit -m "Add feature: description of your changes"
```

6. Push to your fork and create a pull request

### Code Style

- Follow PEP 8 guidelines
- Use meaningful variable and function names
- Add docstrings to all public functions and classes
- Keep functions focused and concise
- Use type hints where appropriate

Example:

```python
def process_commits(commits: List[CommitInfo], repo_name: str) -> str:
    """Process and summarize commits.
    
    Args:
        commits: List of commit information
        repo_name: Name of the repository
        
    Returns:
        Markdown formatted summary
    """
    # Implementation here
    pass
```

### Testing

- Write tests for all new features
- Ensure all tests pass before submitting a PR
- Aim for high test coverage
- Use descriptive test names

Example test structure:

```python
class TestFeature:
    """Test suite for Feature."""
    
    def test_basic_functionality(self):
        """Test basic functionality works as expected."""
        # Arrange
        # Act
        # Assert
        pass
    
    def test_edge_case(self):
        """Test edge case handling."""
        pass
```

### Documentation

- Update README.md if adding new features
- Update USAGE.md with usage examples
- Add docstrings to all new functions and classes
- Update DEPLOY.md if changing deployment procedures

## Pull Request Process

1. **Update Documentation**: Ensure all relevant documentation is updated

2. **Add Tests**: Include tests for new functionality

3. **Run Checks**: Ensure all tests and linters pass:
   ```bash
   pytest tests/ -v
   ruff check src/
   uv build
   ```

4. **Write Clear PR Description**: Explain:
   - What changes were made
   - Why these changes were needed
   - How to test the changes
   - Any breaking changes

5. **Link Issues**: Reference related issues (e.g., "Fixes #123")

6. **Wait for Review**: Maintainers will review your PR and may request changes

## Reporting Bugs

When reporting bugs, please include:

- **Description**: Clear description of the bug
- **Steps to Reproduce**: Detailed steps to reproduce the issue
- **Expected Behavior**: What you expected to happen
- **Actual Behavior**: What actually happened
- **Environment**: OS, Python version, WCG version
- **Logs**: Relevant error messages or logs

Use the issue template when available.

## Suggesting Features

When suggesting features:

- Check if the feature has already been requested
- Explain the use case and benefit
- Provide examples of how it would work
- Consider implementation complexity

## Code Review

All submissions require review. We use GitHub pull requests for this purpose.

Review criteria:
- Code quality and style
- Test coverage
- Documentation
- Performance impact
- Security implications

## Release Process

1. Update version in `pyproject.toml`
2. Update CHANGELOG.md
3. Create a release tag
4. Build and publish to PyPI:
   ```bash
   uv build
   twine upload dist/*
   ```

## Community

- Be respectful and inclusive
- Help others when possible
- Follow the code of conduct
- Ask questions in issues or discussions

## License

By contributing, you agree that your contributions will be licensed under the MIT License.

## Questions?

Feel free to:
- Open an issue for bugs or features
- Start a discussion for questions
- Contact maintainers directly

Thank you for contributing to WCG! 🎉
