# 部署指南 / Deployment Guide

本文档介绍如何部署和发布 WCG 到 PyPI。

## 前置要求

1. 安装 `uv`：
   ```bash
   pip install uv
   ```

2. 获取 PyPI API Token：
   - 登录 https://pypi.org
   - 进入 Account Settings -> API tokens
   - 创建新 token

## 构建包

使用 uv 构建发行包：

```bash
# 确保在项目根目录
cd WCG

# 构建包
uv build

# 这将在 dist/ 目录下生成：
# - wcg-0.1.0.tar.gz (源码包)
# - wcg-0.1.0-py3-none-any.whl (wheel包)
```

## 发布到 PyPI

### 使用 twine (推荐)

1. 安装 twine：
   ```bash
   pip install twine
   ```

2. 上传到 TestPyPI（测试）：
   ```bash
   twine upload --repository testpypi dist/*
   ```

3. 上传到 PyPI（正式发布）：
   ```bash
   twine upload dist/*
   ```

### 使用 uv publish

```bash
uv publish --token YOUR_PYPI_TOKEN
```

## 版本管理

更新版本号需要修改 `pyproject.toml`：

```toml
[project]
name = "wcg"
version = "0.1.1"  # 更新这里
...
```

遵循语义化版本：
- **补丁版本** (0.1.1): 修复 bug
- **次版本** (0.2.0): 添加新功能（向后兼容）
- **主版本** (1.0.0): 重大变更（可能不兼容旧版本）

## 部署为服务

### 使用 systemd (Linux)

创建服务文件 `/etc/systemd/system/wcg.service`：

```ini
[Unit]
Description=WCG - Watch Commits for Git
After=network.target

[Service]
Type=simple
User=your-user
WorkingDirectory=/path/to/wcg
ExecStart=/path/to/.venv/bin/wcg start --config /path/to/config.yaml
Restart=always
RestartSec=10

[Install]
WantedBy=multi-user.target
```

启动服务：

```bash
sudo systemctl daemon-reload
sudo systemctl enable wcg
sudo systemctl start wcg
sudo systemctl status wcg
```

### 使用 Docker

创建 `Dockerfile`：

```dockerfile
FROM python:3.12-slim

WORKDIR /app

# 安装 uv
RUN pip install uv

# 复制项目文件
COPY . .

# 安装依赖
RUN uv pip install --system .

# 创建配置目录
RUN mkdir -p /app/config /app/summaries

# 暴露 Web 端口
EXPOSE 5000

# 默认命令
CMD ["wcg", "start", "--config", "/app/config/config.yaml"]
```

构建和运行：

```bash
# 构建镜像
docker build -t wcg:latest .

# 运行容器
docker run -d \
  --name wcg \
  -v $(pwd)/config:/app/config \
  -v $(pwd)/summaries:/app/summaries \
  -p 5000:5000 \
  wcg:latest
```

### 使用 Docker Compose

创建 `docker-compose.yml`：

```yaml
version: '3.8'

services:
  wcg:
    build: .
    container_name: wcg
    restart: unless-stopped
    volumes:
      - ./config:/app/config
      - ./summaries:/app/summaries
    ports:
      - "5000:5000"
    environment:
      - LOG_LEVEL=INFO
```

运行：

```bash
docker-compose up -d
```

## Web 界面部署

如果只需要运行 Web 配置界面：

### 使用 Gunicorn (生产环境)

```bash
pip install gunicorn

gunicorn -w 4 -b 0.0.0.0:5000 'wcg.web:app'
```

### 使用 Nginx 反向代理

Nginx 配置示例：

```nginx
server {
    listen 80;
    server_name wcg.example.com;

    location / {
        proxy_pass http://127.0.0.1:5000;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
    }
}
```

## 环境变量

支持通过环境变量覆盖配置：

```bash
export WCG_CONFIG_PATH=/path/to/config.yaml
export WCG_LOG_LEVEL=DEBUG
export WCG_GITHUB_TOKEN=your-token
export WCG_LLM_API_KEY=your-api-key
```

## 监控和日志

### 查看日志

使用 systemd：
```bash
journalctl -u wcg -f
```

使用 Docker：
```bash
docker logs -f wcg
```

### 健康检查

可以定期检查 Web 接口：
```bash
curl http://localhost:5000/api/config
```

## 故障排查

### 常见问题

1. **GitHub API 限制**
   - 确保提供了有效的 GitHub token
   - 检查 token 权限（需要 `repo` 权限）

2. **LLM API 错误**
   - 验证 API URL 和 Key 是否正确
   - 检查网络连接和防火墙设置

3. **Webhook 发送失败**
   - 验证 webhook URL 是否可访问
   - 检查 webhook 服务器的日志

4. **定时任务不执行**
   - 检查系统时区设置
   - 验证配置中的时间格式（HH:MM）

## 安全建议

1. **保护敏感信息**
   - 不要将 `config/config.yaml` 提交到版本控制
   - 使用环境变量或密钥管理服务存储 token 和 API keys

2. **网络安全**
   - 在生产环境使用 HTTPS
   - 限制 Web 界面的访问（使用认证）

3. **权限控制**
   - 使用专用用户运行服务
   - 限制配置文件的读写权限

## 更新和维护

### 更新到新版本

```bash
# 停止服务
sudo systemctl stop wcg  # 或 docker-compose down

# 更新包
pip install --upgrade wcg

# 重启服务
sudo systemctl start wcg  # 或 docker-compose up -d
```

### 备份

定期备份重要数据：
```bash
# 备份配置
cp config/config.yaml config/config.yaml.bak

# 备份汇总历史
tar -czf summaries-backup-$(date +%Y%m%d).tar.gz summaries/
```
