# Glovebox CLI Architecture

This document describes the architecture, patterns, and conventions used in the Glovebox CLI.

## Directory Structure

The CLI module is organized as follows:

```
glovebox/cli/
   __init__.py        # Exports app and main function
   app.py             # Main CLI application and entry point
   README.md          # This documentation
   commands/          # CLI command implementations
      __init__.py    # Command registration
      config.py      # Configuration commands
      firmware.py    # Firmware commands
      keymap.py      # Keymap commands
      status.py      # Status command
   decorators/        # Reusable command decorators
      __init__.py    # Exports decorators
      error_handling.py  # Error handling decorator
      profile.py     # Profile handling decorator
   helpers/           # Helper functions
       __init__.py    # Exports helpers
       output.py      # Output formatting helpers
       profile.py     # Profile creation helpers
```

## Key Design Patterns

### 1. Command Registration

Each command group is defined in its own module, and each module registers its commands with the main app:

```python
# In commands/keymap.py
def register_commands(app: typer.Typer) -> None:
    app.add_typer(keymap_app, name="keymap")

# In commands/__init__.py
def register_all_commands(app: typer.Typer) -> None:
    register_keymap_commands(app)
    register_firmware_commands(app)
    register_config_commands(app)
    register_status_commands(app)
```

### 2. Decorator Pattern

Decorators are used to handle common functionality across commands:

#### Error Handling

The `@handle_errors` decorator catches and formats common exceptions:

```python
@handle_errors
def my_command():
    # Command implementation
```

#### Profile Handling

The `@with_profile` decorator simplifies working with keyboard profiles:

```python
@with_profile(default_profile="glove80/default")
def my_command(
    # ... other parameters ...
    profile: str = typer.Option("--profile", "-p"),
    keyboard_profile: KeyboardProfile | None = None,  # Added by decorator
):
    # Command implementation using keyboard_profile
```

### 3. Output Formatting

Consistent output formatting using helper functions:

```python
from glovebox.cli.helpers import (
    print_success_message, 
    print_error_message,
    print_list_item,
    print_result,
)

# Usage
print_success_message("Operation successful")
print_error_message("Operation failed")
print_list_item("Item in a list")
print_result(result_object)
```

## Adding New Commands

To add a new command:

1. Create a function with the command implementation:

```python
@keymap_app.command()
@handle_errors
@with_profile()  # If the command needs a profile
def my_new_command(
    arg1: Annotated[str, typer.Argument(help="Help text")],
    opt1: Annotated[bool, typer.Option("--opt1", help="Help text")] = False,
    profile: Annotated[str, typer.Option("--profile", "-p")] = "default",
    keyboard_profile: KeyboardProfile | None = None,  # If using @with_profile
):
    """Command description for help text."""
    # Command implementation
```

2. If creating a new command group, follow this pattern:

```python
# Create the command group
my_app = typer.Typer(
    name="my_group",
    help="My command group description",
    no_args_is_help=True,
)

# Add commands to the group
@my_app.command()
def my_command():
    # Implementation

# Register with the main app
def register_commands(app: typer.Typer) -> None:
    app.add_typer(my_app, name="my_group")
```

3. Update `commands/__init__.py` to include your new registration function.

## Best Practices

1. **Keep Commands Focused**: Each command should do one thing well.

2. **Consistent Error Handling**: Always use the `@handle_errors` decorator.

3. **Profile Pattern**: Use the `@with_profile` decorator for commands that need a keyboard profile.

4. **Descriptive Help Text**: Provide clear help text for commands and parameters.

5. **Consistent Output Format**: Use the helper functions for output formatting.

6. **Type Annotations**: Always use type annotations with `Annotated[]` for parameters.

7. **Validation**: Validate inputs at the beginning of commands.

8. **Exit Codes**: Use `typer.Exit(1)` for error exits after displaying error messages.

## Command-Line Interface Structure

The general structure of the Glovebox CLI is:

```
glovebox [command] [subcommand] [--options] arguments
```

For example:
- `glovebox keymap compile input.json output/ --profile glove80/v25.05`
- `glovebox firmware flash firmware.uf2 --profile glove80/v25.05`
- `glovebox config list --verbose`
- `glovebox status`

Global options are available to all commands:
- `--verbose` or `-v`: Increase verbosity
- `--log-file FILE`: Log to file
- `--version`: Show version and exit
- `--help` or `-h`: Show help message