"""Tests for firmware CLI command execution."""

import json
import time
from unittest.mock import Mock, patch

import pytest

from glovebox.cli import app
from glovebox.cli.commands import register_all_commands


# Register commands with the app before running tests
register_all_commands(app)


@pytest.mark.skip(reason="Test requires deep mocking of firmware flash commands")
def test_firmware_flash_command(cli_runner, create_keyboard_profile_fixture, tmp_path):
    """Test firmware flash command.

    This test has been skipped because it requires extensive mocking of the firmware flash command,
    which is already tested in the integration tests.
    """
    # This test is now skipped to avoid fixture conflicts
    pass


def test_firmware_devices_command(cli_runner):
    """Test firmware devices command which is easier to mock."""
    # Register commands
    from glovebox.cli.commands import register_all_commands

    register_all_commands(app)

    with (
        patch("glovebox.firmware.flash.create_flash_service") as mock_create_service,
        patch(
            "glovebox.cli.helpers.profile.get_keyboard_profile_from_context"
        ) as mock_get_profile,
        patch(
            "glovebox.cli.helpers.profile.create_profile_from_context"
        ) as mock_create_profile_context,
    ):
        # Mock the keyboard profile creation and context access
        mock_profile = Mock()
        mock_create_profile_context.return_value = mock_profile
        mock_get_profile.return_value = mock_profile

        # Create a simple mock flash service
        mock_flash_service = Mock()
        mock_create_service.return_value = mock_flash_service

        # Set up result with some devices
        from glovebox.firmware.flash.models import FlashResult

        result = FlashResult(success=True)
        result.device_details = [
            {
                "name": "Device 1",
                "status": "success",
                "serial": "GLV80-1234",
                "path": "/dev/sdX",
                "vendor_id": "239a",
                "product_id": "0029",
            },
            {
                "name": "Device 2",
                "status": "success",
                "serial": "GLV80-5678",
                "path": "/dev/sdY",
                "vendor_id": "1b1c",
                "product_id": "1b2f",
            },
        ]
        # Updated to use the correct method name
        mock_flash_service.list_devices.return_value = result

        # Run the command with profile
        cmd_result = cli_runner.invoke(
            app,
            ["firmware", "devices", "--profile", "glove80/v25.05"],
            catch_exceptions=False,
        )

        # Verify results
        assert cmd_result.exit_code == 0
        assert "Device 1" in cmd_result.output
        assert "Device 2" in cmd_result.output
        assert "GLV80-1234" in cmd_result.output
        # Verify vendor_id and product_id are displayed
        assert "VID: 239a" in cmd_result.output
        assert "PID: 0029" in cmd_result.output
        assert "VID: 1b1c" in cmd_result.output
        assert "PID: 1b2f" in cmd_result.output


def test_firmware_devices_json_output(cli_runner):
    """Test firmware devices command with JSON output format."""
    # Register commands
    from glovebox.cli.commands import register_all_commands

    register_all_commands(app)

    with (
        patch("glovebox.firmware.flash.create_flash_service") as mock_create_service,
        patch(
            "glovebox.cli.helpers.profile.get_keyboard_profile_from_context"
        ) as mock_get_profile,
        patch(
            "glovebox.cli.helpers.profile.create_profile_from_context"
        ) as mock_create_profile_context,
    ):
        # Mock the keyboard profile creation and context access
        mock_profile = Mock()
        mock_create_profile_context.return_value = mock_profile
        mock_get_profile.return_value = mock_profile

        # Create a simple mock flash service
        mock_flash_service = Mock()
        mock_create_service.return_value = mock_flash_service

        # Set up result with some devices including vendor_id and product_id
        from glovebox.firmware.flash.models import FlashResult

        result = FlashResult(success=True)
        result.device_details = [
            {
                "name": "Device 1",
                "status": "success",
                "serial": "GLV80-1234",
                "path": "/dev/sdX",
                "vendor_id": "239a",
                "product_id": "0029",
            },
        ]
        mock_flash_service.list_devices.return_value = result

        # Run the command with JSON output format
        cmd_result = cli_runner.invoke(
            app,
            [
                "firmware",
                "devices",
                "--profile",
                "glove80/v25.05",
                "--output-format",
                "json",
            ],
            catch_exceptions=False,
        )

        # Verify results
        assert cmd_result.exit_code == 0

        # Parse JSON output
        output_data = json.loads(cmd_result.output)
        assert output_data["success"] is True
        assert output_data["device_count"] == 1
        assert len(output_data["devices"]) == 1

        # Verify all fields are present in device details
        device = output_data["devices"][0]
        assert device["name"] == "Device 1"
        assert device["serial"] == "GLV80-1234"
        assert device["path"] == "/dev/sdX"
        assert device["vendor_id"] == "239a"
        assert device["product_id"] == "0029"


def test_firmware_devices_wait_mode(cli_runner):
    """Test firmware devices command with --wait flag."""
    # Register commands
    from glovebox.cli.commands import register_all_commands

    register_all_commands(app)

    with (
        patch("glovebox.firmware.flash.create_flash_service") as mock_create_service,
        patch(
            "glovebox.cli.helpers.profile.get_keyboard_profile_from_context"
        ) as mock_get_profile,
        patch(
            "glovebox.cli.helpers.profile.create_profile_from_context"
        ) as mock_create_profile_context,
        patch("signal.signal") as mock_signal,
        patch("time.sleep") as mock_sleep,
    ):
        # Mock the keyboard profile creation and context access
        mock_profile = Mock()
        mock_create_profile_context.return_value = mock_profile
        mock_get_profile.return_value = mock_profile

        # Create a simple mock flash service
        mock_flash_service = Mock()
        mock_create_service.return_value = mock_flash_service

        # Set up initial result with one device
        from glovebox.firmware.flash.models import FlashResult

        initial_result = FlashResult(success=True)
        initial_result.device_details = [
            {
                "name": "Device 1",
                "status": "success",
                "serial": "GLV80-1234",
                "path": "/dev/sdX",
                "vendor_id": "239a",
                "product_id": "0029",
            }
        ]

        # Simulate device addition on second poll
        updated_result = FlashResult(success=True)
        updated_result.device_details = [
            {
                "name": "Device 1",
                "status": "success",
                "serial": "GLV80-1234",
                "path": "/dev/sdX",
                "vendor_id": "239a",
                "product_id": "0029",
            },
            {
                "name": "Device 2",
                "status": "success",
                "serial": "GLV80-5678",
                "path": "/dev/sdY",
                "vendor_id": "1b1c",
                "product_id": "1b2f",
            },
        ]

        # Mock list_devices to return different results
        call_count = 0

        def mock_list_devices(*args, **kwargs):
            nonlocal call_count
            call_count += 1
            if call_count == 1:
                return initial_result
            elif call_count == 2:
                return updated_result
            else:
                # Simulate Ctrl+C after showing the new device
                raise KeyboardInterrupt()

        mock_flash_service.list_devices.side_effect = mock_list_devices

        # Mock sleep to allow quick test execution
        sleep_count = 0

        def mock_sleep_func(duration):
            nonlocal sleep_count
            sleep_count += 1
            if sleep_count > 3:
                # After a few loops, stop the monitoring
                raise KeyboardInterrupt()
            time.sleep(0.01)  # Very short sleep for testing

        mock_sleep.side_effect = mock_sleep_func

        # Run the command with --wait flag
        cmd_result = cli_runner.invoke(
            app,
            ["firmware", "devices", "--wait", "--profile", "glove80/v25.05"],
            catch_exceptions=True,
        )

        # Verify results
        assert cmd_result.exit_code == 0
        assert "Starting continuous device monitoring" in cmd_result.output
        assert "Currently connected devices: 1" in cmd_result.output
        assert "Device 1" in cmd_result.output
        assert "Monitoring for device changes (real-time)..." in cmd_result.output

        # Verify signal handler was registered
        mock_signal.assert_called_once()
        # Verify sleep was called
        assert mock_sleep.called


def test_flash_command_wait_parameters(cli_runner):
    """Test flash command with wait parameters."""
    # Register commands
    from glovebox.cli.commands import register_all_commands

    register_all_commands(app)

    # Test that wait parameters can be parsed without errors
    cmd_result = cli_runner.invoke(
        app,
        [
            "firmware",
            "flash",
            "test.uf2",
            "--wait",
            "--poll-interval",
            "1.0",
            "--show-progress",
            "--profile",
            "glove80/v25.05",
            "--help",  # Use help to avoid actual execution
        ],
        catch_exceptions=False,
    )

    # Should show help text without parameter parsing errors
    assert cmd_result.exit_code == 0


def test_flash_command_help_includes_wait_options(cli_runner):
    """Test that help includes wait-related options."""
    # Register commands
    from glovebox.cli.commands import register_all_commands

    register_all_commands(app)

    cmd_result = cli_runner.invoke(app, ["firmware", "flash", "--help"])

    assert "--wait" in cmd_result.output
    assert "--poll-interval" in cmd_result.output
    assert "--show-progress" in cmd_result.output
    assert "config" in cmd_result.output.lower()  # Mentions configuration


def test_flash_command_wait_parameter_validation(cli_runner):
    """Test wait parameter validation."""
    # Register commands
    from glovebox.cli.commands import register_all_commands

    register_all_commands(app)

    # Test invalid poll-interval (too small)
    cmd_result = cli_runner.invoke(
        app,
        [
            "firmware",
            "flash",
            "test.uf2",
            "--poll-interval",
            "0.05",  # Below minimum of 0.1
            "--profile",
            "glove80/v25.05",
        ],
    )

    # Should fail with validation error
    assert cmd_result.exit_code != 0


def test_flash_command_wait_boolean_flags(cli_runner):
    """Test wait boolean flag variations."""
    # Register commands
    from glovebox.cli.commands import register_all_commands

    register_all_commands(app)

    # Test --wait flag
    cmd_result = cli_runner.invoke(
        app, ["firmware", "flash", "test.uf2", "--wait", "--help"]
    )
    assert cmd_result.exit_code == 0

    # Test --no-wait flag
    cmd_result = cli_runner.invoke(
        app, ["firmware", "flash", "test.uf2", "--no-wait", "--help"]
    )
    assert cmd_result.exit_code == 0

    # Test --show-progress flag
    cmd_result = cli_runner.invoke(
        app, ["firmware", "flash", "test.uf2", "--show-progress", "--help"]
    )
    assert cmd_result.exit_code == 0

    # Test --no-show-progress flag
    cmd_result = cli_runner.invoke(
        app, ["firmware", "flash", "test.uf2", "--no-show-progress", "--help"]
    )
    assert cmd_result.exit_code == 0


def test_firmware_compile_auto_profile_detection(cli_runner, tmp_path):
    """Test firmware compile command with auto-profile detection from JSON."""
    from glovebox.cli.commands import register_all_commands

    register_all_commands(app)

    # Create a test JSON file with keyboard field
    test_json = {"keyboard": "corne", "title": "Test Layout", "layers": []}
    json_file = tmp_path / "test_layout.json"
    json_file.write_text(json.dumps(test_json))

    with (
        patch(
            "glovebox.cli.helpers.auto_profile.resolve_profile_with_auto_detection"
        ) as mock_auto_profile,
        patch(
            "glovebox.cli.helpers.profile.create_profile_from_option"
        ) as mock_create_profile,
        patch(
            "glovebox.cli.commands.firmware.helpers.execute_compilation_from_json"
        ) as mock_compile,
        patch(
            "glovebox.cli.helpers.profile.get_user_config_from_context"
        ) as mock_get_user_config,
    ):
        # Mock auto-profile detection
        mock_auto_profile.return_value = "corne"

        # Mock profile creation
        mock_profile = Mock()
        mock_profile.keyboard_name = "corne"
        mock_profile.firmware_version = None
        mock_profile.keyboard_config.compile_methods = [Mock(method_type="zmk_config")]
        mock_create_profile.return_value = mock_profile

        # Mock user config
        mock_get_user_config.return_value = None

        # Mock compilation result
        from glovebox.firmware.models import BuildResult

        mock_result = BuildResult(success=True)
        mock_result.messages = ["Compilation successful"]
        mock_compile.return_value = mock_result

        # Run command without profile flag (should auto-detect)
        cmd_result = cli_runner.invoke(
            app,
            ["firmware", "compile", str(json_file)],
            catch_exceptions=False,
        )

        # Verify auto-detection was called (user_config will be a UserConfig object, not None)
        assert mock_auto_profile.called
        args = mock_auto_profile.call_args[0]
        assert args[0] == json_file
        assert args[1] is not None  # user_config object

        # Verify profile was created with auto-detected keyboard
        assert mock_create_profile.called
        args = mock_create_profile.call_args[0]
        assert args[0] == "corne"
        assert args[1] is not None  # user_config object

        # Verify compilation was called
        assert mock_compile.called

        assert cmd_result.exit_code == 0


def test_firmware_compile_no_auto_flag_disables_detection(cli_runner, tmp_path):
    """Test that --no-auto flag disables auto-profile detection."""
    from glovebox.cli.commands import register_all_commands

    register_all_commands(app)

    # Create a test JSON file with keyboard field
    test_json = {"keyboard": "corne", "title": "Test Layout", "layers": []}
    json_file = tmp_path / "test_layout.json"
    json_file.write_text(json.dumps(test_json))

    with (
        patch(
            "glovebox.cli.helpers.auto_profile.resolve_profile_with_auto_detection"
        ) as mock_auto_profile,
        patch(
            "glovebox.cli.helpers.profile.create_profile_from_option"
        ) as mock_create_profile,
        patch(
            "glovebox.cli.commands.firmware.helpers.execute_compilation_from_json"
        ) as mock_compile,
        patch(
            "glovebox.cli.helpers.profile.get_user_config_from_context"
        ) as mock_get_user_config,
    ):
        # Mock profile creation
        mock_profile = Mock()
        mock_profile.keyboard_name = "glove80"
        mock_profile.firmware_version = "v25.05"
        mock_profile.keyboard_config.compile_methods = [Mock(method_type="zmk_config")]
        mock_create_profile.return_value = mock_profile

        # Mock user config
        mock_get_user_config.return_value = None

        # Mock compilation result
        from glovebox.firmware.models import BuildResult

        mock_result = BuildResult(success=True)
        mock_result.messages = ["Compilation successful"]
        mock_compile.return_value = mock_result

        # Run command with --no-auto flag
        cmd_result = cli_runner.invoke(
            app,
            [
                "firmware",
                "compile",
                str(json_file),
                "--no-auto",
                "--profile",
                "glove80/v25.05",
            ],
            catch_exceptions=False,
        )

        # Verify auto-detection was NOT called
        mock_auto_profile.assert_not_called()

        # Verify profile was created with explicit profile
        assert mock_create_profile.called
        args = mock_create_profile.call_args[0]
        assert args[0] == "glove80/v25.05"
        assert args[1] is not None  # user_config object

        assert cmd_result.exit_code == 0


def test_firmware_compile_cli_profile_overrides_auto_detection(cli_runner, tmp_path):
    """Test that CLI profile flag overrides auto-detection."""
    from glovebox.cli.commands import register_all_commands

    register_all_commands(app)

    # Create a test JSON file with keyboard field
    test_json = {"keyboard": "corne", "title": "Test Layout", "layers": []}
    json_file = tmp_path / "test_layout.json"
    json_file.write_text(json.dumps(test_json))

    with (
        patch(
            "glovebox.cli.helpers.auto_profile.resolve_profile_with_auto_detection"
        ) as mock_auto_profile,
        patch(
            "glovebox.cli.helpers.profile.create_profile_from_option"
        ) as mock_create_profile,
        patch(
            "glovebox.cli.commands.firmware.helpers.execute_compilation_from_json"
        ) as mock_compile,
        patch(
            "glovebox.cli.helpers.profile.get_user_config_from_context"
        ) as mock_get_user_config,
    ):
        # Mock profile creation
        mock_profile = Mock()
        mock_profile.keyboard_name = "glove80"
        mock_profile.firmware_version = "v25.05"
        mock_profile.keyboard_config.compile_methods = [Mock(method_type="zmk_config")]
        mock_create_profile.return_value = mock_profile

        # Mock user config
        mock_get_user_config.return_value = None

        # Mock compilation result
        from glovebox.firmware.models import BuildResult

        mock_result = BuildResult(success=True)
        mock_result.messages = ["Compilation successful"]
        mock_compile.return_value = mock_result

        # Run command with explicit profile (should NOT auto-detect)
        cmd_result = cli_runner.invoke(
            app,
            ["firmware", "compile", str(json_file), "--profile", "glove80/v25.05"],
            catch_exceptions=False,
        )

        # Verify auto-detection was NOT called (CLI profile takes precedence)
        mock_auto_profile.assert_not_called()

        # Verify profile was created with explicit profile
        assert mock_create_profile.called
        args = mock_create_profile.call_args[0]
        assert args[0] == "glove80/v25.05"
        assert args[1] is not None  # user_config object

        assert cmd_result.exit_code == 0


def test_firmware_compile_auto_detection_only_for_json_files(cli_runner, tmp_path):
    """Test that auto-detection only works with JSON files."""
    from glovebox.cli.commands import register_all_commands

    register_all_commands(app)

    # Create a test .keymap file
    keymap_file = tmp_path / "test.keymap"
    keymap_file.write_text("// Test keymap content")

    # Create a test .conf file
    conf_file = tmp_path / "test.conf"
    conf_file.write_text("CONFIG_TEST=y")

    with (
        patch(
            "glovebox.cli.helpers.auto_profile.resolve_profile_with_auto_detection"
        ) as mock_auto_profile,
        patch(
            "glovebox.cli.helpers.profile.create_profile_from_option"
        ) as mock_create_profile,
        patch(
            "glovebox.cli.commands.firmware.helpers.execute_compilation_service"
        ) as mock_compile,
        patch(
            "glovebox.cli.helpers.profile.get_user_config_from_context"
        ) as mock_get_user_config,
    ):
        # Mock profile creation
        mock_profile = Mock()
        mock_profile.keyboard_name = "glove80"
        mock_profile.firmware_version = "v25.05"
        mock_profile.keyboard_config.compile_methods = [Mock(method_type="zmk_config")]
        mock_create_profile.return_value = mock_profile

        # Mock user config
        mock_get_user_config.return_value = None

        # Mock compilation result
        from glovebox.firmware.models import BuildResult

        mock_result = BuildResult(success=True)
        mock_result.messages = ["Compilation successful"]
        mock_compile.return_value = mock_result

        # Run command with .keymap file (should NOT auto-detect)
        cmd_result = cli_runner.invoke(
            app,
            [
                "firmware",
                "compile",
                str(keymap_file),
                str(conf_file),
                "--profile",
                "glove80/v25.05",
            ],
            catch_exceptions=False,
        )

        # Verify auto-detection was NOT called for non-JSON files
        mock_auto_profile.assert_not_called()

        # Verify profile was created with explicit profile
        assert mock_create_profile.called
        args = mock_create_profile.call_args[0]
        assert args[0] == "glove80/v25.05"
        assert args[1] is not None  # user_config object

        assert cmd_result.exit_code == 0


def test_firmware_compile_help_includes_auto_detection_options(cli_runner):
    """Test that help includes auto-detection related options and documentation."""
    from glovebox.cli.commands import register_all_commands

    register_all_commands(app)

    cmd_result = cli_runner.invoke(app, ["firmware", "compile", "--help"])

    # Check for --no-auto flag
    assert "--no-auto" in cmd_result.output
    assert "Disable automatic profile detection" in cmd_result.output

    # Check for precedence documentation
    assert "Profile precedence" in cmd_result.output
    assert "Auto-detection from JSON keyboard field" in cmd_result.output

    # Check for auto-detection examples
    assert "auto-profile detection" in cmd_result.output

    assert cmd_result.exit_code == 0


def test_firmware_flash_accepts_json_files(cli_runner):
    """Test that firmware flash command accepts JSON files."""
    from glovebox.cli.commands import register_all_commands

    register_all_commands(app)

    # Test that JSON files are accepted in the help
    cmd_result = cli_runner.invoke(app, ["firmware", "flash", "--help"])

    # Should mention JSON files in the help text
    assert cmd_result.exit_code == 0
    assert ".json" in cmd_result.output
    assert "layout" in cmd_result.output.lower()


def test_firmware_flash_json_auto_detection_help(cli_runner):
    """Test that firmware flash help mentions auto-detection for JSON files."""
    from glovebox.cli.commands import register_all_commands

    register_all_commands(app)

    cmd_result = cli_runner.invoke(app, ["firmware", "flash", "--help"])

    # Should mention auto-detection functionality
    assert cmd_result.exit_code == 0
    assert (
        "auto-detect" in cmd_result.output.lower()
        or "automatic" in cmd_result.output.lower()
    )
    assert "keyboard" in cmd_result.output.lower()
