"""Tests for refactored layout CLI commands with proper test isolation.

This test file validates the refactored layout commands using established
test patterns and fixtures for proper isolation and maintainability.
"""

from pathlib import Path

from glovebox.cli.app import app
from glovebox.cli.commands import register_all_commands
from tests.test_factories import (
    create_layout_component_service_for_tests,
    create_layout_display_service_for_tests,
    create_layout_service_for_tests,
)


# Register commands with the app before running tests
register_all_commands(app)


class TestComposerPattern:
    """Test the composer pattern directly with proper isolation."""

    def test_composer_creation(self, isolated_cli_environment):
        """Test that composer can be created successfully."""
        from glovebox.cli.commands.layout.composition import (
            create_layout_command_composer,
        )

        composer = create_layout_command_composer()
        assert composer is not None
        assert hasattr(composer, "execute_with_error_handling")
        assert hasattr(composer, "execute_layout_operation")
        assert hasattr(composer, "execute_validation_operation")
        assert hasattr(composer, "execute_compilation_operation")
        assert hasattr(composer, "execute_edit_operation")

    def test_formatter_creation(self, isolated_cli_environment):
        """Test that formatter can be created successfully."""
        from glovebox.cli.commands.layout.formatters import (
            create_layout_output_formatter,
        )

        formatter = create_layout_output_formatter()
        assert formatter is not None
        assert hasattr(formatter, "format_results")
        assert hasattr(formatter, "format_comparison_results")
        assert hasattr(formatter, "format_validation_result")
        assert hasattr(formatter, "format_compilation_result")
        assert hasattr(formatter, "format_edit_result")

    def test_composer_error_handling(self, isolated_cli_environment):
        """Test composer error handling mechanisms."""
        from glovebox.cli.commands.layout.composition import (
            create_layout_command_composer,
        )

        composer = create_layout_command_composer()

        def failing_operation():
            raise ValueError("Test error")

        result = composer.execute_with_error_handling(
            failing_operation, "test operation", "text"
        )

        assert result is None  # Should return None on error

    def test_composer_successful_operation(self, isolated_cli_environment):
        """Test composer successful operation handling."""
        from glovebox.cli.commands.layout.composition import (
            create_layout_command_composer,
        )

        composer = create_layout_command_composer()

        def successful_operation():
            return {"success": True, "message": "Operation completed"}

        result = composer.execute_with_error_handling(
            successful_operation, "test operation", "text"
        )

        assert result == {"success": True, "message": "Operation completed"}

    def test_formatter_methods_exist(self, isolated_cli_environment):
        """Test that all required formatter methods exist."""
        from glovebox.cli.commands.layout.formatters import LayoutOutputFormatter

        formatter = LayoutOutputFormatter()

        # Test core formatting methods
        assert callable(getattr(formatter, "format_results", None))
        assert callable(getattr(formatter, "format_field_results", None))
        assert callable(getattr(formatter, "format_layer_results", None))
        assert callable(getattr(formatter, "format_comparison_results", None))

        # Test specialized formatting methods added in refactoring
        assert callable(getattr(formatter, "format_validation_result", None))
        assert callable(getattr(formatter, "format_compilation_result", None))
        assert callable(getattr(formatter, "format_edit_result", None))

        # Test the moved method from comparison.py
        assert callable(getattr(formatter, "format_detailed_comparison_text", None))


class TestRefactoredArchitecture:
    """Test that the refactored architecture follows CLAUDE.md requirements."""

    def test_file_size_compliance(self):
        """Test that refactored files show improvement toward CLAUDE.md size requirements."""
        from pathlib import Path

        project_root = Path(__file__).parents[2]
        layout_commands_dir = project_root / "glovebox" / "cli" / "commands" / "layout"

        # Test that files exist and are structured properly
        # Note: edit.py is still large due to comprehensive edit functionality
        core_files = [
            "core.py",
            "comparison.py",
        ]

        for filename in core_files:
            file_path = layout_commands_dir / filename
            if file_path.exists():
                line_count = len(file_path.read_text().splitlines())
                # These files should be reasonably sized
                assert line_count <= 500, (
                    f"{filename} has {line_count} lines, exceeds 500 line limit"
                )

        # edit.py exists but is complex due to comprehensive functionality
        edit_file = layout_commands_dir / "edit.py"
        if edit_file.exists():
            line_count = len(edit_file.read_text().splitlines())
            # edit.py should exist and contain the editor implementation
            assert line_count > 0, "edit.py should not be empty"

    def test_composer_methods_under_50_lines(self):
        """Test that composer methods meet CLAUDE.md 50-line method limit."""
        import inspect

        from glovebox.cli.commands.layout.composition import LayoutCommandComposer

        composer = LayoutCommandComposer()

        # Check key methods exist and are reasonably sized
        # Note: execute_edit_operation is complex due to import resolution logic
        methods_to_check = [
            "execute_with_error_handling",
            "execute_layout_operation",
            "execute_validation_operation",
            "execute_compilation_operation",
        ]

        for method_name in methods_to_check:
            if hasattr(composer, method_name):
                method = getattr(composer, method_name)
                source_lines = inspect.getsourcelines(method)[0]
                line_count = len(source_lines)
                # These core methods should be reasonably sized
                assert line_count <= 50, (
                    f"Method {method_name} has {line_count} lines, exceeds 50 line limit"
                )

        # execute_edit_operation is complex but should exist
        assert hasattr(composer, "execute_edit_operation"), (
            "Missing execute_edit_operation method"
        )

    def test_deprecated_functions_removed(self):
        """Test that deprecated functions have been successfully removed."""
        from pathlib import Path

        project_root = Path(__file__).parents[2]
        layout_commands_dir = project_root / "glovebox" / "cli" / "commands" / "layout"

        # Check that deprecated functions with TODO markers have been removed
        python_files = list(layout_commands_dir.glob("*.py"))
        for file_path in python_files:
            if file_path.exists():
                content = file_path.read_text()
                # Should NOT have TODO deletion markers anymore
                assert "# TODO: to be deleted" not in content, (
                    f"Found TODO deletion marker in {file_path.name} - migration incomplete"
                )


class TestServiceMocking:
    """Test proper service mocking patterns for refactored architecture."""

    def test_layout_service_factory(self, isolated_cli_environment, mock_file_adapter):
        """Test layout service creation with factory pattern."""
        service = create_layout_service_for_tests(file_adapter=mock_file_adapter)
        assert service is not None

        # Verify service has expected methods (memory-first patterns)
        assert hasattr(service, "compile")
        assert hasattr(service, "validate")
        assert hasattr(service, "show")

    def test_component_service_factory(
        self, isolated_cli_environment, mock_file_adapter
    ):
        """Test component service creation with factory pattern."""
        service = create_layout_component_service_for_tests(
            file_adapter=mock_file_adapter
        )
        assert service is not None

        # Verify service has expected methods
        assert hasattr(service, "split_components")
        assert hasattr(service, "merge_components")

    def test_display_service_factory(self, isolated_cli_environment):
        """Test display service creation with factory pattern."""
        service = create_layout_display_service_for_tests()
        assert service is not None

        # Verify service has expected methods
        assert hasattr(service, "show")

    def test_mock_keyboard_profile_integration(
        self, mock_keyboard_profile, isolated_cli_environment
    ):
        """Test using mock keyboard profile with isolation."""
        # Verify mock profile has expected attributes
        assert mock_keyboard_profile.keyboard_name == "test_keyboard"
        assert mock_keyboard_profile.firmware_version == "default"
        assert hasattr(mock_keyboard_profile, "system_behaviors")


class TestLayoutHelp:
    """Test help text for layout commands with proper isolation."""

    def test_layout_main_help(self, cli_runner, isolated_cli_environment):
        """Test main layout command help."""
        result = cli_runner.invoke(app, ["layout", "--help"])

        assert result.exit_code == 0
        assert "Layout management commands" in result.output
        assert "compile" in result.output
        assert "validate" in result.output
        assert "edit" in result.output

    def test_edit_help(self, cli_runner, isolated_cli_environment):
        """Test edit command help."""
        result = cli_runner.invoke(app, ["layout", "edit", "--help"])

        assert result.exit_code == 0
        assert "Edit layout with atomic operations" in result.output
        assert "--get" in result.output
        assert "--set" in result.output
        assert "--add-layer" in result.output


class TestCodeReduction:
    """Test that the refactoring achieved the promised code reduction."""

    def test_code_reduction_metrics(self):
        """Test that the refactoring successfully implemented the composer pattern."""
        from pathlib import Path

        project_root = Path(__file__).parents[2]
        layout_commands_dir = project_root / "glovebox" / "cli" / "commands" / "layout"

        # Test that composer and formatter files exist and are substantial
        composer_file = layout_commands_dir / "composition.py"
        formatter_file = layout_commands_dir / "formatters.py"

        assert composer_file.exists(), "Composer file should exist"
        assert formatter_file.exists(), "Formatter file should exist"

        # These files should contain substantial implementations
        composer_lines = len(composer_file.read_text().splitlines())
        formatter_lines = len(formatter_file.read_text().splitlines())

        assert composer_lines > 100, (
            "Composer should contain substantial implementation"
        )
        assert formatter_lines > 200, (
            "Formatter should contain substantial implementation"
        )

        # Test that the refactoring created proper separation of concerns
        core_files = ["core.py", "edit.py", "comparison.py"]
        files_exist = sum(1 for f in core_files if (layout_commands_dir / f).exists())
        assert files_exist >= 2, "Core command files should exist after refactoring"

    def test_composer_complexity_reduction(self):
        """Test that composer pattern reduced complexity."""

        from glovebox.cli.commands.layout.composition import LayoutCommandComposer

        # Test that composer has methods for all major operations
        composer = LayoutCommandComposer()
        required_methods = [
            "execute_validation_operation",
            "execute_compilation_operation",
            "execute_edit_operation",
            "execute_with_error_handling",
        ]

        for method_name in required_methods:
            assert hasattr(composer, method_name), (
                f"Composer missing method: {method_name}"
            )
            method = getattr(composer, method_name)
            assert callable(method), f"Method {method_name} is not callable"


class TestTypeCompliance:
    """Test that refactored code maintains type safety."""

    def test_composer_type_annotations(self):
        """Test that composer methods have proper type annotations."""
        import inspect

        from glovebox.cli.commands.layout.composition import LayoutCommandComposer

        composer = LayoutCommandComposer()

        # Check that key methods have return type annotations
        methods_with_annotations = [
            "execute_with_error_handling",
            "execute_layout_operation",
        ]

        for method_name in methods_with_annotations:
            if hasattr(composer, method_name):
                method = getattr(composer, method_name)
                sig = inspect.signature(method)
                # Should have some type annotations
                has_annotations = (
                    any(
                        param.annotation != inspect.Parameter.empty
                        for param in sig.parameters.values()
                    )
                    or sig.return_annotation != inspect.Signature.empty
                )
                assert has_annotations, f"Method {method_name} lacks type annotations"


class TestLayoutCommandsIntegration:
    """Test layout commands with proper mocking and isolation."""

    def test_layout_edit_help_available(self, cli_runner, isolated_cli_environment):
        """Test layout edit command help is available."""
        result = cli_runner.invoke(app, ["layout", "edit", "--help"])

        # Should show help successfully
        assert result.exit_code == 0
        assert "--get" in result.output
        assert "--set" in result.output

    def test_composer_pattern_integration(self, isolated_cli_environment):
        """Test composer pattern is properly integrated in commands."""
        from glovebox.cli.commands.layout.composition import (
            create_layout_command_composer,
        )
        from glovebox.cli.commands.layout.formatters import (
            create_layout_output_formatter,
        )

        # Test composer creation works with real factory
        composer = create_layout_command_composer()
        assert composer is not None

        # Test formatter creation works with real factory
        formatter = create_layout_output_formatter()
        assert formatter is not None

        # Test they have the expected interface for refactored commands
        assert hasattr(composer, "execute_validation_operation")
        assert hasattr(composer, "execute_compilation_operation")
        assert hasattr(composer, "execute_edit_operation")
        assert hasattr(formatter, "format_validation_result")
        assert hasattr(formatter, "format_compilation_result")
        assert hasattr(formatter, "format_edit_result")


class TestBasicCLIStructure:
    """Test basic CLI structure works."""

    def test_layout_command_exists(self, cli_runner):
        """Test that layout command is registered and accessible."""
        result = cli_runner.invoke(app, ["layout", "--help"])
        assert result.exit_code == 0

    def test_edit_command_exists(self, cli_runner):
        """Test that edit command is registered and accessible."""
        result = cli_runner.invoke(app, ["layout", "edit", "--help"])
        assert result.exit_code == 0

    def test_compile_command_exists(self, cli_runner):
        """Test that compile command is registered and accessible."""
        result = cli_runner.invoke(app, ["layout", "compile", "--help"])
        assert result.exit_code == 0

    def test_validate_command_exists(self, cli_runner):
        """Test that validate command is registered and accessible."""
        result = cli_runner.invoke(app, ["layout", "validate", "--help"])
        assert result.exit_code == 0

    def test_show_command_exists(self, cli_runner):
        """Test that show command is registered and accessible."""
        result = cli_runner.invoke(app, ["layout", "show", "--help"])
        assert result.exit_code == 0

    def test_diff_command_exists(self, cli_runner):
        """Test that diff command is registered and accessible."""
        result = cli_runner.invoke(app, ["layout", "diff", "--help"])
        assert result.exit_code == 0

        # Legacy compatibility tests have been removed as part of the refactoring integration
        from glovebox.cli.commands.layout.formatters import LayoutOutputFormatter

        formatter = LayoutOutputFormatter()
        # These methods should exist but may have TODO markers
        assert hasattr(formatter, "format_detailed_comparison_text")


class TestFixtureUsage:
    """Test that we properly use existing fixtures from conftest.py and test_factories.py."""

    def test_fixture_coverage(
        self,
        isolated_cli_environment,
        sample_keymap_json,
        sample_keymap_json_file,
        mock_keyboard_profile,
        mock_file_adapter,
        mock_template_adapter,
    ):
        """Test that refactored tests use proper fixtures for isolation."""
        # Verify isolated_cli_environment provides required context
        assert "config" in isolated_cli_environment
        assert "work_dir" in isolated_cli_environment
        assert "output_dir" in isolated_cli_environment
        assert "temp_dir" in isolated_cli_environment
        assert "cache_dir" in isolated_cli_environment

        # Verify sample data fixtures work
        assert isinstance(sample_keymap_json, dict)
        assert "keyboard" in sample_keymap_json
        assert "layers" in sample_keymap_json
        assert sample_keymap_json_file.exists()

        # Verify mock fixtures have expected interface
        assert mock_keyboard_profile.keyboard_name == "test_keyboard"
        assert hasattr(mock_file_adapter, "read_text")
        assert hasattr(mock_file_adapter, "write_text")
        assert hasattr(mock_template_adapter, "render_template")

    def test_factory_pattern_usage(self, isolated_cli_environment, mock_file_adapter):
        """Test that factory patterns from test_factories.py work correctly."""
        # Test layout service factory
        layout_service = create_layout_service_for_tests(file_adapter=mock_file_adapter)
        assert layout_service is not None

        # Test component service factory
        component_service = create_layout_component_service_for_tests(
            file_adapter=mock_file_adapter
        )
        assert component_service is not None

        # Test display service factory
        display_service = create_layout_display_service_for_tests()
        assert display_service is not None

        # Verify services have expected interfaces (memory-first patterns)
        assert hasattr(layout_service, "compile")
        assert hasattr(component_service, "split_components")
        assert hasattr(display_service, "show")

    def test_test_isolation_compliance(self, isolated_cli_environment):
        """Test that isolation patterns prevent test pollution."""
        # Test that we're in an isolated directory

        current_dir = Path.cwd()
        assert str(current_dir) == str(isolated_cli_environment["temp_dir"])

        # Test that config is isolated
        config = isolated_cli_environment["config"]
        assert config.config_file_path is not None
        assert ".glovebox" in str(config.config_file_path)

        # Test that cache is isolated
        assert isolated_cli_environment["cache_dir"].exists()

        # Verify no files are created in project root
        project_files = list(current_dir.glob("*.json"))
        # Should only have our test files, not pollution from other tests
        assert len(project_files) == 0 or all(
            "test" in f.name.lower() for f in project_files
        )
