"""Tests for environment variable integration with Pydantic Settings.

This module tests the comprehensive integration of environment variables
with the configuration system, including precedence, validation, and
complex nested configurations.
"""

import os
from pathlib import Path
from typing import Any

import pytest
from pydantic import ValidationError

from glovebox.config.models import UserConfigData
from glovebox.config.user_config import create_user_config


class TestEnvironmentVariableOverrides:
    """Tests for environment variable override functionality."""

    def test_all_environment_overrides(self, clean_environment):
        """Test that all configuration fields can be overridden by environment variables."""
        # Set all possible environment variables
        env_vars = {
            "GLOVEBOX_PROFILE": "env/test",
            "GLOVEBOX_LOG_LEVEL": "CRITICAL",
            "GLOVEBOX_PROFILES_PATHS": "/env/path1,/env/path2,~/env/path3",
            "GLOVEBOX_FIRMWARE__FLASH__TIMEOUT": "999",
            "GLOVEBOX_FIRMWARE__FLASH__COUNT": "42",
            "GLOVEBOX_FIRMWARE__FLASH__TRACK_FLASHED": "false",
            "GLOVEBOX_FIRMWARE__FLASH__SKIP_EXISTING": "true",
            "GLOVEBOX_FLASH_SKIP_EXISTING": "true",
        }

        # Set environment variables
        for key, value in env_vars.items():
            os.environ[key] = value

        config = UserConfigData()

        # Verify all overrides work
        assert config.profile == "env/test"
        assert config.log_level == "CRITICAL"
        # Check that paths are converted to Path objects and expanded
        expected_paths = [
            Path("/env/path1"),
            Path("/env/path2"),
            Path("~/env/path3").expanduser(),
        ]
        assert config.profiles_paths == expected_paths
        assert config.firmware.flash.timeout == 999
        assert config.firmware.flash.count == 42
        assert config.firmware.flash.track_flashed is False
        assert config.firmware.flash.skip_existing is True

    def test_partial_environment_overrides(self, clean_environment):
        """Test partial environment variable overrides leave defaults intact."""
        # Set only some environment variables
        os.environ["GLOVEBOX_PROFILE"] = "partial/test"
        os.environ["GLOVEBOX_FIRMWARE__FLASH__TIMEOUT"] = "555"

        config = UserConfigData()

        # Overridden values
        assert config.profile == "partial/test"
        assert config.firmware.flash.timeout == 555

        # Default values for non-overridden fields
        assert config.log_level == "INFO"
        assert config.profiles_paths == []
        assert config.firmware.flash.count == 2
        assert config.firmware.flash.track_flashed is True
        assert config.firmware.flash.skip_existing is False

    def test_nested_environment_delimiter(self, clean_environment):
        """Test nested environment variable delimiter handling."""
        # Test various nested configurations
        nested_vars = {
            "GLOVEBOX_FIRMWARE__FLASH__TIMEOUT": "100",
            "GLOVEBOX_FIRMWARE__FLASH__COUNT": "5",
            "GLOVEBOX_FIRMWARE__FLASH__TRACK_FLASHED": "false",
            "GLOVEBOX_FIRMWARE__FLASH__SKIP_EXISTING": "true",
        }

        for key, value in nested_vars.items():
            os.environ[key] = value

        config = UserConfigData()

        # Verify nested structure is correctly parsed
        assert config.firmware.flash.timeout == 100
        assert config.firmware.flash.count == 5
        assert config.firmware.flash.track_flashed is False
        assert config.firmware.flash.skip_existing is True

    def test_case_insensitive_environment_variables(self, clean_environment):
        """Test case insensitive environment variable handling."""
        # Test various case combinations
        case_vars = {
            "glovebox_profile": "lowercase/test",
            "GLOVEBOX_LOG_LEVEL": "debug",
            "Glovebox_Firmware__Flash__Timeout": "123",
        }

        for key, value in case_vars.items():
            os.environ[key] = value

        config = UserConfigData()

        # Should work regardless of case
        assert config.profile == "lowercase/test"
        assert config.log_level == "DEBUG"  # Should be normalized
        assert config.firmware.flash.timeout == 123

    def test_environment_variable_type_conversion(self, clean_environment):
        """Test automatic type conversion for environment variables."""
        # Test different data types
        type_vars = {
            "GLOVEBOX_PROFILE": "string/test",  # String
            "GLOVEBOX_FIRMWARE__FLASH__TIMEOUT": "300",  # String -> Int
            "GLOVEBOX_FIRMWARE__FLASH__COUNT": "0",  # String -> Int (edge case)
            "GLOVEBOX_FIRMWARE__FLASH__TRACK_FLASHED": "false",  # String -> Bool
            "GLOVEBOX_FIRMWARE__FLASH__SKIP_EXISTING": "1",  # String -> Bool (truthy)
            "GLOVEBOX_PROFILES_PATHS": "/path1,/path2,/path3",  # String -> List
        }

        for key, value in type_vars.items():
            os.environ[key] = value

        config = UserConfigData()

        # Verify correct type conversion
        assert isinstance(config.profile, str)
        assert config.profile == "string/test"

        assert isinstance(config.firmware.flash.timeout, int)
        assert config.firmware.flash.timeout == 300

        assert isinstance(config.firmware.flash.count, int)
        assert config.firmware.flash.count == 0

        assert isinstance(config.firmware.flash.track_flashed, bool)
        assert config.firmware.flash.track_flashed is False

        assert isinstance(config.firmware.flash.skip_existing, bool)
        assert config.firmware.flash.skip_existing is True

        assert isinstance(config.profiles_paths, list)
        expected_paths = [Path("/path1"), Path("/path2"), Path("/path3")]
        assert config.profiles_paths == expected_paths

    def test_boolean_environment_variable_values(self, clean_environment):
        """Test various boolean representations in environment variables."""
        # Test different boolean representations
        bool_test_cases = [
            ("true", True),
            ("TRUE", True),
            ("True", True),
            ("1", True),
            ("yes", True),
            ("YES", True),
            ("false", False),
            ("FALSE", False),
            ("False", False),
            ("0", False),
            ("no", False),
            ("NO", False),
            # Note: Empty strings and invalid values cause validation errors in Pydantic
        ]

        for env_value, expected_bool in bool_test_cases:
            # Clear previous env var
            if "GLOVEBOX_FIRMWARE__FLASH__TRACK_FLASHED" in os.environ:
                del os.environ["GLOVEBOX_FIRMWARE__FLASH__TRACK_FLASHED"]

            # Set environment variable
            os.environ["GLOVEBOX_FIRMWARE__FLASH__TRACK_FLASHED"] = env_value

            config = UserConfigData()
            assert config.firmware.flash.track_flashed is expected_bool, (
                f"Failed for value: {env_value}"
            )

    def test_list_environment_variable_parsing(self, clean_environment):
        """Test list parsing from environment variables."""
        list_test_cases = [
            ("", []),  # Empty string -> empty list
            ("/single/path", ["/single/path"]),  # Single item
            ("/path1,/path2", ["/path1", "/path2"]),  # Multiple items
            ("/path1, /path2 , /path3", ["/path1", "/path2", "/path3"]),  # With spaces
            (
                "~/home,/absolute,$ENV_VAR",
                ["~/home", "/absolute", "$ENV_VAR"],
            ),  # Mixed formats
        ]

        for env_value, _expected_list in list_test_cases:
            # Clear previous env var
            if "GLOVEBOX_PROFILES_PATHS" in os.environ:
                del os.environ["GLOVEBOX_PROFILES_PATHS"]

            # Set environment variable
            os.environ["GLOVEBOX_PROFILES_PATHS"] = env_value

            config = UserConfigData()
            # Check that the environment value is parsed to Path objects
            if env_value == "":
                expected_paths = []
            else:
                expected_paths = [
                    Path(path.strip()).expanduser()
                    for path in env_value.split(",")
                    if path.strip()
                ]
            assert config.profiles_paths == expected_paths, (
                f"Failed for value: {env_value}"
            )


class TestEnvironmentVariableValidation:
    """Tests for validation of environment variable values."""

    def test_invalid_profile_environment_variable(self, clean_environment):
        """Test validation error for invalid profile format in environment."""
        os.environ["GLOVEBOX_PROFILE"] = ""  # Empty profile

        with pytest.raises(ValidationError) as exc_info:
            UserConfigData()

        error_msg = str(exc_info.value)
        assert (
            "Profile must be in format 'keyboard/firmware'" in error_msg
            or "cannot be empty" in error_msg
        )

    def test_invalid_log_level_environment_variable(self, clean_environment):
        """Test validation error for invalid log level in environment."""
        os.environ["GLOVEBOX_LOG_LEVEL"] = "INVALID_LEVEL"

        with pytest.raises(ValidationError) as exc_info:
            UserConfigData()

        error_msg = str(exc_info.value)
        assert "Log level must be one of" in error_msg

    def test_invalid_timeout_environment_variable(self, clean_environment):
        """Test validation error for invalid timeout in environment."""
        os.environ["GLOVEBOX_FIRMWARE__FLASH__TIMEOUT"] = "-1"  # Negative value

        with pytest.raises(ValidationError) as exc_info:
            UserConfigData()

        error_msg = str(exc_info.value)
        assert "greater than or equal to 1" in error_msg

    def test_invalid_count_environment_variable(self, clean_environment):
        """Test validation error for invalid count in environment."""
        os.environ["GLOVEBOX_FIRMWARE__FLASH__COUNT"] = "-5"  # Negative value

        with pytest.raises(ValidationError) as exc_info:
            UserConfigData()

        error_msg = str(exc_info.value)
        assert "greater than or equal to 0" in error_msg

    def test_non_numeric_environment_variables(self, clean_environment):
        """Test validation error for non-numeric values in numeric fields."""
        # Test timeout
        os.environ["GLOVEBOX_FIRMWARE__FLASH__TIMEOUT"] = "not_a_number"

        with pytest.raises(ValidationError) as exc_info:
            UserConfigData()

        error_msg = str(exc_info.value)
        assert (
            "Input should be a valid integer" in error_msg
            or "invalid literal for int()" in error_msg
        )


class TestEnvironmentSourceTracking:
    """Tests for tracking environment variable sources in UserConfig."""

    def test_environment_source_tracking_in_user_config(
        self, clean_environment, temp_config_dir
    ):
        """Test that UserConfig properly tracks environment variable sources."""
        # Set environment variables
        os.environ["GLOVEBOX_PROFILE"] = "env/source"
        os.environ["GLOVEBOX_FIRMWARE__FLASH__TIMEOUT"] = "777"

        # Change to temp directory to avoid picking up existing config files
        original_cwd = Path.cwd()
        original_xdg = os.environ.get("XDG_CONFIG_HOME")
        os.chdir(temp_config_dir)
        os.environ["XDG_CONFIG_HOME"] = str(temp_config_dir)

        try:
            config = create_user_config()

            # Should track environment sources
            assert config.get_source("profile") == "environment"
            assert config.get_source("firmware.flash.timeout") == "environment"

            # Non-environment values should show as default
            assert config.get_source("log_level") == "default"
            assert config.get_source("firmware.flash.count") == "default"
        finally:
            os.chdir(original_cwd)
            if original_xdg:
                os.environ["XDG_CONFIG_HOME"] = original_xdg
            elif "XDG_CONFIG_HOME" in os.environ:
                del os.environ["XDG_CONFIG_HOME"]

    def test_mixed_source_tracking_with_environment(
        self, clean_environment, sample_config_dict: dict[str, Any], config_file
    ):
        """Test source tracking with both file and environment sources."""
        # Set environment variables that will override file
        os.environ["GLOVEBOX_PROFILE"] = "env/override"
        os.environ["GLOVEBOX_FIRMWARE__FLASH__COUNT"] = "99"

        config = create_user_config(cli_config_path=config_file)

        # Environment should override file
        assert config._config.profile == "env/override"
        assert config.get_source("profile") == "environment"

        # File values should be tracked for non-overridden settings
        assert config._config.log_level == sample_config_dict["log_level"]
        assert config.get_source("log_level") == f"file:{config_file.name}"

        # Environment-only values
        assert config._config.firmware.flash.count == 99
        assert config.get_source("firmware.flash.count") == "environment"


class TestEnvironmentVariableEdgeCases:
    """Tests for edge cases in environment variable handling."""

    def test_empty_environment_variables(self, clean_environment):
        """Test handling of empty environment variables."""
        # Set empty environment variables
        os.environ["GLOVEBOX_PROFILE"] = ""
        os.environ["GLOVEBOX_LOG_LEVEL"] = ""
        os.environ["GLOVEBOX_PROFILES_PATHS"] = ""

        # Profile should fail validation (empty)
        with pytest.raises(ValidationError):
            UserConfigData()

    def test_whitespace_environment_variables(self, clean_environment):
        """Test handling of whitespace in environment variables."""
        # Set environment variables with whitespace
        os.environ["GLOVEBOX_PROFILE"] = "  whitespace/test  "
        os.environ["GLOVEBOX_LOG_LEVEL"] = "  DEBUG  "

        config = UserConfigData()

        # Should handle whitespace appropriately
        assert config.profile == "  whitespace/test  "  # Profile keeps whitespace
        assert config.log_level == "DEBUG"  # Log level should be trimmed/normalized

    def test_unicode_environment_variables(self, clean_environment):
        """Test handling of unicode characters in environment variables."""
        # Set environment variables with unicode
        os.environ["GLOVEBOX_PROFILE"] = "unicode_🚀/test_✨"
        os.environ["GLOVEBOX_PROFILES_PATHS"] = "/path/with/émojis,~/ユニコード/パス"

        config = UserConfigData()

        # Should handle unicode correctly
        assert config.profile == "unicode_🚀/test_✨"
        expected_paths = [
            Path("/path/with/émojis"),
            Path("~/ユニコード/パス").expanduser(),
        ]
        assert config.profiles_paths == expected_paths

    def test_keyboard_only_profile_environment_variables(self, clean_environment):
        """Test keyboard-only profile format in environment variables."""
        # Set keyboard-only profile
        os.environ["GLOVEBOX_PROFILE"] = "keyboard_only_env"

        config = UserConfigData()

        # Should accept keyboard-only format
        assert config.profile == "keyboard_only_env"

    def test_very_long_environment_variables(self, clean_environment):
        """Test handling of very long environment variable values."""
        # Create very long values
        long_profile = (
            "very_long_keyboard_name_" + "x" * 100 + "/very_long_firmware_" + "y" * 100
        )
        long_paths = ",".join(
            [f"/very/long/path/number/{i}" + "z" * 50 for i in range(10)]
        )

        os.environ["GLOVEBOX_PROFILE"] = long_profile
        os.environ["GLOVEBOX_PROFILES_PATHS"] = long_paths

        config = UserConfigData()

        # Should handle long values
        assert config.profile == long_profile
        assert len(config.profiles_paths) == 10
        assert all("z" * 50 in str(path) for path in config.profiles_paths)

    def test_special_characters_in_environment_variables(self, clean_environment):
        """Test handling of special characters in environment variables."""
        # Test various special characters
        special_profile = "special-chars_123/firmware.version-2.0"
        special_paths = "/path/with spaces,/path/with'quotes,/path/with\"double-quotes"

        os.environ["GLOVEBOX_PROFILE"] = special_profile
        os.environ["GLOVEBOX_PROFILES_PATHS"] = special_paths

        config = UserConfigData()

        # Should handle special characters
        assert config.profile == special_profile
        expected_paths = [
            Path("/path/with spaces"),
            Path("/path/with'quotes"),
            Path('/path/with"double-quotes'),
        ]
        assert config.profiles_paths == expected_paths


class TestEnvironmentVariablePrecedence:
    """Tests for environment variable precedence rules."""

    def test_environment_overrides_defaults(self, clean_environment):
        """Test that environment variables override default values."""
        os.environ["GLOVEBOX_PROFILE"] = "env/override"
        os.environ["GLOVEBOX_LOG_LEVEL"] = "CRITICAL"

        config = UserConfigData()

        # Environment should override defaults
        assert config.profile == "env/override"  # Not default "glove80/v25.05"
        assert config.log_level == "CRITICAL"  # Not default "INFO"

    def test_environment_overrides_file_config(
        self, clean_environment, config_file, sample_config_dict: dict[str, Any]
    ):
        """Test that environment variables override file configuration."""
        # Set environment variables that conflict with file
        os.environ["GLOVEBOX_PROFILE"] = "env_keyboard/file_firmware"
        os.environ["GLOVEBOX_LOG_LEVEL"] = "CRITICAL"

        config = create_user_config(cli_config_path=config_file)

        # Environment should override file
        assert config._config.profile == "env_keyboard/file_firmware"  # Not from file
        assert config._config.log_level == "CRITICAL"  # Not from file

        # File values should still be used for non-overridden settings
        assert (
            config._config.firmware.flash.timeout
            == sample_config_dict["firmware"]["flash"]["timeout"]
        )

    def test_complete_precedence_chain(
        self, clean_environment, config_file, sample_config_dict: dict[str, Any]
    ):
        """Test complete precedence: environment > file > defaults."""
        # Environment variable overrides everything
        os.environ["GLOVEBOX_PROFILE"] = "env_kb/highest_precedence"

        config = create_user_config(cli_config_path=config_file)

        # Environment wins
        assert config._config.profile == "env_kb/highest_precedence"
        assert config.get_source("profile") == "environment"

        # File overrides defaults (where no environment var is set)
        assert config._config.log_level == sample_config_dict["log_level"]
        assert config.get_source("log_level") == f"file:{config_file.name}"

        # Test shows that file values override defaults (precedence working correctly)
        # The sample config explicitly sets firmware.flash.skip_existing = True
        assert config._config.firmware.flash.skip_existing is True
