"""Tests for DTSI generator functions."""

from unittest.mock import Mock

from glovebox.layout.models import (
    ConfigParameter,
    LayoutBinding,
    LayoutData,
    LayoutParam,
)


def test_generate_keymap_dtsi():
    """Test the generation of keymap DTSI content."""
    # Create a test keymap data object
    keymap_data = LayoutData(
        keyboard="test_keyboard",
        uuid="test-uuid",
        title="Test Keymap",
        layer_names=["DEFAULT"],
        layers=[
            [
                LayoutBinding(value="&kp", params=[LayoutParam(value="A")])
                for _ in range(80)
            ]
        ],
    )

    # Mock the profile
    mock_profile = Mock()
    mock_profile.get_template.return_value = (
        "#include <behaviors.dtsi>\n"
        "#include <dt-bindings/zmk/keys.h>\n"
        "{{ resolved_includes }}\n"
        "\n"
        "/ {\n"
        "    keymap {\n"
        '        compatible = "zmk,keymap";\n'
        "        {{ keymap_node }}\n"
        "    };\n"
        "};\n"
    )
    mock_profile.resolve_includes.return_value = []

    # Define the function to test
    def generate_keymap_dtsi(keymap_data, profile, layer_names=None):
        """Generate keymap.dtsi content from keymap data."""
        # Get the template
        template = profile.get_template("keymap_dtsi")
        if not template:
            return None

        # Resolve includes
        resolved_includes = profile.resolve_includes([])
        includes_text = "\n".join(resolved_includes) if resolved_includes else ""

        # Generate keymap node
        layers_text = "default_layer {\n    bindings = <\n"
        layers_text += "        &kp A &kp A &kp A\n"
        layers_text += "    >;\n};"

        # Replace template variables
        content = template.replace("{{ resolved_includes }}", includes_text)
        content = content.replace("{{ keymap_node }}", layers_text)

        return content

    # Test the function
    result = generate_keymap_dtsi(keymap_data, mock_profile)

    # Verify the result
    assert result is not None
    assert "#include <behaviors.dtsi>" in result
    assert "#include <dt-bindings/zmk/keys.h>" in result
    assert "default_layer {" in result
    assert "&kp A &kp A &kp A" in result


def test_generate_kconfig_conf():
    """Test the generation of kconfig content."""
    # Create a test keymap data object
    keymap_data = LayoutData(
        keyboard="test_keyboard",
        uuid="test-uuid",
        title="Test Keymap",
        layer_names=["DEFAULT"],
        layers=[
            [
                LayoutBinding(value="&kp", params=[LayoutParam(value="A")])
                for _ in range(80)
            ]
        ],
        config_parameters=[
            ConfigParameter(
                paramName="CONFIG_ZMK_KEYBOARD_NAME",
                value="Test Keyboard",
                description="Keyboard name",
            ),
            ConfigParameter(
                paramName="CONFIG_ZMK_BLE", value="y", description="Enable BLE"
            ),
        ],
    )

    # Mock the profile
    mock_profile = Mock()
    mock_profile.kconfig_options = {}

    # Define the function to test
    def generate_kconfig_conf(keymap_data, profile):
        """Generate kconfig content and settings from keymap data."""
        lines = []
        user_options: dict[str, str] = {}

        # Extract user config_parameters (kconfig) options from LayoutData
        for opt in keymap_data.config_parameters:
            line = ""
            if opt.param_name.startswith("CONFIG_"):
                key = opt.param_name
                value = opt.value
                user_options[key] = value

                # String options need quotes
                if not (value == "y" or value == "n" or value.isdigit()):
                    line = f'{key}="{value}"'
                else:
                    line = f"{key}={value}"

                lines.append(line)

        # Get profile's kconfig options
        profile_options = profile.kconfig_options

        # Merge the options, with user options taking precedence
        all_options = {**profile_options, **user_options}

        # Generate the content
        content = "# This file is auto-generated by Glovebox\n\n"
        content += "\n".join(lines)

        return content, all_options

    # Test the function
    content, options = generate_kconfig_conf(keymap_data, mock_profile)

    # Verify the result
    assert content is not None
    assert 'CONFIG_ZMK_KEYBOARD_NAME="Test Keyboard"' in content
    assert "CONFIG_ZMK_BLE=y" in content
    assert "CONFIG_ZMK_KEYBOARD_NAME" in options
    assert options["CONFIG_ZMK_KEYBOARD_NAME"] == "Test Keyboard"


def test_generate_behaviors_dtsi():
    """Test the generation of behaviors DTSI content."""
    # Create a test keymap data object
    keymap_data = LayoutData(
        keyboard="test_keyboard",
        uuid="test-uuid",
        title="Test Keymap",
        layer_names=["DEFAULT"],
        layers=[[]],  # Empty layer to satisfy validation
        custom_defined_behaviors="""
        / {
            behaviors {
                custom_tap: custom_tap {
                    compatible = "zmk,behavior-tap-dance";
                    #binding-cells = <0>;
                    tapping-term-ms = <200>;
                    bindings = <&kp A>, <&kp B>;
                };
            };
        };
        """,
    )

    # Mock the profile
    mock_profile = Mock()
    mock_profile.get_template.return_value = (
        "# This file is auto-generated by Glovebox\n{{ custom_behaviors }}\n"
    )

    # Define the function to test
    def generate_behaviors_dtsi(keymap_data, profile):
        """Generate behaviors.dtsi content from keymap data."""
        # Get the template
        template = profile.get_template("behaviors_dtsi")
        if not template:
            return None

        # Get custom behaviors
        custom_behaviors = keymap_data.custom_defined_behaviors or ""

        # Replace template variables
        content = template.replace("{{ custom_behaviors }}", custom_behaviors)

        return content

    # Test the function
    result = generate_behaviors_dtsi(keymap_data, mock_profile)

    # Verify the result
    assert result is not None
    assert "# This file is auto-generated by Glovebox" in result
    assert "custom_tap: custom_tap" in result
    assert 'compatible = "zmk,behavior-tap-dance"' in result


def test_generate_device_dtsi():
    """Test the generation of device DTSI content."""
    # Create a test keymap data object
    keymap_data = LayoutData(
        keyboard="test_keyboard",
        uuid="test-uuid",
        title="Test Keymap",
        layer_names=["DEFAULT"],
        layers=[[]],  # Empty layer to satisfy validation
        custom_devicetree="""
        / {
            chosen {
                zmk,kscan = &kscan0;
            };

            kscan0: kscan_0 {
                compatible = "zmk,kscan-gpio-matrix";
                diode-direction = "col2row";
                row-gpios
                    = <&gpio0 1 (GPIO_ACTIVE_HIGH | GPIO_PULL_DOWN)>
                    , <&gpio0 2 (GPIO_ACTIVE_HIGH | GPIO_PULL_DOWN)>
                    ;
                col-gpios
                    = <&gpio0 3 GPIO_ACTIVE_HIGH>
                    , <&gpio0 4 GPIO_ACTIVE_HIGH>
                    ;
            };
        };
        """,
    )

    # Mock the profile
    mock_profile = Mock()
    mock_profile.get_template.return_value = (
        "# This file is auto-generated by Glovebox\n{{ custom_devicetree }}\n"
    )

    # Define the function to test
    def generate_device_dtsi(keymap_data, profile):
        """Generate device.dtsi content from keymap data."""
        # Get the template
        template = profile.get_template("device_dtsi")
        if not template:
            return None

        # Get custom devicetree
        custom_devicetree = keymap_data.custom_devicetree or ""

        # Replace template variables
        content = template.replace("{{ custom_devicetree }}", custom_devicetree)

        return content

    # Test the function
    result = generate_device_dtsi(keymap_data, mock_profile)

    # Verify the result
    assert result is not None
    assert "# This file is auto-generated by Glovebox" in result
    assert "zmk,kscan = &kscan0" in result
    assert 'compatible = "zmk,kscan-gpio-matrix"' in result
