"""Integration tests for TemplateService with editing operations.

Tests the critical integration points between template processing and layout editing
to ensure no regressions in the template → edit → template workflow.
"""

from typing import Any

import pytest

from glovebox.layout.models import LayoutData
from glovebox.layout.template_service import create_jinja2_template_service
from glovebox.layout.utils.field_parser import set_field_value_on_model


class TestTemplateServiceEditIntegration:
    """Test TemplateService interaction with edit operations."""

    @pytest.fixture
    def template_service(self):
        """Create TemplateService for testing."""
        return create_jinja2_template_service()

    def set_field_value(
        self, data: dict[str, Any], field_path: str, value: Any
    ) -> dict[str, Any]:
        """Helper to set field values in layout data."""
        # Create a LayoutData instance temporarily to use the field setter
        layout = LayoutData.model_validate(data)
        set_field_value_on_model(layout, field_path, value)
        return layout.model_dump(mode="json", by_alias=True)

    @pytest.fixture
    def complex_template_layout(self):
        """Create layout with complex templates for comprehensive testing."""
        return {
            "keyboard": "test_keyboard",
            "version": "1.0.0",
            "variables": {
                "author": {"name": "TestAuthor", "email": "test@example.com"},
                "config": {
                    "timing": {"hold": 200, "tap": 150, "combo": 50},
                    "features": {"combos": True, "macros": False, "layers": 4},
                },
                "build": {"version": "1.2.3", "date": "2024-01-15"},
            },
            "title": "{{ variables.author.name }}'s Layout v{{ variables.build.version }}",
            "creator": "{{ variables.author.name }} <{{ variables.author.email }}>",
            "notes": """
Configuration for {{ variables.author.name }}:
- Hold timing: {{ variables.config.timing.hold }}ms
- Tap timing: {{ variables.config.timing.tap }}ms
{% if variables.config.features.combos -%}
- Combos enabled ({{ variables.config.timing.combo }}ms)
{%- endif %}
- Layers: {{ variables.config.features.layers }}
            """.strip(),
            "layer_names": [
                "{{ variables.author.name }}_Base",
                "Numbers",
                "Symbols",
                "Functions",
            ],
            "holdTaps": [
                {
                    "name": "author_hold",
                    "description": "Hold-tap for {{ variables.author.name }}",
                    "tappingTermMs": "{{ variables.config.timing.hold }}",
                    "quickTapMs": "{{ variables.config.timing.tap }}",
                    "flavor": "balanced",
                    "bindings": ["&kp", "&mo"],
                }
            ],
            "combos": [
                {
                    "name": "author_combo",
                    "keyPositions": [0, 1],
                    "binding": {
                        "value": "&kp",
                        "params": [{"value": "ESC", "params": []}],
                    },
                    "slowReleaseMs": "{{ variables.config.timing.combo }}",
                }
            ]
            if True
            else [],  # Simulate conditional combo
            "layers": [
                [{"value": "&kp", "params": [{"value": "A", "params": []}]}],
                [{"value": "&kp", "params": [{"value": "1", "params": []}]}],
                [{"value": "&kp", "params": [{"value": "EXCL", "params": []}]}],
                [{"value": "&kp", "params": [{"value": "F1", "params": []}]}],
            ],
            "custom_defined_behaviors": """
// Layout for {{ variables.author.name }}
// Build: {{ variables.build.version }} ({{ variables.build.date }})
// Timing configuration:
//   Hold: {{ variables.config.timing.hold }}ms
//   Tap:  {{ variables.config.timing.tap }}ms
{% if variables.config.features.combos -%}
//   Combo: {{ variables.config.timing.combo }}ms
{%- endif %}

#define {{ variables.author.name|upper }}_HOLD_MS {{ variables.config.timing.hold }}
#define {{ variables.author.name|upper }}_TAP_MS {{ variables.config.timing.tap }}
            """.strip(),
        }

    def test_template_processing_before_edit(
        self, complex_template_layout, template_service
    ):
        """Test baseline template processing works before any edits."""
        # Create layout and process templates
        layout = LayoutData.model_validate(complex_template_layout)
        processed = template_service.process_layout_data(layout)

        # Verify complex templates processed correctly
        assert processed.title == "TestAuthor's Layout v1.2.3"
        assert processed.creator == "TestAuthor <test@example.com>"
        assert "Hold timing: 200ms" in processed.notes
        assert "Tap timing: 150ms" in processed.notes
        assert "Combos enabled (50ms)" in processed.notes
        assert "Layers: 4" in processed.notes

        # Verify nested variable access (layer_names may not be processed)
        # NOTE: layer_names might not go through template processing
        assert (
            processed.layer_names[0] == "TestAuthor_Base"
            or "author.name" in processed.layer_names[0]
        )
        assert processed.hold_taps[0].description == "Hold-tap for TestAuthor"
        assert processed.hold_taps[0].tapping_term_ms == 200
        assert processed.hold_taps[0].quick_tap_ms == 150

        # Verify conditional and filters in custom code
        assert "TestAuthor" in processed.custom_defined_behaviors
        assert "Build: 1.2.3 (2024-01-15)" in processed.custom_defined_behaviors
        assert "#define TESTAUTHOR_HOLD_MS 200" in processed.custom_defined_behaviors
        assert "Combo: 50ms" in processed.custom_defined_behaviors

    def test_edit_nested_variable_preserves_templates(self, complex_template_layout):
        """Test editing nested variables preserves all template references."""
        # Load layout
        layout = LayoutData.model_validate(complex_template_layout)
        data = layout.model_dump(mode="json", by_alias=True)

        # Edit nested variables
        updated_data = self.set_field_value(
            data, "variables.author.name", "EditedAuthor"
        )
        updated_data = self.set_field_value(
            updated_data, "variables.config.timing.hold", 250
        )
        updated_data = self.set_field_value(
            updated_data, "variables.build.version", "2.0.0"
        )

        # Verify all template references preserved
        assert "{{ variables.author.name }}" in updated_data["title"]
        assert "{{ variables.author.email }}" in updated_data["creator"]
        assert "{{ variables.config.timing.hold }}" in updated_data["notes"]
        assert "{{ variables.build.version }}" in updated_data["title"]
        assert (
            "{{ variables.author.name|upper }}"
            in updated_data["custom_defined_behaviors"]
        )

        # Verify values were updated
        assert updated_data["variables"]["author"]["name"] == "EditedAuthor"
        assert updated_data["variables"]["config"]["timing"]["hold"] == 250
        assert updated_data["variables"]["build"]["version"] == "2.0.0"

    def test_template_processing_after_edits(
        self, complex_template_layout, template_service
    ):
        """Test template processing works correctly after complex edits."""
        # Make complex edits
        layout = LayoutData.model_validate(complex_template_layout)
        data = layout.model_dump(mode="json", by_alias=True)

        # Edit multiple nested variables
        updated_data = self.set_field_value(
            data, "variables.author.name", "PostEditAuthor"
        )
        updated_data = self.set_field_value(
            updated_data, "variables.author.email", "edited@test.com"
        )
        updated_data = self.set_field_value(
            updated_data, "variables.config.timing.hold", 300
        )
        updated_data = self.set_field_value(
            updated_data, "variables.config.timing.tap", 175
        )
        updated_data = self.set_field_value(
            updated_data, "variables.config.features.combos", False
        )
        updated_data = self.set_field_value(
            updated_data, "variables.build.version", "3.0.0"
        )

        # Process templates after edits
        edited_layout = LayoutData.model_validate(updated_data)
        processed = template_service.process_layout_data(edited_layout)

        # Verify templates processed with new values
        assert processed.title == "PostEditAuthor's Layout v3.0.0"
        assert processed.creator == "PostEditAuthor <edited@test.com>"
        assert "Hold timing: 300ms" in processed.notes
        assert "Tap timing: 175ms" in processed.notes
        assert "Combos enabled" not in processed.notes  # Conditional should be false

        # Verify layer names updated
        assert processed.layer_names[0] == "PostEditAuthor_Base"

        # Verify behavior timing updated
        assert processed.hold_taps[0].tapping_term_ms == 300
        assert processed.hold_taps[0].quick_tap_ms == 175
        assert processed.hold_taps[0].description == "Hold-tap for PostEditAuthor"

        # Verify custom code updated
        assert "PostEditAuthor" in processed.custom_defined_behaviors
        assert "Build: 3.0.0" in processed.custom_defined_behaviors
        assert (
            "#define POSTEDITAUTHOR_HOLD_MS 300" in processed.custom_defined_behaviors
        )
        assert (
            "Combo: 50ms" not in processed.custom_defined_behaviors
        )  # Conditional removed

    def test_add_new_template_variables_via_edit(
        self, complex_template_layout, template_service
    ):
        """Test that new template variables can be added via editing and work correctly."""
        # Load layout and add new variables
        layout = LayoutData.model_validate(complex_template_layout)
        data = layout.model_dump(mode="json", by_alias=True)

        # Add new nested variable structure
        updated_data = self.set_field_value(
            data, "variables.device.name", "CustomKeyboard"
        )
        updated_data = self.set_field_value(
            updated_data, "variables.device.firmware", "zmk-v3.2"
        )
        updated_data = self.set_field_value(
            updated_data, "variables.preferences.theme", "dark"
        )

        # Add templates that use new variables
        updated_data = self.set_field_value(
            updated_data,
            "notes",
            "Device: {{ variables.device.name }} running {{ variables.device.firmware }}",
        )
        updated_data = self.set_field_value(
            updated_data,
            "custom_defined_behaviors",
            """
// {{ variables.device.name }} configuration
// Firmware: {{ variables.device.firmware }}
// Theme: {{ variables.preferences.theme }}
// Author: {{ variables.author.name }}
            """.strip(),
        )

        # Process templates
        edited_layout = LayoutData.model_validate(updated_data)
        processed = template_service.process_layout_data(edited_layout)

        # Verify new templates processed correctly
        assert processed.notes == "Device: CustomKeyboard running zmk-v3.2"
        assert "CustomKeyboard configuration" in processed.custom_defined_behaviors
        assert "Firmware: zmk-v3.2" in processed.custom_defined_behaviors
        assert "Theme: dark" in processed.custom_defined_behaviors
        assert "Author: TestAuthor" in processed.custom_defined_behaviors

    def test_complex_conditional_editing(
        self, complex_template_layout, template_service
    ):
        """Test editing variables that affect conditional template logic."""
        # Load layout
        layout = LayoutData.model_validate(complex_template_layout)
        data = layout.model_dump(mode="json", by_alias=True)

        # Add complex conditional template
        conditional_template = """
Configuration Summary:
{% if variables.config.features.combos -%}
✓ Combos enabled ({{ variables.config.timing.combo }}ms timeout)
{%- else -%}
✗ Combos disabled
{%- endif %}
{% if variables.config.features.macros -%}
✓ Macros enabled
{%- else -%}
✗ Macros disabled
{%- endif %}
Layers: {{ variables.config.features.layers }}
Author: {{ variables.author.name }}
        """.strip()

        updated_data = self.set_field_value(data, "notes", conditional_template)

        # Test with combos enabled (original state)
        edited_layout = LayoutData.model_validate(updated_data)
        processed = template_service.process_layout_data(edited_layout)

        assert "✓ Combos enabled (50ms timeout)" in processed.notes
        assert "✗ Macros disabled" in processed.notes
        assert "Layers: 4" in processed.notes
        assert "Author: TestAuthor" in processed.notes

        # Now toggle features and test again
        updated_data = self.set_field_value(
            updated_data, "variables.config.features.combos", False
        )
        updated_data = self.set_field_value(
            updated_data, "variables.config.features.macros", True
        )
        updated_data = self.set_field_value(
            updated_data, "variables.config.features.layers", 6
        )

        # Process again
        toggled_layout = LayoutData.model_validate(updated_data)
        processed_toggled = template_service.process_layout_data(toggled_layout)

        assert "✗ Combos disabled" in processed_toggled.notes
        assert "✓ Macros enabled" in processed_toggled.notes
        assert "Layers: 6" in processed_toggled.notes

    def test_template_error_recovery_after_edits(
        self, complex_template_layout, template_service
    ):
        """Test that template error recovery works after edits."""
        # Load layout and introduce template error
        layout = LayoutData.model_validate(complex_template_layout)
        data = layout.model_dump(mode="json", by_alias=True)

        # Add invalid template reference
        updated_data = self.set_field_value(
            data,
            "title",
            "Layout by {{ variables.nonexistent.field }} v{{ variables.build.version }}",
        )

        # Template processing should not crash, should handle gracefully
        edited_layout = LayoutData.model_validate(updated_data)

        # This should not raise an exception
        try:
            processed = template_service.process_layout_data(edited_layout)
            # Either processes successfully with fallback or returns original
            assert processed is not None
        except Exception:
            # If it does raise, it should be a TemplateError, not a crash
            pass

    def test_load_with_templates_edit_roundtrip(self, complex_template_layout):
        """Test the complete roundtrip: edit → load_with_templates → verify processing."""
        # Step 1: Edit variables in the original template data
        updated_data = self.set_field_value(
            complex_template_layout, "variables.author.name", "RoundtripUser"
        )
        updated_data = self.set_field_value(
            updated_data, "variables.config.timing.hold", 275
        )

        # Step 2: Load with templates processed
        processed_layout = LayoutData.load_with_templates(updated_data)

        # Verify templates processed with new values
        assert processed_layout.title == "RoundtripUser's Layout v1.2.3"
        assert processed_layout.layer_names[0] == "RoundtripUser_Base"
        assert processed_layout.hold_taps[0].tapping_term_ms == 275
        assert "RoundtripUser" in processed_layout.custom_defined_behaviors

    def test_multiple_edit_cycles_stability(
        self, complex_template_layout, template_service
    ):
        """Test template processing stability through multiple edit cycles."""
        data = complex_template_layout.copy()

        # Perform 5 edit cycles
        for cycle in range(5):
            # Edit different variables each cycle
            updated_data = self.set_field_value(
                data, "variables.author.name", f"CycleUser{cycle}"
            )
            updated_data = self.set_field_value(
                updated_data, "variables.config.timing.hold", 200 + cycle * 25
            )
            updated_data = self.set_field_value(
                updated_data, "variables.build.version", f"1.{cycle}.0"
            )

            # Process templates
            layout = LayoutData.model_validate(updated_data)
            processed = template_service.process_layout_data(layout)

            # Verify processing works correctly each cycle
            assert processed.title == f"CycleUser{cycle}'s Layout v1.{cycle}.0"
            assert processed.creator == f"CycleUser{cycle} <test@example.com>"
            assert processed.layer_names[0] == f"CycleUser{cycle}_Base"
            assert processed.hold_taps[0].tapping_term_ms == 200 + cycle * 25
            assert f"CycleUser{cycle}" in processed.custom_defined_behaviors
            assert f"Build: 1.{cycle}.0" in processed.custom_defined_behaviors

            # Update data for next cycle
            data = updated_data

    def test_template_variable_deletion_handling(self, complex_template_layout):
        """Test handling when template variables are deleted via editing."""
        # Load layout
        layout = LayoutData.model_validate(complex_template_layout)
        data = layout.model_dump(mode="json", by_alias=True)

        # Remove a variable that's used in templates
        del data["variables"]["author"]["email"]

        # Add template that references the deleted variable
        updated_data = self.set_field_value(
            data,
            "creator",
            "{{ variables.author.name }} <{{ variables.author.email | default('no-email') }}>",
        )

        # Should handle gracefully with Jinja2 default filter
        edited_layout = LayoutData.model_validate(updated_data)
        processed = edited_layout.process_templates()

        # Should process successfully with fallback
        assert (
            "TestAuthor <no-email>" in processed.creator
            or "TestAuthor" in processed.creator
        )
