"""Comprehensive tests for processor compatibility and chunk tracking."""

import datetime as _datetime
import io
import shutil
import tempfile
from datetime import datetime
from pathlib import Path
from unittest.mock import Mock, patch

import pyarrow as pa
import pyarrow.parquet as pq
import pytest
from PIL import Image

# Import pytest-asyncio
pytest_plugins = ("pytest_asyncio",)
import pytest_asyncio
from caption_flow.models import Caption, JobId
from caption_flow.processors import ProcessorConfig, WorkResult, WorkUnit
from caption_flow.processors.huggingface import (
    HuggingFaceDatasetOrchestratorProcessor,
    HuggingFaceDatasetWorkerProcessor,
)
from caption_flow.processors.local_filesystem import (
    LocalFilesystemOrchestratorProcessor,
    LocalFilesystemWorkerProcessor,
)

# Import processor implementations
from caption_flow.processors.webdataset import (
    WebDatasetOrchestratorProcessor,
    WebDatasetWorkerProcessor,
)

# Import the modules to test
from caption_flow.storage import StorageManager
from caption_flow.utils.chunk_tracker import ChunkTracker


class ProcessorTestBase:
    """Base class for processor compatibility tests."""

    @pytest.fixture
    def temp_dir(self):
        """Create a temporary directory for testing."""
        temp_dir = tempfile.mkdtemp()
        yield Path(temp_dir)
        # More robust cleanup
        try:
            shutil.rmtree(temp_dir, ignore_errors=True)
        except Exception:
            pass

    @pytest_asyncio.fixture
    async def storage_manager(self, temp_dir):
        """Create a StorageManager instance."""
        storage = StorageManager(temp_dir, caption_buffer_size=5)
        await storage.initialize()
        return storage

    @pytest.fixture
    def chunk_tracker(self, temp_dir):
        """Create a ChunkTracker instance."""
        return ChunkTracker(temp_dir / "chunks.json")

    def create_mock_caption(self, shard_id: str, chunk_id: str, sample_id: str, item_index: int):
        """Helper to create a mock caption."""
        job_id = JobId(shard_id=shard_id, chunk_id=chunk_id, sample_id=sample_id)
        return Caption(
            job_id=job_id,
            dataset="test_dataset",
            shard=shard_id,
            chunk_id=chunk_id,
            item_key=f"item_{sample_id}",
            caption=f"test caption {sample_id}",
            outputs={"captions": [f"test caption {sample_id}"]},
            contributor_id="test_worker",
            timestamp=datetime.now(_datetime.UTC),
            caption_count=1,
            metadata={"_item_index": item_index},
        )


# ============= WebDataset Processor Tests =============


class TestWebDatasetProcessors(ProcessorTestBase):
    """Test suite for WebDataset processor compatibility."""

    @pytest.fixture
    def orchestrator_config(self, temp_dir):
        """Create test orchestrator configuration."""
        return ProcessorConfig(
            processor_type="webdataset",
            config={
                "dataset": {
                    "dataset_path": "s3://test-bucket/dataset",
                    "metadata_path": None,
                },
                "chunk_size": 100,
                "checkpoint_dir": str(temp_dir / "checkpoints"),
                "cache_dir": str(temp_dir / "cache"),
                "shard_cache_gb": 1.0,
            },
        )

    @pytest.fixture
    def worker_config(self):
        """Create test worker configuration."""
        return ProcessorConfig(
            processor_type="webdataset",
            config={
                "dataset": {
                    "dataset_path": "s3://test-bucket/dataset",
                    "mock_results": True,  # Enable mock mode for testing
                },
                "cache_dir": "./test_cache",
                "buffer_size": 5,
            },
        )

    @pytest.fixture
    def mock_webshart_dataset(self):
        """Mock webshart dataset for testing."""
        mock_dataset = Mock()
        mock_dataset.num_shards = 2
        mock_dataset.name = "test_dataset"
        mock_dataset.dataset_format = "webdataset"

        # Mock get_shard_info
        def get_shard_info(idx):
            return {
                "name": f"shard_{idx}",
                "path": f"s3://test-bucket/shard_{idx}.tar",
                "num_files": 200,
            }

        mock_dataset.get_shard_info = Mock(side_effect=get_shard_info)
        mock_dataset.get_stats = Mock(
            return_value={
                "total_shards": 2,
                "total_files": 400,
            }
        )
        mock_dataset.enable_metadata_cache = Mock()
        mock_dataset.enable_shard_cache = Mock()

        return mock_dataset

    def test_orchestrator_initialization(
        self, orchestrator_config, storage_manager, mock_webshart_dataset
    ):
        """Test WebDataset orchestrator initialization."""
        with patch("webshart.discover_dataset") as mock_discover:
            mock_discover.return_value = mock_webshart_dataset

            orchestrator = WebDatasetOrchestratorProcessor()
            orchestrator.initialize(orchestrator_config, storage_manager)

            assert orchestrator.dataset is not None
            assert orchestrator.chunk_tracker is not None
            assert orchestrator.chunk_size == 100
            assert orchestrator.dataset.num_shards == 2

    def test_worker_initialization(self, worker_config):
        """Test WebDataset worker initialization."""
        worker = WebDatasetWorkerProcessor()
        worker.gpu_id = 0

        # Initialize in mock mode
        worker.initialize(worker_config)

        assert worker.mock_results is True
        # In mock mode, dataset and loader remain None
        assert worker.dataset is None
        assert worker.loader is None

    def test_work_unit_creation_and_assignment(
        self, orchestrator_config, storage_manager, mock_webshart_dataset
    ):
        """Test work unit creation and assignment flow."""
        with patch("webshart.discover_dataset") as mock_discover:
            mock_discover.return_value = mock_webshart_dataset

            orchestrator = WebDatasetOrchestratorProcessor()
            orchestrator.initialize(orchestrator_config, storage_manager)

            # Let background thread create some units
            import time

            time.sleep(0.1)

            # Request work units
            units = orchestrator.get_work_units(2, "worker1")

            assert len(units) <= 2
            for unit in units:
                assert unit.unit_id.startswith("shard_")
                assert unit.source_id.startswith("shard_")
                assert "unprocessed_ranges" in unit.data
                assert unit.unit_size > 0

    def test_chunk_tracking_integration(
        self, orchestrator_config, storage_manager, mock_webshart_dataset, chunk_tracker
    ):
        """Test chunk tracker integration with WebDataset."""
        with patch("webshart.discover_dataset") as mock_discover:
            mock_discover.return_value = mock_webshart_dataset

            orchestrator = WebDatasetOrchestratorProcessor()
            # Don't override chunk_tracker - let it create its own
            orchestrator.initialize(orchestrator_config, storage_manager)

            # Wait for background thread to create units
            import time

            time.sleep(0.2)

            # Create and assign a work unit
            units = orchestrator.get_work_units(1, "worker1")
            if units:
                unit = units[0]

                # Simulate processing some items
                result = WorkResult(
                    unit_id=unit.unit_id,
                    source_id=unit.source_id,
                    chunk_id=unit.chunk_id,
                    sample_id="0",
                    outputs={"captions": ["test caption"]},
                    metadata={"_item_index": 0},
                )

                orchestrator.handle_result(result)

                # Check that the orchestrator's chunk tracker was updated
                chunk_state = orchestrator.chunk_tracker.chunks.get(unit.chunk_id)
                assert chunk_state is not None
                assert chunk_state.status == "assigned"

    def test_storage_synchronization(
        self, orchestrator_config, storage_manager, mock_webshart_dataset, temp_dir
    ):
        """Test synchronization between storage and chunk tracker."""
        with patch("webshart.discover_dataset") as mock_discover:
            mock_discover.return_value = mock_webshart_dataset

            orchestrator = WebDatasetOrchestratorProcessor()

            # Pre-populate storage with some processed items
            processed_job_ids = set()
            for i in range(50, 150, 10):
                job_id = JobId(
                    shard_id="shard_0", chunk_id="0" if i < 100 else "1", sample_id=str(i)
                )
                processed_job_ids.add(str(job_id))  # Use str() instead of to_str()

            storage_manager.get_all_processed_job_ids = Mock(return_value=processed_job_ids)

            orchestrator.initialize(orchestrator_config, storage_manager)

            # Check that processed items were recognized
            assert orchestrator.chunk_tracker is not None
            # Verify some chunks exist (exact behavior depends on implementation)

    def test_worker_processing_mock_mode(self, worker_config):
        """Test worker processing in mock mode."""
        worker = WebDatasetWorkerProcessor()
        worker.gpu_id = 0

        # Initialize in mock mode
        worker.initialize(worker_config)

        # Create a work unit with all required data fields
        unit = WorkUnit(
            unit_id="shard_0:chunk:0",
            chunk_id="shard_0:chunk:0",
            source_id="shard_0",
            unit_size=10,
            data={
                "shard_name": "shard_0",
                "shard_idx": 0,
                "start_index": 0,  # Required field
                "chunk_size": 10,  # Required field
                "unprocessed_ranges": [(0, 9)],
            },
            metadata={"chunk_index": 0},
        )

        # Process unit
        context = {}
        items = list(worker.process_unit(unit, context))

        assert len(items) == 10
        for idx, item in enumerate(items):
            assert item["image"] is not None
            assert item["job_id"] == f"shard_0:chunk:0:idx:{idx}"
            assert item["metadata"]["_mock"] is True


# ============= HuggingFace Dataset Processor Tests =============


class TestHuggingFaceDatasetProcessors(ProcessorTestBase):
    """Test suite for HuggingFace dataset processor compatibility."""

    @pytest.fixture
    def orchestrator_config(self, temp_dir):
        """Create test orchestrator configuration."""
        return ProcessorConfig(
            processor_type="huggingface",
            config={
                "dataset": {
                    "dataset_path": "test_dataset",
                    "dataset_config": "default",
                    "dataset_split": "train",
                },
                "chunk_size": 100,
                "checkpoint_dir": str(temp_dir / "checkpoints"),
                "min_chunk_buffer": 5,
                "chunk_buffer_multiplier": 2,
            },
        )

    @pytest.fixture
    def worker_config(self):
        """Create test worker configuration."""
        return ProcessorConfig(
            processor_type="huggingface",
            config={
                "dataset": {
                    "dataset_path": "test/dataset",
                    "dataset_config": "default",
                    "dataset_split": "train",
                    "dataset_image_column": "image",
                    "mock_results": True,
                },
            },
        )

    @pytest.fixture
    def mock_hf_dataset_builder(self):
        """Mock HuggingFace dataset builder."""
        mock_builder = Mock()
        mock_builder.config.data_files = {"train": ["file1.parquet", "file2.parquet"]}
        return mock_builder

    def test_orchestrator_initialization_with_auto_detect(
        self, orchestrator_config, storage_manager
    ):
        """Test HF orchestrator initialization with config auto-detection."""
        with patch("datasets.get_dataset_config_names") as mock_configs:
            with patch("datasets.get_dataset_split_names") as mock_splits:
                with patch("datasets.load_dataset_builder") as mock_builder:
                    mock_configs.return_value = ["default", "en", "fr"]
                    mock_splits.return_value = ["train", "validation", "test"]

                    mock_builder_instance = Mock()
                    mock_builder_instance.config.data_files = {"train": ["data.parquet"]}
                    mock_builder.return_value = mock_builder_instance

                    orchestrator = HuggingFaceDatasetOrchestratorProcessor()
                    orchestrator.initialize(orchestrator_config, storage_manager)

                    assert orchestrator.dataset_name == "test_dataset"
                    assert orchestrator.config == "default"
                    assert orchestrator.split == "train"
                    assert orchestrator.chunk_tracker is not None

    def test_work_unit_dynamic_creation(self, orchestrator_config, storage_manager):
        """Test dynamic work unit creation based on demand."""
        with patch(
            "caption_flow.processors.huggingface.HuggingFaceDatasetOrchestratorProcessor._get_data_files_from_builder"
        ) as mock_get_files:
            with patch("huggingface_hub.hf_hub_download") as mock_download:
                with patch("pyarrow.parquet.read_metadata") as mock_metadata:
                    mock_get_files.return_value = ["data.parquet"]
                    mock_download.return_value = "/tmp/data.parquet"
                    mock_meta = Mock()
                    mock_meta.num_rows = 500
                    mock_metadata.return_value = mock_meta

                    orchestrator = HuggingFaceDatasetOrchestratorProcessor()
                    orchestrator.initialize(orchestrator_config, storage_manager)

                    # Let background thread run
                    import time

                    time.sleep(0.2)

                    # Request units from multiple workers
                    units1 = orchestrator.get_work_units(3, "worker1")
                    units2 = orchestrator.get_work_units(2, "worker2")

                    all_unit_ids = [u.unit_id for u in units1 + units2]
                    assert len(set(all_unit_ids)) == len(all_unit_ids)  # No duplicates

    def test_shard_discovery_and_caching(self, orchestrator_config, storage_manager, temp_dir):
        """Test shard discovery and caching mechanism."""
        # Create a dummy parquet file first
        dummy_parquet = temp_dir / "data.parquet"

        # Create a simple parquet file with PyArrow

        # Create a dummy table
        table = pa.table({"dummy": pa.array([1] * 1000)})
        pq.write_table(table, dummy_parquet)

        with patch(
            "caption_flow.processors.huggingface.HuggingFaceDatasetOrchestratorProcessor._get_data_files_from_builder"
        ) as mock_get_files:
            with patch("caption_flow.processors.huggingface.hf_hub_download") as mock_download:
                # Setup mocks
                mock_get_files.return_value = ["data-00000.parquet"]
                mock_download.return_value = str(dummy_parquet)

                orchestrator = HuggingFaceDatasetOrchestratorProcessor()
                orchestrator.initialize(orchestrator_config, storage_manager)

                assert orchestrator.total_items == 1000
                assert len(orchestrator.shard_info) == 1
                assert 0 in orchestrator.shard_info

                # Check cache file was created
                cache_files = list(temp_dir.glob("**/checkpoints/*_shard_info.json"))
                assert len(cache_files) > 0

    # For test_worker_processing_with_ranges, replace with:
    def test_worker_processing_with_ranges(self, worker_config, temp_dir):
        """Test worker processing with specific unprocessed ranges."""
        worker = HuggingFaceDatasetWorkerProcessor()
        worker.gpu_id = 0

        # Create a dummy parquet file
        dummy_parquet = temp_dir / "data.parquet"

        # Create a parquet file with test data

        # Create test data with image bytes
        data = []
        for i in range(100):
            # Create a tiny test image
            import io

            from PIL import Image

            img = Image.new("RGB", (10, 10), color=(i, i, i))
            img_bytes = io.BytesIO()
            img.save(img_bytes, format="PNG")
            data.append({"image": {"bytes": img_bytes.getvalue()}, "idx": i})

        table = pa.Table.from_pylist(data)
        pq.write_table(table, dummy_parquet)

        # Patch hf_hub_download to return our dummy file
        with patch(
            "caption_flow.processors.huggingface.hf_hub_download", return_value=str(dummy_parquet)
        ):
            worker.initialize(worker_config)

            # Create work unit with specific ranges
            unit = WorkUnit(
                unit_id="data:chunk:0",
                chunk_id="data:chunk:0",
                source_id="data",
                unit_size=100,
                data={
                    "dataset_name": "test/dataset",
                    "config": "default",
                    "split": "train",
                    "start_index": 0,
                    "chunk_size": 100,
                    "unprocessed_ranges": [(10, 19), (50, 59)],  # Only process 20 items
                    "shard_ids": [0],
                    "data_files": ["data.parquet"],
                },
                metadata={"chunk_index": 0, "shard_name": "data"},
            )

            context = {}
            items = list(worker.process_unit(unit, context))

            # Should only process items in the unprocessed ranges
            assert len(items) == 20

            # Check processed indices
            processed_indices = context.get("_processed_indices", [])
            assert len(processed_indices) == 20

    @pytest.mark.asyncio
    async def test_storage_update_flow(self, orchestrator_config, storage_manager, temp_dir):
        """Test updating chunk tracker from storage."""
        with patch(
            "caption_flow.processors.huggingface.HuggingFaceDatasetOrchestratorProcessor._get_data_files_from_builder"
        ) as mock_get_files:
            with patch("huggingface_hub.hf_hub_download") as mock_download:
                with patch("pyarrow.parquet.read_metadata") as mock_metadata:
                    mock_get_files.return_value = ["data.parquet"]
                    mock_download.return_value = "/tmp/data.parquet"
                    mock_meta = Mock()
                    mock_meta.num_rows = 200
                    mock_metadata.return_value = mock_meta

                    # Add some captions to storage
                    for i in range(0, 50, 5):
                        caption = self.create_mock_caption("data", "0", str(i), i)
                        await storage_manager.save_caption(caption)

                    await storage_manager.checkpoint()

                    orchestrator = HuggingFaceDatasetOrchestratorProcessor()
                    orchestrator.initialize(orchestrator_config, storage_manager)

                    # Verify chunk tracker was updated
                    if orchestrator.chunk_tracker:
                        # Check that some processing was recorded
                        stats = orchestrator.chunk_tracker.get_stats()
                        assert stats["total_in_memory"] >= 0


# ============= Local Filesystem Processor Tests =============


class TestLocalFilesystemProcessors(ProcessorTestBase):
    """Test suite for Local Filesystem processor compatibility."""

    @pytest.fixture
    def test_images_dir(self, temp_dir):
        """Create a directory with test images."""
        images_dir = temp_dir / "images"
        images_dir.mkdir()

        # Create subdirectories
        (images_dir / "subdir1").mkdir()
        (images_dir / "subdir2").mkdir()

        # Create test images
        for i in range(5):
            img = Image.new("RGB", (100, 100), color=(i * 50, i * 50, i * 50))
            img.save(images_dir / f"image_{i}.jpg")

        for i in range(3):
            img = Image.new("RGB", (100, 100), color=(100, i * 80, i * 80))
            img.save(images_dir / "subdir1" / f"sub_image_{i}.png")

        # Create non-image files (should be ignored)
        (images_dir / "readme.txt").write_text("Test dataset")

        return images_dir

    @pytest.fixture
    def orchestrator_config(self, test_images_dir, temp_dir):
        """Create test orchestrator configuration."""
        return ProcessorConfig(
            processor_type="local_filesystem",
            config={
                "dataset": {
                    "dataset_path": str(test_images_dir),
                    "recursive": True,
                    "follow_symlinks": False,
                    "http_bind_address": "127.0.0.1",
                    "http_port": 8767,
                    "public_address": "localhost",
                },
                "chunk_size": 3,
                "checkpoint_dir": str(temp_dir / "checkpoints"),
            },
        )

    @pytest.fixture
    def worker_config(self, test_images_dir):
        """Create test worker configuration."""
        return ProcessorConfig(
            processor_type="local_filesystem",
            config={
                "dataset": {
                    "dataset_path": str(test_images_dir),
                },
                "worker": {
                    "local_storage_path": str(test_images_dir),  # Simulate local access
                },
            },
        )

    def test_orchestrator_image_discovery(self, orchestrator_config, storage_manager):
        """Test local filesystem image discovery."""
        orchestrator = LocalFilesystemOrchestratorProcessor()

        # Mock HTTP server start
        orchestrator._start_http_server = Mock()

        orchestrator.initialize(orchestrator_config, storage_manager)

        # Should find 8 images (5 in root + 3 in subdir1)
        assert orchestrator.total_images == 8
        assert len(orchestrator.all_images) == 8

        # Check sorting
        image_names = [img[0].name for img in orchestrator.all_images]
        assert image_names == sorted(image_names)

    def test_http_server_integration(self, orchestrator_config, storage_manager):
        """Test HTTP server setup for image serving."""
        orchestrator = LocalFilesystemOrchestratorProcessor()

        with patch("uvicorn.Config"):
            with patch("uvicorn.Server"):
                with patch(
                    "caption_flow.processors.local_filesystem.threading.Thread"
                ) as mock_thread:
                    # Mock the threading to prevent actual thread creation
                    mock_thread.return_value.start = Mock()

                    orchestrator.initialize(orchestrator_config, storage_manager)

                    # Check HTTP app was created
                    assert orchestrator.http_app is not None

                    # Check server config
                    assert orchestrator.http_bind_address == "127.0.0.1"
                    assert orchestrator.http_port == 8767

    def test_chunk_creation_with_ranges(self, orchestrator_config, storage_manager):
        """Test chunk creation with proper range management."""
        orchestrator = LocalFilesystemOrchestratorProcessor()
        orchestrator._start_http_server = Mock()
        orchestrator.initialize(orchestrator_config, storage_manager)

        # Let background thread create units
        import time

        time.sleep(0.1)

        # Get work units
        units = orchestrator.get_work_units(3, "worker1")

        assert len(units) > 0
        for unit in units:
            assert "start_index" in unit.data
            assert "chunk_size" in unit.data
            assert "unprocessed_ranges" in unit.data
            assert "http_url" in unit.data
            assert "filenames" in unit.data

    def test_worker_local_vs_http_access(self, worker_config):
        """Test worker with local storage access vs HTTP fallback."""
        worker = LocalFilesystemWorkerProcessor()
        worker.gpu_id = 0
        worker.initialize(worker_config)

        # Check local access was detected
        assert worker.dataset_path is not None

        # Create work unit
        unit = WorkUnit(
            unit_id="local:chunk:0",
            chunk_id="local:chunk:0",
            source_id="local",
            unit_size=3,
            data={
                "start_index": 0,
                "chunk_size": 3,
                "unprocessed_ranges": [(0, 2)],
                "http_url": "http://localhost:8767",
                "filenames": {0: "image_0.jpg", 1: "image_1.jpg", 2: "image_2.jpg"},
            },
            metadata={"chunk_index": 0},
        )

        # Test HTTP fallback
        with patch("requests.get") as mock_get:
            mock_response = Mock()
            mock_response.content = io.BytesIO(b"fake_image_data").read()
            mock_response.raise_for_status = Mock()
            mock_get.return_value = mock_response

            # Remove local access to test HTTP
            worker.dataset_path = None

            context = {}
            items = list(worker.process_unit(unit, context))

            # Should have made HTTP requests
            assert mock_get.called
            assert len(items) <= 3

    def test_chunk_tracker_persistence(self, orchestrator_config, storage_manager, temp_dir):
        """Test chunk tracker persistence and restoration."""
        # Create first orchestrator instance
        orch1 = LocalFilesystemOrchestratorProcessor()
        orch1._start_http_server = Mock()
        orch1.initialize(orchestrator_config, storage_manager)

        # Wait for unit creation
        import time

        time.sleep(0.2)

        # Get and complete some units
        units = orch1.get_work_units(1, "worker1")
        if units:
            unit = units[0]
            orch1.mark_completed(unit.unit_id, "worker1")

        # Save checkpoint
        if orch1.chunk_tracker:
            orch1.chunk_tracker.save()
            saved_path = orch1.chunk_tracker.checkpoint_path

            # Create second orchestrator instance with new chunk tracker
            orch2 = LocalFilesystemOrchestratorProcessor()
            orch2._start_http_server = Mock()

            # Load from same checkpoint file
            orch2.chunk_tracker = ChunkTracker(saved_path)

            # Should have restored chunks
            assert len(orch2.chunk_tracker.chunks) > 0

            # If unit was completed, check its status
            if units:
                chunk_state = orch2.chunk_tracker.chunks.get(unit.unit_id)
                if chunk_state:
                    assert chunk_state.status == "completed"


# ============= Cross-Processor Compatibility Tests =============


class TestCrossProcessorCompatibility(ProcessorTestBase):
    """Test compatibility across all processor types."""

    @pytest.mark.parametrize("processor_type", ["webdataset", "huggingface", "local_filesystem"])
    def test_job_id_consistency(self, processor_type):
        """Test that all processors generate consistent JobId formats."""
        # Test job ID parsing for each processor type
        test_cases = {
            "webdataset": "shard_0:chunk:0:idx:42",
            "huggingface": "data:chunk:0:idx:42",
            "local_filesystem": "local:chunk:0:idx:42",
        }

        job_id_str = test_cases[processor_type]
        job_id = JobId.from_str(job_id_str)

        assert job_id.get_sample_str() == job_id_str
        assert job_id.get_chunk_str() in job_id_str

    @pytest.mark.asyncio
    async def test_storage_compatibility(self, temp_dir):
        """Test that all processors work with the same storage format."""
        storage = StorageManager(temp_dir, caption_buffer_size=5)
        await storage.initialize()

        # Create captions from different processor types
        processor_shards = {
            "webdataset": "shard_0",
            "huggingface": "data",
            "local_filesystem": "local",
        }

        for _proc_type, shard_id in processor_shards.items():
            for i in range(3):
                caption = self.create_mock_caption(shard_id, "0", str(i), i)
                await storage.save_caption(caption)

        await storage.checkpoint()

        # Verify all captions were saved
        stats = await storage.get_storage_stats()
        assert stats["total_rows"] >= 9

    def test_chunk_tracker_format_compatibility(self, temp_dir):
        """Test chunk tracker works with all processor chunk ID formats."""
        tracker = ChunkTracker(temp_dir / "chunks.json")

        # Add chunks from different processors
        tracker.add_chunk("shard_0:chunk:0", "shard_0", "http://example.com/shard_0.tar", 0, 100)
        tracker.add_chunk("data:chunk:0", "data", "", 0, 100)
        tracker.add_chunk("local:chunk:0", "local", "/path/to/images", 0, 100)

        # Mark some items as processed
        tracker.mark_items_processed("shard_0:chunk:0", 10, 20)
        tracker.mark_items_processed("data:chunk:0", 30, 40)
        tracker.mark_items_processed("local:chunk:0", 50, 60)

        # Save and reload
        tracker.save()

        new_tracker = ChunkTracker(temp_dir / "chunks.json")

        assert len(new_tracker.chunks) == 3
        assert all(
            chunk_id in new_tracker.chunks
            for chunk_id in ["shard_0:chunk:0", "data:chunk:0", "local:chunk:0"]
        )

    def test_work_result_format_compatibility(self):
        """Test WorkResult format is consistent across processors."""
        # Create work results as each processor would
        results = []

        # WebDataset style
        results.append(
            WorkResult(
                unit_id="shard_0:chunk:0",
                source_id="shard_0",
                chunk_id="shard_0:chunk:0",
                sample_id="42",
                outputs={"captions": ["A webdataset image"]},
                metadata={"_item_index": 42},
            )
        )

        # HuggingFace style
        results.append(
            WorkResult(
                unit_id="data:chunk:0",
                source_id="data",
                chunk_id="data:chunk:0",
                sample_id="42",
                outputs={"captions": ["A huggingface image"]},
                metadata={"_item_index": 42},
            )
        )

        # Local filesystem style
        results.append(
            WorkResult(
                unit_id="local:chunk:0",
                source_id="local",
                chunk_id="local:chunk:0",
                sample_id="42",
                outputs={"captions": ["A local image"]},
                metadata={"_item_index": 42},
            )
        )

        # All should have consistent structure
        for result in results:
            assert result.is_success()
            assert "captions" in result.outputs
            assert "_item_index" in result.metadata

    @pytest.mark.parametrize(
        "processor_pairs",
        [
            (WebDatasetOrchestratorProcessor, WebDatasetWorkerProcessor),
            (HuggingFaceDatasetOrchestratorProcessor, HuggingFaceDatasetWorkerProcessor),
            (LocalFilesystemOrchestratorProcessor, LocalFilesystemWorkerProcessor),
        ],
    )
    def test_orchestrator_worker_communication(self, processor_pairs):
        """Test that orchestrator and worker pairs can communicate properly."""
        OrchestratorClass, WorkerClass = processor_pairs

        # Create instances
        orchestrator = OrchestratorClass()
        worker = WorkerClass()

        # Verify they have compatible interfaces
        assert hasattr(orchestrator, "get_work_units")
        assert hasattr(orchestrator, "handle_result")
        assert hasattr(worker, "process_unit")
        assert hasattr(worker, "prepare_result")


# ============= Integration Tests =============


class TestProcessorIntegration(ProcessorTestBase):
    """Integration tests for complete workflows."""

    @pytest.mark.asyncio
    async def test_complete_processing_workflow(self, temp_dir):
        """Test complete workflow from work assignment to storage."""
        storage = StorageManager(temp_dir, caption_buffer_size=10)
        await storage.initialize()

        # Setup WebDataset orchestrator (using mock mode)
        config = ProcessorConfig(
            processor_type="webdataset",
            config={
                "dataset": {"dataset_path": "mock://test", "metadata_path": None},
                "chunk_size": 10,
                "checkpoint_dir": str(temp_dir / "checkpoints"),
                "cache_dir": str(temp_dir / "cache"),
            },
        )

        with patch("webshart.discover_dataset") as mock_discover:
            # Mock dataset
            mock_dataset = Mock()
            mock_dataset.num_shards = 1
            mock_dataset.get_shard_info = Mock(
                return_value={
                    "name": "shard_0",
                    "path": "mock://shard_0.tar",
                    "num_files": 20,
                }
            )
            mock_dataset.enable_metadata_cache = Mock()
            mock_dataset.enable_shard_cache = Mock()
            mock_discover.return_value = mock_dataset

            orchestrator = WebDatasetOrchestratorProcessor()
            orchestrator.initialize(config, storage)

            # Create worker
            worker_config = ProcessorConfig(
                processor_type="webdataset",
                config={"dataset": {"dataset_path": "mock://test", "mock_results": True}},
            )
            worker = WebDatasetWorkerProcessor()
            worker.gpu_id = 0
            worker.initialize(worker_config)

            # Simulate workflow
            import time

            time.sleep(0.1)  # Let background thread create units

            # Worker requests work
            units = orchestrator.get_work_units(1, "test_worker")
            assert len(units) > 0

            unit = units[0]

            # Add required fields to unit data if missing
            if "start_index" not in unit.data:
                unit.data["start_index"] = 0
            if "chunk_size" not in unit.data:
                unit.data["chunk_size"] = unit.unit_size

            # Worker processes unit
            context = {}
            items = list(worker.process_unit(unit, context))
            assert len(items) > 0

            # Simulate caption generation
            outputs = []
            for item in items:
                outputs.append(
                    {
                        "captions": [f"Generated caption for {item['item_key']}"],
                        "metadata": item["metadata"],
                    }
                )

            # Create result
            result = worker.prepare_result(unit, outputs, 1000.0)

            # Submit result to orchestrator
            orchestrator.handle_result(result)

            # Create and save caption
            job_id = JobId.from_str(items[0]["job_id"])
            caption = Caption(
                job_id=job_id,
                dataset="test_dataset",
                shard=unit.source_id,
                chunk_id=unit.chunk_id,
                item_key=items[0]["item_key"],
                caption=outputs[0]["captions"][0],
                outputs={"captions": outputs[0]["captions"]},
                contributor_id="test_worker",
                timestamp=datetime.now(_datetime.UTC),
                caption_count=1,
                metadata=items[0]["metadata"],
            )

            await storage.save_caption(caption)
            await storage.checkpoint()

            # Verify complete workflow
            stats = await storage.get_storage_stats()
            assert stats["total_rows"] >= 1

    def test_failure_recovery(self, temp_dir):
        """Test recovery from worker failures."""
        storage = Mock()
        storage.get_all_processed_job_ids = Mock(return_value=set())

        config = ProcessorConfig(
            processor_type="local_filesystem",
            config={
                "dataset": {"dataset_path": str(temp_dir)},
                "chunk_size": 5,
                "checkpoint_dir": str(temp_dir / "checkpoints"),
            },
        )

        # Create some test files
        for i in range(10):
            (temp_dir / f"image_{i}.jpg").touch()

        orchestrator = LocalFilesystemOrchestratorProcessor()
        orchestrator._start_http_server = Mock()
        orchestrator.initialize(config, storage)

        # Wait for background thread to create units
        import time

        time.sleep(0.5)  # Give more time for unit creation

        # Simulate work assignment
        units = orchestrator.get_work_units(2, "worker1")

        # Check that we got units before proceeding
        if not units:
            # If still no units, manually trigger unit creation
            with orchestrator.lock:
                orchestrator.current_index = 0
            time.sleep(0.5)
            units = orchestrator.get_work_units(2, "worker1")

        assert len(units) > 0, "No work units were created"
        unit_ids = [u.unit_id for u in units]

        # Simulate worker failure
        orchestrator.mark_failed(unit_ids[0], "worker1", "Worker crashed")

        # Unit should be available again
        new_units = orchestrator.get_work_units(1, "worker2")
        assert any(u.unit_id == unit_ids[0] for u in new_units)

        # Simulate worker disconnect
        orchestrator.release_assignments("worker1")

        # Remaining units should be available
        available_units = orchestrator.get_work_units(10, "worker3")
        available_ids = [u.unit_id for u in available_units]
        assert any(uid in available_ids for uid in unit_ids[1:])


if __name__ == "__main__":
    pytest.main([__file__, "-v", "-s"])
