listobj = ''''from ..imports import *\nimport traceback\nimport ast\nimport re\nimport os\n\ndef _clear_layout(layout: QtWidgets.QLayout):\n    """Recursively delete all widgets in a layout (Qt-safe)."""\n    if layout is None:\n        return\n    while layout.count():\n        item = layout.takeAt(0)\n        if item.widget():\n            item.widget().setParent(None)      # detach\n            item.widget().deleteLater()        # mark for deletion\n        elif item.layout():\n            _clear_layout(item.layout())\n\nclass FileDropArea(QtWidgets.QWidget):\n    """\n    Right‐hand pane: “Drag‐Drop → Clipboard.”\n    """\n    function_selected = QtCore.pyqtSignal(dict)\n    file_selected = QtCore.pyqtSignal(dict)\n\n    def __init__(self, log_widget: QtWidgets.QTextEdit, parent=None):\n        super().__init__(parent)\n        self.log_widget = log_widget\n\n        self.setWindowTitle("Drag - Drop → Clipboard")\n        self.resize(600, 420)\n        self.setAcceptDrops(True)\n\n        # Map ".ext" → QCheckBox\n        self.ext_checks: dict[str, QtWidgets.QCheckBox] = {}\n        # Remember the raw list of dropped/browsed paths\n        self._last_raw_paths: List[str] = []\n        # Store function info\n        self.functions: List[dict] = []\n        # Store Python file info\n        self.python_files: List[dict] = []\n\n        # Main vertical layout\n        lay = QtWidgets.QVBoxLayout(self)\n\n        # Instruction label\n        self.info = QtWidgets.QLabel(\n            "Drag one or more supported files here,\\n"\n            "or click “Browse…”",\n            alignment=QtCore.Qt.AlignCenter,\n        )\n        self.info.setStyleSheet("font-size: 14px; color: #555;")\n        lay.addWidget(self.info)\n\n        # “Browse…” button\n        browse_btn = QtWidgets.QPushButton("Browse Files…")\n        browse_btn.clicked.connect(self.browse_files)\n        lay.addWidget(browse_btn, alignment=QtCore.Qt.AlignHCenter)\n\n        # ── extension‐filter row (initially hidden) ─────────────────────\n        self.ext_row = QtWidgets.QScrollArea(widgetResizable=True)\n        self.ext_row.setFixedHeight(45)\n        self.ext_row.setVisible(False)\n\n        self.ext_row_w = QtWidgets.QWidget()\n        self.ext_row.setWidget(self.ext_row_w)\n        self.ext_row_lay = QtWidgets.QHBoxLayout(self.ext_row_w)\n        self.ext_row_lay.setContentsMargins(4, 4, 4, 4)\n        self.ext_row_lay.setSpacing(10)\n\n        lay.addWidget(self.ext_row)\n\n        # ── function listbox ─────────────────────\n        self.function_list = QtWidgets.QListWidget()\n        self.function_list.setVisible(False)\n        self.function_list.itemClicked.connect(self.on_function_clicked)\n        lay.addWidget(self.function_list)\n\n        # ── python file listbox ─────────────────────\n        self.python_file_list = QtWidgets.QListWidget()\n        self.python_file_list.setVisible(False)\n        self.python_file_list.itemClicked.connect(self.on_python_file_clicked)\n        lay.addWidget(self.python_file_list)\n\n        # Status line at bottom\n        self.status = QtWidgets.QLabel("No files selected.", alignment=QtCore.Qt.AlignCenter)\n        self.status.setStyleSheet("color: #333; font-size: 12px;")\n        lay.addWidget(self.status)\n\n    def dragEnterEvent(self, event: QtGui.QDragEnterEvent):\n        if event.mimeData().hasUrls():\n            event.acceptProposedAction()\n        else:\n            event.ignore()\n\n    def dropEvent(self, event: QtGui.QDropEvent):\n        """\n        Wrap in try/except so invalid drops don’t crash. Log everything.\n        """\n        try:\n            # 1) Extract local file paths from the drop event\n            urls = event.mimeData().urls()\n            paths = [url.toLocalFile() for url in urls if url.isLocalFile()]\n            self._log(f"Received raw drop paths: {paths!r}")\n\n            if not paths:\n                raise ValueError("No local files detected on drop.")\n\n            # 2) Expand directories, filter out unwanted files\n            filtered = self._filtered_file_list(paths)\n            self._log(f"_filtered_file_list returned {len(filtered)} path(s): {filtered!r}")\n\n            # 3) Before proceeding, inspect each element’s type\n            for idx, p in enumerate(filtered):\n                if not isinstance(p, str):\n                    self._log(f"  >>> filtered[{idx}] is a {type(p).__name__}: {p!r}")\n\n            # 4) If filtering produced no valid strings, abort\n            if not filtered:\n                self.status.setText("⚠️ No valid files detected in drop.")\n                self._log("No valid paths after filtering.")\n                return\n\n            # 5) Log how many files will be processed\n            self._log(f"dropEvent: Proceeding to process {len(filtered)} file(s).")\n            self.process_files(filtered)\n\n        except Exception as e:\n            # 6) Catch any exception, log a full traceback, and update status\n            tb = traceback.format_exc()\n            self.status.setText(f"⚠️ Error during drop: {e}")\n            self._log("dropEvent ERROR:\\n" + tb)\n\n    def _rebuild_ext_row(self, paths: List[str]):\n        """\n        1. Collect unique extensions from paths (only actual files).\n        2. Wipe any existing checkboxes.\n        3. Create a checkbox for each extension, preserving prior states.\n        4. Show the row, or hide if no extensions found.\n        """\n        exts = {os.path.splitext(p)[1].lower() for p in paths if os.path.isfile(p)}\n        exts.discard("")   # drop empty ext (e.g. directories)\n        self._log(f"Unique extensions: {exts}")\n\n        if not exts:\n            # No file‐extensions → hide the row\n            self.ext_row.setVisible(False)\n            self.ext_checks.clear()\n            return\n\n        # Remove any existing checkboxes from the layout\n        _clear_layout(self.ext_row_lay)\n\n        new_checks: dict[str, QtWidgets.QCheckBox] = {}\n        for ext in sorted(exts):\n            cb = QtWidgets.QCheckBox(ext)\n            # Keep prior check state if exists, else default True\n            prev_cb = self.ext_checks.get(ext)\n            cb.setChecked(prev_cb.isChecked() if prev_cb else True)\n            cb.stateChanged.connect(self._apply_ext_filter)\n            self.ext_row_lay.addWidget(cb)\n            new_checks[ext] = cb\n\n        # Replace with new map\n        self.ext_checks = new_checks\n        # Make the row visible now that it has content\n        self.ext_row.setVisible(True)\n\n    def _apply_ext_filter(self):\n        """\n        Called when any extension‐checkbox toggles.\n        Simply re-process the last raw paths, but skip rebuilding the ext‐row.\n        """\n        if not self._last_raw_paths:\n            return\n        self.process_files(self._last_raw_paths, rebuild_ext_row=False)\n\n    def process_files(self, raw_paths: List[str], *, rebuild_ext_row: bool = True):\n        """\n        1. Expand directories → flat list of file‐paths.\n        2. If rebuild_ext_row: re-derive the set of extensions.\n        3. Filter by checked extensions.\n        4. Read each file, concatenate, copy to clipboard.\n        5. Parse Python files for functions and files, display in listboxes.\n        """\n        self._last_raw_paths = raw_paths\n        # Expand directories, apply exclusion rules\n        all_paths = collect_filepaths(\n            raw_paths,\n            exclude_dirs=DEFAULT_EXCLUDE_DIRS,\n            exclude_file_patterns=[\'__init__\']\n        )\n        self._log(f"{len(all_paths)} total path(s) after expansion")\n\n        # Step 2: maybe rebuild the extension-filter row\n        if rebuild_ext_row:\n            self._rebuild_ext_row(paths=all_paths)\n\n        # Step 3: determine which extensions remain checked\n        if self.ext_checks:\n            visible_exts = {ext for ext, cb in self.ext_checks.items() if cb.isChecked()}\n            self._log(f"Visible extensions: {visible_exts}")\n            # Filter out any file that isn’t a directory or whose ext isn’t in visible_exts\n            filtered_paths = [\n                p for p in all_paths\n                if os.path.isdir(p) or os.path.splitext(p)[1].lower() in visible_exts\n            ]\n        else:\n            # No checkboxes exist ⇒ no filtering\n            filtered_paths = all_paths\n\n        if not filtered_paths:\n            self.status.setText("⚠️ No files match current extension filter.")\n            return\n\n        # Step 4: Read + combine\n        count = len(filtered_paths)\n        msg = f"Reading {count} file(s)…"\n        self.status.setText(msg)\n        self._log(msg)\n        QtWidgets.QApplication.processEvents()\n\n        combined: list[str] = []\n        self.functions = []\n        self.python_files = []\n        for idx, p in enumerate(filtered_paths):\n            combined.append(f"=== {p} ===\\n")\n            try:\n                text = read_file_as_text(p)\n                combined.append(str(text) or "")\n                # Parse Python files for functions and store file info\n                if p.endswith(\'.py\'):\n                    self.python_files.append({\'path\': p, \'text\': text})\n                    self._parse_functions(p, text)\n            except Exception as exc:\n                combined.append(f"[Error reading {os.path.basename(p)}: {exc}]\\n")\n                self._log(f"Error reading {p} → {exc}")\n            if idx < count - 1:\n                combined.append("\\n\\n――――――――――――――――――\\n\\n")\n\n        # Copy to clipboard\n        QtWidgets.QApplication.clipboard().setText("".join(combined))\n        self.status.setText(f"✅ Copied {count} file(s) to clipboard!")\n        self._log(f"Copied {count} file(s) successfully")\n\n        # Step 5: Update listboxes\n        self._update_function_list()\n        self._update_python_file_list()\n\n    def _parse_functions(self, file_path: str, text: str):\n        """\n        Parse Python file to extract function definitions.\n        """\n        try:\n            tree = ast.parse(text, filename=file_path)\n            for node in ast.walk(tree):\n                if isinstance(node, ast.FunctionDef):\n                    self.functions.append({\n                        \'name\': node.name,\n                        \'file\': file_path,\n                        \'line\': node.lineno,\n                        \'code\': \'\\n\'.join(text.splitlines()[node.lineno-1:node.end_lineno]),\n                        \'imports\': self._extract_imports(tree)\n                    })\n        except SyntaxError as e:\n            self._log(f"Syntax error in {file_path}: {e}")\n\n    def _extract_imports(self, tree: ast.AST) -> List[str]:\n        """\n        Extract import statements from AST.\n        """\n        imports = []\n        for node in ast.walk(tree):\n            if isinstance(node, ast.Import):\n                for alias in node.names:\n                    imports.append(f"import {alias.name}")\n            elif isinstance(node, ast.ImportFrom):\n                module = node.module or \'\'\n                for alias in node.names:\n                    imports.append(f"from {module} import {alias.name}")\n        return imports\n\n    def _update_function_list(self):\n        """\n        Populate the function listbox with parsed functions.\n        """\n        self.function_list.clear()\n        if self.functions:\n            for func in self.functions:\n                self.function_list.addItem(f"{func[\'name\']} ({func[\'file\']})")\n            self.function_list.setVisible(True)\n        else:\n            self.function_list.setVisible(False)\n\n    def _update_python_file_list(self):\n        """\n        Populate the Python file listbox with parsed Python files.\n        """\n        self.python_file_list.clear()\n        if self.python_files:\n            for file_info in self.python_files:\n                self.python_file_list.addItem(os.path.basename(file_info[\'path\']))\n            self.python_file_list.setVisible(True)\n        else:\n            self.python_file_list.setVisible(False)\n\n    def on_function_clicked(self, item: QtWidgets.QListWidgetItem):\n        """\n        Emit signal when a function is selected.\n        """\n        index = self.function_list.row(item)\n        self.function_selected.emit(self.functions[index])\n\n    def on_python_file_clicked(self, item: QtWidgets.QListWidgetItem):\n        """\n        Emit signal when a Python file is selected.\n        """\n        index = self.python_file_list.row(item)\n        self.file_selected.emit(self.python_files[index])\n\n    def map_function_dependencies(self, function_info: dict):\n        """\n        Map the selected function\'s imports and project reach, then copy to clipboard.\n        """\n        combined: list[str] = []\n        combined.append(f"=== Function: {function_info[\'name\']} ===\\n")\n        combined.append(function_info[\'code\'])\n        combined.append("\\n\\n=== Imports ===\\n")\n        combined.extend(function_info[\'imports\'])\n        \n        # Find project reach: files that might be related\n        project_files = collect_filepaths(\n            [os.path.dirname(function_info[\'file\'])],\n            exclude_dirs=DEFAULT_EXCLUDE_DIRS,\n            exclude_file_patterns=DEFAULT_EXCLUDE_FILE_PATTERNS\n        )\n        combined.append("\\n\\n=== Project Reach ===\\n")\n        for file_path in project_files:\n            if file_path != function_info[\'file\'] and file_path.endswith(\'.py\'):\n                combined.append(f"--- {file_path} ---\\n")\n                try:\n                    text = read_file_as_text(file_path)\n                    combined.append(text)\n                except Exception as exc:\n                    combined.append(f"[Error reading {os.path.basename(file_path)}: {exc}]\\n")\n                combined.append("\\n")\n\n        QtWidgets.QApplication.clipboard().setText("\\n".join(combined))\n        self.status.setText(f"✅ Copied function {function_info[\'name\']} and dependencies to clipboard!")\n        self._log(f"Copied function {function_info[\'name\']} with dependencies")\n\n    def map_import_chain(self, file_info: dict):\n        """\n        Map the import chain for the selected Python file, then copy to clipboard.\n        """\n        project_root = os.path.dirname(file_info[\'path\'])\n        import_chain = self._build_import_chain(file_info[\'path\'], project_root, set())\n        combined: list[str] = []\n        combined.append(f"=== Import Chain for {file_info[\'path\']} ===\\n")\n        if import_chain:\n            combined.append("\\n".join([f"- {item}" for item in import_chain]))\n        else:\n            combined.append("No imports detected.")\n\n        QtWidgets.QApplication.clipboard().setText("\\n".join(combined))\n        self.status.setText(f"✅ Copied import chain for {os.path.basename(file_info[\'path\'])} to clipboard!")\n        self._log(f"Copied import chain for {file_info[\'path\']}")\n\n    def _build_import_chain(self, file_path: str, project_root: str, visited: set) -> List[str]:\n        """\n        Recursively build the import chain for a Python file.\n        """\n        if file_path in visited:\n            return []\n        visited.add(file_path)\n\n        chain = []\n        try:\n            text = read_file_as_text(file_path)\n            tree = ast.parse(text, filename=file_path)\n            for node in ast.walk(tree):\n                if isinstance(node, ast.Import):\n                    for alias in node.names:\n                        chain.append(f"import {alias.name}")\n                elif isinstance(node, ast.ImportFrom):\n                    module = node.module or \'\'\n                    for alias in node.names:\n                        chain.append(f"from {module} import {alias.name}")\n                        # Check if the module is a local file in the project\n                        if not module.startswith(\'.\'):  # Ignore relative imports for simplicity\n                            module_path = os.path.join(project_root, f"{module}.py")\n                            if os.path.isfile(module_path) and module_path.endswith(\'.py\'):\n                                chain.extend(self._build_import_chain(module_path, project_root, visited))\n        except (SyntaxError, Exception) as e:\n            self._log(f"Error parsing {file_path} for imports: {e}")\n\n        return chain\n\n    def browse_files(self):\n        files, _ = QtWidgets.QFileDialog.getOpenFileNames(\n            self,\n            "Select Files to Copy",\n            "",\n            "All Supported Files (*.txt *.md *.csv *.tsv *.log "\n            "*.xls *.xlsx *.ods *.parquet *.geojson *.shp);;All Files (*)"\n        )\n        if files:\n            filtered = self._filtered_file_list(files)\n            self._log(f"browse_files: Selected {len(filtered)} file(s): {filtered!r}")\n            self.process_files(filtered)\n\n    def _filtered_file_list(self, raw_paths: List[str]) -> List[str]:\n        """\n        Recursively collect files under directories (excluding node_modules/__pycache__, etc).\n        Always returns a flat List[str].\n        """\n        filtered = collect_filepaths(\n            raw_paths,\n            exclude_dirs=DEFAULT_EXCLUDE_DIRS,\n            exclude_file_patterns=DEFAULT_EXCLUDE_FILE_PATTERNS\n        )\n        self._log(f"_filtered_file_list: Expanded to {len(filtered)} file(s).")\n        return filtered\n\n    def _log(self, message: str):\n        """Write out to the shared log widget (with timestamp)."""\n        log_it(self=self, message=message)\n'][Error reading file_drop_area.py: compile() arg 1 must be a string, bytes or AST object'''

input(listobj)
