import pytest
from bluesky import plan_stubs as bps
from bluesky.run_engine import RunEngine
from ophyd_async.testing import set_mock_value

from dodal.common.enums import OnOffUpper
from dodal.devices.i24.dual_backlight import (
    BacklightPositions,
    DualBacklight,
)


@pytest.fixture
async def fake_backlight() -> DualBacklight:
    fake_backlight = DualBacklight("", name="fake_backlight")
    await fake_backlight.connect(mock=True)
    return fake_backlight


async def test_dual_backlight_can_be_written_and_read_from(
    fake_backlight: DualBacklight,
    run_engine: RunEngine,
):
    run_engine(bps.abs_set(fake_backlight.frontlight_state, OnOffUpper.OFF, wait=True))
    assert await fake_backlight.frontlight_state.get_value() == OnOffUpper.OFF


async def test_backlight_position(
    fake_backlight: DualBacklight,
    run_engine: RunEngine,
):
    run_engine(
        bps.abs_set(
            fake_backlight.backlight_position.pos_level,
            BacklightPositions.IN,
            wait=True,
        )
    )
    assert (
        await fake_backlight.backlight_position.pos_level.get_value()
        == BacklightPositions.IN
    )


async def test_when_backlight_out_it_switches_off(
    fake_backlight: DualBacklight, run_engine: RunEngine
):
    set_mock_value(fake_backlight.backlight_state, OnOffUpper.ON)
    run_engine(bps.abs_set(fake_backlight, BacklightPositions.OUT, wait=True))
    assert (
        await fake_backlight.backlight_position.pos_level.get_value()
        == BacklightPositions.OUT
    )
    assert await fake_backlight.backlight_state.get_value() == OnOffUpper.OFF


async def test_when_backlight_not_out_it_switches_on(
    fake_backlight: DualBacklight, run_engine: RunEngine
):
    run_engine(bps.abs_set(fake_backlight, "OAV2", wait=True))
    assert await fake_backlight.backlight_state.get_value() == OnOffUpper.ON


async def test_frontlight_independent_from_backlight_position(
    fake_backlight: DualBacklight, run_engine: RunEngine
):
    set_mock_value(fake_backlight.frontlight_state, OnOffUpper.OFF)
    run_engine(bps.abs_set(fake_backlight, BacklightPositions.IN, wait=True))
    assert await fake_backlight.backlight_state.get_value() == OnOffUpper.ON
    assert await fake_backlight.frontlight_state.get_value() == OnOffUpper.OFF
