# Kerykeion v5.0.0 – Data-first astrology

> **The fifth major version of Kerykeion is ready**: a ground-up redesign that separates computation, data, and rendering, introduces a coherent ecosystem of factories and Pydantic models, and brings SVG chart generation to a new level of quality.

## Executive summary
- **Modernised pipeline**: everything now revolves around `AstrologicalSubjectFactory`, `ChartDataFactory`, and `ChartDrawer`, with strongly typed models in `kerykeion.schemas`.
- **Deeper analysis**: a unified aspects engine, full planetary return workflow, house comparison, and transit time ranges arrive as first-class citizens.
- **Documentation and tooling**: modular docs in `site-docs/`, snippet verification via `scripts/test_markdown_snippets.py`, and runnable examples in `examples/`.
- **Guided compatibility**: the `kerykeion.backword` module keeps the legacy classes alive with deprecation warnings for a gradual migration from v4.

## Headline features
### Factory-centred architecture
- New `kerykeion.astrological_subject_factory` replaces the old `AstrologicalSubject` class and handles **tropical/sidereal zodiacs, multiple house systems, geocentric/heliocentric/topocentric perspectives, cached GeoNames lookup**, and active point preselection.
- All data now lives in `kerykeion.schemas`: Pydantic 2 models (`AstrologicalSubjectModel`, `ChartDataModel`, `SingleChartAspectsModel`, `DualChartAspectsModel`, `PlanetReturnModel`, etc.) with literals consolidated in `kr_literals`.
- The models are subscriptable and serialisable; every API returns coherent structures designed for AI integrations, web apps, and CLIs.

### Next-generation chart renderer
- `ChartDataFactory` produces data enriched with **element/quality distributions, relationship scores, house comparison**, and convenience helpers for each chart type (natal, synastry, transit, composite, returns).
- `ChartDrawer` now focuses solely on SVG rendering: it supports minification, CSS inlining, and wheel/aspect-only exports (`save_svg`, `save_wheel_only_svg_file`, `save_aspect_grid_only_svg_file`).
- Visual assets were refreshed with updated themes (`classic`, `dark`, `dark-high-contrast`, `light`, `strawberry`), polished XML templates, and a refreshed fixed stars dataset (`sweph/sefstars.txt`).
- A brand-new Black & White theme renders glyphs, rings, and aspects in solid black, making natal charts printer-friendly on monochrome devices.

### Upgraded astrological analysis
- `AspectsFactory` unifies single- and dual-chart aspects, keeps legacy aliases (`NatalAspectsModel`, `SynastryAspectsModel`) for compatibility, and introduces the `axis_orb_limit` keyword to optionally reintroduce traditional axis orbs while defaulting to modern practice (axes share the same orb as planets).
- `HouseComparisonFactory`, `RelationshipScoreFactory`, and the new `ChartDataFactory` integrate house overlays, affinity scoring, and elemental statistics.
- `PlanetaryReturnFactory` computes Solar and Lunar returns via Swiss Ephemeris (methods `next_return_from_year`, `next_return_from_month_and_year`, `next_return_from_iso_formatted_time`).
- `EphemerisDataFactory` + `TransitsTimeRangeFactory` build subject time series and transit snapshots with safeguards on step size and range.
- Element and modality distributions now default to a curated weighted map (Sun/Moon/Asc = 2.0, angles 1.5, personal planets 1.5, social planets 1.0, outers 0.5, etc.) and expose `distribution_method`/`custom_distribution_weights` hooks for equal counts or bespoke weighting across single and dual charts.

### Documentation, examples, and tooling
- Documentation has been rewritten in `site-docs/*.md`, with each factory owning a detailed guide and example outputs.
- `scripts/test_markdown_snippets.py` validates documentation snippets.
- The `examples/` folder now ships runnable scripts (see *Quickstart examples* below) aligned with the docs.
- Development guides (`DEVELOPMENT.md`, `IMPROVE.md`) have been updated with the modern workflow.

## Compatibility & migration from v4
- Import directly from the package root (`from kerykeion import AstrologicalSubjectFactory, ChartDataFactory, ChartDrawer, ...`).
- `kerykeion.backword` provides deprecated wrappers (`AstrologicalSubject`, `KerykeionChartSVG`, `NatalAspects`, `SynastryAspects`) so existing code keeps running while pointing to the new APIs.
- `kr_types` module path still works but shows deprecation warnings; update to `kerykeion.schemas`.
- Type aliases `Planet` and `AxialCusps` unified as `AstrologicalPoint`; legacy aliases available in v5 without warnings when imported from `schemas`.
- Backward compatibility layer (`kr_types` module and legacy wrappers) will be **removed in v6.0**.
- Relationship scoring now lives in `RelationshipScoreFactory` and the models returned by `ChartDataFactory`.
- Regeneration scripts (`scripts/regenerate_*`) realign fixtures and custom SVG assets with the new pipeline.

## Breaking changes
- Legacy classes (`AstrologicalSubject`, `NatalAspects`, `SynastryAspects`, `KerykeionChartSVG`, `relationship_score`, `transits_time_range`, `ephemeris_data`) have been removed in favour of their factory-based counterparts.
- `kerykeion/enums.py`, the entire `kr_types/` tree, and scattered config files were removed; everything is now centralised in `schemas` and `settings`.
- Packaging migrated from Poetry to PEP 621 + Hatchling (`pyproject.toml`), with dependency locking handled via `uv.lock` and dependency groups.
- `DEFAULT_ACTIVE_POINTS` and `DEFAULT_ACTIVE_ASPECTS` are more restrictive; chart axes no longer have a forced 1° orb—use the new `axis_orb_limit` keyword argument when you need a traditional axis threshold.
- **Lunar nodes naming**: All lunar node-related fields have been renamed for clarity (see Migration Guide below).

## Migration Guide: Lunar Nodes Naming

### Breaking Changes Summary

In version 5.0, we have renamed all lunar node-related fields and constants to be more explicit and consistent:

| Old Name (v4.x) | New Name (v5.0) |
|-----------------|-----------------|
| `Mean_Node` | `Mean_North_Lunar_Node` |
| `True_Node` | `True_North_Lunar_Node` |
| `Mean_South_Node` | `Mean_South_Lunar_Node` |
| `True_South_Node` | `True_South_Lunar_Node` |

### Why This Change?

The previous naming was ambiguous:
- `Mean_Node` and `True_Node` didn't specify they were **North** nodes
- This caused confusion when working with both North and South nodes
- The new names are self-documenting and follow astrological conventions

### Migration Steps

#### 1. Update Model Field Access

**Old Code:**
```python
from kerykeion import AstrologicalSubjectFactory

subject = AstrologicalSubjectFactory.from_birth_data(
    "John Doe", 1990, 1, 1, 12, 0, "London", "GB"
)

# Old way
print(subject.mean_node)
print(subject.true_node)
print(subject.mean_south_node)
print(subject.true_south_node)
```

**New Code:**
```python
from kerykeion import AstrologicalSubjectFactory

subject = AstrologicalSubjectFactory.from_birth_data(
    "John Doe", 1990, 1, 1, 12, 0, "London", "GB"
)

# New way
print(subject.mean_north_lunar_node)
print(subject.true_north_lunar_node)
print(subject.mean_south_lunar_node)
print(subject.true_south_lunar_node)
```

#### 2. Update Active Points Lists

**Old Code:**
```python
from kerykeion import AstrologicalSubjectFactory

active_points = [
    "Sun", "Moon", "Mercury", "Venus", "Mars",
    "Mean_Node", "True_Node", "Mean_South_Node", "True_South_Node"
]

subject = AstrologicalSubjectFactory.from_birth_data(
    "John Doe", 1990, 1, 1, 12, 0, "London", "GB",
    active_points=active_points
)
```

**New Code:**
```python
from kerykeion import AstrologicalSubjectFactory

active_points = [
    "Sun", "Moon", "Mercury", "Venus", "Mars",
    "Mean_North_Lunar_Node", "True_North_Lunar_Node",
    "Mean_South_Lunar_Node", "True_South_Lunar_Node"
]

subject = AstrologicalSubjectFactory.from_birth_data(
    "John Doe", 1990, 1, 1, 12, 0, "London", "GB",
    active_points=active_points
)
```

#### 3. Update Type Hints and Literals

**Old Code:**
```python
from kerykeion.schemas.kr_literals import AstrologicalPoint
from typing import List

def get_nodes(subject) -> List[AstrologicalPoint]:
    return ["Mean_Node", "True_Node"]  # This will cause a type error in v5
```

**New Code:**
```python
from kerykeion.schemas.kr_literals import AstrologicalPoint
from typing import List

def get_nodes(subject) -> List[AstrologicalPoint]:
    return ["Mean_North_Lunar_Node", "True_North_Lunar_Node"]
```

#### 4. Update Custom Settings/Translations

If you ship a custom `LANGUAGE_SETTINGS` dictionary (or pass overrides to `ChartDrawer`), update lunar node keys to the new names:

```python
LANGUAGE_SETTINGS = {
    "EN": {
        "celestial_points": {
            # Old -> new key names
            "Mean_North_Lunar_Node": "Mean North Lunar Node",
            "True_North_Lunar_Node": "True North Lunar Node",
            "Mean_South_Lunar_Node": "Mean South Lunar Node",
            "True_South_Lunar_Node": "True South Lunar Node",
        }
    }
}
```

Custom packs can be supplied at runtime via:

```python
ChartDrawer(chart_data=data, chart_language="PT", language_pack=custom_dict)
```

### Backward Compatibility

For backward compatibility, the legacy `AstrologicalSubject` wrapper in `kerykeion.backword` provides deprecated properties:

```python
from kerykeion import AstrologicalSubject  # Legacy wrapper

subject = AstrologicalSubject("John Doe", 1990, 1, 1, 12, 0, "London", "GB")

# This still works but shows a DeprecationWarning
print(subject.mean_node)  # Maps to mean_north_lunar_node
print(subject.true_node)  # Maps to true_north_lunar_node
```

**⚠️ Warning:** This compatibility layer is deprecated and will be removed in v6.0. Please migrate to the new naming as soon as possible.

### Automated Migration

You can use the following sed commands to update your codebase:

```bash
# Update Python files
find . -name "*.py" -type f -exec sed -i.bak \
    -e 's/\.mean_node/.mean_north_lunar_node/g' \
    -e 's/\.true_node/.true_north_lunar_node/g' \
    -e 's/\.mean_south_node/.mean_south_lunar_node/g' \
    -e 's/\.true_south_node/.true_south_lunar_node/g' \
    -e 's/"Mean_Node"/"Mean_North_Lunar_Node"/g' \
    -e 's/"True_Node"/"True_North_Lunar_Node"/g' \
    -e 's/"Mean_South_Node"/"Mean_South_Lunar_Node"/g' \
    -e 's/"True_South_Node"/"True_South_Lunar_Node"/g' \
    -e "s/'Mean_Node'/'Mean_North_Lunar_Node'/g" \
    -e "s/'True_Node'/'True_North_Lunar_Node'/g" \
    -e "s/'Mean_South_Node'/'Mean_South_Lunar_Node'/g" \
    -e "s/'True_South_Node'/'True_South_Lunar_Node'/g" \
    {} \;
```

**Note:** Always review automated changes and test thoroughly before committing.

### Testing Your Migration

After migrating, ensure your tests pass:

```bash
pytest tests/
```

If you encounter any issues, please:
1. Check that all lunar node references have been updated
2. Verify custom configuration files use new names
3. Update any saved/cached data structures that reference nodes

### Need Help?

If you encounter migration issues:
- Check the [GitHub Discussions](https://github.com/g-battaglia/kerykeion/discussions)
- Open an issue on [GitHub](https://github.com/g-battaglia/kerykeion/issues)
- Review the updated documentation at [kerykeion.readthedocs.io](https://kerykeion.readthedocs.io)

## Other notable changes
- `kerykeion/settings` consolidates chart presets in `settings/chart_defaults.py` and caches configuration merges via `functools.lru_cache`.
- `utilities.py` now hosts helpers for distributing percentages, inlining CSS, computing element/quality stats, and normalising active points.
- Test datasets and fixtures were regenerated (new reference SVGs, updated expected aspects, complete factory coverage).
- README and CHANGELOG received a full rewrite with Pydantic 2 examples, theming guidance, and LLM integration notes.

## Quickstart examples (verified)
Each snippet lives in `examples/` and was executed while preparing the release.

```python
# File: examples/quickstart_natal_chart.py
"""Quickstart example for Kerykeion v5: natal chart calculation and SVG export."""
from pathlib import Path

from kerykeion import AstrologicalSubjectFactory, ChartDataFactory, ChartDrawer


def main() -> None:
    output_dir = Path(__file__).parent / "output"
    output_dir.mkdir(parents=True, exist_ok=True)

    ada = AstrologicalSubjectFactory.from_birth_data(
        name="Ada Lovelace",
        year=1815,
        month=12,
        day=10,
        hour=4,
        minute=20,
        lng=-0.1246,
        lat=51.5014,
        tz_str="Europe/London",
        city="London",
        nation="GB",
        online=False,
    )

    chart_data = ChartDataFactory.create_natal_chart_data(ada)

    drawer = ChartDrawer(chart_data=chart_data, theme="classic")
    drawer.save_svg(output_path=output_dir, filename="ada_lovelace_natal", minify=True, remove_css_variables=True)

    element_distribution = chart_data.element_distribution
    print(
        "Saved natal chart for Ada Lovelace. Element balance: "
        f"Fire {element_distribution.fire_percentage:.1f}%, "
        f"Earth {element_distribution.earth_percentage:.1f}%, "
        f"Air {element_distribution.air_percentage:.1f}%, "
        f"Water {element_distribution.water_percentage:.1f}%"
    )


if __name__ == "__main__":
    main()
```

```python
# File: examples/aspects_synastry.py
"""Synastry aspect calculation using the unified AspectsFactory."""
from kerykeion import AstrologicalSubjectFactory, AspectsFactory


def main() -> None:
    john = AstrologicalSubjectFactory.from_birth_data(
        name="John",
        year=1990,
        month=1,
        day=1,
        hour=12,
        minute=0,
        lng=-0.1276,
        lat=51.5072,
        tz_str="Europe/London",
        city="London",
        nation="GB",
        online=False,
    )

    jane = AstrologicalSubjectFactory.from_birth_data(
        name="Jane",
        year=1992,
        month=6,
        day=15,
        hour=14,
        minute=30,
        lng=2.3522,
        lat=48.8566,
        tz_str="Europe/Paris",
        city="Paris",
        nation="FR",
        online=False,
    )

    synastry = AspectsFactory.dual_chart_aspects(john, jane)

    print(f"Total aspects computed: {len(synastry.all_aspects)}")
    print(f"Relevant aspects: {len(synastry.relevant_aspects)}")

    if synastry.relevant_aspects:
        first = synastry.relevant_aspects[0]
        print(
            "Example relevant aspect: "
            f"{first.p1_name} {first.aspect} {first.p2_name} "
            f"with orb {first.orbit:.2f}°"
        )


if __name__ == "__main__":
    main()
```

```python
# File: examples/planetary_return.py
"""Solar return calculation and visualization with Kerykeion v5."""
from pathlib import Path

from kerykeion import (
    AstrologicalSubjectFactory,
    ChartDataFactory,
    ChartDrawer,
    PlanetaryReturnFactory,
)


def main() -> None:
    output_dir = Path(__file__).parent / "output"
    output_dir.mkdir(parents=True, exist_ok=True)

    natal = AstrologicalSubjectFactory.from_birth_data(
        name="Grace Hopper",
        year=1906,
        month=12,
        day=9,
        hour=11,
        minute=30,
        lng=-73.9857,
        lat=40.7484,
        tz_str="America/New_York",
        city="New York",
        nation="US",
        online=False,
    )

    factory = PlanetaryReturnFactory(
        natal,
        lng=-73.9857,
        lat=40.7484,
        tz_str="America/New_York",
        online=False,
    )

    solar_return = factory.next_return_from_year(2025, "Solar")

    chart_data = ChartDataFactory.create_return_chart_data(natal, solar_return)
    drawer = ChartDrawer(chart_data=chart_data, theme="light")
    drawer.save_svg(output_path=output_dir, filename="grace_hopper_solar_return", minify=True)

    print("Solar return cast for:", solar_return.iso_formatted_local_datetime)
    print("Return location:", solar_return.city, solar_return.nation)
    print("Total relevant aspects:", len(chart_data.aspects.relevant_aspects))


if __name__ == "__main__":
    main()
```

```python
# File: examples/transits_time_range.py
"""Compute weekly transits for a natal chart using the new time range factory."""
from datetime import datetime, timedelta, timezone

from kerykeion import (
    AstrologicalSubjectFactory,
    EphemerisDataFactory,
    TransitsTimeRangeFactory,
)


def main() -> None:
    natal = AstrologicalSubjectFactory.from_birth_data(
        name="Research Subject",
        year=1988,
        month=4,
        day=12,
        hour=9,
        minute=15,
        lng=12.4964,
        lat=41.9028,
        tz_str="Europe/Rome",
        city="Rome",
        nation="IT",
        online=False,
    )

    start = datetime(2025, 1, 1, tzinfo=timezone.utc)
    end = start + timedelta(days=21)

    ephemeris_factory = EphemerisDataFactory(
        start,
        end,
        step_type="days",
        step=3,
        lat=natal.lat or 41.9028,
        lng=natal.lng or 12.4964,
        tz_str=natal.tz_str or "Europe/Rome",
    )
    data_points = ephemeris_factory.get_ephemeris_data_as_astrological_subjects()

    transit_factory = TransitsTimeRangeFactory(natal, data_points)
    transits = transit_factory.get_transit_moments()

    date_range = transits.dates or []
    if date_range:
        print(
            "Generated {count} transit snapshots between {start} and {end}.".format(
                count=len(transits.transits),
                start=date_range[0],
                end=date_range[-1],
            )
        )
    else:
        print(f"Generated {len(transits.transits)} transit snapshots.")

    if transits.transits:
        first_transit = transits.transits[0]
        print("First snapshot:", first_transit.date)
        if first_transit.aspects:
            aspect = first_transit.aspects[0]
            print(
                "Example transit aspect:",
                f"{aspect.p1_name} {aspect.aspect} {aspect.p2_name}",
                f"orb {aspect.orbit:.2f}°",
            )


if __name__ == "__main__":
    main()
```

## Tooling for developers
- New CLI scripts (`scripts/regenerate_expected_aspects.py`, `scripts/regenerate_test_charts.py`, `scripts/regenerate_synastry_*.py`) keep fixtures and outputs aligned with the latest algorithms.
- `scripts/test_markdown_snippets.py` integrates Markdown snippet verification into CI.
- The `tests/` folder now includes complete coverage for every factory and the compatibility layer (`tests/test_backword.py`).

## Quality & verification
- The whole test suite targets pytest 8.x with `--cov`; regenerated SVG snapshots and aspect fixtures reflect the tighter orb defaults.
- All `examples/` scripts were executed while drafting these notes to ensure the published documentation remains runnable.

## Useful resources
- Detailed documentation: `site-docs/`
- Upgrade quickstart: see the *Migration Guide* inside `README.md`
- Issues & support: [GitHub](https://github.com/g-battaglia/kerykeion)

Many thanks to the community for the steady v5 beta feedback: this release is production-ready and lays the groundwork for advanced modules (web APIs, automation, AI integration) powered by a modern, data-first core.
