"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Template = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const core_1 = require("../../core");
const fs = require("fs-extra");
const match_1 = require("./match");
const matcher_1 = require("./matcher");
const conditions_1 = require("./private/conditions");
const cyclic_1 = require("./private/cyclic");
const mappings_1 = require("./private/mappings");
const outputs_1 = require("./private/outputs");
const parameters_1 = require("./private/parameters");
const resources_1 = require("./private/resources");
/**
 * Suite of assertions that can be run on a CDK stack.
 * Typically used, as part of unit tests, to validate that the rendered
 * CloudFormation template has expected resources and properties.
 */
class Template {
    /**
     * Base your assertions on the CloudFormation template synthesized by a CDK `Stack`.
     * @param stack the CDK Stack to run assertions on
     * @param templateParsingOptions Optional param to configure template parsing behavior, such as disregarding circular
     * dependencies.
     */
    static fromStack(stack, templateParsingOptions) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_Stack(stack);
            jsiiDeprecationWarnings.aws_cdk_lib_assertions_TemplateParsingOptions(templateParsingOptions);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromStack);
            }
            throw error;
        }
        return new Template(toTemplate(stack), templateParsingOptions);
    }
    /**
     * Base your assertions from an existing CloudFormation template formatted as an in-memory
     * JSON object.
     * @param template the CloudFormation template formatted as a nested set of records
     * @param templateParsingOptions Optional param to configure template parsing behavior, such as disregarding circular
     * dependencies.
     */
    static fromJSON(template, templateParsingOptions) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_assertions_TemplateParsingOptions(templateParsingOptions);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromJSON);
            }
            throw error;
        }
        return new Template(template, templateParsingOptions);
    }
    /**
     * Base your assertions from an existing CloudFormation template formatted as a
     * JSON string.
     * @param template the CloudFormation template in
     * @param templateParsingOptions Optional param to configure template parsing behavior, such as disregarding circular
     * dependencies.
     */
    static fromString(template, templateParsingOptions) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_assertions_TemplateParsingOptions(templateParsingOptions);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromString);
            }
            throw error;
        }
        return new Template(JSON.parse(template), templateParsingOptions);
    }
    constructor(template, templateParsingOptions = {}) {
        this.template = template;
        if (!templateParsingOptions?.skipCyclicalDependenciesCheck ?? true) {
            (0, cyclic_1.checkTemplateForCyclicDependencies)(this.template);
        }
    }
    /**
     * The CloudFormation template deserialized into an object.
     */
    toJSON() {
        return this.template;
    }
    /**
     * Assert that the given number of resources of the given type exist in the
     * template.
     * @param type the resource type; ex: `AWS::S3::Bucket`
     * @param count number of expected instances
     */
    resourceCountIs(type, count) {
        const counted = (0, resources_1.countResources)(this.template, type);
        if (counted !== count) {
            throw new Error(`Expected ${count} resources of type ${type} but found ${counted}`);
        }
    }
    /**
     * Assert that the given number of resources of the given type and properties exists in the
     * CloudFormation template.
     * @param type the resource type; ex: `AWS::S3::Bucket`
     * @param props the 'Properties' section of the resource as should be expected in the template.
     * @param count number of expected instances
     */
    resourcePropertiesCountIs(type, props, count) {
        const counted = (0, resources_1.countResourcesProperties)(this.template, type, props);
        if (counted !== count) {
            throw new Error(`Expected ${count} resources of type ${type} but found ${counted}`);
        }
    }
    /**
     * Assert that a resource of the given type and properties exists in the
     * CloudFormation template.
     * By default, performs partial matching on the `Properties` key of the resource, via the
     * `Match.objectLike()`. To configure different behavior, use other matchers in the `Match` class.
     * @param type the resource type; ex: `AWS::S3::Bucket`
     * @param props the 'Properties' section of the resource as should be expected in the template.
     */
    hasResourceProperties(type, props) {
        const matchError = (0, resources_1.hasResourceProperties)(this.template, type, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Assert that a resource of the given type and given definition exists in the
     * CloudFormation template.
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavior, use other matchers in the `Match` class.
     * @param type the resource type; ex: `AWS::S3::Bucket`
     * @param props the entire definition of the resource as should be expected in the template.
     */
    hasResource(type, props) {
        const matchError = (0, resources_1.hasResource)(this.template, type, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching resources of a given type and properties in the CloudFormation template.
     * @param type the type to match in the CloudFormation template
     * @param props by default, matches all resources with the given type.
     * When a literal is provided, performs a partial match via `Match.objectLike()`.
     * Use the `Match` APIs to configure a different behaviour.
     */
    findResources(type, props = {}) {
        return (0, resources_1.findResources)(this.template, type, props);
    }
    /**
     * Assert that all resources of the given type contain the given definition in the
     * CloudFormation template.
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavior, use other matchers in the `Match` class.
     * @param type the resource type; ex: `AWS::S3::Bucket`
     * @param props the entire definition of the resources as they should be expected in the template.
     */
    allResources(type, props) {
        const matchError = (0, resources_1.allResources)(this.template, type, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Assert that all resources of the given type contain the given properties
     * CloudFormation template.
     * By default, performs partial matching on the `Properties` key of the resource, via the
     * `Match.objectLike()`. To configure different behavior, use other matchers in the `Match` class.
     * @param type the resource type; ex: `AWS::S3::Bucket`
     * @param props the 'Properties' section of the resource as should be expected in the template.
     */
    allResourcesProperties(type, props) {
        const matchError = (0, resources_1.allResourcesProperties)(this.template, type, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Assert that a Parameter with the given properties exists in the CloudFormation template.
     * By default, performs partial matching on the parameter, via the `Match.objectLike()`.
     * To configure different behavior, use other matchers in the `Match` class.
     * @param logicalId the name of the parameter. Provide `'*'` to match all parameters in the template.
     * @param props the parameter as should be expected in the template.
     */
    hasParameter(logicalId, props) {
        const matchError = (0, parameters_1.hasParameter)(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Parameters that match the given properties in the CloudFormation template.
     * @param logicalId the name of the parameter. Provide `'*'` to match all parameters in the template.
     * @param props by default, matches all Parameters in the template.
     * When a literal object is provided, performs a partial match via `Match.objectLike()`.
     * Use the `Match` APIs to configure a different behaviour.
     */
    findParameters(logicalId, props = {}) {
        return (0, parameters_1.findParameters)(this.template, logicalId, props);
    }
    /**
     * Assert that an Output with the given properties exists in the CloudFormation template.
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavior, use other matchers in the `Match` class.
     * @param logicalId the name of the output. Provide `'*'` to match all outputs in the template.
     * @param props the output as should be expected in the template.
     */
    hasOutput(logicalId, props) {
        const matchError = (0, outputs_1.hasOutput)(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Outputs that match the given properties in the CloudFormation template.
     * @param logicalId the name of the output. Provide `'*'` to match all outputs in the template.
     * @param props by default, matches all Outputs in the template.
     * When a literal object is provided, performs a partial match via `Match.objectLike()`.
     * Use the `Match` APIs to configure a different behaviour.
     */
    findOutputs(logicalId, props = {}) {
        return (0, outputs_1.findOutputs)(this.template, logicalId, props);
    }
    /**
     * Assert that a Mapping with the given properties exists in the CloudFormation template.
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavior, use other matchers in the `Match` class.
     * @param logicalId the name of the mapping. Provide `'*'` to match all mappings in the template.
     * @param props the output as should be expected in the template.
     */
    hasMapping(logicalId, props) {
        const matchError = (0, mappings_1.hasMapping)(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Mappings that match the given properties in the CloudFormation template.
     * @param logicalId the name of the mapping. Provide `'*'` to match all mappings in the template.
     * @param props by default, matches all Mappings in the template.
     * When a literal object is provided, performs a partial match via `Match.objectLike()`.
     * Use the `Match` APIs to configure a different behaviour.
     */
    findMappings(logicalId, props = {}) {
        return (0, mappings_1.findMappings)(this.template, logicalId, props);
    }
    /**
     * Assert that a Condition with the given properties exists in the CloudFormation template.
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavior, use other matchers in the `Match` class.
     * @param logicalId the name of the mapping. Provide `'*'` to match all conditions in the template.
     * @param props the output as should be expected in the template.
     */
    hasCondition(logicalId, props) {
        const matchError = (0, conditions_1.hasCondition)(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Conditions that match the given properties in the CloudFormation template.
     * @param logicalId the name of the condition. Provide `'*'` to match all conditions in the template.
     * @param props by default, matches all Conditions in the template.
     * When a literal object is provided, performs a partial match via `Match.objectLike()`.
     * Use the `Match` APIs to configure a different behaviour.
     */
    findConditions(logicalId, props = {}) {
        return (0, conditions_1.findConditions)(this.template, logicalId, props);
    }
    /**
     * Assert that the CloudFormation template matches the given value
     * @param expected the expected CloudFormation template as key-value pairs.
     */
    templateMatches(expected) {
        const matcher = matcher_1.Matcher.isMatcher(expected) ? expected : match_1.Match.objectLike(expected);
        const result = matcher.test(this.template);
        if (result.hasFailed()) {
            throw new Error([
                'Template did not match as expected. The following mismatches were found:',
                ...result.toHumanStrings().map(s => `\t${s}`),
            ].join('\n'));
        }
    }
}
_a = JSII_RTTI_SYMBOL_1;
Template[_a] = { fqn: "aws-cdk-lib.assertions.Template", version: "2.74.0" };
exports.Template = Template;
function toTemplate(stack) {
    const root = stack.node.root;
    if (!core_1.Stage.isStage(root)) {
        throw new Error('unexpected: all stacks must be part of a Stage or an App');
    }
    const assembly = root.synth();
    if (stack.nestedStackParent) {
        // if this is a nested stack (it has a parent), then just read the template as a string
        return JSON.parse(fs.readFileSync(path.join(assembly.directory, stack.templateFile)).toString('utf-8'));
    }
    return assembly.getStackArtifact(stack.artifactId).template;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVtcGxhdGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZW1wbGF0ZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSw2QkFBNkI7QUFDN0IscUNBQTBDO0FBQzFDLCtCQUErQjtBQUMvQixtQ0FBZ0M7QUFDaEMsdUNBQW9DO0FBQ3BDLHFEQUFvRTtBQUNwRSw2Q0FBc0U7QUFDdEUsaURBQThEO0FBQzlELCtDQUEyRDtBQUMzRCxxREFBb0U7QUFDcEUsbURBQXdLO0FBR3hLOzs7O0dBSUc7QUFDSCxNQUFhLFFBQVE7SUFFbkI7Ozs7O09BS0c7SUFDSSxNQUFNLENBQUMsU0FBUyxDQUFDLEtBQVksRUFBRSxzQkFBK0M7Ozs7Ozs7Ozs7O1FBQ25GLE9BQU8sSUFBSSxRQUFRLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxFQUFFLHNCQUFzQixDQUFDLENBQUM7S0FDaEU7SUFFRDs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsUUFBUSxDQUFDLFFBQWlDLEVBQUUsc0JBQStDOzs7Ozs7Ozs7O1FBQ3ZHLE9BQU8sSUFBSSxRQUFRLENBQUMsUUFBUSxFQUFFLHNCQUFzQixDQUFDLENBQUM7S0FDdkQ7SUFFRDs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsVUFBVSxDQUFDLFFBQWdCLEVBQUUsc0JBQStDOzs7Ozs7Ozs7O1FBQ3hGLE9BQU8sSUFBSSxRQUFRLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO0tBQ25FO0lBSUQsWUFBb0IsUUFBZ0MsRUFBRSx5QkFBaUQsRUFBRTtRQUN2RyxJQUFJLENBQUMsUUFBUSxHQUFHLFFBQXdCLENBQUM7UUFDekMsSUFBSSxDQUFDLHNCQUFzQixFQUFFLDZCQUE2QixJQUFJLElBQUksRUFBRTtZQUNsRSxJQUFBLDJDQUFrQyxFQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQztTQUNuRDtLQUNGO0lBRUQ7O09BRUc7SUFDSSxNQUFNO1FBQ1gsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDO0tBQ3RCO0lBRUQ7Ozs7O09BS0c7SUFDSSxlQUFlLENBQUMsSUFBWSxFQUFFLEtBQWE7UUFDaEQsTUFBTSxPQUFPLEdBQUcsSUFBQSwwQkFBYyxFQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDcEQsSUFBSSxPQUFPLEtBQUssS0FBSyxFQUFFO1lBQ3JCLE1BQU0sSUFBSSxLQUFLLENBQUMsWUFBWSxLQUFLLHNCQUFzQixJQUFJLGNBQWMsT0FBTyxFQUFFLENBQUMsQ0FBQztTQUNyRjtLQUNGO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCLENBQUMsSUFBWSxFQUFFLEtBQVUsRUFBRSxLQUFhO1FBQ3RFLE1BQU0sT0FBTyxHQUFHLElBQUEsb0NBQXdCLEVBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxJQUFJLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDckUsSUFBSSxPQUFPLEtBQUssS0FBSyxFQUFFO1lBQ3JCLE1BQU0sSUFBSSxLQUFLLENBQUMsWUFBWSxLQUFLLHNCQUFzQixJQUFJLGNBQWMsT0FBTyxFQUFFLENBQUMsQ0FBQztTQUNyRjtLQUNGO0lBRUQ7Ozs7Ozs7T0FPRztJQUNJLHFCQUFxQixDQUFDLElBQVksRUFBRSxLQUFVO1FBQ25ELE1BQU0sVUFBVSxHQUFHLElBQUEsaUNBQXFCLEVBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxJQUFJLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDckUsSUFBSSxVQUFVLEVBQUU7WUFDZCxNQUFNLElBQUksS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQzdCO0tBQ0Y7SUFFRDs7Ozs7OztPQU9HO0lBQ0ksV0FBVyxDQUFDLElBQVksRUFBRSxLQUFVO1FBQ3pDLE1BQU0sVUFBVSxHQUFHLElBQUEsdUJBQVcsRUFBQyxJQUFJLENBQUMsUUFBUSxFQUFFLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztRQUMzRCxJQUFJLFVBQVUsRUFBRTtZQUNkLE1BQU0sSUFBSSxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUM7U0FDN0I7S0FDRjtJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWEsQ0FBQyxJQUFZLEVBQUUsUUFBYSxFQUFFO1FBQ2hELE9BQU8sSUFBQSx5QkFBYSxFQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsSUFBSSxFQUFFLEtBQUssQ0FBQyxDQUFDO0tBQ2xEO0lBRUQ7Ozs7Ozs7T0FPRztJQUNJLFlBQVksQ0FBQyxJQUFZLEVBQUUsS0FBVTtRQUMxQyxNQUFNLFVBQVUsR0FBRyxJQUFBLHdCQUFZLEVBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxJQUFJLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDNUQsSUFBSSxVQUFVLEVBQUU7WUFDZCxNQUFNLElBQUksS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQzdCO0tBQ0Y7SUFFRDs7Ozs7OztPQU9HO0lBQ0ksc0JBQXNCLENBQUMsSUFBWSxFQUFFLEtBQVU7UUFDcEQsTUFBTSxVQUFVLEdBQUcsSUFBQSxrQ0FBc0IsRUFBQyxJQUFJLENBQUMsUUFBUSxFQUFFLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztRQUN0RSxJQUFJLFVBQVUsRUFBRTtZQUNkLE1BQU0sSUFBSSxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUM7U0FDN0I7S0FDRjtJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVksQ0FBQyxTQUFpQixFQUFFLEtBQVU7UUFDL0MsTUFBTSxVQUFVLEdBQUcsSUFBQSx5QkFBWSxFQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ2pFLElBQUksVUFBVSxFQUFFO1lBQ2QsTUFBTSxJQUFJLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQztTQUM3QjtLQUNGO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYyxDQUFDLFNBQWlCLEVBQUUsUUFBYSxFQUFFO1FBQ3RELE9BQU8sSUFBQSwyQkFBYyxFQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO0tBQ3hEO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksU0FBUyxDQUFDLFNBQWlCLEVBQUUsS0FBVTtRQUM1QyxNQUFNLFVBQVUsR0FBRyxJQUFBLG1CQUFTLEVBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxTQUFTLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDOUQsSUFBSSxVQUFVLEVBQUU7WUFDZCxNQUFNLElBQUksS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQzdCO0tBQ0Y7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXLENBQUMsU0FBaUIsRUFBRSxRQUFhLEVBQUU7UUFDbkQsT0FBTyxJQUFBLHFCQUFXLEVBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxTQUFTLEVBQUUsS0FBSyxDQUFDLENBQUM7S0FDckQ7SUFFRDs7Ozs7O09BTUc7SUFDSSxVQUFVLENBQUMsU0FBaUIsRUFBRSxLQUFVO1FBQzdDLE1BQU0sVUFBVSxHQUFHLElBQUEscUJBQVUsRUFBQyxJQUFJLENBQUMsUUFBUSxFQUFFLFNBQVMsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUMvRCxJQUFJLFVBQVUsRUFBRTtZQUNkLE1BQU0sSUFBSSxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUM7U0FDN0I7S0FDRjtJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVksQ0FBQyxTQUFpQixFQUFFLFFBQWEsRUFBRTtRQUNwRCxPQUFPLElBQUEsdUJBQVksRUFBQyxJQUFJLENBQUMsUUFBUSxFQUFFLFNBQVMsRUFBRSxLQUFLLENBQUMsQ0FBQztLQUN0RDtJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVksQ0FBQyxTQUFpQixFQUFFLEtBQVU7UUFDL0MsTUFBTSxVQUFVLEdBQUcsSUFBQSx5QkFBWSxFQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ2pFLElBQUksVUFBVSxFQUFFO1lBQ2QsTUFBTSxJQUFJLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQztTQUM3QjtLQUNGO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYyxDQUFDLFNBQWlCLEVBQUUsUUFBYSxFQUFFO1FBQ3RELE9BQU8sSUFBQSwyQkFBYyxFQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO0tBQ3hEO0lBRUQ7OztPQUdHO0lBQ0ksZUFBZSxDQUFDLFFBQWE7UUFDbEMsTUFBTSxPQUFPLEdBQUcsaUJBQU8sQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsYUFBSyxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUNwRixNQUFNLE1BQU0sR0FBRyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUUzQyxJQUFJLE1BQU0sQ0FBQyxTQUFTLEVBQUUsRUFBRTtZQUN0QixNQUFNLElBQUksS0FBSyxDQUFDO2dCQUNkLDBFQUEwRTtnQkFDMUUsR0FBRyxNQUFNLENBQUMsY0FBYyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQzthQUM5QyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1NBQ2Y7S0FDRjs7OztBQXRRVSw0QkFBUTtBQXVSckIsU0FBUyxVQUFVLENBQUMsS0FBWTtJQUM5QixNQUFNLElBQUksR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQztJQUM3QixJQUFJLENBQUMsWUFBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRTtRQUN4QixNQUFNLElBQUksS0FBSyxDQUFDLDBEQUEwRCxDQUFDLENBQUM7S0FDN0U7SUFFRCxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDOUIsSUFBSSxLQUFLLENBQUMsaUJBQWlCLEVBQUU7UUFDM0IsdUZBQXVGO1FBQ3ZGLE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRSxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztLQUN6RztJQUNELE9BQU8sUUFBUSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxRQUFRLENBQUM7QUFDOUQsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBTdGFjaywgU3RhZ2UgfSBmcm9tICcuLi8uLi9jb3JlJztcbmltcG9ydCAqIGFzIGZzIGZyb20gJ2ZzLWV4dHJhJztcbmltcG9ydCB7IE1hdGNoIH0gZnJvbSAnLi9tYXRjaCc7XG5pbXBvcnQgeyBNYXRjaGVyIH0gZnJvbSAnLi9tYXRjaGVyJztcbmltcG9ydCB7IGZpbmRDb25kaXRpb25zLCBoYXNDb25kaXRpb24gfSBmcm9tICcuL3ByaXZhdGUvY29uZGl0aW9ucyc7XG5pbXBvcnQgeyBjaGVja1RlbXBsYXRlRm9yQ3ljbGljRGVwZW5kZW5jaWVzIH0gZnJvbSAnLi9wcml2YXRlL2N5Y2xpYyc7XG5pbXBvcnQgeyBmaW5kTWFwcGluZ3MsIGhhc01hcHBpbmcgfSBmcm9tICcuL3ByaXZhdGUvbWFwcGluZ3MnO1xuaW1wb3J0IHsgZmluZE91dHB1dHMsIGhhc091dHB1dCB9IGZyb20gJy4vcHJpdmF0ZS9vdXRwdXRzJztcbmltcG9ydCB7IGZpbmRQYXJhbWV0ZXJzLCBoYXNQYXJhbWV0ZXIgfSBmcm9tICcuL3ByaXZhdGUvcGFyYW1ldGVycyc7XG5pbXBvcnQgeyBhbGxSZXNvdXJjZXMsIGFsbFJlc291cmNlc1Byb3BlcnRpZXMsIGNvdW50UmVzb3VyY2VzLCBjb3VudFJlc291cmNlc1Byb3BlcnRpZXMsIGZpbmRSZXNvdXJjZXMsIGhhc1Jlc291cmNlLCBoYXNSZXNvdXJjZVByb3BlcnRpZXMgfSBmcm9tICcuL3ByaXZhdGUvcmVzb3VyY2VzJztcbmltcG9ydCB7IFRlbXBsYXRlIGFzIFRlbXBsYXRlVHlwZSB9IGZyb20gJy4vcHJpdmF0ZS90ZW1wbGF0ZSc7XG5cbi8qKlxuICogU3VpdGUgb2YgYXNzZXJ0aW9ucyB0aGF0IGNhbiBiZSBydW4gb24gYSBDREsgc3RhY2suXG4gKiBUeXBpY2FsbHkgdXNlZCwgYXMgcGFydCBvZiB1bml0IHRlc3RzLCB0byB2YWxpZGF0ZSB0aGF0IHRoZSByZW5kZXJlZFxuICogQ2xvdWRGb3JtYXRpb24gdGVtcGxhdGUgaGFzIGV4cGVjdGVkIHJlc291cmNlcyBhbmQgcHJvcGVydGllcy5cbiAqL1xuZXhwb3J0IGNsYXNzIFRlbXBsYXRlIHtcblxuICAvKipcbiAgICogQmFzZSB5b3VyIGFzc2VydGlvbnMgb24gdGhlIENsb3VkRm9ybWF0aW9uIHRlbXBsYXRlIHN5bnRoZXNpemVkIGJ5IGEgQ0RLIGBTdGFja2AuXG4gICAqIEBwYXJhbSBzdGFjayB0aGUgQ0RLIFN0YWNrIHRvIHJ1biBhc3NlcnRpb25zIG9uXG4gICAqIEBwYXJhbSB0ZW1wbGF0ZVBhcnNpbmdPcHRpb25zIE9wdGlvbmFsIHBhcmFtIHRvIGNvbmZpZ3VyZSB0ZW1wbGF0ZSBwYXJzaW5nIGJlaGF2aW9yLCBzdWNoIGFzIGRpc3JlZ2FyZGluZyBjaXJjdWxhclxuICAgKiBkZXBlbmRlbmNpZXMuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21TdGFjayhzdGFjazogU3RhY2ssIHRlbXBsYXRlUGFyc2luZ09wdGlvbnM/OiBUZW1wbGF0ZVBhcnNpbmdPcHRpb25zKTogVGVtcGxhdGUge1xuICAgIHJldHVybiBuZXcgVGVtcGxhdGUodG9UZW1wbGF0ZShzdGFjayksIHRlbXBsYXRlUGFyc2luZ09wdGlvbnMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEJhc2UgeW91ciBhc3NlcnRpb25zIGZyb20gYW4gZXhpc3RpbmcgQ2xvdWRGb3JtYXRpb24gdGVtcGxhdGUgZm9ybWF0dGVkIGFzIGFuIGluLW1lbW9yeVxuICAgKiBKU09OIG9iamVjdC5cbiAgICogQHBhcmFtIHRlbXBsYXRlIHRoZSBDbG91ZEZvcm1hdGlvbiB0ZW1wbGF0ZSBmb3JtYXR0ZWQgYXMgYSBuZXN0ZWQgc2V0IG9mIHJlY29yZHNcbiAgICogQHBhcmFtIHRlbXBsYXRlUGFyc2luZ09wdGlvbnMgT3B0aW9uYWwgcGFyYW0gdG8gY29uZmlndXJlIHRlbXBsYXRlIHBhcnNpbmcgYmVoYXZpb3IsIHN1Y2ggYXMgZGlzcmVnYXJkaW5nIGNpcmN1bGFyXG4gICAqIGRlcGVuZGVuY2llcy5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUpTT04odGVtcGxhdGU6IHsgW2tleTogc3RyaW5nXSA6IGFueSB9LCB0ZW1wbGF0ZVBhcnNpbmdPcHRpb25zPzogVGVtcGxhdGVQYXJzaW5nT3B0aW9ucyk6IFRlbXBsYXRlIHtcbiAgICByZXR1cm4gbmV3IFRlbXBsYXRlKHRlbXBsYXRlLCB0ZW1wbGF0ZVBhcnNpbmdPcHRpb25zKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBCYXNlIHlvdXIgYXNzZXJ0aW9ucyBmcm9tIGFuIGV4aXN0aW5nIENsb3VkRm9ybWF0aW9uIHRlbXBsYXRlIGZvcm1hdHRlZCBhcyBhXG4gICAqIEpTT04gc3RyaW5nLlxuICAgKiBAcGFyYW0gdGVtcGxhdGUgdGhlIENsb3VkRm9ybWF0aW9uIHRlbXBsYXRlIGluXG4gICAqIEBwYXJhbSB0ZW1wbGF0ZVBhcnNpbmdPcHRpb25zIE9wdGlvbmFsIHBhcmFtIHRvIGNvbmZpZ3VyZSB0ZW1wbGF0ZSBwYXJzaW5nIGJlaGF2aW9yLCBzdWNoIGFzIGRpc3JlZ2FyZGluZyBjaXJjdWxhclxuICAgKiBkZXBlbmRlbmNpZXMuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21TdHJpbmcodGVtcGxhdGU6IHN0cmluZywgdGVtcGxhdGVQYXJzaW5nT3B0aW9ucz86IFRlbXBsYXRlUGFyc2luZ09wdGlvbnMpOiBUZW1wbGF0ZSB7XG4gICAgcmV0dXJuIG5ldyBUZW1wbGF0ZShKU09OLnBhcnNlKHRlbXBsYXRlKSwgdGVtcGxhdGVQYXJzaW5nT3B0aW9ucyk7XG4gIH1cblxuICBwcml2YXRlIHJlYWRvbmx5IHRlbXBsYXRlOiBUZW1wbGF0ZVR5cGU7XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3Rvcih0ZW1wbGF0ZTogeyBba2V5OiBzdHJpbmddOiBhbnkgfSwgdGVtcGxhdGVQYXJzaW5nT3B0aW9uczogVGVtcGxhdGVQYXJzaW5nT3B0aW9ucyA9IHt9KSB7XG4gICAgdGhpcy50ZW1wbGF0ZSA9IHRlbXBsYXRlIGFzIFRlbXBsYXRlVHlwZTtcbiAgICBpZiAoIXRlbXBsYXRlUGFyc2luZ09wdGlvbnM/LnNraXBDeWNsaWNhbERlcGVuZGVuY2llc0NoZWNrID8/IHRydWUpIHtcbiAgICAgIGNoZWNrVGVtcGxhdGVGb3JDeWNsaWNEZXBlbmRlbmNpZXModGhpcy50ZW1wbGF0ZSk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBDbG91ZEZvcm1hdGlvbiB0ZW1wbGF0ZSBkZXNlcmlhbGl6ZWQgaW50byBhbiBvYmplY3QuXG4gICAqL1xuICBwdWJsaWMgdG9KU09OKCk6IHsgW2tleTogc3RyaW5nXTogYW55IH0ge1xuICAgIHJldHVybiB0aGlzLnRlbXBsYXRlO1xuICB9XG5cbiAgLyoqXG4gICAqIEFzc2VydCB0aGF0IHRoZSBnaXZlbiBudW1iZXIgb2YgcmVzb3VyY2VzIG9mIHRoZSBnaXZlbiB0eXBlIGV4aXN0IGluIHRoZVxuICAgKiB0ZW1wbGF0ZS5cbiAgICogQHBhcmFtIHR5cGUgdGhlIHJlc291cmNlIHR5cGU7IGV4OiBgQVdTOjpTMzo6QnVja2V0YFxuICAgKiBAcGFyYW0gY291bnQgbnVtYmVyIG9mIGV4cGVjdGVkIGluc3RhbmNlc1xuICAgKi9cbiAgcHVibGljIHJlc291cmNlQ291bnRJcyh0eXBlOiBzdHJpbmcsIGNvdW50OiBudW1iZXIpOiB2b2lkIHtcbiAgICBjb25zdCBjb3VudGVkID0gY291bnRSZXNvdXJjZXModGhpcy50ZW1wbGF0ZSwgdHlwZSk7XG4gICAgaWYgKGNvdW50ZWQgIT09IGNvdW50KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEV4cGVjdGVkICR7Y291bnR9IHJlc291cmNlcyBvZiB0eXBlICR7dHlwZX0gYnV0IGZvdW5kICR7Y291bnRlZH1gKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQXNzZXJ0IHRoYXQgdGhlIGdpdmVuIG51bWJlciBvZiByZXNvdXJjZXMgb2YgdGhlIGdpdmVuIHR5cGUgYW5kIHByb3BlcnRpZXMgZXhpc3RzIGluIHRoZVxuICAgKiBDbG91ZEZvcm1hdGlvbiB0ZW1wbGF0ZS5cbiAgICogQHBhcmFtIHR5cGUgdGhlIHJlc291cmNlIHR5cGU7IGV4OiBgQVdTOjpTMzo6QnVja2V0YFxuICAgKiBAcGFyYW0gcHJvcHMgdGhlICdQcm9wZXJ0aWVzJyBzZWN0aW9uIG9mIHRoZSByZXNvdXJjZSBhcyBzaG91bGQgYmUgZXhwZWN0ZWQgaW4gdGhlIHRlbXBsYXRlLlxuICAgKiBAcGFyYW0gY291bnQgbnVtYmVyIG9mIGV4cGVjdGVkIGluc3RhbmNlc1xuICAgKi9cbiAgcHVibGljIHJlc291cmNlUHJvcGVydGllc0NvdW50SXModHlwZTogc3RyaW5nLCBwcm9wczogYW55LCBjb3VudDogbnVtYmVyKTogdm9pZCB7XG4gICAgY29uc3QgY291bnRlZCA9IGNvdW50UmVzb3VyY2VzUHJvcGVydGllcyh0aGlzLnRlbXBsYXRlLCB0eXBlLCBwcm9wcyk7XG4gICAgaWYgKGNvdW50ZWQgIT09IGNvdW50KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEV4cGVjdGVkICR7Y291bnR9IHJlc291cmNlcyBvZiB0eXBlICR7dHlwZX0gYnV0IGZvdW5kICR7Y291bnRlZH1gKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQXNzZXJ0IHRoYXQgYSByZXNvdXJjZSBvZiB0aGUgZ2l2ZW4gdHlwZSBhbmQgcHJvcGVydGllcyBleGlzdHMgaW4gdGhlXG4gICAqIENsb3VkRm9ybWF0aW9uIHRlbXBsYXRlLlxuICAgKiBCeSBkZWZhdWx0LCBwZXJmb3JtcyBwYXJ0aWFsIG1hdGNoaW5nIG9uIHRoZSBgUHJvcGVydGllc2Aga2V5IG9mIHRoZSByZXNvdXJjZSwgdmlhIHRoZVxuICAgKiBgTWF0Y2gub2JqZWN0TGlrZSgpYC4gVG8gY29uZmlndXJlIGRpZmZlcmVudCBiZWhhdmlvciwgdXNlIG90aGVyIG1hdGNoZXJzIGluIHRoZSBgTWF0Y2hgIGNsYXNzLlxuICAgKiBAcGFyYW0gdHlwZSB0aGUgcmVzb3VyY2UgdHlwZTsgZXg6IGBBV1M6OlMzOjpCdWNrZXRgXG4gICAqIEBwYXJhbSBwcm9wcyB0aGUgJ1Byb3BlcnRpZXMnIHNlY3Rpb24gb2YgdGhlIHJlc291cmNlIGFzIHNob3VsZCBiZSBleHBlY3RlZCBpbiB0aGUgdGVtcGxhdGUuXG4gICAqL1xuICBwdWJsaWMgaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKHR5cGU6IHN0cmluZywgcHJvcHM6IGFueSk6IHZvaWQge1xuICAgIGNvbnN0IG1hdGNoRXJyb3IgPSBoYXNSZXNvdXJjZVByb3BlcnRpZXModGhpcy50ZW1wbGF0ZSwgdHlwZSwgcHJvcHMpO1xuICAgIGlmIChtYXRjaEVycm9yKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IobWF0Y2hFcnJvcik7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEFzc2VydCB0aGF0IGEgcmVzb3VyY2Ugb2YgdGhlIGdpdmVuIHR5cGUgYW5kIGdpdmVuIGRlZmluaXRpb24gZXhpc3RzIGluIHRoZVxuICAgKiBDbG91ZEZvcm1hdGlvbiB0ZW1wbGF0ZS5cbiAgICogQnkgZGVmYXVsdCwgcGVyZm9ybXMgcGFydGlhbCBtYXRjaGluZyBvbiB0aGUgcmVzb3VyY2UsIHZpYSB0aGUgYE1hdGNoLm9iamVjdExpa2UoKWAuXG4gICAqIFRvIGNvbmZpZ3VyZSBkaWZmZXJlbnQgYmVoYXZpb3IsIHVzZSBvdGhlciBtYXRjaGVycyBpbiB0aGUgYE1hdGNoYCBjbGFzcy5cbiAgICogQHBhcmFtIHR5cGUgdGhlIHJlc291cmNlIHR5cGU7IGV4OiBgQVdTOjpTMzo6QnVja2V0YFxuICAgKiBAcGFyYW0gcHJvcHMgdGhlIGVudGlyZSBkZWZpbml0aW9uIG9mIHRoZSByZXNvdXJjZSBhcyBzaG91bGQgYmUgZXhwZWN0ZWQgaW4gdGhlIHRlbXBsYXRlLlxuICAgKi9cbiAgcHVibGljIGhhc1Jlc291cmNlKHR5cGU6IHN0cmluZywgcHJvcHM6IGFueSk6IHZvaWQge1xuICAgIGNvbnN0IG1hdGNoRXJyb3IgPSBoYXNSZXNvdXJjZSh0aGlzLnRlbXBsYXRlLCB0eXBlLCBwcm9wcyk7XG4gICAgaWYgKG1hdGNoRXJyb3IpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihtYXRjaEVycm9yKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogR2V0IHRoZSBzZXQgb2YgbWF0Y2hpbmcgcmVzb3VyY2VzIG9mIGEgZ2l2ZW4gdHlwZSBhbmQgcHJvcGVydGllcyBpbiB0aGUgQ2xvdWRGb3JtYXRpb24gdGVtcGxhdGUuXG4gICAqIEBwYXJhbSB0eXBlIHRoZSB0eXBlIHRvIG1hdGNoIGluIHRoZSBDbG91ZEZvcm1hdGlvbiB0ZW1wbGF0ZVxuICAgKiBAcGFyYW0gcHJvcHMgYnkgZGVmYXVsdCwgbWF0Y2hlcyBhbGwgcmVzb3VyY2VzIHdpdGggdGhlIGdpdmVuIHR5cGUuXG4gICAqIFdoZW4gYSBsaXRlcmFsIGlzIHByb3ZpZGVkLCBwZXJmb3JtcyBhIHBhcnRpYWwgbWF0Y2ggdmlhIGBNYXRjaC5vYmplY3RMaWtlKClgLlxuICAgKiBVc2UgdGhlIGBNYXRjaGAgQVBJcyB0byBjb25maWd1cmUgYSBkaWZmZXJlbnQgYmVoYXZpb3VyLlxuICAgKi9cbiAgcHVibGljIGZpbmRSZXNvdXJjZXModHlwZTogc3RyaW5nLCBwcm9wczogYW55ID0ge30pOiB7IFtrZXk6IHN0cmluZ106IHsgW2tleTogc3RyaW5nXTogYW55IH0gfSB7XG4gICAgcmV0dXJuIGZpbmRSZXNvdXJjZXModGhpcy50ZW1wbGF0ZSwgdHlwZSwgcHJvcHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFzc2VydCB0aGF0IGFsbCByZXNvdXJjZXMgb2YgdGhlIGdpdmVuIHR5cGUgY29udGFpbiB0aGUgZ2l2ZW4gZGVmaW5pdGlvbiBpbiB0aGVcbiAgICogQ2xvdWRGb3JtYXRpb24gdGVtcGxhdGUuXG4gICAqIEJ5IGRlZmF1bHQsIHBlcmZvcm1zIHBhcnRpYWwgbWF0Y2hpbmcgb24gdGhlIHJlc291cmNlLCB2aWEgdGhlIGBNYXRjaC5vYmplY3RMaWtlKClgLlxuICAgKiBUbyBjb25maWd1cmUgZGlmZmVyZW50IGJlaGF2aW9yLCB1c2Ugb3RoZXIgbWF0Y2hlcnMgaW4gdGhlIGBNYXRjaGAgY2xhc3MuXG4gICAqIEBwYXJhbSB0eXBlIHRoZSByZXNvdXJjZSB0eXBlOyBleDogYEFXUzo6UzM6OkJ1Y2tldGBcbiAgICogQHBhcmFtIHByb3BzIHRoZSBlbnRpcmUgZGVmaW5pdGlvbiBvZiB0aGUgcmVzb3VyY2VzIGFzIHRoZXkgc2hvdWxkIGJlIGV4cGVjdGVkIGluIHRoZSB0ZW1wbGF0ZS5cbiAgICovXG4gIHB1YmxpYyBhbGxSZXNvdXJjZXModHlwZTogc3RyaW5nLCBwcm9wczogYW55KTogdm9pZCB7XG4gICAgY29uc3QgbWF0Y2hFcnJvciA9IGFsbFJlc291cmNlcyh0aGlzLnRlbXBsYXRlLCB0eXBlLCBwcm9wcyk7XG4gICAgaWYgKG1hdGNoRXJyb3IpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihtYXRjaEVycm9yKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQXNzZXJ0IHRoYXQgYWxsIHJlc291cmNlcyBvZiB0aGUgZ2l2ZW4gdHlwZSBjb250YWluIHRoZSBnaXZlbiBwcm9wZXJ0aWVzXG4gICAqIENsb3VkRm9ybWF0aW9uIHRlbXBsYXRlLlxuICAgKiBCeSBkZWZhdWx0LCBwZXJmb3JtcyBwYXJ0aWFsIG1hdGNoaW5nIG9uIHRoZSBgUHJvcGVydGllc2Aga2V5IG9mIHRoZSByZXNvdXJjZSwgdmlhIHRoZVxuICAgKiBgTWF0Y2gub2JqZWN0TGlrZSgpYC4gVG8gY29uZmlndXJlIGRpZmZlcmVudCBiZWhhdmlvciwgdXNlIG90aGVyIG1hdGNoZXJzIGluIHRoZSBgTWF0Y2hgIGNsYXNzLlxuICAgKiBAcGFyYW0gdHlwZSB0aGUgcmVzb3VyY2UgdHlwZTsgZXg6IGBBV1M6OlMzOjpCdWNrZXRgXG4gICAqIEBwYXJhbSBwcm9wcyB0aGUgJ1Byb3BlcnRpZXMnIHNlY3Rpb24gb2YgdGhlIHJlc291cmNlIGFzIHNob3VsZCBiZSBleHBlY3RlZCBpbiB0aGUgdGVtcGxhdGUuXG4gICAqL1xuICBwdWJsaWMgYWxsUmVzb3VyY2VzUHJvcGVydGllcyh0eXBlOiBzdHJpbmcsIHByb3BzOiBhbnkpOiB2b2lkIHtcbiAgICBjb25zdCBtYXRjaEVycm9yID0gYWxsUmVzb3VyY2VzUHJvcGVydGllcyh0aGlzLnRlbXBsYXRlLCB0eXBlLCBwcm9wcyk7XG4gICAgaWYgKG1hdGNoRXJyb3IpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihtYXRjaEVycm9yKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQXNzZXJ0IHRoYXQgYSBQYXJhbWV0ZXIgd2l0aCB0aGUgZ2l2ZW4gcHJvcGVydGllcyBleGlzdHMgaW4gdGhlIENsb3VkRm9ybWF0aW9uIHRlbXBsYXRlLlxuICAgKiBCeSBkZWZhdWx0LCBwZXJmb3JtcyBwYXJ0aWFsIG1hdGNoaW5nIG9uIHRoZSBwYXJhbWV0ZXIsIHZpYSB0aGUgYE1hdGNoLm9iamVjdExpa2UoKWAuXG4gICAqIFRvIGNvbmZpZ3VyZSBkaWZmZXJlbnQgYmVoYXZpb3IsIHVzZSBvdGhlciBtYXRjaGVycyBpbiB0aGUgYE1hdGNoYCBjbGFzcy5cbiAgICogQHBhcmFtIGxvZ2ljYWxJZCB0aGUgbmFtZSBvZiB0aGUgcGFyYW1ldGVyLiBQcm92aWRlIGAnKidgIHRvIG1hdGNoIGFsbCBwYXJhbWV0ZXJzIGluIHRoZSB0ZW1wbGF0ZS5cbiAgICogQHBhcmFtIHByb3BzIHRoZSBwYXJhbWV0ZXIgYXMgc2hvdWxkIGJlIGV4cGVjdGVkIGluIHRoZSB0ZW1wbGF0ZS5cbiAgICovXG4gIHB1YmxpYyBoYXNQYXJhbWV0ZXIobG9naWNhbElkOiBzdHJpbmcsIHByb3BzOiBhbnkpOiB2b2lkIHtcbiAgICBjb25zdCBtYXRjaEVycm9yID0gaGFzUGFyYW1ldGVyKHRoaXMudGVtcGxhdGUsIGxvZ2ljYWxJZCwgcHJvcHMpO1xuICAgIGlmIChtYXRjaEVycm9yKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IobWF0Y2hFcnJvcik7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEdldCB0aGUgc2V0IG9mIG1hdGNoaW5nIFBhcmFtZXRlcnMgdGhhdCBtYXRjaCB0aGUgZ2l2ZW4gcHJvcGVydGllcyBpbiB0aGUgQ2xvdWRGb3JtYXRpb24gdGVtcGxhdGUuXG4gICAqIEBwYXJhbSBsb2dpY2FsSWQgdGhlIG5hbWUgb2YgdGhlIHBhcmFtZXRlci4gUHJvdmlkZSBgJyonYCB0byBtYXRjaCBhbGwgcGFyYW1ldGVycyBpbiB0aGUgdGVtcGxhdGUuXG4gICAqIEBwYXJhbSBwcm9wcyBieSBkZWZhdWx0LCBtYXRjaGVzIGFsbCBQYXJhbWV0ZXJzIGluIHRoZSB0ZW1wbGF0ZS5cbiAgICogV2hlbiBhIGxpdGVyYWwgb2JqZWN0IGlzIHByb3ZpZGVkLCBwZXJmb3JtcyBhIHBhcnRpYWwgbWF0Y2ggdmlhIGBNYXRjaC5vYmplY3RMaWtlKClgLlxuICAgKiBVc2UgdGhlIGBNYXRjaGAgQVBJcyB0byBjb25maWd1cmUgYSBkaWZmZXJlbnQgYmVoYXZpb3VyLlxuICAgKi9cbiAgcHVibGljIGZpbmRQYXJhbWV0ZXJzKGxvZ2ljYWxJZDogc3RyaW5nLCBwcm9wczogYW55ID0ge30pOiB7IFtrZXk6IHN0cmluZ106IHsgW2tleTogc3RyaW5nXTogYW55IH0gfSB7XG4gICAgcmV0dXJuIGZpbmRQYXJhbWV0ZXJzKHRoaXMudGVtcGxhdGUsIGxvZ2ljYWxJZCwgcHJvcHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFzc2VydCB0aGF0IGFuIE91dHB1dCB3aXRoIHRoZSBnaXZlbiBwcm9wZXJ0aWVzIGV4aXN0cyBpbiB0aGUgQ2xvdWRGb3JtYXRpb24gdGVtcGxhdGUuXG4gICAqIEJ5IGRlZmF1bHQsIHBlcmZvcm1zIHBhcnRpYWwgbWF0Y2hpbmcgb24gdGhlIHJlc291cmNlLCB2aWEgdGhlIGBNYXRjaC5vYmplY3RMaWtlKClgLlxuICAgKiBUbyBjb25maWd1cmUgZGlmZmVyZW50IGJlaGF2aW9yLCB1c2Ugb3RoZXIgbWF0Y2hlcnMgaW4gdGhlIGBNYXRjaGAgY2xhc3MuXG4gICAqIEBwYXJhbSBsb2dpY2FsSWQgdGhlIG5hbWUgb2YgdGhlIG91dHB1dC4gUHJvdmlkZSBgJyonYCB0byBtYXRjaCBhbGwgb3V0cHV0cyBpbiB0aGUgdGVtcGxhdGUuXG4gICAqIEBwYXJhbSBwcm9wcyB0aGUgb3V0cHV0IGFzIHNob3VsZCBiZSBleHBlY3RlZCBpbiB0aGUgdGVtcGxhdGUuXG4gICAqL1xuICBwdWJsaWMgaGFzT3V0cHV0KGxvZ2ljYWxJZDogc3RyaW5nLCBwcm9wczogYW55KTogdm9pZCB7XG4gICAgY29uc3QgbWF0Y2hFcnJvciA9IGhhc091dHB1dCh0aGlzLnRlbXBsYXRlLCBsb2dpY2FsSWQsIHByb3BzKTtcbiAgICBpZiAobWF0Y2hFcnJvcikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKG1hdGNoRXJyb3IpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBHZXQgdGhlIHNldCBvZiBtYXRjaGluZyBPdXRwdXRzIHRoYXQgbWF0Y2ggdGhlIGdpdmVuIHByb3BlcnRpZXMgaW4gdGhlIENsb3VkRm9ybWF0aW9uIHRlbXBsYXRlLlxuICAgKiBAcGFyYW0gbG9naWNhbElkIHRoZSBuYW1lIG9mIHRoZSBvdXRwdXQuIFByb3ZpZGUgYCcqJ2AgdG8gbWF0Y2ggYWxsIG91dHB1dHMgaW4gdGhlIHRlbXBsYXRlLlxuICAgKiBAcGFyYW0gcHJvcHMgYnkgZGVmYXVsdCwgbWF0Y2hlcyBhbGwgT3V0cHV0cyBpbiB0aGUgdGVtcGxhdGUuXG4gICAqIFdoZW4gYSBsaXRlcmFsIG9iamVjdCBpcyBwcm92aWRlZCwgcGVyZm9ybXMgYSBwYXJ0aWFsIG1hdGNoIHZpYSBgTWF0Y2gub2JqZWN0TGlrZSgpYC5cbiAgICogVXNlIHRoZSBgTWF0Y2hgIEFQSXMgdG8gY29uZmlndXJlIGEgZGlmZmVyZW50IGJlaGF2aW91ci5cbiAgICovXG4gIHB1YmxpYyBmaW5kT3V0cHV0cyhsb2dpY2FsSWQ6IHN0cmluZywgcHJvcHM6IGFueSA9IHt9KTogeyBba2V5OiBzdHJpbmddOiB7IFtrZXk6IHN0cmluZ106IGFueSB9IH0ge1xuICAgIHJldHVybiBmaW5kT3V0cHV0cyh0aGlzLnRlbXBsYXRlLCBsb2dpY2FsSWQsIHByb3BzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBc3NlcnQgdGhhdCBhIE1hcHBpbmcgd2l0aCB0aGUgZ2l2ZW4gcHJvcGVydGllcyBleGlzdHMgaW4gdGhlIENsb3VkRm9ybWF0aW9uIHRlbXBsYXRlLlxuICAgKiBCeSBkZWZhdWx0LCBwZXJmb3JtcyBwYXJ0aWFsIG1hdGNoaW5nIG9uIHRoZSByZXNvdXJjZSwgdmlhIHRoZSBgTWF0Y2gub2JqZWN0TGlrZSgpYC5cbiAgICogVG8gY29uZmlndXJlIGRpZmZlcmVudCBiZWhhdmlvciwgdXNlIG90aGVyIG1hdGNoZXJzIGluIHRoZSBgTWF0Y2hgIGNsYXNzLlxuICAgKiBAcGFyYW0gbG9naWNhbElkIHRoZSBuYW1lIG9mIHRoZSBtYXBwaW5nLiBQcm92aWRlIGAnKidgIHRvIG1hdGNoIGFsbCBtYXBwaW5ncyBpbiB0aGUgdGVtcGxhdGUuXG4gICAqIEBwYXJhbSBwcm9wcyB0aGUgb3V0cHV0IGFzIHNob3VsZCBiZSBleHBlY3RlZCBpbiB0aGUgdGVtcGxhdGUuXG4gICAqL1xuICBwdWJsaWMgaGFzTWFwcGluZyhsb2dpY2FsSWQ6IHN0cmluZywgcHJvcHM6IGFueSk6IHZvaWQge1xuICAgIGNvbnN0IG1hdGNoRXJyb3IgPSBoYXNNYXBwaW5nKHRoaXMudGVtcGxhdGUsIGxvZ2ljYWxJZCwgcHJvcHMpO1xuICAgIGlmIChtYXRjaEVycm9yKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IobWF0Y2hFcnJvcik7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEdldCB0aGUgc2V0IG9mIG1hdGNoaW5nIE1hcHBpbmdzIHRoYXQgbWF0Y2ggdGhlIGdpdmVuIHByb3BlcnRpZXMgaW4gdGhlIENsb3VkRm9ybWF0aW9uIHRlbXBsYXRlLlxuICAgKiBAcGFyYW0gbG9naWNhbElkIHRoZSBuYW1lIG9mIHRoZSBtYXBwaW5nLiBQcm92aWRlIGAnKidgIHRvIG1hdGNoIGFsbCBtYXBwaW5ncyBpbiB0aGUgdGVtcGxhdGUuXG4gICAqIEBwYXJhbSBwcm9wcyBieSBkZWZhdWx0LCBtYXRjaGVzIGFsbCBNYXBwaW5ncyBpbiB0aGUgdGVtcGxhdGUuXG4gICAqIFdoZW4gYSBsaXRlcmFsIG9iamVjdCBpcyBwcm92aWRlZCwgcGVyZm9ybXMgYSBwYXJ0aWFsIG1hdGNoIHZpYSBgTWF0Y2gub2JqZWN0TGlrZSgpYC5cbiAgICogVXNlIHRoZSBgTWF0Y2hgIEFQSXMgdG8gY29uZmlndXJlIGEgZGlmZmVyZW50IGJlaGF2aW91ci5cbiAgICovXG4gIHB1YmxpYyBmaW5kTWFwcGluZ3MobG9naWNhbElkOiBzdHJpbmcsIHByb3BzOiBhbnkgPSB7fSk6IHsgW2tleTogc3RyaW5nXTogeyBba2V5OiBzdHJpbmddOiBhbnkgfSB9IHtcbiAgICByZXR1cm4gZmluZE1hcHBpbmdzKHRoaXMudGVtcGxhdGUsIGxvZ2ljYWxJZCwgcHJvcHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFzc2VydCB0aGF0IGEgQ29uZGl0aW9uIHdpdGggdGhlIGdpdmVuIHByb3BlcnRpZXMgZXhpc3RzIGluIHRoZSBDbG91ZEZvcm1hdGlvbiB0ZW1wbGF0ZS5cbiAgICogQnkgZGVmYXVsdCwgcGVyZm9ybXMgcGFydGlhbCBtYXRjaGluZyBvbiB0aGUgcmVzb3VyY2UsIHZpYSB0aGUgYE1hdGNoLm9iamVjdExpa2UoKWAuXG4gICAqIFRvIGNvbmZpZ3VyZSBkaWZmZXJlbnQgYmVoYXZpb3IsIHVzZSBvdGhlciBtYXRjaGVycyBpbiB0aGUgYE1hdGNoYCBjbGFzcy5cbiAgICogQHBhcmFtIGxvZ2ljYWxJZCB0aGUgbmFtZSBvZiB0aGUgbWFwcGluZy4gUHJvdmlkZSBgJyonYCB0byBtYXRjaCBhbGwgY29uZGl0aW9ucyBpbiB0aGUgdGVtcGxhdGUuXG4gICAqIEBwYXJhbSBwcm9wcyB0aGUgb3V0cHV0IGFzIHNob3VsZCBiZSBleHBlY3RlZCBpbiB0aGUgdGVtcGxhdGUuXG4gICAqL1xuICBwdWJsaWMgaGFzQ29uZGl0aW9uKGxvZ2ljYWxJZDogc3RyaW5nLCBwcm9wczogYW55KTogdm9pZCB7XG4gICAgY29uc3QgbWF0Y2hFcnJvciA9IGhhc0NvbmRpdGlvbih0aGlzLnRlbXBsYXRlLCBsb2dpY2FsSWQsIHByb3BzKTtcbiAgICBpZiAobWF0Y2hFcnJvcikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKG1hdGNoRXJyb3IpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBHZXQgdGhlIHNldCBvZiBtYXRjaGluZyBDb25kaXRpb25zIHRoYXQgbWF0Y2ggdGhlIGdpdmVuIHByb3BlcnRpZXMgaW4gdGhlIENsb3VkRm9ybWF0aW9uIHRlbXBsYXRlLlxuICAgKiBAcGFyYW0gbG9naWNhbElkIHRoZSBuYW1lIG9mIHRoZSBjb25kaXRpb24uIFByb3ZpZGUgYCcqJ2AgdG8gbWF0Y2ggYWxsIGNvbmRpdGlvbnMgaW4gdGhlIHRlbXBsYXRlLlxuICAgKiBAcGFyYW0gcHJvcHMgYnkgZGVmYXVsdCwgbWF0Y2hlcyBhbGwgQ29uZGl0aW9ucyBpbiB0aGUgdGVtcGxhdGUuXG4gICAqIFdoZW4gYSBsaXRlcmFsIG9iamVjdCBpcyBwcm92aWRlZCwgcGVyZm9ybXMgYSBwYXJ0aWFsIG1hdGNoIHZpYSBgTWF0Y2gub2JqZWN0TGlrZSgpYC5cbiAgICogVXNlIHRoZSBgTWF0Y2hgIEFQSXMgdG8gY29uZmlndXJlIGEgZGlmZmVyZW50IGJlaGF2aW91ci5cbiAgICovXG4gIHB1YmxpYyBmaW5kQ29uZGl0aW9ucyhsb2dpY2FsSWQ6IHN0cmluZywgcHJvcHM6IGFueSA9IHt9KTogeyBba2V5OiBzdHJpbmddOiB7IFtrZXk6IHN0cmluZ106IGFueSB9IH0ge1xuICAgIHJldHVybiBmaW5kQ29uZGl0aW9ucyh0aGlzLnRlbXBsYXRlLCBsb2dpY2FsSWQsIHByb3BzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBc3NlcnQgdGhhdCB0aGUgQ2xvdWRGb3JtYXRpb24gdGVtcGxhdGUgbWF0Y2hlcyB0aGUgZ2l2ZW4gdmFsdWVcbiAgICogQHBhcmFtIGV4cGVjdGVkIHRoZSBleHBlY3RlZCBDbG91ZEZvcm1hdGlvbiB0ZW1wbGF0ZSBhcyBrZXktdmFsdWUgcGFpcnMuXG4gICAqL1xuICBwdWJsaWMgdGVtcGxhdGVNYXRjaGVzKGV4cGVjdGVkOiBhbnkpOiB2b2lkIHtcbiAgICBjb25zdCBtYXRjaGVyID0gTWF0Y2hlci5pc01hdGNoZXIoZXhwZWN0ZWQpID8gZXhwZWN0ZWQgOiBNYXRjaC5vYmplY3RMaWtlKGV4cGVjdGVkKTtcbiAgICBjb25zdCByZXN1bHQgPSBtYXRjaGVyLnRlc3QodGhpcy50ZW1wbGF0ZSk7XG5cbiAgICBpZiAocmVzdWx0Lmhhc0ZhaWxlZCgpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoW1xuICAgICAgICAnVGVtcGxhdGUgZGlkIG5vdCBtYXRjaCBhcyBleHBlY3RlZC4gVGhlIGZvbGxvd2luZyBtaXNtYXRjaGVzIHdlcmUgZm91bmQ6JyxcbiAgICAgICAgLi4ucmVzdWx0LnRvSHVtYW5TdHJpbmdzKCkubWFwKHMgPT4gYFxcdCR7c31gKSxcbiAgICAgIF0uam9pbignXFxuJykpO1xuICAgIH1cbiAgfVxufVxuXG4vKipcbiAqIE9wdGlvbnMgdG8gY29uZmlndXJlIHRlbXBsYXRlIHBhcnNpbmcgYmVoYXZpb3IsIHN1Y2ggYXMgZGlzcmVnYXJkaW5nIGNpcmN1bGFyXG4gKiBkZXBlbmRlbmNpZXMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVGVtcGxhdGVQYXJzaW5nT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBJZiBzZXQgdG8gdHJ1ZSwgd2lsbCBza2lwIGNoZWNraW5nIGZvciBjeWNsaWNhbCAvIGNpcmN1bGFyIGRlcGVuZGVuY2llcy4gU2hvdWxkIGJlIHNldCB0byBmYWxzZSBvdGhlciB0aGFuIGZvclxuICAgKiB0ZW1wbGF0ZXMgdGhhdCBhcmUgdmFsaWQgZGVzcGl0ZSBjb250YWluaW5nIGN5Y2xlcywgc3VjaCBhcyB1bnByb2Nlc3NlZCB0cmFuc2Zvcm0gc3RhY2tzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgc2tpcEN5Y2xpY2FsRGVwZW5kZW5jaWVzQ2hlY2s/OiBib29sZWFuO1xufVxuXG5mdW5jdGlvbiB0b1RlbXBsYXRlKHN0YWNrOiBTdGFjayk6IGFueSB7XG4gIGNvbnN0IHJvb3QgPSBzdGFjay5ub2RlLnJvb3Q7XG4gIGlmICghU3RhZ2UuaXNTdGFnZShyb290KSkge1xuICAgIHRocm93IG5ldyBFcnJvcigndW5leHBlY3RlZDogYWxsIHN0YWNrcyBtdXN0IGJlIHBhcnQgb2YgYSBTdGFnZSBvciBhbiBBcHAnKTtcbiAgfVxuXG4gIGNvbnN0IGFzc2VtYmx5ID0gcm9vdC5zeW50aCgpO1xuICBpZiAoc3RhY2submVzdGVkU3RhY2tQYXJlbnQpIHtcbiAgICAvLyBpZiB0aGlzIGlzIGEgbmVzdGVkIHN0YWNrIChpdCBoYXMgYSBwYXJlbnQpLCB0aGVuIGp1c3QgcmVhZCB0aGUgdGVtcGxhdGUgYXMgYSBzdHJpbmdcbiAgICByZXR1cm4gSlNPTi5wYXJzZShmcy5yZWFkRmlsZVN5bmMocGF0aC5qb2luKGFzc2VtYmx5LmRpcmVjdG9yeSwgc3RhY2sudGVtcGxhdGVGaWxlKSkudG9TdHJpbmcoJ3V0Zi04JykpO1xuICB9XG4gIHJldHVybiBhc3NlbWJseS5nZXRTdGFja0FydGlmYWN0KHN0YWNrLmFydGlmYWN0SWQpLnRlbXBsYXRlO1xufVxuIl19