"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RateLimitedApiKey = exports.ApiKey = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const apigateway_generated_1 = require("./apigateway.generated");
const usage_plan_1 = require("./usage-plan");
/**
 * Base implementation that is common to the various implementations of IApiKey
 */
class ApiKeyBase extends core_1.Resource {
    /**
     * Permits the IAM principal all read operations through this key
     *
     * @param grantee The principal to grant access to
     */
    grantRead(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: readPermissions,
            resourceArns: [this.keyArn],
        });
    }
    /**
     * Permits the IAM principal all write operations through this key
     *
     * @param grantee The principal to grant access to
     */
    grantWrite(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: writePermissions,
            resourceArns: [this.keyArn],
        });
    }
    /**
     * Permits the IAM principal all read and write operations through this key
     *
     * @param grantee The principal to grant access to
     */
    grantReadWrite(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: [...readPermissions, ...writePermissions],
            resourceArns: [this.keyArn],
        });
    }
}
/**
 * An API Gateway ApiKey.
 *
 * An ApiKey can be distributed to API clients that are executing requests
 * for Method resources that require an Api Key.
 */
class ApiKey extends ApiKeyBase {
    /**
     * Import an ApiKey by its Id
     */
    static fromApiKeyId(scope, id, apiKeyId) {
        class Import extends ApiKeyBase {
            constructor() {
                super(...arguments);
                this.keyId = apiKeyId;
                this.keyArn = core_1.Stack.of(this).formatArn({
                    service: 'apigateway',
                    account: '',
                    resource: '/apikeys',
                    arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME,
                    resourceName: apiKeyId,
                });
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.apiKeyName,
        });
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_ApiKeyProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ApiKey);
            }
            throw error;
        }
        const resource = new apigateway_generated_1.CfnApiKey(this, 'Resource', {
            customerId: props.customerId,
            description: props.description,
            enabled: props.enabled ?? true,
            generateDistinctId: props.generateDistinctId,
            name: this.physicalName,
            stageKeys: this.renderStageKeys(props.resources, props.stages),
            value: props.value,
        });
        this.keyId = resource.ref;
        this.keyArn = core_1.Stack.of(this).formatArn({
            service: 'apigateway',
            account: '',
            resource: '/apikeys',
            arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME,
            resourceName: this.keyId,
        });
    }
    renderStageKeys(resources, stages) {
        if (!resources && !stages) {
            return undefined;
        }
        if (resources && stages) {
            throw new Error('Only one of "resources" or "stages" should be provided');
        }
        return resources
            ? resources.map((resource) => {
                const restApi = resource;
                if (!restApi.deploymentStage) {
                    throw new Error('Cannot add an ApiKey to a RestApi that does not contain a "deploymentStage".\n' +
                        'Either set the RestApi.deploymentStage or create an ApiKey from a Stage');
                }
                const restApiId = restApi.restApiId;
                const stageName = restApi.deploymentStage.stageName.toString();
                return { restApiId, stageName };
            })
            : stages ? stages.map((stage => {
                return { restApiId: stage.restApi.restApiId, stageName: stage.stageName };
            })) : undefined;
    }
}
_a = JSII_RTTI_SYMBOL_1;
ApiKey[_a] = { fqn: "aws-cdk-lib.aws_apigateway.ApiKey", version: "2.74.0" };
exports.ApiKey = ApiKey;
/**
 * An API Gateway ApiKey, for which a rate limiting configuration can be specified.
 *
 * @resource AWS::ApiGateway::ApiKey
 */
class RateLimitedApiKey extends ApiKeyBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.apiKeyName,
        });
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_RateLimitedApiKeyProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, RateLimitedApiKey);
            }
            throw error;
        }
        const resource = new ApiKey(this, 'Resource', props);
        if (props.apiStages || props.quota || props.throttle) {
            const usageplan = new usage_plan_1.UsagePlan(this, 'UsagePlanResource', {
                apiStages: props.apiStages,
                quota: props.quota,
                throttle: props.throttle,
            });
            usageplan.addApiKey(resource);
        }
        this.keyId = resource.keyId;
        this.keyArn = resource.keyArn;
    }
}
_b = JSII_RTTI_SYMBOL_1;
RateLimitedApiKey[_b] = { fqn: "aws-cdk-lib.aws_apigateway.RateLimitedApiKey", version: "2.74.0" };
exports.RateLimitedApiKey = RateLimitedApiKey;
const readPermissions = [
    'apigateway:GET',
];
const writePermissions = [
    'apigateway:POST',
    'apigateway:PUT',
    'apigateway:PATCH',
    'apigateway:DELETE',
];
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBpLWtleS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImFwaS1rZXkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBQUEscUNBQXFDO0FBQ3JDLHFDQUFvRjtBQUVwRixpRUFBbUQ7QUFJbkQsNkNBQWdHO0FBcUZoRzs7R0FFRztBQUNILE1BQWUsVUFBVyxTQUFRLGVBQVE7SUFJeEM7Ozs7T0FJRztJQUNJLFNBQVMsQ0FBQyxPQUF1QjtRQUN0QyxPQUFPLEdBQUcsQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDO1lBQzlCLE9BQU87WUFDUCxPQUFPLEVBQUUsZUFBZTtZQUN4QixZQUFZLEVBQUUsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDO1NBQzVCLENBQUMsQ0FBQztLQUNKO0lBRUQ7Ozs7T0FJRztJQUNJLFVBQVUsQ0FBQyxPQUF1QjtRQUN2QyxPQUFPLEdBQUcsQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDO1lBQzlCLE9BQU87WUFDUCxPQUFPLEVBQUUsZ0JBQWdCO1lBQ3pCLFlBQVksRUFBRSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUM7U0FDNUIsQ0FBQyxDQUFDO0tBQ0o7SUFFRDs7OztPQUlHO0lBQ0ksY0FBYyxDQUFDLE9BQXVCO1FBQzNDLE9BQU8sR0FBRyxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUM7WUFDOUIsT0FBTztZQUNQLE9BQU8sRUFBRSxDQUFDLEdBQUcsZUFBZSxFQUFFLEdBQUcsZ0JBQWdCLENBQUM7WUFDbEQsWUFBWSxFQUFFLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQztTQUM1QixDQUFDLENBQUM7S0FDSjtDQUNGO0FBRUQ7Ozs7O0dBS0c7QUFDSCxNQUFhLE1BQU8sU0FBUSxVQUFVO0lBRXBDOztPQUVHO0lBQ0ksTUFBTSxDQUFDLFlBQVksQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUFnQjtRQUN2RSxNQUFNLE1BQU8sU0FBUSxVQUFVO1lBQS9COztnQkFDUyxVQUFLLEdBQUcsUUFBUSxDQUFDO2dCQUNqQixXQUFNLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7b0JBQ3ZDLE9BQU8sRUFBRSxZQUFZO29CQUNyQixPQUFPLEVBQUUsRUFBRTtvQkFDWCxRQUFRLEVBQUUsVUFBVTtvQkFDcEIsU0FBUyxFQUFFLGdCQUFTLENBQUMsbUJBQW1CO29CQUN4QyxZQUFZLEVBQUUsUUFBUTtpQkFDdkIsQ0FBQyxDQUFDO1lBQ0wsQ0FBQztTQUFBO1FBRUQsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7S0FDOUI7SUFLRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQXFCLEVBQUc7UUFDaEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixZQUFZLEVBQUUsS0FBSyxDQUFDLFVBQVU7U0FDL0IsQ0FBQyxDQUFDOzs7Ozs7K0NBMUJNLE1BQU07Ozs7UUE0QmYsTUFBTSxRQUFRLEdBQUcsSUFBSSxnQ0FBUyxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDL0MsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVO1lBQzVCLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztZQUM5QixPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU8sSUFBSSxJQUFJO1lBQzlCLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxrQkFBa0I7WUFDNUMsSUFBSSxFQUFFLElBQUksQ0FBQyxZQUFZO1lBQ3ZCLFNBQVMsRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsS0FBSyxDQUFDLE1BQU0sQ0FBQztZQUM5RCxLQUFLLEVBQUUsS0FBSyxDQUFDLEtBQUs7U0FDbkIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLEtBQUssR0FBRyxRQUFRLENBQUMsR0FBRyxDQUFDO1FBQzFCLElBQUksQ0FBQyxNQUFNLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7WUFDckMsT0FBTyxFQUFFLFlBQVk7WUFDckIsT0FBTyxFQUFFLEVBQUU7WUFDWCxRQUFRLEVBQUUsVUFBVTtZQUNwQixTQUFTLEVBQUUsZ0JBQVMsQ0FBQyxtQkFBbUI7WUFDeEMsWUFBWSxFQUFFLElBQUksQ0FBQyxLQUFLO1NBQ3pCLENBQUMsQ0FBQztLQUNKO0lBRU8sZUFBZSxDQUFDLFNBQXNCLEVBQUUsTUFBaUI7UUFDL0QsSUFBSSxDQUFDLFNBQVMsSUFBSSxDQUFDLE1BQU0sRUFBRTtZQUN6QixPQUFPLFNBQVMsQ0FBQztTQUNsQjtRQUVELElBQUksU0FBUyxJQUFJLE1BQU0sRUFBRTtZQUN2QixNQUFNLElBQUksS0FBSyxDQUFDLHdEQUF3RCxDQUFDLENBQUM7U0FDM0U7UUFFRCxPQUFPLFNBQVM7WUFDZCxDQUFDLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxDQUFDLFFBQWtCLEVBQUUsRUFBRTtnQkFDckMsTUFBTSxPQUFPLEdBQUcsUUFBUSxDQUFDO2dCQUN6QixJQUFJLENBQUMsT0FBTyxDQUFDLGVBQWUsRUFBRTtvQkFDNUIsTUFBTSxJQUFJLEtBQUssQ0FBQyxnRkFBZ0Y7d0JBQ2hHLHlFQUF5RSxDQUFDLENBQUM7aUJBQzVFO2dCQUNELE1BQU0sU0FBUyxHQUFHLE9BQU8sQ0FBQyxTQUFTLENBQUM7Z0JBQ3BDLE1BQU0sU0FBUyxHQUFHLE9BQU8sQ0FBQyxlQUFnQixDQUFDLFNBQVMsQ0FBQyxRQUFRLEVBQUUsQ0FBQztnQkFDaEUsT0FBTyxFQUFFLFNBQVMsRUFBRSxTQUFTLEVBQUUsQ0FBQztZQUNsQyxDQUFDLENBQUM7WUFDRixDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsS0FBSyxDQUFDLEVBQUU7Z0JBQzdCLE9BQU8sRUFBRSxTQUFTLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxTQUFTLEVBQUUsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTLEVBQUUsQ0FBQztZQUM1RSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7S0FDbkI7Ozs7QUF2RVUsd0JBQU07QUFpR25COzs7O0dBSUc7QUFDSCxNQUFhLGlCQUFrQixTQUFRLFVBQVU7SUFJL0MsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUFnQyxFQUFHO1FBQzNFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsWUFBWSxFQUFFLEtBQUssQ0FBQyxVQUFVO1NBQy9CLENBQUMsQ0FBQzs7Ozs7OytDQVBNLGlCQUFpQjs7OztRQVMxQixNQUFNLFFBQVEsR0FBRyxJQUFJLE1BQU0sQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRXJELElBQUksS0FBSyxDQUFDLFNBQVMsSUFBSSxLQUFLLENBQUMsS0FBSyxJQUFJLEtBQUssQ0FBQyxRQUFRLEVBQUU7WUFDcEQsTUFBTSxTQUFTLEdBQUcsSUFBSSxzQkFBUyxDQUFDLElBQUksRUFBRSxtQkFBbUIsRUFBRTtnQkFDekQsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTO2dCQUMxQixLQUFLLEVBQUUsS0FBSyxDQUFDLEtBQUs7Z0JBQ2xCLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUTthQUN6QixDQUFDLENBQUM7WUFDSCxTQUFTLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQy9CO1FBRUQsSUFBSSxDQUFDLEtBQUssR0FBRyxRQUFRLENBQUMsS0FBSyxDQUFDO1FBQzVCLElBQUksQ0FBQyxNQUFNLEdBQUcsUUFBUSxDQUFDLE1BQU0sQ0FBQztLQUMvQjs7OztBQXRCVSw4Q0FBaUI7QUF5QjlCLE1BQU0sZUFBZSxHQUFHO0lBQ3RCLGdCQUFnQjtDQUNqQixDQUFDO0FBRUYsTUFBTSxnQkFBZ0IsR0FBRztJQUN2QixpQkFBaUI7SUFDakIsZ0JBQWdCO0lBQ2hCLGtCQUFrQjtJQUNsQixtQkFBbUI7Q0FDcEIsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGlhbSBmcm9tICcuLi8uLi9hd3MtaWFtJztcbmltcG9ydCB7IEFybkZvcm1hdCwgSVJlc291cmNlIGFzIElSZXNvdXJjZUJhc2UsIFJlc291cmNlLCBTdGFjayB9IGZyb20gJy4uLy4uL2NvcmUnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBDZm5BcGlLZXkgfSBmcm9tICcuL2FwaWdhdGV3YXkuZ2VuZXJhdGVkJztcbmltcG9ydCB7IFJlc291cmNlT3B0aW9ucyB9IGZyb20gJy4vcmVzb3VyY2UnO1xuaW1wb3J0IHsgSVJlc3RBcGkgfSBmcm9tICcuL3Jlc3RhcGknO1xuaW1wb3J0IHsgSVN0YWdlIH0gZnJvbSAnLi9zdGFnZSc7XG5pbXBvcnQgeyBRdW90YVNldHRpbmdzLCBUaHJvdHRsZVNldHRpbmdzLCBVc2FnZVBsYW4sIFVzYWdlUGxhblBlckFwaVN0YWdlIH0gZnJvbSAnLi91c2FnZS1wbGFuJztcblxuLyoqXG4gKiBBUEkga2V5cyBhcmUgYWxwaGFudW1lcmljIHN0cmluZyB2YWx1ZXMgdGhhdCB5b3UgZGlzdHJpYnV0ZSB0b1xuICogYXBwIGRldmVsb3BlciBjdXN0b21lcnMgdG8gZ3JhbnQgYWNjZXNzIHRvIHlvdXIgQVBJXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSUFwaUtleSBleHRlbmRzIElSZXNvdXJjZUJhc2Uge1xuICAvKipcbiAgICogVGhlIEFQSSBrZXkgSUQuXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGtleUlkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBBUEkga2V5IEFSTi5cbiAgICovXG4gIHJlYWRvbmx5IGtleUFybjogc3RyaW5nO1xufVxuXG4vKipcbiAqIFRoZSBvcHRpb25zIGZvciBjcmVhdGluZyBhbiBBUEkgS2V5LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEFwaUtleU9wdGlvbnMgZXh0ZW5kcyBSZXNvdXJjZU9wdGlvbnMge1xuICAvKipcbiAgICogQSBuYW1lIGZvciB0aGUgQVBJIGtleS4gSWYgeW91IGRvbid0IHNwZWNpZnkgYSBuYW1lLCBBV1MgQ2xvdWRGb3JtYXRpb24gZ2VuZXJhdGVzIGEgdW5pcXVlIHBoeXNpY2FsIElEIGFuZCB1c2VzIHRoYXQgSUQgZm9yIHRoZSBBUEkga2V5IG5hbWUuXG4gICAqIEBsaW5rIGh0dHA6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXJlc291cmNlLWFwaWdhdGV3YXktYXBpa2V5Lmh0bWwjY2ZuLWFwaWdhdGV3YXktYXBpa2V5LW5hbWVcbiAgICogQGRlZmF1bHQgYXV0b21pY2FsbHkgZ2VuZXJhdGVkIG5hbWVcbiAgICovXG4gIHJlYWRvbmx5IGFwaUtleU5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSB2YWx1ZSBvZiB0aGUgQVBJIGtleS4gTXVzdCBiZSBhdCBsZWFzdCAyMCBjaGFyYWN0ZXJzIGxvbmcuXG4gICAqIEBsaW5rIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2F3cy1yZXNvdXJjZS1hcGlnYXRld2F5LWFwaWtleS5odG1sI2Nmbi1hcGlnYXRld2F5LWFwaWtleS12YWx1ZVxuICAgKiBAZGVmYXVsdCBub25lXG4gICAqL1xuICByZWFkb25seSB2YWx1ZT86IHN0cmluZztcblxuICAvKipcbiAgICogQSBkZXNjcmlwdGlvbiBvZiB0aGUgcHVycG9zZSBvZiB0aGUgQVBJIGtleS5cbiAgICogQGxpbmsgaHR0cDovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTQ2xvdWRGb3JtYXRpb24vbGF0ZXN0L1VzZXJHdWlkZS9hd3MtcmVzb3VyY2UtYXBpZ2F0ZXdheS1hcGlrZXkuaHRtbCNjZm4tYXBpZ2F0ZXdheS1hcGlrZXktZGVzY3JpcHRpb25cbiAgICogQGRlZmF1bHQgbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgZGVzY3JpcHRpb24/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogQXBpS2V5IFByb3BlcnRpZXMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXBpS2V5UHJvcHMgZXh0ZW5kcyBBcGlLZXlPcHRpb25zIHtcbiAgLyoqXG4gICAqIEEgbGlzdCBvZiByZXNvdXJjZXMgdGhpcyBhcGkga2V5IGlzIGFzc29jaWF0ZWQgd2l0aC5cbiAgICogQGRlZmF1bHQgbm9uZVxuICAgKiBAZGVwcmVjYXRlZCAtIHVzZSBgc3RhZ2VzYCBpbnN0ZWFkXG4gICAqL1xuICByZWFkb25seSByZXNvdXJjZXM/OiBJUmVzdEFwaVtdO1xuXG4gIC8qKlxuICAgKiBBIGxpc3Qgb2YgU3RhZ2VzIHRoaXMgYXBpIGtleSBpcyBhc3NvY2lhdGVkIHdpdGguXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdGhlIGFwaSBrZXkgaXMgbm90IGFzc29jaWF0ZWQgd2l0aCBhbnkgc3RhZ2VzXG4gICAqL1xuICByZWFkb25seSBzdGFnZXM/OiBJU3RhZ2VbXTtcblxuICAvKipcbiAgICogQW4gQVdTIE1hcmtldHBsYWNlIGN1c3RvbWVyIGlkZW50aWZpZXIgdG8gdXNlIHdoZW4gaW50ZWdyYXRpbmcgd2l0aCB0aGUgQVdTIFNhYVMgTWFya2V0cGxhY2UuXG4gICAqIEBsaW5rIGh0dHA6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXJlc291cmNlLWFwaWdhdGV3YXktYXBpa2V5Lmh0bWwjY2ZuLWFwaWdhdGV3YXktYXBpa2V5LWN1c3RvbWVyaWRcbiAgICogQGRlZmF1bHQgbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgY3VzdG9tZXJJZD86IHN0cmluZztcblxuICAvKipcbiAgICogSW5kaWNhdGVzIHdoZXRoZXIgdGhlIEFQSSBrZXkgY2FuIGJlIHVzZWQgYnkgY2xpZW50cy5cbiAgICogQGxpbmsgaHR0cDovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTQ2xvdWRGb3JtYXRpb24vbGF0ZXN0L1VzZXJHdWlkZS9hd3MtcmVzb3VyY2UtYXBpZ2F0ZXdheS1hcGlrZXkuaHRtbCNjZm4tYXBpZ2F0ZXdheS1hcGlrZXktZW5hYmxlZFxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBlbmFibGVkPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogU3BlY2lmaWVzIHdoZXRoZXIgdGhlIGtleSBpZGVudGlmaWVyIGlzIGRpc3RpbmN0IGZyb20gdGhlIGNyZWF0ZWQgQVBJIGtleSB2YWx1ZS5cbiAgICogQGxpbmsgaHR0cDovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTQ2xvdWRGb3JtYXRpb24vbGF0ZXN0L1VzZXJHdWlkZS9hd3MtcmVzb3VyY2UtYXBpZ2F0ZXdheS1hcGlrZXkuaHRtbCNjZm4tYXBpZ2F0ZXdheS1hcGlrZXktZ2VuZXJhdGVkaXN0aW5jdGlkXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBnZW5lcmF0ZURpc3RpbmN0SWQ/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIEJhc2UgaW1wbGVtZW50YXRpb24gdGhhdCBpcyBjb21tb24gdG8gdGhlIHZhcmlvdXMgaW1wbGVtZW50YXRpb25zIG9mIElBcGlLZXlcbiAqL1xuYWJzdHJhY3QgY2xhc3MgQXBpS2V5QmFzZSBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSUFwaUtleSB7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBrZXlJZDogc3RyaW5nO1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkga2V5QXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFBlcm1pdHMgdGhlIElBTSBwcmluY2lwYWwgYWxsIHJlYWQgb3BlcmF0aW9ucyB0aHJvdWdoIHRoaXMga2V5XG4gICAqXG4gICAqIEBwYXJhbSBncmFudGVlIFRoZSBwcmluY2lwYWwgdG8gZ3JhbnQgYWNjZXNzIHRvXG4gICAqL1xuICBwdWJsaWMgZ3JhbnRSZWFkKGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHtcbiAgICByZXR1cm4gaWFtLkdyYW50LmFkZFRvUHJpbmNpcGFsKHtcbiAgICAgIGdyYW50ZWUsXG4gICAgICBhY3Rpb25zOiByZWFkUGVybWlzc2lvbnMsXG4gICAgICByZXNvdXJjZUFybnM6IFt0aGlzLmtleUFybl0sXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogUGVybWl0cyB0aGUgSUFNIHByaW5jaXBhbCBhbGwgd3JpdGUgb3BlcmF0aW9ucyB0aHJvdWdoIHRoaXMga2V5XG4gICAqXG4gICAqIEBwYXJhbSBncmFudGVlIFRoZSBwcmluY2lwYWwgdG8gZ3JhbnQgYWNjZXNzIHRvXG4gICAqL1xuICBwdWJsaWMgZ3JhbnRXcml0ZShncmFudGVlOiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudCB7XG4gICAgcmV0dXJuIGlhbS5HcmFudC5hZGRUb1ByaW5jaXBhbCh7XG4gICAgICBncmFudGVlLFxuICAgICAgYWN0aW9uczogd3JpdGVQZXJtaXNzaW9ucyxcbiAgICAgIHJlc291cmNlQXJuczogW3RoaXMua2V5QXJuXSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBQZXJtaXRzIHRoZSBJQU0gcHJpbmNpcGFsIGFsbCByZWFkIGFuZCB3cml0ZSBvcGVyYXRpb25zIHRocm91Z2ggdGhpcyBrZXlcbiAgICpcbiAgICogQHBhcmFtIGdyYW50ZWUgVGhlIHByaW5jaXBhbCB0byBncmFudCBhY2Nlc3MgdG9cbiAgICovXG4gIHB1YmxpYyBncmFudFJlYWRXcml0ZShncmFudGVlOiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudCB7XG4gICAgcmV0dXJuIGlhbS5HcmFudC5hZGRUb1ByaW5jaXBhbCh7XG4gICAgICBncmFudGVlLFxuICAgICAgYWN0aW9uczogWy4uLnJlYWRQZXJtaXNzaW9ucywgLi4ud3JpdGVQZXJtaXNzaW9uc10sXG4gICAgICByZXNvdXJjZUFybnM6IFt0aGlzLmtleUFybl0sXG4gICAgfSk7XG4gIH1cbn1cblxuLyoqXG4gKiBBbiBBUEkgR2F0ZXdheSBBcGlLZXkuXG4gKlxuICogQW4gQXBpS2V5IGNhbiBiZSBkaXN0cmlidXRlZCB0byBBUEkgY2xpZW50cyB0aGF0IGFyZSBleGVjdXRpbmcgcmVxdWVzdHNcbiAqIGZvciBNZXRob2QgcmVzb3VyY2VzIHRoYXQgcmVxdWlyZSBhbiBBcGkgS2V5LlxuICovXG5leHBvcnQgY2xhc3MgQXBpS2V5IGV4dGVuZHMgQXBpS2V5QmFzZSB7XG5cbiAgLyoqXG4gICAqIEltcG9ydCBhbiBBcGlLZXkgYnkgaXRzIElkXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21BcGlLZXlJZChzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBhcGlLZXlJZDogc3RyaW5nKTogSUFwaUtleSB7XG4gICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgQXBpS2V5QmFzZSB7XG4gICAgICBwdWJsaWMga2V5SWQgPSBhcGlLZXlJZDtcbiAgICAgIHB1YmxpYyBrZXlBcm4gPSBTdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oe1xuICAgICAgICBzZXJ2aWNlOiAnYXBpZ2F0ZXdheScsXG4gICAgICAgIGFjY291bnQ6ICcnLFxuICAgICAgICByZXNvdXJjZTogJy9hcGlrZXlzJyxcbiAgICAgICAgYXJuRm9ybWF0OiBBcm5Gb3JtYXQuU0xBU0hfUkVTT1VSQ0VfTkFNRSxcbiAgICAgICAgcmVzb3VyY2VOYW1lOiBhcGlLZXlJZCxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHJldHVybiBuZXcgSW1wb3J0KHNjb3BlLCBpZCk7XG4gIH1cblxuICBwdWJsaWMgcmVhZG9ubHkga2V5SWQ6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGtleUFybjogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBBcGlLZXlQcm9wcyA9IHsgfSkge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgcGh5c2ljYWxOYW1lOiBwcm9wcy5hcGlLZXlOYW1lLFxuICAgIH0pO1xuXG4gICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgQ2ZuQXBpS2V5KHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIGN1c3RvbWVySWQ6IHByb3BzLmN1c3RvbWVySWQsXG4gICAgICBkZXNjcmlwdGlvbjogcHJvcHMuZGVzY3JpcHRpb24sXG4gICAgICBlbmFibGVkOiBwcm9wcy5lbmFibGVkID8/IHRydWUsXG4gICAgICBnZW5lcmF0ZURpc3RpbmN0SWQ6IHByb3BzLmdlbmVyYXRlRGlzdGluY3RJZCxcbiAgICAgIG5hbWU6IHRoaXMucGh5c2ljYWxOYW1lLFxuICAgICAgc3RhZ2VLZXlzOiB0aGlzLnJlbmRlclN0YWdlS2V5cyhwcm9wcy5yZXNvdXJjZXMsIHByb3BzLnN0YWdlcyksXG4gICAgICB2YWx1ZTogcHJvcHMudmFsdWUsXG4gICAgfSk7XG5cbiAgICB0aGlzLmtleUlkID0gcmVzb3VyY2UucmVmO1xuICAgIHRoaXMua2V5QXJuID0gU3RhY2sub2YodGhpcykuZm9ybWF0QXJuKHtcbiAgICAgIHNlcnZpY2U6ICdhcGlnYXRld2F5JyxcbiAgICAgIGFjY291bnQ6ICcnLFxuICAgICAgcmVzb3VyY2U6ICcvYXBpa2V5cycsXG4gICAgICBhcm5Gb3JtYXQ6IEFybkZvcm1hdC5TTEFTSF9SRVNPVVJDRV9OQU1FLFxuICAgICAgcmVzb3VyY2VOYW1lOiB0aGlzLmtleUlkLFxuICAgIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSByZW5kZXJTdGFnZUtleXMocmVzb3VyY2VzPzogSVJlc3RBcGlbXSwgc3RhZ2VzPzogSVN0YWdlW10pOiBDZm5BcGlLZXkuU3RhZ2VLZXlQcm9wZXJ0eVtdIHwgdW5kZWZpbmVkIHtcbiAgICBpZiAoIXJlc291cmNlcyAmJiAhc3RhZ2VzKSB7XG4gICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cblxuICAgIGlmIChyZXNvdXJjZXMgJiYgc3RhZ2VzKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ09ubHkgb25lIG9mIFwicmVzb3VyY2VzXCIgb3IgXCJzdGFnZXNcIiBzaG91bGQgYmUgcHJvdmlkZWQnKTtcbiAgICB9XG5cbiAgICByZXR1cm4gcmVzb3VyY2VzXG4gICAgICA/IHJlc291cmNlcy5tYXAoKHJlc291cmNlOiBJUmVzdEFwaSkgPT4ge1xuICAgICAgICBjb25zdCByZXN0QXBpID0gcmVzb3VyY2U7XG4gICAgICAgIGlmICghcmVzdEFwaS5kZXBsb3ltZW50U3RhZ2UpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0Nhbm5vdCBhZGQgYW4gQXBpS2V5IHRvIGEgUmVzdEFwaSB0aGF0IGRvZXMgbm90IGNvbnRhaW4gYSBcImRlcGxveW1lbnRTdGFnZVwiLlxcbicrXG4gICAgICAgICAgJ0VpdGhlciBzZXQgdGhlIFJlc3RBcGkuZGVwbG95bWVudFN0YWdlIG9yIGNyZWF0ZSBhbiBBcGlLZXkgZnJvbSBhIFN0YWdlJyk7XG4gICAgICAgIH1cbiAgICAgICAgY29uc3QgcmVzdEFwaUlkID0gcmVzdEFwaS5yZXN0QXBpSWQ7XG4gICAgICAgIGNvbnN0IHN0YWdlTmFtZSA9IHJlc3RBcGkuZGVwbG95bWVudFN0YWdlIS5zdGFnZU5hbWUudG9TdHJpbmcoKTtcbiAgICAgICAgcmV0dXJuIHsgcmVzdEFwaUlkLCBzdGFnZU5hbWUgfTtcbiAgICAgIH0pXG4gICAgICA6IHN0YWdlcyA/IHN0YWdlcy5tYXAoKHN0YWdlID0+IHtcbiAgICAgICAgcmV0dXJuIHsgcmVzdEFwaUlkOiBzdGFnZS5yZXN0QXBpLnJlc3RBcGlJZCwgc3RhZ2VOYW1lOiBzdGFnZS5zdGFnZU5hbWUgfTtcbiAgICAgIH0pKSA6IHVuZGVmaW5lZDtcbiAgfVxufVxuXG4vKipcbiAqIFJhdGVMaW1pdGVkQXBpS2V5IHByb3BlcnRpZXMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUmF0ZUxpbWl0ZWRBcGlLZXlQcm9wcyBleHRlbmRzIEFwaUtleVByb3BzIHtcbiAgLyoqXG4gICAqIEFQSSBTdGFnZXMgdG8gYmUgYXNzb2NpYXRlZCB3aXRoIHRoZSBSYXRlTGltaXRlZEFwaUtleS5cbiAgICogQGRlZmF1bHQgbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgYXBpU3RhZ2VzPzogVXNhZ2VQbGFuUGVyQXBpU3RhZ2VbXTtcblxuICAvKipcbiAgICogTnVtYmVyIG9mIHJlcXVlc3RzIGNsaWVudHMgY2FuIG1ha2UgaW4gYSBnaXZlbiB0aW1lIHBlcmlvZC5cbiAgICogQGRlZmF1bHQgbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgcXVvdGE/OiBRdW90YVNldHRpbmdzO1xuXG4gIC8qKlxuICAgKiBPdmVyYWxsIHRocm90dGxlIHNldHRpbmdzIGZvciB0aGUgQVBJLlxuICAgKiBAZGVmYXVsdCBub25lXG4gICAqL1xuICByZWFkb25seSB0aHJvdHRsZT86IFRocm90dGxlU2V0dGluZ3M7XG59XG5cbi8qKlxuICogQW4gQVBJIEdhdGV3YXkgQXBpS2V5LCBmb3Igd2hpY2ggYSByYXRlIGxpbWl0aW5nIGNvbmZpZ3VyYXRpb24gY2FuIGJlIHNwZWNpZmllZC5cbiAqXG4gKiBAcmVzb3VyY2UgQVdTOjpBcGlHYXRld2F5OjpBcGlLZXlcbiAqL1xuZXhwb3J0IGNsYXNzIFJhdGVMaW1pdGVkQXBpS2V5IGV4dGVuZHMgQXBpS2V5QmFzZSB7XG4gIHB1YmxpYyByZWFkb25seSBrZXlJZDogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkga2V5QXJuOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFJhdGVMaW1pdGVkQXBpS2V5UHJvcHMgPSB7IH0pIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHBoeXNpY2FsTmFtZTogcHJvcHMuYXBpS2V5TmFtZSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHJlc291cmNlID0gbmV3IEFwaUtleSh0aGlzLCAnUmVzb3VyY2UnLCBwcm9wcyk7XG5cbiAgICBpZiAocHJvcHMuYXBpU3RhZ2VzIHx8IHByb3BzLnF1b3RhIHx8IHByb3BzLnRocm90dGxlKSB7XG4gICAgICBjb25zdCB1c2FnZXBsYW4gPSBuZXcgVXNhZ2VQbGFuKHRoaXMsICdVc2FnZVBsYW5SZXNvdXJjZScsIHtcbiAgICAgICAgYXBpU3RhZ2VzOiBwcm9wcy5hcGlTdGFnZXMsXG4gICAgICAgIHF1b3RhOiBwcm9wcy5xdW90YSxcbiAgICAgICAgdGhyb3R0bGU6IHByb3BzLnRocm90dGxlLFxuICAgICAgfSk7XG4gICAgICB1c2FnZXBsYW4uYWRkQXBpS2V5KHJlc291cmNlKTtcbiAgICB9XG5cbiAgICB0aGlzLmtleUlkID0gcmVzb3VyY2Uua2V5SWQ7XG4gICAgdGhpcy5rZXlBcm4gPSByZXNvdXJjZS5rZXlBcm47XG4gIH1cbn1cblxuY29uc3QgcmVhZFBlcm1pc3Npb25zID0gW1xuICAnYXBpZ2F0ZXdheTpHRVQnLFxuXTtcblxuY29uc3Qgd3JpdGVQZXJtaXNzaW9ucyA9IFtcbiAgJ2FwaWdhdGV3YXk6UE9TVCcsXG4gICdhcGlnYXRld2F5OlBVVCcsXG4gICdhcGlnYXRld2F5OlBBVENIJyxcbiAgJ2FwaWdhdGV3YXk6REVMRVRFJyxcbl07XG4iXX0=