"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RequestAuthorizer = exports.TokenAuthorizer = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../../aws-iam");
const lambda = require("../../../aws-lambda");
const core_1 = require("../../../core");
const cx_api_1 = require("../../../cx-api");
const apigateway_generated_1 = require("../apigateway.generated");
const authorizer_1 = require("../authorizer");
class LambdaAuthorizer extends authorizer_1.Authorizer {
    constructor(scope, id, props) {
        super(scope, id);
        this.handler = props.handler;
        this.role = props.assumeRole;
        if (props.resultsCacheTtl && props.resultsCacheTtl?.toSeconds() > 3600) {
            throw new Error('Lambda authorizer property \'resultsCacheTtl\' must not be greater than 3600 seconds (1 hour)');
        }
    }
    /**
     * Attaches this authorizer to a specific REST API.
     * @internal
     */
    _attachToApi(restApi) {
        if (this.restApiId && this.restApiId !== restApi.restApiId) {
            throw new Error('Cannot attach authorizer to two different rest APIs');
        }
        this.restApiId = restApi.restApiId;
        const deployment = restApi.latestDeployment;
        const addToLogicalId = core_1.FeatureFlags.of(this).isEnabled(cx_api_1.APIGATEWAY_AUTHORIZER_CHANGE_DEPLOYMENT_LOGICAL_ID);
        if (deployment && addToLogicalId) {
            let functionName;
            if (this.handler instanceof lambda.Function) {
                // if not imported, attempt to get the function name, which
                // may be a token
                functionName = this.handler.node.defaultChild.functionName;
            }
            else {
                // if imported, the function name will be a token
                functionName = this.handler.functionName;
            }
            deployment.node.addDependency(this);
            deployment.addToLogicalId({
                authorizer: this.authorizerProps,
                authorizerToken: functionName,
            });
        }
    }
    /**
     * Sets up the permissions necessary for the API Gateway service to invoke the Lambda function.
     */
    setupPermissions() {
        if (!this.role) {
            this.addDefaultPermisionRole();
        }
        else if (iam.Role.isRole(this.role)) {
            this.addLambdaInvokePermission(this.role);
        }
    }
    /**
     * Add Default Permission Role for handler
     */
    addDefaultPermisionRole() {
        this.handler.addPermission(`${core_1.Names.uniqueId(this)}:Permissions`, {
            principal: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            sourceArn: this.authorizerArn,
        });
    }
    /**
     * Add Lambda Invoke Permission for LambdaAurhorizer's role
     */
    addLambdaInvokePermission(role) {
        role.attachInlinePolicy(new iam.Policy(this, 'authorizerInvokePolicy', {
            statements: [
                new iam.PolicyStatement({
                    resources: this.handler.resourceArnsForGrantInvoke,
                    actions: ['lambda:InvokeFunction'],
                }),
            ],
        }));
    }
    /**
     * Returns a token that resolves to the Rest Api Id at the time of synthesis.
     * Throws an error, during token resolution, if no RestApi is attached to this authorizer.
     */
    lazyRestApiId() {
        return core_1.Lazy.string({
            produce: () => {
                if (!this.restApiId) {
                    throw new Error(`Authorizer (${this.node.path}) must be attached to a RestApi`);
                }
                return this.restApiId;
            },
        });
    }
}
/**
 * Token based lambda authorizer that recognizes the caller's identity as a bearer token,
 * such as a JSON Web Token (JWT) or an OAuth token.
 * Based on the token, authorization is performed by a lambda function.
 *
 * @resource AWS::ApiGateway::Authorizer
 */
class TokenAuthorizer extends LambdaAuthorizer {
    constructor(scope, id, props) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_TokenAuthorizerProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, TokenAuthorizer);
            }
            throw error;
        }
        const restApiId = this.lazyRestApiId();
        const authorizerProps = {
            name: props.authorizerName ?? core_1.Names.uniqueId(this),
            restApiId,
            type: 'TOKEN',
            authorizerUri: lambdaAuthorizerArn(props.handler),
            authorizerCredentials: props.assumeRole?.roleArn,
            authorizerResultTtlInSeconds: props.resultsCacheTtl?.toSeconds(),
            identitySource: props.identitySource || 'method.request.header.Authorization',
            identityValidationExpression: props.validationRegex,
        };
        this.authorizerProps = authorizerProps;
        const resource = new apigateway_generated_1.CfnAuthorizer(this, 'Resource', authorizerProps);
        this.authorizerId = resource.ref;
        this.authorizerArn = core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: restApiId,
            resourceName: `authorizers/${this.authorizerId}`,
        });
        this.setupPermissions();
    }
}
_a = JSII_RTTI_SYMBOL_1;
TokenAuthorizer[_a] = { fqn: "aws-cdk-lib.aws_apigateway.TokenAuthorizer", version: "2.74.0" };
exports.TokenAuthorizer = TokenAuthorizer;
/**
 * Request-based lambda authorizer that recognizes the caller's identity via request parameters,
 * such as headers, paths, query strings, stage variables, or context variables.
 * Based on the request, authorization is performed by a lambda function.
 *
 * @resource AWS::ApiGateway::Authorizer
 */
class RequestAuthorizer extends LambdaAuthorizer {
    constructor(scope, id, props) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_RequestAuthorizerProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, RequestAuthorizer);
            }
            throw error;
        }
        if ((props.resultsCacheTtl === undefined || props.resultsCacheTtl.toSeconds() !== 0) && props.identitySources.length === 0) {
            throw new Error('At least one Identity Source is required for a REQUEST-based Lambda authorizer if caching is enabled.');
        }
        const restApiId = this.lazyRestApiId();
        const authorizerProps = {
            name: props.authorizerName ?? core_1.Names.uniqueId(this),
            restApiId,
            type: 'REQUEST',
            authorizerUri: lambdaAuthorizerArn(props.handler),
            authorizerCredentials: props.assumeRole?.roleArn,
            authorizerResultTtlInSeconds: props.resultsCacheTtl?.toSeconds(),
            identitySource: props.identitySources.map(is => is.toString()).join(','),
        };
        this.authorizerProps = authorizerProps;
        const resource = new apigateway_generated_1.CfnAuthorizer(this, 'Resource', authorizerProps);
        this.authorizerId = resource.ref;
        this.authorizerArn = core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: restApiId,
            resourceName: `authorizers/${this.authorizerId}`,
        });
        this.setupPermissions();
    }
}
_b = JSII_RTTI_SYMBOL_1;
RequestAuthorizer[_b] = { fqn: "aws-cdk-lib.aws_apigateway.RequestAuthorizer", version: "2.74.0" };
exports.RequestAuthorizer = RequestAuthorizer;
/**
 * constructs the authorizerURIArn.
 */
function lambdaAuthorizerArn(handler) {
    const { region, partition } = core_1.Arn.split(handler.functionArn, core_1.ArnFormat.COLON_RESOURCE_NAME);
    return `arn:${partition}:apigateway:${region}:lambda:path/2015-03-31/functions/${handler.functionArn}/invocations`;
}
//# sourceMappingURL=data:application/json;base64,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