"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DomainName = exports.SecurityPolicy = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const apigwv2 = require("../../aws-apigatewayv2");
const core_1 = require("../../core");
const apigateway_generated_1 = require("./apigateway.generated");
const base_path_mapping_1 = require("./base-path-mapping");
const restapi_1 = require("./restapi");
/**
 * The minimum version of the SSL protocol that you want API Gateway to use for HTTPS connections.
 */
var SecurityPolicy;
(function (SecurityPolicy) {
    /** Cipher suite TLS 1.0 */
    SecurityPolicy["TLS_1_0"] = "TLS_1_0";
    /** Cipher suite TLS 1.2 */
    SecurityPolicy["TLS_1_2"] = "TLS_1_2";
})(SecurityPolicy = exports.SecurityPolicy || (exports.SecurityPolicy = {}));
class DomainName extends core_1.Resource {
    /**
     * Imports an existing domain name.
     */
    static fromDomainNameAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_DomainNameAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromDomainNameAttributes);
            }
            throw error;
        }
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.domainName = attrs.domainName;
                this.domainNameAliasDomainName = attrs.domainNameAliasTarget;
                this.domainNameAliasHostedZoneId = attrs.domainNameAliasHostedZoneId;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id);
        this.basePaths = new Set();
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_DomainNameProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, DomainName);
            }
            throw error;
        }
        this.endpointType = props.endpointType || restapi_1.EndpointType.REGIONAL;
        const edge = this.endpointType === restapi_1.EndpointType.EDGE;
        this.securityPolicy = props.securityPolicy;
        if (!core_1.Token.isUnresolved(props.domainName) && /[A-Z]/.test(props.domainName)) {
            throw new Error(`Domain name does not support uppercase letters. Got: ${props.domainName}`);
        }
        const mtlsConfig = this.configureMTLS(props.mtls);
        const resource = new apigateway_generated_1.CfnDomainName(this, 'Resource', {
            domainName: props.domainName,
            certificateArn: edge ? props.certificate.certificateArn : undefined,
            regionalCertificateArn: edge ? undefined : props.certificate.certificateArn,
            endpointConfiguration: { types: [this.endpointType] },
            mutualTlsAuthentication: mtlsConfig,
            securityPolicy: props.securityPolicy,
        });
        this.domainName = resource.ref;
        this.domainNameAliasDomainName = edge
            ? resource.attrDistributionDomainName
            : resource.attrRegionalDomainName;
        this.domainNameAliasHostedZoneId = edge
            ? resource.attrDistributionHostedZoneId
            : resource.attrRegionalHostedZoneId;
        const multiLevel = this.validateBasePath(props.basePath);
        if (props.mapping && !multiLevel) {
            this.addBasePathMapping(props.mapping, {
                basePath: props.basePath,
            });
        }
        else if (props.mapping && multiLevel) {
            this.addApiMapping(props.mapping.deploymentStage, {
                basePath: props.basePath,
            });
        }
    }
    validateBasePath(path) {
        if (this.isMultiLevel(path)) {
            if (this.endpointType === restapi_1.EndpointType.EDGE) {
                throw new Error('multi-level basePath is only supported when endpointType is EndpointType.REGIONAL');
            }
            if (this.securityPolicy && this.securityPolicy !== SecurityPolicy.TLS_1_2) {
                throw new Error('securityPolicy must be set to TLS_1_2 if multi-level basePath is provided');
            }
            return true;
        }
        return false;
    }
    isMultiLevel(path) {
        return (path?.split('/').filter(x => !!x) ?? []).length >= 2;
    }
    /**
     * Maps this domain to an API endpoint.
     *
     * This uses the BasePathMapping from ApiGateway v1 which does not support multi-level paths.
     *
     * If you need to create a mapping for a multi-level path use `addApiMapping` instead.
     *
     * @param targetApi That target API endpoint, requests will be mapped to the deployment stage.
     * @param options Options for mapping to base path with or without a stage
     */
    addBasePathMapping(targetApi, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_IRestApi(targetApi);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_BasePathMappingOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addBasePathMapping);
            }
            throw error;
        }
        if (this.basePaths.has(options.basePath)) {
            throw new Error(`DomainName ${this.node.id} already has a mapping for path ${options.basePath}`);
        }
        if (this.isMultiLevel(options.basePath)) {
            throw new Error('BasePathMapping does not support multi-level paths. Use "addApiMapping instead.');
        }
        this.basePaths.add(options.basePath);
        const basePath = options.basePath || '/';
        const id = `Map:${basePath}=>${core_1.Names.nodeUniqueId(targetApi.node)}`;
        return new base_path_mapping_1.BasePathMapping(this, id, {
            domainName: this,
            restApi: targetApi,
            ...options,
        });
    }
    /**
     * Maps this domain to an API endpoint.
     *
     * This uses the ApiMapping from ApiGatewayV2 which supports multi-level paths, but
     * also only supports:
     * - SecurityPolicy.TLS_1_2
     * - EndpointType.REGIONAL
     *
     * @param targetStage the target API stage.
     * @param options Options for mapping to a stage
     */
    addApiMapping(targetStage, options = {}) {
        try {
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_IStage(targetStage);
            jsiiDeprecationWarnings.aws_cdk_lib_aws_apigateway_ApiMappingOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addApiMapping);
            }
            throw error;
        }
        if (this.basePaths.has(options.basePath)) {
            throw new Error(`DomainName ${this.node.id} already has a mapping for path ${options.basePath}`);
        }
        this.validateBasePath(options.basePath);
        this.basePaths.add(options.basePath);
        const id = `Map:${options.basePath ?? 'none'}=>${core_1.Names.nodeUniqueId(targetStage.node)}`;
        new apigwv2.CfnApiMapping(this, id, {
            apiId: targetStage.restApi.restApiId,
            stage: targetStage.stageName,
            domainName: this.domainName,
            apiMappingKey: options.basePath,
        });
    }
    configureMTLS(mtlsConfig) {
        if (!mtlsConfig)
            return undefined;
        return {
            truststoreUri: mtlsConfig.bucket.s3UrlForObject(mtlsConfig.key),
            truststoreVersion: mtlsConfig.version,
        };
    }
}
_a = JSII_RTTI_SYMBOL_1;
DomainName[_a] = { fqn: "aws-cdk-lib.aws_apigateway.DomainName", version: "2.74.0" };
exports.DomainName = DomainName;
//# sourceMappingURL=data:application/json;base64,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